/******************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *-------------------------------------------------------------------
 * OPIS:
 * Dialog sluzacy do prezentacji postepow jakiegos dlugo
 * dzialajacego procesu (nie wazne jakiego).
 * Informacje o wykonywanych dzialaniach inne moduly moga
 * przysylac poprzez BscEvent.
 * Uzytkownik moze zarzadac przerwania procesu poprzez nacisniecie
 * klawisza 'break'. Akcja ta spowoduje wyslanie sygnalu
 * 'break_work'. Zadaniem obiektu nadrzednego jest podlaczenie sie
 * do tego sygnalu i jego obsluzenie.
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "PleaseWait.h"
#include "Shared.h"
#include "Events.h"
#include <qlabel.h>
#include <qlayout.h>
#include <qframe.h>
#include <qpushbutton.h>
#include <qcursor.h>
#include <qapplication.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString PleaseWait::PleaseWaitMsg = QT_TR_NOOP( "Please wait..." );


//*******************************************************************
// PleaseWait                                           CONSTRUCTOR
//*******************************************************************
PleaseWait::PleaseWait( QWidget* const in_parent )
: QDialog       ( in_parent )
, d_main_layout ( new QVBoxLayout( this ))
, d_info_layout ( new QHBoxLayout )
, d_btn_layout  ( new QHBoxLayout )
, d_sep_layout  ( new QHBoxLayout )
, d_message_lbl ( new QLabel( tr(PleaseWaitMsg), this ))
, d_break_btn   ( new QPushButton( Shared::BreakBtnLabel, this ))
{
	d_main_layout->setMargin( Shared::LayoutMargin  );
	d_main_layout->setSpacing( Shared::LayoutSpacing );
	
    d_info_layout->addStretch( Shared::OverStretch );
    d_info_layout->addWidget( d_message_lbl );
    d_info_layout->addStretch( Shared::OverStretch );
    d_main_layout->addLayout( d_info_layout );

    d_sep_layout->addStretch( 10 );
    QFrame* const sep = new QFrame( this );
    sep->setFrameShape( QFrame::HLine );
    d_sep_layout->addWidget( sep );
    d_sep_layout->setStretchFactor( sep, 80 ); 
    d_sep_layout->addStretch( 10 );
    d_main_layout->addLayout( d_sep_layout );

    d_btn_layout->addStretch( Shared::OverStretch );
    Shared::add_icon( d_break_btn, "break.png" );
    d_btn_layout->addWidget( d_break_btn );
    d_main_layout->addLayout( d_btn_layout );
    
    connect( d_break_btn, SIGNAL( clicked() ), this, SLOT( break_work_slot() ));
    setModal( TRUE );
}
// end of PleaseWait

//*******************************************************************
// ~PleaseWait                                            DESTRUCTOR
//*******************************************************************
PleaseWait::~PleaseWait()
{
    QApplication::restoreOverrideCursor();
    Shared::idle();
}
// end of ~PleaseWait

//*******************************************************************
// show                                             PUBLIC inherited
//*******************************************************************
void PleaseWait::show()
{
    QDialog::show();
    Shared::polish_width( this, 30 );
    QApplication::setOverrideCursor( Qt::WaitCursor );
    Shared::idle();
}
// end of show

//*******************************************************************
// message                                                    PUBLIC
//*******************************************************************
void PleaseWait::message( const QString& in_message )
{
    d_message_lbl->setText( in_message );
}
// end of message

//*******************************************************************
// caption                                                    PUBLIC
//*******************************************************************
void PleaseWait::caption( const QString& in_caption )
{
    setCaption( in_caption );
}
// end of message

//*******************************************************************
// break_work_slot                                      PRIVATE slot
//*******************************************************************
void PleaseWait::break_work_slot()
{
    d_break_btn->setEnabled( FALSE );
    d_message_lbl->setText( tr(PleaseWaitMsg) );
    emit break_work();
}
// end of break_work_slot

//*******************************************************************
// customEvent                                     PRIVATE inherited
//-------------------------------------------------------------------
// Obsluga wlasnego eventsa 'BscEvent'.
// Jedyne czego oczekujemy od niego to lancuch tekstowy z infomacja
// o postepach procesu. Tekst ten jest wyswietlany w dialogu.
//*******************************************************************
void PleaseWait::customEvent( QCustomEvent* in_event )
{
    const int event_type = static_cast<int>( in_event->type() );

    switch( event_type )
    {
        case DeleteFileEvent:
        case WipeFileEvent:
        case JoinFileEvent:
            {
                BscEvent* event = dynamic_cast<BscEvent*>( in_event );
                if( event ) {
                    d_message_lbl->setText( event->m_message );
                    Shared::idle();
                }
            }
            break;
        default:
            QDialog::customEvent( in_event );
    }
}
// end of customEvent

//*******************************************************************
// keyPressEvent                                   PRIVATE inherited
//-------------------------------------------------------------------
// Alternatywnym sposobem zakonczenia procesu, oprocz klawisza
// 'break', jest nacisniecie przez uzytkownika klawisza 'Esc'.
//*******************************************************************
void PleaseWait::keyPressEvent( QKeyEvent* e )
{
    if( Qt::Key_Escape == e->key() ) {
        e->accept();
        break_work_slot();        
    }
    else {    	
		QDialog::keyPressEvent( e );
	}
}
// end of keyPressEvent
