/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Editor.h"
#include "Shared.h"
#include "Busy.h"
#include "Shared.h"
#include "Config.h"
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qapplication.h>
#include <qdesktopwidget.h>
#include <qfile.h>
#include <qmessagebox.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString Editor::CAPTION           = QT_TR_NOOP( "Editor" );
const QString Editor::MSG_READ_ERROR    = QT_TR_NOOP( "Reading failure %1:\n%2" );
const QString Editor::MSG_SAVE_ERROR    = QT_TR_NOOP( "Saving failure %1:\n%2" );
const QString Editor::MSG_FILE_MODIFIED = QT_TR_NOOP( "File: %1\nis modified. Maybe you first save ?" );


//*******************************************************************
// Editor                                                CONSTRUCTOR
//*******************************************************************
Editor::Editor( QWidget* const in_parent, const QString& in_fname )
: QDialog     ( in_parent )
, d_editor    ( new QTextEdit( this ))
, d_reload_btn( new QPushButton( tr(Shared::ReloadBtnLabel), this ))
, d_save_btn  ( new QPushButton( tr(Shared::SaveBtnLabel), this ))
, d_return_btn( new QPushButton( tr(Shared::CloseBtnLabel), this ))
, d_fname     ( in_fname )
{
	setCaption( tr(CAPTION) );
	setFont( Config::instance()->lfs_default_font() );
	
	set_edit_config();
	read_file();

	Shared::add_icon( d_reload_btn, Shared::ReloadIcon );
	Shared::add_icon( d_save_btn  , Shared::SaveIcon );
	Shared::add_icon( d_return_btn, Shared::CloseIcon );
	
	connect( d_reload_btn, SIGNAL( clicked() ), this, SLOT( reload() ));
	connect( d_save_btn  , SIGNAL( clicked() ), this, SLOT( save  () ));
	connect( d_return_btn, SIGNAL( clicked() ), this, SLOT( accept() ));
	
	QVBoxLayout* const main_layout = new QVBoxLayout( this );
	main_layout->setMargin( Shared::LayoutMargin );
	main_layout->setSpacing( Shared::LayoutSpacing );
	main_layout->addWidget( d_editor );
		
	QHBoxLayout* const btn_layout = new QHBoxLayout;
	btn_layout->addStretch( Shared::OverStretch );
	btn_layout->addWidget( d_reload_btn );
	btn_layout->addWidget( d_save_btn );
	btn_layout->addWidget( d_return_btn );
	main_layout->addLayout( btn_layout );
}
// end of Editor

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void Editor::polish()
{
	Shared::polish( this, 60, 75 );
}
// end of polish

//*******************************************************************
// set_edit_config                                           PRIVATE
//*******************************************************************
void Editor::set_edit_config()
{
	d_editor->setPaletteBackgroundColor( Config::instance()->lfs_default_bkg_color() );
	d_editor->setWordWrap( QTextEdit::NoWrap );
	QFontMetrics fm( font() );
	d_editor->setTabStopWidth( 3 * fm.width( "X" ));
}
// end of set_edit_config

//********************************************************************
// read_file
//********************************************************************
bool Editor::read_file()
{
	bool retval = TRUE;
	QFile file( d_fname );

	if( file.open( IO_ReadOnly ) ) {
		QTextStream in( &file );
		Busy::set_busy( TRUE );
		d_editor->setText( in.read() );
		Busy::set_busy( FALSE );
		file.close();
	}
	else {
		QMessageBox::warning( this, tr(CAPTION), tr(MSG_READ_ERROR).arg( file.name() ).arg( file.errorString() ));
		retval = FALSE;
	}

	return retval;
}
// end of read_file

//********************************************************************
// save_file                                                  PRIVATE
//********************************************************************
bool Editor::save_file()
{
	bool retval = TRUE;
	QFile file( d_fname );

	if( file.open( IO_WriteOnly ) ) {
		QTextStream out( &file );
		Busy::set_busy( TRUE );
		out << d_editor->text();
		Busy::set_busy( FALSE );
		file.close();
	}
	else {
		QMessageBox::warning( this, tr(CAPTION), tr(MSG_SAVE_ERROR).arg( file.name() ).arg( file.errorString() ));
		retval = FALSE;
	}

	return retval;
}
// end of save_file

//*******************************************************************
// reload                                                    PRIVATE
//*******************************************************************
void Editor::reload()
{
	d_editor->clear();
	read_file();
}
// end of reload

//*******************************************************************
// save                                                      PRIVATE
//*******************************************************************
void Editor::save()
{
	save_file();
	d_editor->setModified( FALSE );
}
// end of save

//*******************************************************************
// accept                                                    PRIVATE
//*******************************************************************
void Editor::accept()
{
	bool can_exit = TRUE;
	
	if( TRUE == d_editor->isModified() ) {
		const int retval = QMessageBox::warning( this, tr(CAPTION),
									tr(MSG_FILE_MODIFIED).arg( d_fname ),
									tr(Shared::NoBtnLabel),
									tr(Shared::YesBtnLabel),
									tr(Shared::CancelBtnLabel), 2 );
									
		     if( 1 == retval ) save_file();
		else if( 2 == retval ) can_exit = FALSE;
	}
	
	if( TRUE == can_exit ) {
		QDialog::accept();
	}
}
// end of accept
