#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/mount.h>
#include <syslog.h> 
#include "util.h"
#include "dbootstrap.h"
#include "net-fetch.h"
#include "lang.h"

extern int fdflush(const char *filename);

extern char *pattern[];
extern char *altpattern[];
extern int patidx;

char *driver_filenames[] = { "drv12", "drv14", "drv14compact", "drv14idepci", 
	"drv14ide", "drv14bf2.4", "drv14pmac", "drv14chrp", "drv14prep", 
	"drv14apus", "drv14sun4cdm", "drv14sun4u",
        "drv14generic", "drv14jensen", "drv14nautilus", "drv14riscpc", "drv14netwinder", NULL };

static char*
copy_to_local(const char* diskimage)
{
  char* dev;
  int status;
  if (is_nfs_partition("/target")) {
    /* if nfs-root, cannot copy to local.  Instead use a ramdisk */
    sprintf(prtbuf, "dd < %s > /dev/ram 2> /dev/null", diskimage);
    status = execlog(prtbuf, LOG_INFO);
    dev = "/dev/ram";
  }
  else {
    int ro=1;
    sprintf(prtbuf,"cp %s /target/image.bin", diskimage);
    status = execlog(prtbuf, LOG_INFO);
    if (status < 0)
      return NULL;
    dev = find_unused_loop_device();
    if (dev == NULL)
      return NULL;
    status = set_loop(dev,"/target/image.bin",0,&ro); 
  }
  if (status)
    return NULL;
  return dev;
}

char*
get_device(const char* diskimage)
{
  char* device = NULL;
  if (is_nfs_partition("/target") || is_nfs_partition("/instmnt")) { 
    sprintf(prtbuf, _("NFS install detected. Copying %s to local before processing..."), 
	    diskimage);
    pleaseWaitBox(prtbuf);
    device = copy_to_local(diskimage);
    boxPopWindow();
  }
  else {
    int ro = 1;
    device = find_unused_loop_device();
    if (set_loop(device, diskimage, 0, &ro)) {
      ERRMSG("set_loop couldn't create a loop device for %s", prtbuf);
      return NULL;
    }
  }
  return device;
}

void
release_device(const char* device)
{
  if (strncmp(device,"/dev/loop",8) ) {
    int f=open(device,O_RDWR);
    if (f >= 0) {
      ioctl(f,BLKFLSBUF);
      close(f);
    }
  }
  else {
    del_loop(device);
    unlink("/target/image.bin");
  }
}


int install_floppy (char *device, const char *type, const char *text)
{
  int status;

  status = mount_and_check_floppy(device, type, text);
  if (status == DLG_CANCEL)
    return DLG_CANCEL;
  else if (status)
    return 1;
  if (text != NULL && strlen(text))
    sprintf(prtbuf, _("Installing the %s "), text);
  else
    sprintf(prtbuf, _("Installing "));
  if (strncmp(device, "/dev/loop", 8)) {
    strcat(prtbuf, "...");
  } else {
    strcat(prtbuf, _("from floppy images on mounted medium ..."));
  }
  pleaseWaitBox(prtbuf);
  chdir("/floppy");
  status = execlog("./install.sh /target", LOG_INFO);
  chdir("/");
  execlog("umount /floppy", LOG_DEBUG);
  boxPopWindow();

  if (status)
    return 1;

  return 0;
}

int
install_from_file (const char *filename, const char *descr)
{
  int          status;
  struct stat  statbuf;
  const char   tmpdir [] = "/target/tmp";
  char         instdir [128];

  snprintf (instdir, sizeof instdir - 1, "%s/drivers", tmpdir);
  if (!NAME_ISDIR (tmpdir, &statbuf))
    {
      if (!mkdir (tmpdir, 0777)) {
	chown (tmpdir, 0, 3); /* root_uid= 0 , sys_gid= 3 */
        chmod (tmpdir, 01777);
      }
    }

  if (!NAME_ISDIR (instdir, &statbuf)) {
      if (!mkdir (instdir, 0777)) {
	chown (instdir, 0, 3); /* root_uid= 0 , sys_gid= 3 */
      }
    }

  if (!NAME_ISDIR (instdir, &statbuf)) {
      vaproblemBox(_("Non-existent temporary directory"),
                   _("There was a problem installing the %s from %s"), descr, filename);
      unmount_dir("/instmnt");
      return 1;
  }

  sprintf (prtbuf, _("Installing %s from %s..."), descr, filename);
  pleaseWaitBox (prtbuf);
  chdir (instdir);

  sprintf (prtbuf, "zcat %s | tar -x", filename);
  status = execlog (prtbuf, LOG_INFO);

  if (! status) { /* tar went OK, try install.sh */
    status = execlog ("./install.sh /target", LOG_INFO);
  }
  chdir("/");
  boxPopWindow();
  if (status) {
    vaproblemBox(_("Installation Error"), 
                 _("There was a problem installing the %s from %s"), descr, filename);
    unmount_dir("/instmnt");
    return 1;
  }

  return 0;
}

static int
install_floppies (const char *device, const char *filename, const char *descr)
{
  int          status;
  struct stat  statbuf;
  const char   tmpdir [] = "/target/tmp";
  char         instdir [128];

  snprintf (instdir, sizeof instdir - 1, "%s/drivers", tmpdir);

  if (!NAME_ISDIR (tmpdir, &statbuf)) {
    if (!mkdir (tmpdir, 0777)) {
      chown (tmpdir, 0, 3); /* root_uid= 0, sys_gid= 3 */
      chmod (tmpdir, 01777);
    }
  }

  if (!NAME_ISDIR (instdir, &statbuf)) {
    if (!mkdir (instdir, 0777)) {
      chown (instdir, 0, 3); /* root_uid= 0, sys_gid= 3 */
    }
  }

  status = getFloppies (_("The drivers are being installed from the %s floppy drive.\n"
                          "\n"
                          "Please insert driver disk %d."),
		        device, filename, driver_filenames, _("the driver series"));

  if (status == 0) {
    status = install_from_file (filename, descr);
  }
  else if (status != DLG_CANCEL) {
    vaproblemBox(prtbuf, _("Floppy Error"),
                 _("There was a problem reading the %s from %s"), descr, device); 
  }

  if (NAME_ISREG (filename, &statbuf)) {
    unlink (filename);
  }

  return status;
}

void maybefdflush(const char *fddev) 
{
#ifdef SCSI_FLOPPY
  if (strncmp(fddev, "/dev/sfd", 8)) /* not scsi floppy */
#endif
    fdflush(fddev);
}

int choose_and_install(void) {
  int status=0;
  char* device;
  char filename [512];
  struct stat statbuf;

  if ( choose_medium() )
    return 1;

  /* set up dbootstrap settings and net config files on target root */
  check_pending_config();

  INFOMSG("installing kernel and modules from %s", Archive_Dir);
  if (! strcmp(Archive_Dir, "netfetch")) {
    status = nf_install_os();
  } else {
    /*
     * in case we're installing from a floppy device ...
     */
    if (! strncmp(Archive_Dir,"/dev/fd",7)
#ifdef SCSI_FLOPPY
	|| ! strncmp(Archive_Dir,"/dev/sfd",8)
#endif
	) {

      maybefdflush(Archive_Dir);
      status = install_floppy(Archive_Dir, "rescue", _("Rescue Floppy"));
      maybefdflush(Archive_Dir);
      eject_floppy(Archive_Dir);

      if (status) {
	if (status == DLG_CANCEL)
	  return 0;
	ERRMSG(_("extracting %s from floppy device %s failed with %d"),
	       _("Rescue Floppy"),
	       Archive_Dir, status);
	vaproblemBox(_("Floppy Error"), 
		     _("The attempt to extract the %s failed."),
		     _("Rescue Floppy"));
      } else {
	/*
	 * now proceed with the drivers floppy
	 */
	strcpy(filename, "/target/" DRVTGZFILE);
	status = install_floppies(Archive_Dir, filename, _("Drivers"));
	maybefdflush(Archive_Dir);
	eject_floppy(Archive_Dir);
	if (status) {
	  ERRMSG(_("extracting %s from floppy device %s failed with %d"),
		 _("Drivers"),
		 Archive_Dir, status);
	}
      }
    } else {
      /* we are not installing from a floppy, maybe it's a file on a local
       * disk or NFS partition */
      char *diskimage = concat_paths(Archive_Dir, kernel_image_path);
      device = get_device(diskimage);
      free(diskimage);
      if (device) {
	status = install_floppy(device, "rescue", _("Rescue Floppy Image"));
	release_device(device);
      } else {
	sprintf(prtbuf, "%s/%s", Archive_Dir, KERDISKFILE);
	device = get_device(prtbuf);
	if (device) {
	  status = install_floppy(device, "rescue", _("Rescue Floppy Image"));
	  release_device(device);
	} else
	  status = -1;
      }

      if (status) {
	if (status == DLG_CANCEL)
	  return 0;
	ERRMSG(_("extracting %s from non-floppy device %s failed with %d"),
	       _("Rescue Floppy"),
	       Archive_Dir, status);
	vaproblemBox(_("Floppy Error"),
		     _("The attempt to extract the %s from disk failed."),
		     _("Rescue Floppy"));
      } else {
	/*
	 * now proceed with the drivers floppy
	 */

	char  *filename;

	sprintf(prtbuf, "%s/%s", Archive_Dir, drivers_path);
	if (! NAME_ISREG(prtbuf, &statbuf)) {
	  sprintf(prtbuf, "%s/" DRVTGZFILE, Archive_Dir);
	}
	filename = strdup(prtbuf);
	status = install_from_file(filename, _("Drivers"));
	free(filename);
	if (status) {
	  ERRMSG(_("extracting %s from non-floppy device %s failed with %d"),
		 _("Drivers"),
		 Archive_Dir, status);
	}
      }
    }
  }
  unmount_dir("/instmnt");
  return status;
}

int extract_kernel_and_modules (void)
{
  DIR* dp;
  struct dirent *dirp;
  int found=0,status=0;
  struct fdisk_partition *p;

  disqtype=kernel;

  /* Only verify root choice if it is not empty. */

  dp = opendir("/target/.");
  while (NULL != (dirp = readdir(dp))) {
    if (( 0 != strcmp(dirp->d_name, ".")) &&
        ( 0 != strcmp(dirp->d_name, "..")) &&
        ( 0 != strcmp(dirp->d_name, "lost+found"))) {
      found = 1;
      break;
    }
  }
  closedir(dp);

  if (found && bootargs.isverbose) {
    sprintf(prtbuf,_("The following filesystems are mounted, and will have the kernel, the modules and the base system installed to them:"));
    p = mounted_partitions;
    while (p) {
      if (! strncmp(p->mount_point, "/target",
		    strlen("/target")) ) {
	char *begg;
        strcat(prtbuf, "\n");
        strcat(prtbuf, p->name);
        strcat(prtbuf, _(" on "));
	begg = (p->mount_point) + strlen("/target");
	if (*begg == '\0')
	  strcat(prtbuf, "/");
	else
	  strcat(prtbuf, begg);
      }
      p = p->next_in_use;
    }
    if ( yesNoBox(prtbuf,_("Verify Filesystem Choice?")) == DLG_NO )
      return 1;
  }

  status = choose_and_install();
  if ( status == 0 )
    INFOMSG("kernel and module install was successful");
  
  frob_lib_modules();

#if #cpu(s390)
  add_modules_from_floppy();
#endif

  /*
   * run depmod while we're at it (PCMCIA will need it)
   */
  execlog("depmod -a", LOG_INFO);

  return status;
}

/* vi: set sw=2 ts=8: */
