#!/bin/bash
# maintained by the boot-floppies team, debian-boot@lists.debian.org
#   copyright (c) 1999 -- 2002
# This is free software under the GNU General Public License.

# Create release directory and copy files over
# Made into a shell script to add flexibility

# See setup_image_names() in utilities/dbootstrap/main.c to get the
# idea of what sort of directory structure you want to end up with.
# The most important things to get right are the rescue disk and the
# drivers.tgz file, as these are what's used by network installs.

# Basically the rescue disk lives in
# [subarch]/images-size/[flavor]/rescue.bin ([] stuff is optional -
# note that images-size (i.e. images-1.44, images-1.20) is *not*
# optional, as it will default to 1.44 if not set in the bootloader
# configuration)

# In contrast, drivers.tgz and linux.bin (the bare kernel) live in
# [subarch]/[flavor].  This really isn't the best layout but it works
# well enough in most cases.

. ./common.sh

set -e

# parse command line params
if [ $# -lt 3 ]; then
  echo "release.sh <arch> <dist> <version> [<arch-specific args>]"
  exit 1
fi

arch="$1"
dist="$2"
version="$3"
shift;shift;shift
# arch specific settings
otherargs="$*"

# Set up the environment
MAKE=${MAKE:-make}
tools_dir=${TOOLS_DIR:-/archive/debian/tools}
TOP=$(pwd)
release=$TOP/release
pkgver="`dpkg-parsechangelog | awk -- '/^Version/ {print $2}'`"
suffix="_${pkgver}_${arch}"
bfinstall="bf-archive-install${suffix}.sh"

do_common_pretasks()
{
  rm -rf $release
  mkdir -p $release

  $MAKE release/README.txt

  # dynamically deal with l10n of README-Users
  for file in README-Users.??.m4; do
    $MAKE release/$(echo $file | sed -e 's/ME-Users.\(..\).m4/-\1.txt/')
  done

  #---------------------------------------------------------------------
  # standard image handling
  #---------------------------------------------------------------------
  # strategy is to copy files to 'release' then pick off from there
  cp *.bin drivers*.tgz $release/

  # Unfortunately there's not a standard way of specifying
  # subarchitectures and flavors in the image file names... so we work
  # around this on a per-architecture basis below.  Yes this will be
  # fixed.

  cd $release

  mkdir images-2.88
  mkdir images-1.44
  mkdir images-1.20

  for f in resc2880*.bin; do
    [ ! -f $f ] || mv $f images-2.88/rescue${f#resc2880}
  done

  for f in resc1440*.bin; do
    [ ! -f $f ] || mv $f images-1.44/rescue${f#resc1440}
  done
  for f in root1440*.bin; do
    [ ! -f $f ] || mv $f images-1.44/root${f#root1440}
  done
  for f in drv14*.bin; do	
    [ ! -f $f ] || mv $f images-1.44/driver${f#drv14}
  done

  for f in resc1200*.bin; do
    [ ! -f $f ] || mv $f images-1.20/rescue${f#resc1200}
  done
  for f in root1200*.bin; do
    [ ! -f $f ] || mv $f images-1.20/root${f#root1200}
  done
  for f in drv12*.bin; do
    [ ! -f $f ] || mv $f images-1.20/driver${f#drv12}
  done

  rmdir images-2.88 2>/dev/null || true	# in case empty
  rmdir images-1.44 2>/dev/null || true	# in case empty
  rmdir images-1.20 2>/dev/null || true	# in case empty

  cd $TOP

  # make the stub for the archive installer script
  sed "s/%arch%/$arch/g; s/%dist%/$dist/g; s/%version%/$version/g;" \
	< debian/bf-archive-install.sh > $release/$bfinstall
  chmod a+x $release/$bfinstall

  #---------------------------------------------------------------------
  # docs and doc tar-ball
  #---------------------------------------------------------------------
  mkdir $release/doc
  make -C documentation DESTDIR=$release/doc install

  tar cz --remove-files -f $release/bf-doc${suffix}.tar.gz -C $release doc
  echo "tar -xzf bf-doc${suffix}.tar.gz ; rm -f bf-doc${suffix}.tar.gz" >>	\
	$release/$bfinstall
  rm -rf $release/doc
}


do_archspecific()
{
  # Architecture specific stuff
  case $arch in
  i386)
  	do_arch_i386 $otherargs
	;;
  sparc)
  	do_arch_sparc $otherargs
	;;
  powerpc)
  	do_arch_powerpc $otherargs
	;;
  arm)
  	do_arch_arm $otherargs
	;;
  m68k)
  	do_arch_m68k $otherargs
	;;
  alpha)
  	do_arch_alpha $otherargs
	;;
  ia64)
	do_arch_ia64 $otherargs
	;;
  hppa)
	do_arch_hppa $otherargs
	;;
  mips)
	do_arch_mips $otherargs
	;;
  mipsel)
	do_arch_mipsel $otherargs
	;;
  s390)
	do_arch_s390 $otherargs
	;;
  *)
  	echo "Unsupported arch $arch"
	exit 1
  esac
}

do_common_posttasks()
{
  #--------------------------------------------------------
  # standard post stuff
  #--------------------------------------------------------
  # this script is self-destructing
  echo "find . -type d -exec chmod 775 \{\} \;" >> $release/$bfinstall
  echo "find . -type f -exec chmod 664 \{\} \;" >> $release/$bfinstall
  do_archspecific_posttasks
  echo "rm -f $bfinstall" >> $release/$bfinstall
}

do_archspecific_posttasks()
{
  #--------------------------------------------------------
  # Architecture specific posttasks
  #--------------------------------------------------------
  case $arch in
  m68k)
       do_arch_m68k_posttasks
       ;;
  esac
}

do_arch_m68k_posttasks()
{
  #--------------------------------------------------------
  # m68k posttasks
  #--------------------------------------------------------
  echo "chmod +x amiga/amiboot-5.6 dmesg" >> $release/$bfinstall
}

do_arch_i386()
{
  #--------------------------------------------------------
  # i386 installation
  #--------------------------------------------------------

  # Language support for rescue disk
  # text files and fonts for syslinux and a batch file
  langs=`cd scripts/rescue/messages; ls -d ??`
  langfiles="readme.txt debian.txt f1.txt f2.txt f3.txt f4.txt f5.txt f6.txt f7.txt f8.txt f9.txt f10.txt"

  mkdir -p $release/dosutils
  mkdir -p $release/lang
  mkdir -p $release/lang/fonts

  sed "s/LANGUAGES/`echo $langs`/" < scripts/dos/setlang.bat > $release/dosutils/setlang.bat
  cp scripts/rootdisk/consolefonts/*.psf $release/lang/fonts/
  for lang in ${langs}
  do
    debug "Processing files for language ${lang}"
    mkdir -p $release/lang/${lang}
    for file in ${langfiles}
    do
      if [ -f scripts/rescue/messages/${lang}/${file} ]; then
        debug "m4 processing scripts/rescue/messages/${lang}/${file}"
        m4 -EP -D__kernel_version__=$kver \
              -D__debianversion__=$version \
              -D__arch__=i386 scripts/rescue/messages/${lang}/${file} \
                > $release/lang/${lang}/${file}.tmp
	sed s/$/$'\r'/ $release/lang/${lang}/${file}.tmp \
            > $release/lang/${lang}/${file}
        rm $release/lang/${lang}/${file}.tmp
      fi
    done
  done

  cp xlp.tgz $release

  # DOS tools provided for i386 only
  recode latin1..ibmpc <scripts/dos/install.bat >$release/install.bat

  cd $release/dosutils
  miniunzip $tools_dir/rawrite2.zip rawrite2.exe > /dev/null
  miniunzip $tools_dir/rawrite2.zip rawrite2.txt > /dev/null
  miniunzip $tools_dir/lodlin16.zip LODLIN16/LOADLIN.EXE > /dev/null
  cp -f LODLIN16/LOADLIN.EXE loadlin.exe
  rm -Rf LODLIN16
  cd $TOP

  gunzip -c $TOP/config.gz > $release/kernel-config
  gunzip -c $TOP/config.gz > $release/images-1.44/kernel-config
  cp $release/images-1.44/kernel-config $release/images-2.88/kernel-config
  # safe "flavor"
  for size in 1.20 1.44; do
    safedir=$release/images-$size/safe
    mkdir -p $safedir
    (cd $safedir && 
	ln -s ../driver-* . &&
	mv ../rescue-s.bin rescue.bin)
    cp $TOP/documentation/README-safe $safedir/README.txt
  done

  for _flavor in compact idepci bf2.4; do
    mkdir  -p $release/$_flavor
    mv $release/linux$_flavor.bin $release/$_flavor/linux.bin
    recode latin1..ibmpc <scripts/dos/install-$_flavor.bat >$release/$_flavor/install.bat
    mv $release/drivers$_flavor.tgz $release/$_flavor/drivers.tgz
    gunzip -c $TOP/config$_flavor.gz > $release/$_flavor/kernel-config  
    
    if [ -f tftpboot-$_flavor.img ] ; then
      cp tftpboot-$_flavor.img $release/$_flavor/tftpboot.img
    fi

    for size in 1.44 2.88; do
      fromdir=$release/images-$size
      destdir=$release/images-$size/$_flavor
  
      mkdir -p $destdir
      mv $fromdir/rescue$_flavor.bin $destdir/rescue.bin
      cp $TOP/documentation/README-$_flavor $destdir/README.txt
      cp $TOP/documentation/README-$_flavor $release/$_flavor/README.txt
      gunzip -c $TOP/config$_flavor.gz > $destdir/kernel-config
      
      if [ -f $fromdir/root$_flavor.bin ]; then
  	mv $fromdir/root$_flavor.bin $destdir/root.bin
      fi
      for i in $fromdir/driver$_flavor-?.bin; do
        if [ -f $i ]; then
          base=$(basename $i | sed -e s,$_flavor,,)
          mv $i $destdir/$base
        fi
      done
    done
  
    rm -f release/images-1.20/driver$_flavor-{1,2,3,4}.bin   # what's this file doing?
  
    rm -f $release/root$_flavor.bin
  done

  #--------------------------------------------------------
  # last minute cleanup
  #--------------------------------------------------------
  rm -f $release/root*.bin

  # set permissions
  chmod -R a+r $release
  
  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f | 			\
    grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  ### make tarballs, and the script steps to undo them
  cd $release
  for size in 1.20 1.44 2.88; do
    tar cz --remove-files -f bf-images-${size}${suffix}.tar.gz images-$size
    echo "tar -xzf bf-images-$size${suffix}.tar.gz ; rm -f bf-images-$size${suffix}.tar.gz" >> \
            $bfinstall
    rm -rf images-$size
  done

  tar cz --remove-files -f bf-common${suffix}.tar.gz *.tgz compact idepci bf2.4 dosutils kernel-config
  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
  	$bfinstall
  rmdir compact
  rmdir idepci
  rmdir dosutils
  rmdir bf2.4

  tar cz --remove-files -f bf-misc${suffix}.tar.gz \
  	README.txt install.bat linux.bin md5sum.txt \
        lang \
	READ-??.txt
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
  	$bfinstall
  rm -rf lang

  cd $TOP
}

do_arch_sparc()
{
  rootdisk=$1
  info "using $rootdisk as generic root filesystem image"
  mv $release/$rootdisk $release/images-1.44/root.bin

  for subarch in sun4u sun4cdm; do
    # Per subarch images
    info "setting up subarchitecture '$subarch'"
    mkdir -p $release/${subarch}/images-1.44/
    mv $release/images-1.44/rescue${subarch}.bin $release/${subarch}/images-1.44/rescue.bin
    (cd $release/images-1.44 &&
      for f in driver${subarch}*.bin; do
        mkdir -p ../${subarch}/images-1.44/ # just to be sure
	[ ! -f $f ] || mv $f ../${subarch}/images-1.44/driver${f#driver${subarch}};
      done)
    mv $release/drivers${subarch}.tgz $release/${subarch}/drivers.tgz
    gunzip -c config${subarch} > $release/${subarch}/kernel-config

    # the linux.bin kernel is mainly dedicated to net boot: convert it to
    # a.out (required by the SUN TFTP boot process) and pad the result to
    # be 4-bytes aligned (required by some old bootproms).
    info "making linux${subarch}.bin safe for TFTP uses"
    gzip -dcf $release/linux${subarch}.bin > $release/${subarch}/linux.bin
    elftoaout -o $release/${subarch}/linux-a.out $release/${subarch}/linux.bin ||
        mv $release/${subarch}/linux.bin $release/${subarch}/linux-a.out
    rm -f $release/linux${subarch}.bin $release/${subarch}/linux.bin
    size=`ls -l release/${subarch}/linux-a.out | awk '{print $5}'`
    if rem=`expr \( 4 - $size % 4 \) % 4`; then
        dd if=/dev/zero bs=1 count=$rem >> $release/${subarch}/linux-a.out
    fi
    # Install the per subarch tftpboot images
    cp tftpboot-${subarch}.img $release/${subarch}/tftpboot.img
  done

  # install netserver miniroot
  cp root.tar.gz $release/

  # make md5sum before tarring up all the contents
  info "making md5sum"
  (cd $release && md5sum `find . -type f | 			\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # make tarballs for subarches, etc.
  for dir in images-1.44 sun4cdm sun4u; do
    (cd $release && tar cz --remove-files -f bf-${dir}${suffix}.tar.gz $dir)
    echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
	$release/$bfinstall
    rm -rf $release/${dir}
  done

  (cd $release && tar cz --remove-files -f bf-common${suffix}.tar.gz \
	*.txt root.tar.gz)
  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
	$release/$bfinstall

}

do_arch_m68k()
{
  #--------------------------------------------------------
  # m68k installation
  #--------------------------------------------------------
  # m68k particulars; we support 4 m68k platforms so far, all different. If 
  # you ate lunch just recently, simply skip this section.
  #

  rootfiles="README.txt READ-??.txt md5sum.txt"

  for subarch in amiga atari bvme6000 mvme16x mvme147 mac; do
    for size in 1.44 2.88; do
      mkdir -p $release/$subarch/images-$size/
      [ ! -f $release/images-$size/rescue$subarch.bin ] || \
          mv $release/images-$size/rescue$subarch.bin \
               $release/$subarch/images-$size/rescue.bin
      [ ! -f $release/images-$size/root$subarch.bin   ] || \
          mv $release/images-$size/root$subarch.bin \
               $release/$subarch/images-$size/root.bin
      if [ -d $release/images-$size ]; then
        cd $release/images-$size
        for f in driver$subarch*.bin; do
          [ ! -f $f ] || mv $f ../$subarch/images-$size/driver${f#driver$subarch};
        done
        cd ../$subarch/images-$size
        if [ -f driver-1.bin -a ! -f driver-2.bin ]; then
          mv driver-1.bin driver.bin;
        fi
        cd $TOP
      fi
      rmdir $release/$subarch/images-$size 2>/dev/null || true	# in case empty
      rmdir $release/images-$size          2>/dev/null || true	# in case empty
    done
    [ ! -f $release/drivers$subarch.tgz ] || \
	    mv $release/drivers$subarch.tgz $release/$subarch/drivers.tgz
    [ ! -f $release/root$subarch.bin    ] || \
        mv $release/root$subarch.bin    $release/$subarch/root.bin
    [ ! -f $release/linux$subarch.bin   ] || \
        mv $release/linux$subarch.bin   $release/$subarch/linux.bin
    [ ! -f $TOP/sys_map$subarch.gz ] || \
        cp $TOP/sys_map$subarch.gz $release/$subarch/sysmap.gz
    rmdir $release/$subarch  2>/dev/null || true	# in case empty
  done

  #--------------------------------------------------------
  # m68k/atari 
  #--------------------------------------------------------
  if [ -d $release/atari ]; then
    #
    # Atari install kit: lzh archive is no more, we use tgz now
    #
    cp m68k-specials/bootstra.ttp             $release/atari/bootstra.prg
    echo "-s -k linux -r root.bin root=/dev/ram video=keep load_ramdisk=1" \
                                            > $release/atari/bootargs
#    cp m68k-specials/debian-atari.txt         $release/atari/install.doc

    # 
    # Atari: loose files
    #
    cp m68k-specials/rawwrite.ttp $release
    rootfiles="$rootfiles rawwrite.ttp"
  fi

  #--------------------------------------------------------
  # m68k/amiga
  #--------------------------------------------------------
  if [ -d $release/amiga ]; then
    #
    # Amiga install kit: lzh archive is no more, we use tgz now
    #
    # unpack AmigaOS magic from m68k support directory
    (cd $release && tar xvfz $TOP/m68k-specials/amiga-kit.tgz)
#    cp m68k-specials/debian-amiga.txt         $release/amiga/install.doc
    rm $release/amiga/install.doc.info

    #
    # Amiga: loose files
    #
    cp m68k-specials/dmesg		$release
    chmod a+x $release/dmesg
    cp m68k-specials/dmesg.readme	$release
    rootfiles="$rootfiles dmesg dmesg.readme amiga.info"
  fi

  #--------------------------------------------------------
  # m68k/mac
  #--------------------------------------------------------
  if [ -d $release/mac ]; then
    # get Penguin-19 HQX from: 
    #  http://people.debian.org/~cts/debian-m68k/misc/
    # which was made from this file:
    #  http://prdownloads.sourceforge.net/linux-mac68k/Penguin-19.sit

    if [ -d do_not_build_hfs_stuff ]; then
    # Mac install kit: HFS floppy image, to be packaged as StuffIt archive
    #

    # create HFS floppy image
    dd if=/dev/zero of=$release/mac/macinstall-floppy-hfs.img bs=1024 count=2880
    $ROOTCMD hformat -l "Debian/68k" $release/mac/macinstall-floppy-hfs.img

    # mount it (':' is top dir)
    $ROOTCMD hmount $release/mac/macinstall-floppy-hfs.img

    # copy kernel/sysmap/ramdisk as raw
    # FIXME: correct?  should use $release/*.bin ?
    $ROOTCMD hcopy -r linuxmac.bin :linux.bin
    $ROOTCMD hcopy -r sys_mapmac.gz :System-Map.gz
    $ROOTCMD hcopy -r rootmac.bin :root.bin

    # MacOS magic; useful for HFS CD I hope
    #
    # hybrid CD note:
    # mount this HFS floppy image using the mount command
    # mount -t hfs -o loop,fork=netatalk,afpd
    # copy over both Penguin-15 and .AppleDouble/Penguin-15
    # copy over both Penguin Prefs and .AppleDouble/Penguin Prefs
    # (.AppleDouble files go into .AppleDouble subdir of target dir)
    #
    # make CD image using the command
    # mkhybrid -map <mapfile> --netatalk -a -j -hfs -r -V <label> -o <image> <source dir>
    #
    # map file contains stuff like
    # # ext. xlate  creator  type    comment
    # .hqx   Ascii  'BnHx'   'TEXT'  "BinHex file"
    # .mov   Raw    'TVOD'   'MooV'  "QuickTime Movie"
    # .deb   Raw    'Debn'   'bina'  "Debian package"
    # .bin   Raw    'Debn'   'bina'  "Floppy or ramdisk image"
    # *      Ascii  'ttxt'   'TEXT'  "Text file"
    #
    # Note that we can't use MacBinary mapping for .bin files due to
    # name clashes with the binary image files, so we have to provide 
    # all encoded Mac files in BinHex format.
    #
    # Kudos Brad Midgley, <brad@pht.com> for the tricks!
    #
    # copy booter as BinHex
    # hcopy -b m68k-specials/Penguin-17.hqx :
    $ROOTCMD hcopy -b m68k-specials/Penguin-19.hqx :
    $ROOTCMD hcopy -b m68k-specials/Penguin-Colors.hqx :
    $ROOTCMD hcopy -b m68k-specials/Penguin.doc.hqx :

    # copy Preferences as BinHex
    # hcopy -b m68k-specials/Penguin_Prefs_floppy.hqx :"Penguin Prefs"

    # copy install doc as text
#    $ROOTCMD hcopy -t m68k-specials/debian-mac.txt :Install.doc
    $ROOTCMD humount $release/mac/macinstall-floppy-hfs.img

# FIXME:
# why do we do all this hfs magic, if we do not need the image in the end?
# will the image be needed for an improved debian-cd?
    rm $release/mac/macinstall-floppy-hfs.img
    fi

    # package Mac booter stuff for the CD install dir
    # (tar format until I can come up with something nicer)
#    mkdir -p $release/mac/install/
#    cp m68k-specials/Penguin-17.hqx $release/mac/
#    cp m68k-specials/Penguin-18.hqx $release/mac/
    cp m68k-specials/Penguin-19.hqx $release/mac/
    cp m68k-specials/Penguin-Colors.hqx $release/mac/
    cp m68k-specials/Penguin.doc.hqx $release/mac/
#    mkdir -p $release/mac/install/mac/
    # cp m68k-specials/Penguin-17.hqx $release/mac/install/mac/
    # cp m68k-specials/Penguin_Prefs_CD.hqx $release/mac/install/mac/
    # cp m68k-specials/Penguin_Prefs_CD_autoboot.hqx $release/mac/install/mac/

#    dd if=/dev/zero of=$release/macinstall-temp.img bs=1024 count=1440
#    hformat $release/macinstall-temp.img
#
#    # mount it (':' is top dir)
#    hmount $release/macinstall-temp.img
#    hcopy -b m68k-specials/Penguin_Prefs_CD.hqx :"Penguin Prefs"
#    hcopy -b m68k-specials/Penguin_Prefs_CD_autoboot.hqx :"Penguin Prefs (autoboot)"
#    hcopy -b :"Penguin Prefs" $release/mac/install/mac/
#    hcopy -b :"Penguin Prefs (autoboot)" $release/mac/install/mac/
#    humount $release/macinstall-temp.img
#    rm $release/macinstall-temp.img

#    cp m68k-specials/debian-mac.txt $release/mac/

    #
    # Mac: loose files
    #
    rootfiles="$rootfiles"
  fi

  #--------------------------------------------------------
  # m68k/vme bvme6000, mvme16x (no tftp support for mvme147)
  #--------------------------------------------------------
  # VME stuff
  #

  extractdir=${tmpdir}/boot-floppies/extract-tmp-$$

  rm -rf $extractdir
  make_tmpdir $extractdir

  info "downloading m68k-vme-tftplilo"
  package=`grab_paths m68k-vme-tftplilo`

  info "extracting $package to $extractdir"
  if [ -z "$package" ] || [ ! -f $package ]; then
      error "unable to find m68k-vme-tftplilo package path"
  fi


  if [ -d $release/bvme6000 -o -d $release/mvme16x ]; then
      dpkg-deb --extract $package $extractdir
  fi

  #--------------------------------------------------------
  # m68k/vme: bvme6000
  #--------------------------------------------------------
  if [ -d $release/bvme6000 ]; then
    cp $extractdir/boot/tftplilo.bvme $release/bvme6000
    zcat $extractdir/usr/share/doc/m68k-vme-tftplilo/README.gz \
                                                >$release/bvme6000/tftplilo.txt
    cp m68k-specials/tftplilo.conf               $release/bvme6000/tftplilo.conf
    cp m68k-specials/bvmbug-G.bin                $release/bvme6000/bvmbug-G.bin
    cp m68k-specials/bvmbug.txt                  $release/bvme6000/bvmbug.txt
#    cp m68k-specials/debian-vme.txt              $release/bvme6000/install.txt
    mv $release/bvme6000/linux.bin               $release/bvme6000/linuxbvme6000.bin
    mv $release/bvme6000/root.bin                $release/bvme6000/rootbvme6000.bin
  fi

  #--------------------------------------------------------
  # m68k/vme: mvme16x
  #--------------------------------------------------------
  if [ -d $release/mvme16x ]; then
    cp $extractdir/boot/tftplilo.mvme $release/mvme16x
    zcat $extractdir/usr/share/doc/m68k-vme-tftplilo/README.gz \
                                                >$release/mvme16x/tftplilo.txt
    cp m68k-specials/tftplilo.conf               $release/mvme16x/tftplilo.conf
#    cp m68k-specials/debian-vme.txt              $release/mvme16x/install.txt
    mv $release/mvme16x/linux.bin                $release/mvme16x/linuxmvme16x.bin
    mv $release/mvme16x/root.bin                 $release/mvme16x/rootmvme16x.bin
  fi

  rm -rf $extractdir

  #--------------------------------------------------------
  # m68k/vme: mvme147
  #--------------------------------------------------------
  if [ -d $release/mvme147 ]; then
#    cp m68k-specials/debian-vme.txt              $release/mvme147/install.txt
    mv $release/mvme147/linux.bin                $release/mvme147/linuxmvme147.bin
    mv $release/mvme147/root.bin                 $release/mvme147/rootmvme147.bin
  fi

  if [ -d $release/bvme6000 -o -d $release/mvme16x -o -d $release/mvme147 ];
  then
    #
    # VME: loose files
    #
    cp m68k-specials/rawrite2.exe               $release/
    rootfiles="$rootfiles rawrite2.exe"

    # clean up
    rm -rf /var/tmp/extract-tftplilo
  fi

  #--------------------------------------------------------
  # m68k magic
  #--------------------------------------------------------
  #
  # Copy the m68k magic used for this build to $release/
  #
  mkdir -p $release/source
  ( cd m68k-specials
    tar czf $release/source/m68k-support.tar.gz * --exclude CVS
  )

  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f | 			\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # make tarball of root files
  ( cd $release
    tar cz --remove-files -f bf-root${suffix}.tar.gz $rootfiles
    echo "tar -xzf bf-root${suffix}.tar.gz ; rm -f bf-root${suffix}.tar.gz" >> \
        $release/$bfinstall
  )

  # make tarballs for subarches, etc.
  for dir in images-1.44 source \
             amiga atari bvme6000 mvme16x mvme147 mac; do
	if [ -d $release/$dir ]; then
      (cd $release && tar cz --remove-files -f bf-${dir}${suffix}.tar.gz $dir)
      echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
        $release/$bfinstall
      rm -rf $release/${dir}
    fi
  done

  # create [amiga|atari|mac]install.tar.gz "on the fly"
  for dir in amiga atari mac; do
    case $dir in
	amiga)	
	    extra="amiga.info dmesg dmesg.readme"
	    echo "chmod +x amiga/amiboot-5.6 dmesg" >> $release/$bfinstall
	    ;;
	    *)
	    extra=""
	    ;;
    esac
    echo "tar -czf ${dir}install.tar.gz ${dir} ${extra}" >> $release/$bfinstall
  done
}

do_arch_powerpc()
{
  #--------------------------------------------------------
  # PowerPC install documentation
  #--------------------------------------------------------
  $MAKE -C powerpc-specials clean
  $MAKE -C powerpc-specials

  #--------------------------------------------------------
  # CHRP install kit
  #--------------------------------------------------------
  if [ -f rootchrp.bin ]; then
      mkdir -p $release/chrp/images-1.44
      [ ! -f linuxchrp.bin ] || cp linuxchrp.bin $release/chrp/linux.bin
      [ ! -f sys_mapchrp.gz ] || cp sys_mapchrp.gz $release/chrp/sysmap.gz
      mv $release/images-1.44/rescuechrp.bin $release/chrp/images-1.44/rescue.bin
      mv $release/images-1.44/rootchrp.bin $release/chrp/images-1.44/root.bin
      mv $release/driverschrp.tgz $release/chrp/drivers.tgz
      for dfile in $release/images-1.44/driverchrp-?.bin; do 
	  bdfile=`basename $dfile`
	  dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
	  mv $release/images-1.44/$bdfile $release/chrp/images-1.44/driver-$dnum.bin
      done
      cp powerpc-specials/bootargs-chrp $release/chrp/bootargs
      cp powerpc-specials/install-chrp.en.txt $release/chrp/install.txt
  fi

  #--------------------------------------------------------
  # PMac install kit: HFS floppy image
  #--------------------------------------------------------
  mkdir -p $release/powermac/images-1.44
  if [ -f linuxnewpmac.bin.gz ]; then
      mkdir -p $release/new-powermac/images-1.44
      mkdir -p $release/new-powermac/images-2.88
  fi

  if [ -f linuxpmac.bin.gz ]; then zcat linuxpmac.bin.gz > $release/powermac/linux.bin; fi
  if [ -f sys_mappmac.gz ]; then cp sys_mappmac.gz $release/powermac/sysmap.gz ; fi
  if [ -f linuxnewpmac.bin.gz ]; then zcat linuxnewpmac.bin.gz > $release/new-powermac/linux.bin; fi
  if [ -f sys_mapnewpmac.gz ]; then cp sys_mapnewpmac.gz $release/new-powermac/sysmap.gz ; fi

  sh powerpc-specials/miBoot/mkboot.sh $release/powermac/images-1.44/boot-floppy-hfs.img

  mv $release/images-1.44/rescuepmac.bin $release/powermac/images-1.44/rescue.bin
  mv $release/images-1.44/rootpmac.bin $release/powermac/images-1.44/root.bin
  mv $release/driverspmac.tgz $release/powermac/drivers.tgz
  for dfile in $release/images-1.44/driverpmac-?.bin; do 
    bdfile=`basename $dfile`
    dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
    mv $release/images-1.44/$bdfile $release/powermac/images-1.44/driver-$dnum.bin
  done
  if [ -f linuxnewpmac.bin.gz ]; then
      mv $release/images-2.88/rescuenewpmac.bin $release/new-powermac/images-2.88/rescue.bin
      (cd $release/new-powermac/images-1.44 && ln -s ../../powermac/images-1.44/root.bin .)
	  mv $release/driversnewpmac.tgz $release/new-powermac/drivers.tgz
	  for dfile in $release/images-1.44/drivernewpmac-?.bin; do 
	      bdfile=`basename $dfile`
	      dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
	      mv $release/images-1.44/$bdfile $release/new-powermac/images-1.44/driver-$dnum.bin
	  done
  fi
  #	$MAKE -C powerpc-specials/coffboot ../../rescue-pmac.coff
  #	cp rescue-pmac.coff $release/powermac/rescue-pmac.coff
  ar -p "$(grab_paths yaboot)" data.tar.gz | tar -zxp -C $tmpdir/boot-floppies ./usr/lib/yaboot/yaboot
  mv $tmpdir/boot-floppies/usr/lib/yaboot/yaboot $release/powermac/
  (cd $tmpdir/boot-floppies && rmdir -p usr/lib/yaboot)
  (cd $release/powermac && ln -s images-1.44/root.bin ramdisk.image.gz)
  if [ -f linuxnewpmac.bin.gz ]; then
      (cd $release/new-powermac && ln -s ../powermac/yaboot .)
  fi
  cp ${tools_dir}/BootX*.sit $release/powermac/
  cp ${tools_dir}/bootvars*sit.hqx $release/powermac/

  cp powerpc-specials/bootargs-pmac $release/powermac/bootargs
  cp powerpc-specials/install-pmac.en.txt $release/powermac/install.txt
  cp powerpc-specials/yaboot.conf $release/powermac/yaboot.conf
  if [ -f linuxnewpmac.bin.gz ]; then
      (cd $release/new-powermac && ln -s ../powermac/install.txt .)
      cp powerpc-specials/yaboot.conf.newpmac $release/new-powermac/yaboot.conf
  fi
  #--------------------------------------------------------
  # PReP install kit
  #--------------------------------------------------------
  if [ -f rootprep.bin ]; then
      mkdir -p $release/prep/images-1.44
      if [ -f linuxprep.bin ]; then cp linuxprep.bin $release/prep/linux.bin ; fi
      if [ -f sys_mapprep.gz ]; then cp sys_mapprep.gz $release/prep/sysmap.gz ; fi
      mv $release/bootprepfull.bin $release/prep/bootfull.bin
      mv $release/bootprep.bin $release/prep/images-1.44/boot.bin
      mv $release/images-1.44/rescueprep.bin $release/prep/images-1.44/rescue.bin
      mv $release/images-1.44/rootprep.bin $release/prep/images-1.44/root.bin
      mv $release/driversprep.tgz $release/prep/drivers.tgz
      for dfile in $release/images-1.44/driverprep-?.bin; do 
	  bdfile=`basename $dfile`; \
	      dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
	  mv $release/images-1.44/$bdfile $release/prep/images-1.44/driver-$dnum.bin
      done 
      cp powerpc-specials/install-prep.en.txt $release/prep/install.txt
  fi

  #--------------------------------------------------------
  # APUS stuff
  #--------------------------------------------------------
  if [ -f rootapus.bin ]; then
      mkdir -p $release/apus/images-1.44
      tar xzvf powerpc-specials/bootstrap-apus.tgz -C $release/apus
      if [ -f linuxapus.bin ]; then cp linuxapus.bin $release/apus/linux.bin ; fi
      if [ -f sys_mapapus.gz ]; then cp sys_mapapus.gz $release/apus/sysmap.gz ; fi
      mv $release/images-1.44/rescueapus.bin $release/apus/images-1.44/rescue.bin
      mv $release/images-1.44/rootapus.bin $release/apus/images-1.44/root.bin
      mv $release/driversapus.tgz $release/apus/drivers.tgz
      for dfile in $release/images-1.44/driverapus-?.bin; do 
	  bdfile=`basename $dfile`; \
	      dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
	  mv $release/images-1.44/$bdfile $release/apus/images-1.44/driver-$dnum.bin
      done 
      cp powerpc-specials/install-apus.en.txt $release/apus/install.txt    
  fi
  #--------------------------------------------------------
  # last minute cleanup
  #--------------------------------------------------------
  rm -f $release/root*.bin $release/linux*.bin

  #--------------------------------------------------------
  # make md5sum before we tar up the contents
  (cd $release && md5sum `find . -type f | 		\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)
  #--------------------------------------------------------
  # Multitudinous inner tarballs
  #--------------------------------------------------------
  built=""
  for x in apus chrp prep new-powermac; do
      if test -d $release/$x; then
	  built="$built $x"
      fi
  done
  (cd $release && tar cz --remove-files -f bf-arch${suffix}.tar.gz powermac $built)
  echo "tar -xzf bf-arch${suffix}.tar.gz ; rm -f bf-arch${suffix}.tar.gz" >> \
  	$release/$bfinstall
  rm -rf $release/{chrp,powermac,new-powermac,prep,apus}

## there's nothing in images-1.44, kill it
  rmdir $release/images-1.44
  if [ -d $release/images-2.88 ]; then
    rmdir $release/images-2.88
  fi

#  (cd $release && tar cz --remove-files -f bf-common${suffix}.tar.gz images-1.44 )
#  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
#        $release/$bfinstall
#  (cd $release && rm -rf images-1.44)

  (cd $release && tar cz --remove-files -f bf-misc${suffix}.tar.gz \
	README.txt READ-??.txt md5sum.txt)
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
  $release/$bfinstall
}

do_arch_alpha()
{

# Alpha directory structure should look like:
#
# APB/
#     (apb various stuff - might be moved around a bit)
# MILO/
#      linload.exe
#      ldmilo.exe
#      (milo various stuff)
# ${subarch}/				(jensen/nautilus)
#            drivers.tgz
#            linux.bin
#            images-1.44/
#	                 rescue.bin
#                        driver-?.bin
# READ*
# bootlx (needed for CDs)
# doc/
# drivers.tgz
# images-1.44/
#		root.bin (generic)
#		rescue.bin (generic)
#		drivers.tgz (generic)
#		driver-?.bin (generic)
# md5sum.txt
# root.tar.gz	(for netboot)
# tftpboot.img

  mv $release/images-1.44/rootgeneric.bin $release/images-1.44/root.bin
  mkdir -p $release/MILO
  cp linload.exe ldmilo.exe $release/MILO

  #$APB=APB

  if [ $APB ]; then  
    mkdir -p $release/APB
    if [ -L apb.exe ]; then
      cp apb.exe up1000.pal $release/APB/
      for i in apb_*; do cp $i $release/APB/${i#apb_}; done
      cat >$release/APB/apb.cfg <<EOF
debian_install|bootl up1000.pal linux load_ramdisk=1
EOF
    fi
  fi

  for subarch in $*
  do
    targetdir=$release
    # Some subarchs have extra rescue disks (jensen and nautilus currently)
    if [ -f $release/images-1.44/rescue_${subarch}.bin ]; then
      if [ "$subarch" != "generic" ]; then
        targetdir=$release/$subarch
	subarch_dirs="$subarch_dirs $subarch"
      fi
      mkdir -p $targetdir/images-1.44
      mv $release/images-1.44/rescue_${subarch}.bin \
       $targetdir/images-1.44/rescue.bin
      mv release/drivers${subarch}.tgz \
       $targetdir/drivers.tgz
      cp linux${subarch}.bin $targetdir/linux.bin
      for drvdisc in $release/images-1.44/driver${subarch}-?.bin
      do
	drv_number=`echo $drvdisc|sed "s/.*driver${subarch}-\([1-9]\).bin/\1/"`
	mv $drvdisc $targetdir/images-1.44/driver-${drv_number}.bin
      done
    fi
    if [ -f $release/milo_${subarch}.bin ]; then
      mv $release/milo_${subarch}.bin $targetdir/MILO/milo_${subarch}.bin
      cp milo_${subarch} $targetdir/MILO/${subarch}
    fi
  done

  if [ -f tftpboot.img ]; then
    # install netserver miniroot
    cp root.tar.gz $release/
    [ ! -f root-graphical.tar.gz ] || cp root-graphical.tar.gz $release/

    # install tftpboot image
    cp tftpboot.img $release/
    [ ! -f tftpboot-graphical.img ] || cp tftpboot-graphical.img $release/

    netbootfiles="tftpboot.img root.tar.gz"
  fi

  # install bootlx for CDs (yes, you must have aboot 0.7 installed)
  cp /boot/bootlx $release/

  # make md5sum before tarring up
  (cd $release && md5sum `find . -type f | 		\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # Now, a plethora of inner tarballs
  (cd $release && tar cz --remove-files -f bf-generic${suffix}.tar.gz \
	images-1.44 linux.bin $netbootfiles *.tgz)
  echo "tar -xzf bf-generic${suffix}.tar.gz ; rm -f bf-generic${suffix}.tar.gz" >> \
	$release/$bfinstall
  rmdir $release/images-1.44

  (cd $release && tar cz --remove-files -f bf-others${suffix}.tar.gz $subarch_dirs)
  echo "tar -xzf bf-others${suffix}.tar.gz ; rm -f bf-others${suffix}.tar.gz" >> \
	$release/$bfinstall
  for subarch in $subarch_dirs; do rm -rf $release/$subarch; done

  (cd $release && tar cz --remove-files -f bf-misc${suffix}.tar.gz \
	README.txt READ-??.txt md5sum.txt MILO $APB bootlx)
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
	$release/$bfinstall
  (cd $release && rm -rf MILO $APB)
  (cd $release && rm root.bin)	# remove unpadded image
}

do_arch_arm()
{
  for subarch in $arm_arches cats; do
    mkdir -p $release/$subarch
  done

  # install drivers and rescue images
  for subarch in $arm_arches; do
    mv $release/drivers$subarch.tgz $release/$subarch/drivers.tgz
    mkdir -p $release/$subarch/images-1.44
    ( cd $release/images-1.44
    for f in driver${subarch}*.bin; do
      mv $f $release/${subarch}/images-1.44/driver${f#driver${subarch}};
    done
    )
    if [ -f $release/images-1.44/rescue$subarch.bin ]; then
      mv $release/images-1.44/rescue$subarch.bin $release/$subarch/images-1.44/rescue.bin
    fi
    if [ -f $release/images-2.88/rescue$subarch.bin ]; then
      mkdir -p $release/$subarch/images-2.88/
      mv $release/images-2.88/rescue$subarch.bin $release/$subarch/images-2.88/rescue.bin
    fi
  done

  # install subarch root.bin
  for subarch in $arm_arches_rootbin; do
    mkdir -p $release/$subarch/images-1.44
    mv $release/root$subarch.bin $release/$subarch/images-1.44/root.bin
  done

  # delete unwanted root.bins
  for subarch in $arm_arches; do
    rm -f $release/root$subarch.bin
  done

  # install subarch root.tar.gz
  for subarch in $arm_arches_nfsroot; do
    cp root$subarch.tar.gz $release/$subarch/root.tar.gz
  done

  # install subarch raw kernels
  for subarch in $arm_arches_linux; do
    cp linux$subarch.bin $release/$subarch/linux.bin
  done

  # install tftp images
  for subarch in $arm_arches_tftp cats; do
    cp tftpboot-$subarch.img $release/$subarch/tftpboot.img
  done

  # build RiscPC install kit
  echo building riscpc install kit
  cp arm-specials/rpc-skel-kit.zip $release/riscpc/rpc-skel-kit.zip
  rm -rf $release/riscpc/!dInstall
  mkdir $release/riscpc/!dInstall
  cp linuxriscpc.bin $release/riscpc/!dInstall/linux
  cp rootriscpc.bin $release/riscpc/!dInstall/root
  ( cd $release/riscpc; zip -ur0X rpc-skel-kit.zip !dInstall )
  rm -rf $release/riscpc/!dInstall
  mv $release/riscpc/rpc-skel-kit.zip $release/riscpc/dinstall.zip
  chmod 644 $release/riscpc/dinstall.zip
  ls -l $release/riscpc/dinstall.zip

  # make md5sum before tarring up all the contents
  echo generating md5sum.txt
  (cd $release && md5sum `find . -type f | 			\
    grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  ### make tarballs, and the script steps to undo them
  echo making tarballs
  ( cd $release
  for dir in cats $arm_arches ; do
  echo "  bf-${dir}${suffix}.tar.gz"
  tar cz --remove-files -f bf-${dir}${suffix}.tar.gz ${dir}
  echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
  	$bfinstall
  rm -rf ${dir}
  done

  echo "  bf-misc${suffix}.tar.gz"
  tar cz --remove-files -f bf-misc${suffix}.tar.gz \
  	README.txt READ-??.txt md5sum.txt
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
  	$bfinstall

  # delete empty directories
  rmdir $release/images-1.44 $release/images-2.88
  )
}

do_arch_ia64()
{
  #--------------------------------------------------------
  # ia64 installation
  #--------------------------------------------------------

  # root.bin
  # linux.bin
  # config.gz
  # sys_map.gz
  # drivers.tgz
  # images-1.44/rescue.bin

  subarch=mckinley-smp
  mkdir -p $release/images-1.44
  mv $release/resc10000${subarch}.bin $release/images-1.44/rescue.bin
  mv $release/linux${subarch}.bin $release/linux.bin
  mv config${subarch}.gz $release/config.gz
  mv sys_map${subarch}.gz $release/sys_map.gz
  mv $release/drivers${subarch}.tgz $release/drivers.tgz
  mv $release/root${subarch}.bin $release/root.bin
  rm -f $release/images-1.44/driver*.bin

  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f |                      \
    grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  ### make tarballs, and the script steps to undo them
  ( cd $release
  tar cz --remove-files -f bf-images${suffix}.tar.gz images-1.44 linux.bin config.gz sys_map.gz drivers.tgz root.bin
  echo "tar -xzf bf-images${suffix}.tar.gz ; rm -f bf-images${suffix}.tar.gz" >> \
        $bfinstall
  rm -rf images-1.44

  tar cz --remove-files -f bf-misc${suffix}.tar.gz \
        README.txt READ-??.txt md5sum.txt
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
        $bfinstall
  )
}

do_arch_hppa()
{
  #--------------------------------------------------------
  # hppa installation
  #--------------------------------------------------------

  # 32/linux.bin
  # 32/config.gz
  # 32/sys_map.gz
  # 32/drivers.tgz
  # 32/images-1.44/rescue.bin
  # lifimage
  # root.bin

  for subarch in 32 64; do
    mkdir -p $release/$subarch/images-1.44
    mv $release/images-2.88/rescue$subarch.bin $release/$subarch/images-1.44/rescue.bin
    mv $release/linux$subarch.bin $release/$subarch/linux.bin
    mv config$subarch.gz $release/$subarch/config.gz
    mv sys_map$subarch.gz $release/$subarch/sys_map.gz
    mv $release/drivers$subarch.tgz $release/$subarch/drivers.tgz
  done
  mv root32.bin $release/root.bin
  cd $release
  palo -f /dev/null -k 32/linux.bin -k 64/linux.bin -r root.bin \
	-s lifimage \
 	-c "0/linux HOME=/ ramdisk_size=8192 root=/dev/ram initrd=0/ramdisk"
  cd $TOP
  rm -rf $release/images-2.88 $release/images-1.44
  rm -f $release/root{32,64}.bin

  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f |                      \
    grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  ### make tarballs, and the script steps to undo them
  ( cd $release
  tar cz --remove-files -f bf-images${suffix}.tar.gz 32 64 lifimage root.bin
  echo "tar -xzf bf-images${suffix}.tar.gz ; rm -f bf-images${suffix}.tar.gz" >> \
        $bfinstall
  rm -rf 32 64

  tar cz --remove-files -f bf-misc${suffix}.tar.gz \
        README.txt READ-??.txt md5sum.txt
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
        $bfinstall
  )
}

do_arch_mips()
{ 
# use r4k-ip22 as generic root-disk:
  cp -a rootr4k-ip22.tar.gz $release/root.tar.gz
  mv $release/rootr4k-ip22.bin $release/root.bin

# overkill but keeps things simple for new subarches
  for subarch in r4k-ip22; do
    info "setting up subarchitecture '$subarch'"
    mkdir -p $release/${subarch}/images-1.44/
    # rescue.bin + drivers*.bin
    mv $release/images-1.44/rescue${subarch}.bin \
    		$release/${subarch}/images-1.44/rescue.bin
    (cd $release/images-1.44 &&
      for f in driver${subarch}*.bin; do
        mkdir -p ../${subarch}/images-1.44/ # just to be sure
	[ ! -f $f ] || mv $f ../${subarch}/images-1.44/driver${f#driver${subarch}};
      done)
    mv $release/drivers${subarch}.tgz $release/${subarch}/drivers.tgz
    # kernel + tftp boot image
    gunzip -c config${subarch} > $release/${subarch}/kernel-config
    cp -a  linux${subarch}.bin $release/${subarch}/linux.bin
    cp tftpboot-${subarch}.img $release/${subarch}/tftpboot.img
  done

  rmdir $release/images-1.44
  if [ -d $release/images-2.88 ]; then
    rmdir $release/images-2.88
  fi

  # make md5sum before tarring up all the contents
  info "making md5sum"
  (cd $release && md5sum `find . -type f | 			\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # make tarballs for subarches, etc.
  for dir in r4k-ip22; do
    (cd $release && tar cz --remove-files -f bf-${dir}${suffix}.tar.gz $dir)
    echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
	$release/$bfinstall
    rm -rf $release/${dir}
  done

  (cd $release && tar cz --remove-files -f bf-common${suffix}.tar.gz \
	*.txt root.tar.gz root.bin)
  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
	$release/$bfinstall
  info "done with mips specific part."
}
	
do_arch_mipsel()
{

  mkdir -p $release
  
  mv rootr4k-kn04.tar.gz $release/root.tar.gz
  mv rootr4k-kn04.bin $release/root.bin
  
  # overkill but keeps things simple for new subarches
  for subarch in r4k-kn04 r3k-kn02; do
    # Per subarch images
    info "setting up subarchitecture '$subarch'"
    mkdir -p $release/${subarch}/
    mkdir -p $release/${subarch}/images-1.44
    mkdir -p $release/${subarch}/images-2.88
    mv $release/images-1.44/rescue${subarch}.bin $release/${subarch}/images-1.44/rescue.bin
    (cd $release/images-1.44 &&
      for f in driver${subarch}*.bin; do
        mkdir -p ../${subarch}/images-1.44/ # just to be sure
	[ ! -f $f ] || mv $f ../${subarch}/images-1.44/driver${f#driver${subarch}};
      done)
    mv $release/images-2.88/rescue${subarch}.bin $release/${subarch}/images-2.88/rescue.bin
    mv $release/drivers${subarch}.tgz $release/${subarch}/drivers.tgz
    gunzip -c config${subarch} > $release/${subarch}/kernel-config
    echo "contents of `pwd`"
    ls -l
    cp -a linux${subarch}.bin $release/${subarch}/linux.bin
    if [ -f linux${subarch}.ecoff ]; then
      echo "ECOFF kernel image found"
      echo "copying linux${subarch}.ecoff to $release/${subarch}/linux.ecoff"
      cp -a  linux${subarch}.ecoff $release/${subarch}/linux.ecoff
    fi
  done

  # clean up
  # a) we use a unified root.bin, so the subarch-specific ramdisk-images can
  #    be removed
  # b) remove the linux${subarch}.bin files resulting from the linux ->
  #    linux.bin transition
  for subarch in r4k-kn04 r3k-kn02; do
    rm -f $release/root${subarch}.bin
    rm -f $release/linux${subarch}.bin
  done

  # make md5sum before tarring up all the contents
  info "making md5sum"
  (cd $release && md5sum `find . -type f |                      \
        grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # make tarballs for subarches, etc.
  for dir in r4k-kn04 r3k-kn02; do
    (cd $release && tar cz --remove-files -f bf-${dir}${suffix}.tar.gz $dir)
    echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
        $release/$bfinstall
    rm -rf $release/${dir}
  done

  (cd $release && tar cz --remove-files -f bf-common${suffix}.tar.gz \
        *.txt root.tar.gz root.bin)
  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
        $release/$bfinstall
 rm -rf $release/images-2.88
 rm -rf $release/images-1.44

 echo "contents of release-directory"
 find $release

 info "done with mipsel specific part."
 
}

do_arch_s390()
{
  #--------------------------------------------------------
  # s390 installation
  #--------------------------------------------------------
 
  # drivers.tgz
  # images-1.44/rescue.bin
  # tape/kernel.debian
  # tape/kernel.debian-nolabel
  # tape/parmfile.debian
  # tape/initrd.debian
  # vmrdr/debian.exec
  # vmrdr/kernel.debian
  # vmrdr/parmfile.debian -> ../tape/parmfile.debian
  # vmrdr/initrd.debian -> ../tape/initrd.debian

  mkdir -p $release/images-1.44 $release/tape $release/vmrdr
  mv $release/driverss390.tgz $release/drivers.tgz
  mv $release/images-2.88/rescues390.bin $release/images-1.44/rescue.bin

  mv linuxs390-tape $release/tape/kernel.debian
  mv linuxs390-tape-nolabel $release/tape/kernel.debian-nolabel
  cp s390-specials/parmfile $release/tape/parmfile.debian
  mv $release/roots390.bin $release/tape/initrd.debian

  cp s390-specials/iplrdr.exec $release/vmrdr/debian.exec
  mv linuxs390.bin $release/vmrdr/kernel.debian
  ln -s ../tape/parmfile.debian $release/vmrdr/parmfile.debian
  ln -s ../tape/initrd.debian $release/vmrdr/initrd.debian
 
  # not needed on s390
  rm -rf $release/images-2.88
  rm -f $release/images-1.44/drivers390-1.bin

  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f |                      \
    grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)
 
  ### make tarballs, and the script steps to undo them
  ( cd $release
  tar cz --remove-files -f bf-images${suffix}.tar.gz drivers.tgz images-1.44 tape vmrdr
  echo "tar -xzf bf-images${suffix}.tar.gz ; rm -f bf-images${suffix}.tar.gz" >> \
        $bfinstall

  rmdir images-1.44 tape vmrdr
 
  tar cz --remove-files -f bf-misc${suffix}.tar.gz \
        README.txt READ-??.txt md5sum.txt
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
        $bfinstall
  )
}
                                                                                

# main
do_common_pretasks
do_archspecific
do_common_posttasks
exit 0
