/** 
 * @namespace   biew_plugins_auto
 * @file        plugins/bin/coff386.c
 * @brief       This file contains implementation of COFF-i386 file format decoder.
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Nick Kurshev
 * @since       1999
 * @note        Development, fixes and improvements
**/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "reg_form.h"
#include "tstrings.h"
#include "codeguid.h"
#include "colorset.h"
#include "bconsole.h"
#include "bin_util.h"
#include "bmfile.h"
#include "biewutil.h"
#include "biewhelp.h"
#include "plugins/bin/coff386.h"
#include "plugins/disasm.h"
#include "biewlib/kbd_code.h"
#include "biewlib/pmalloc.h"

static struct external_filehdr coff386hdr;
static AOUTHDR coff386ahdr;
static SCNHDR *coff386so;
static unsigned short nsections;
static BGLOBAL coff_cache;
static unsigned long strings_ptr;

static void __FASTCALL__ coff_ReadPubNameList(BGLOBAL handle,void (__FASTCALL__ *mem_out)(const char *));
static void __FASTCALL__ coff_ReadPubName(BGLOBAL b_cache,const struct PubName *it,
                           char *buff,unsigned cb_buff);
static unsigned __FASTCALL__ coff386_GetObjAttr(unsigned long pa,char *name,unsigned cb_name,
                             unsigned long *start,unsigned long *end,int *class,int *bitness);

static tBool __NEAR__ __FASTCALL__ FindPubName(char *buff,unsigned cb_buff,unsigned long pa)
{
  return fmtFindPubName(coff_cache,buff,cb_buff,pa,
                        coff_ReadPubNameList,
                        coff_ReadPubName);
}


static void __NEAR__ __FASTCALL__ coffReadLongName(BGLOBAL handle,unsigned long offset,
                                      char *str, unsigned slen)
{
  unsigned i;
  unsigned long fpos;
   fpos = bioTell(handle);
   bioSeek(handle,offset+strings_ptr,BIO_SEEK_SET);
   for(i = 0;i < slen;i++)
   {
     unsigned char ch;
     ch = bioReadByte(handle);
     if(ch && !bioEOF(handle)) str[i] = ch;
     else  break;
   }
   str[i] = 0;
   bioSeek(handle,fpos,BIO_SEEK_SET);
}

static unsigned long __FASTCALL__ coff386_VA2PA(unsigned long va)
{
  unsigned long i;
  for(i = 0;i < nsections;i++)
  {
    if(va >= *(unsigned long *)&coff386so[i].s_vaddr && va <= *(unsigned long *)&coff386so[i].s_vaddr+*(unsigned long *)&coff386so[i].s_size)
    {
      return *(unsigned long *)&coff386so[i].s_scnptr + (va - *(unsigned long *)&coff386so[i].s_vaddr);
    }
  }
  return 0L;
}

static unsigned long __FASTCALL__ coff386_PA2VA(unsigned long pa)
{
  unsigned long i;
  for(i = 0;i < nsections;i++)
  {
    if(pa >= *(unsigned long *)&coff386so[i].s_scnptr && pa <= *(unsigned long *)&coff386so[i].s_scnptr+*(unsigned long *)&coff386so[i].s_size)
    {
      return *(unsigned long *)coff386so[i].s_vaddr + (pa - *(unsigned long *)&coff386so[i].s_scnptr);
    }
  }
  return 0L;
}


static void __FASTCALL__ coffObjPaint(TWindow * win,const void ** names,unsigned start,unsigned nlist)
{
 char buffer[81];
 const SCNHDR ** obj = (const SCNHDR **)names;
 const SCNHDR *  objs = obj[start];
 twUseWin(win);
 twFreezeWin(win);
 twClearWin();
 sprintf(buffer," Object Table [ %u / %u ] ",start + 1,nlist);
 twSetTitleAttr(win,buffer,TW_TMODE_CENTER,dialog_cset.title);
 twSetFooterAttr(win,PAGEBOX_SUB,TW_TMODE_RIGHT,dialog_cset.selfooter);
 twGotoXY(1,1);
 twPrintF("Object Name                    = %8s\n"
          "Physical address               = %08lXH\n"
          "Virtual address                = %08lXH\n"
          "Section size                   = %08lXH\n"
          "File pointer to raw data       = %08lXH\n"
          "File pointer to relocations    = %08lXH\n"
          "File pointer to line numbers   = %08lXH\n"
          "Number of relocations          = %04XH\n"
          "Number of line numbers         = %04XH\n"
          "Flags                          = %08lXH\n"
          "  [%c] - section type is text\n"
          "  [%c] - section type is data\n"
          "  [%c] - section type is bss"
          ,objs->s_name
          ,*(unsigned long *)&objs->s_paddr
          ,*(unsigned long *)&objs->s_vaddr
          ,*(unsigned long *)&objs->s_size
          ,*(unsigned long *)&objs->s_scnptr
          ,*(unsigned long *)&objs->s_relptr
          ,*(unsigned long *)&objs->s_lnnoptr
          ,*(unsigned short *)&objs->s_nreloc
          ,*(unsigned short *)&objs->s_nlnno
          ,*(unsigned long *)&objs->s_flags
          ,GetBool((*(unsigned long *)&objs->s_flags & STYP_TEXT) == STYP_TEXT)
          ,GetBool((*(unsigned long *)&objs->s_flags & STYP_DATA) == STYP_DATA)
          ,GetBool((*(unsigned long *)&objs->s_flags & STYP_BSS) == STYP_BSS)
          );
 twRefreshFullWin(win);
}

static tBool __NEAR__ __FASTCALL__ __coffReadObjects(BGLOBAL handle,memArray * obj,unsigned n)
{
 size_t i;
  for(i = 0;i < n;i++)
  {
    SCNHDR po;
    if(IsKbdTerminate() || bioEOF(handle)) break;
    bioReadBuffer(handle,&po,sizeof(SCNHDR));
    if(!ma_AddData(obj,&po,sizeof(SCNHDR),True)) break;
  }
  return True;
}

static unsigned long __FASTCALL__ coffShowObjects( void )
{
 BGLOBAL handle;
 unsigned nnames;
 unsigned long fpos,off;
 memArray * obj;
 fpos = BMGetCurrFilePos();
 nnames = *(unsigned short *)&coff386hdr.f_nscns;
 if(!nnames) { NotifyBox(NOT_ENTRY," Objects Table "); return fpos; }
 if(!(obj = ma_Build(nnames,True))) return fpos;
 handle = coff_cache;
 off = sizeof(coff386hdr);
 if(*(unsigned short *)&coff386hdr.f_opthdr) off += *(unsigned short *)&coff386hdr.f_opthdr;
 bioSeek(handle,off,SEEK_SET);
 if(__coffReadObjects(handle,obj,nnames))
 {
  int ret;
    ret = PageBox(70,13,(const void **)obj->data,obj->nItems,coffObjPaint);
    if(ret != -1)  fpos = *(unsigned long *)(((SCNHDR *)obj->data[ret])->s_scnptr);
 }
 ma_Destroy(obj);
 return fpos;
}

static const char * __NEAR__ __FASTCALL__ coff386_encode_hdr(unsigned short info)
{
   switch(info)
   {
     case I386MAGIC: return " i386 coff ";
     case I386PTXMAGIC: return " i386 PTX coff ";
     case I386AIXMAGIC: return " i386 AIX coff ";
     case LYNXCOFFMAGIC: return " Lynx coff ";
     default:    return " Unknown coff i386 format ";
   }
}

static unsigned long __FASTCALL__ ShowCoff386Header( void )
{
  unsigned long fpos,entry,v_entry;
  unsigned keycode;
  TWindow * w;
  fpos = BMGetCurrFilePos();
  v_entry = entry = 0L;
  if(*(unsigned short *)coff386ahdr.magic == ZMAGIC)
  {
    v_entry = entry = *(unsigned long *)&coff386ahdr.entry;
    entry = coff386_VA2PA(v_entry);
  }
  w = CrtDlgWndnls(coff386_encode_hdr(*((unsigned short *)&coff386hdr.f_magic)),54,12);
  twGotoXY(1,1);
  twPrintF("Number of sections          = %04XH\n"
           "Time & data stamp           = %08lXH\n"
           "File pointer to symtab      = %08lXH\n"
           "Number of symtab entries    = %08lXH\n"
           "Size of optional header     = %04XH\n"
           "COFF flags                  = %04XH\n"
           "  [%c] - Rel. info stripped\n"
           "  [%c] - File is executable (no unresolv. refs.)\n"
           "  [%c] - Line numbers stripped\n"
           "  [%c] - Local symbols stripped\n"
           "  [%c] - File is 32-bits little endian\n"
           ,*((unsigned short *)&coff386hdr.f_nscns)
           ,*((unsigned long *)&coff386hdr.f_timdat)
           ,*((unsigned long *)&coff386hdr.f_symptr)
           ,*((unsigned long *)&coff386hdr.f_nsyms)
           ,*((unsigned short *)&coff386hdr.f_opthdr)
           ,*((unsigned short *)&coff386hdr.f_flags)
           ,GetBool((*((unsigned long *)&coff386hdr.f_flags) & F_RELFLG) == F_RELFLG)
           ,GetBool((*((unsigned long *)&coff386hdr.f_flags) & F_EXEC) == F_EXEC)
           ,GetBool((*((unsigned long *)&coff386hdr.f_flags) & F_LNNO) == F_LNNO)
           ,GetBool((*((unsigned long *)&coff386hdr.f_flags) & F_LSYMS) == F_LSYMS)
           ,GetBool((*((unsigned long *)&coff386hdr.f_flags) & F_AR32WR) == F_AR32WR));
  twSetColorAttr(dialog_cset.entry);
  twPrintF("Entry point                 = %08lXH (VA=%08lXH)"
           ,entry,v_entry);
  twClrEOL();
  twSetColorAttr(dialog_cset.main);
  while(1)
  {
    keycode = GetEvent(drawEmptyPrompt,w);
    if(keycode == KE_ENTER) { fpos = entry; break; }
    else
      if(keycode == KE_ESCAPE || keycode == KE_F(10)) break;
  }
  CloseWnd(w);
  return fpos;
}

static const char * __NEAR__ __FASTCALL__ coffEncodeType(unsigned type)
{
  const char *ret;
  switch(type)
  {
    case T_NULL: ret = "T_NULL"; break;
    case T_VOID: ret = "T_VOID"; break;
    case T_CHAR: ret = "T_CHAR"; break;
    case T_SHORT: ret = "T_SHORT"; break;
    case T_INT: ret = "T_INT"; break;
    case T_LONG: ret = "T_LONG"; break;
    case T_FLOAT: ret = "T_FLOAT"; break;
    case T_DOUBLE: ret = "T_DOUBLE"; break;
    case T_STRUCT: ret = "T_STRUCT"; break;
    case T_UNION: ret = "T_UNION"; break;
    case T_ENUM: ret = "T_ENUM"; break;
    case T_MOE: ret = "T_MOE"; break;
    case T_UCHAR: ret = "T_UCHAR"; break;
    case T_USHORT: ret = "T_USHORT"; break;
    case T_UINT: ret = "T_UINT"; break;
    case T_ULONG: ret = "T_ULONG"; break;
    case T_LNGDBL: ret = "T_LNGDBL"; break;
    default: ret = "T_UNK"; break;
  }
  return ret;
}

static const char * __NEAR__ __FASTCALL__ coffEncodeClass(unsigned _class)
{
  const char *ret;
  switch(_class)
  {
    case C_EFCN: ret = "C_EFCN"; break;
    case C_NULL: ret = "C_NULL"; break;
    case C_AUTO: ret = "C_AUTO"; break;
    case C_EXT: ret = "C_EXT"; break;
    case C_STAT: ret = "C_STAT"; break;
    case C_REG: ret = "C_REG"; break;
    case C_EXTDEF: ret = "C_EXTDEF"; break;
    case C_LABEL: ret = "C_LABEL"; break;
    case C_ULABEL: ret = "C_ULABEL"; break;
    case C_MOS: ret = "C_MOS"; break;
    case C_ARG: ret = "C_ARG"; break;
    case C_STRTAG: ret = "C_STRTAG"; break;
    case C_MOU: ret = "C_MOU"; break;
    case C_UNTAG: ret = "C_UNTAG"; break;
    case C_TPDEF: ret = "C_TPDEF"; break;
    case C_USTATIC: ret = "C_USTATIC"; break;
    case C_ENTAG: ret = "C_ENTAG"; break;
    case C_MOE: ret = "C_MOE"; break;
    case C_REGPARM: ret = "C_REGPARM"; break;
    case C_FIELD: ret = "C_FIELD"; break;
    case C_AUTOARG: ret = "C_AUTOARG"; break;
    case C_LASTENT: ret = "C_LASTENT"; break;
    case C_BLOCK: ret = "C_BLOCK"; break;
    case C_FCN: ret = "C_FCN"; break;
    case C_EOS: ret = "C_EOS"; break;
    case C_FILE: ret = "C_FILE"; break;
    case C_LINE: ret = "C_LINE"; break;
    case C_ALIAS: ret = "C_ALIAS"; break;
    case C_HIDDEN: ret = "C_HIDDEN"; break;
    default: ret = "C_UNK"; break;
  }
  return ret;
}


static unsigned __FASTCALL__ coffSymTabNumItems(BGLOBAL handle)
{
  UNUSED(handle);
  return (unsigned)(*(unsigned long *)&coff386hdr.f_nsyms);
}

static tBool  __FASTCALL__ coffSymTabReadItems(BGLOBAL handle,memArray * obj,unsigned nnames)
{
 unsigned i;
 unsigned length;
 bioSeek(handle,*(unsigned long *)&coff386hdr.f_symptr,SEEKF_START);
 for(i = 0;i < nnames;i++)
 {
   struct external_syment cse;
   char stmp[256];
   memset(stmp,0,sizeof(stmp));
   bioReadBuffer(handle,&cse,sizeof(struct external_syment));
   if(*(unsigned long *)&cse.e.e.e_zeroes == 0L &&
      *(unsigned long *)&cse.e.e.e_offset >= 4)
   {
     coffReadLongName(handle,*(unsigned long *)&cse.e.e.e_offset,stmp,sizeof(stmp));
   }
   else
   {
     memcpy(stmp,cse.e.e_name,E_SYMNMLEN);
     stmp[E_SYMNMLEN] = 0;
   }
   if(IsKbdTerminate() || bioEOF(handle)) break;
   length = strlen(stmp);
   if(length > 38) strcpy(&stmp[38],">>>");
   else  {  memset(&stmp[length],' ',38-length);  stmp[38] = 0; }
   sprintf(&stmp[strlen(stmp)-1],"(%04hX.%08lX) %s.%s"
           ,*(unsigned short *)&cse.e_scnum
           ,*(unsigned long *)&cse.e_value
           ,coffEncodeClass(cse.e_sclass[0])
           ,coffEncodeType(*(unsigned short *)&cse.e_type));
   if(!ma_AddString(obj,stmp,True)) break;
 }
 return True;
}

static unsigned long __NEAR__ __FASTCALL__ CalcEntryCoff(unsigned long idx,tBool display_msg)
{
  struct external_syment cse;
  unsigned short sec_num;
  unsigned long fpos;
  fpos = 0L;
  bmSeek(*(unsigned long *)&coff386hdr.f_symptr+idx*sizeof(struct external_syment),BM_SEEK_SET);
  bmReadBuffer(&cse,sizeof(struct external_syment));
  sec_num = *(unsigned short *)&cse.e_scnum;
  if(sec_num && sec_num <= *(unsigned short *)coff386hdr.f_nscns &&
    ((*(unsigned char *)&cse.e_sclass == C_EXT ||
      *(unsigned char *)&cse.e_sclass == C_STAT ||
      *(unsigned char *)&cse.e_sclass == C_HIDDEN) ||
     (*(unsigned char *)&cse.e_sclass == C_LABEL &&
      *(unsigned long *)&cse.e_value)))
  {
    fpos = *(unsigned long *)&coff386so[sec_num-1].s_scnptr+*(unsigned long *)&cse.e_value;
  }
  else
    if(display_msg)
      ErrMessageBox("This entry does not refer inside the file",NULL);
  return fpos;
}

static unsigned long __FASTCALL__ coffShowSymTab( void )
{
  unsigned long fpos = BMGetCurrFilePos();
  int ret;
  ret = fmtShowList(coffSymTabNumItems,coffSymTabReadItems,
                    "Symbol Table",
                    LB_SELECTIVE,NULL);
  if(ret != -1)
  {
    fpos = CalcEntryCoff(ret,True);
  }
  return fpos;
}

/***************************************************************************/
/*********************  REFS COFF386  **************************************/
/***************************************************************************/
static char __codelen;

typedef struct tagRELOC_COFF386
{
  unsigned long offset;
  unsigned long nameoff;
  unsigned      type;
}RELOC_COFF386;

static linearArray *RelocCoff386 = NULL;

static tCompare __FASTCALL__ coff386_compare_rels(const void __HUGE__ *e1,const void __HUGE__ *e2)
{
  const RELOC_COFF386 __HUGE__ *r1,__HUGE__ *r2;
  tCompare ret;
  r1 = e1;
  r2 = e2;
  if(r1->offset < r2->offset) ret = -1;
  else
    if(r1->offset > r2->offset) ret = 1;
    else                        ret = 0;
  return ret;
}

static void __NEAR__ __FASTCALL__ BuildRelocCoff386( void )
{
  TWindow * w,*usd;
  unsigned j,segcount;
  RELOC_COFF386 rel;
  usd = twUsedWin();
  if(!(RelocCoff386 = la_Build(0,sizeof(RELOC_COFF386),MemOutBox))) return;
  w = CrtDlgWndnls(" System is busy ",49,1);
  if(!PubNames) coff_ReadPubNameList(bmbioHandle(),MemOutBox);
  twUseWin(w);
  twGotoXY(1,1);
  twPutS(" Building reference chains for Coff386 format");
  twUseWin(usd);
  for(segcount = 0;segcount < *(unsigned short *)&coff386hdr.f_nscns;segcount++)
  {
    tBool is_eof;
    is_eof = False;
    bmSeek(*(unsigned long *)&coff386so[segcount].s_relptr,BM_SEEK_SET);
    for(j = 0;j < *(unsigned short *)&coff386so[segcount].s_nreloc;j++)
    {
      struct external_reloc er;
      bmReadBuffer(&er,sizeof(struct external_reloc));
      if((is_eof = bmEOF()) != 0) break;
      rel.offset = er.r_vaddr + *(unsigned long *)&coff386so[segcount].s_scnptr;
      rel.nameoff = er.r_symndx;
      rel.type = er.r_type;
      if(!la_AddData(RelocCoff386,&rel,MemOutBox)) goto next;
    }
    if(is_eof) break;
  }
  next:
  la_Sort(RelocCoff386,coff386_compare_rels);
  CloseWnd(w);
}

static tBool  __NEAR__ __FASTCALL__ coffSymTabReadItemsIdx(BGLOBAL handle,unsigned long idx,
                                            char *name,unsigned cb_name,
                                            unsigned short *secnum,
                                            unsigned long *offset)
{
 struct external_syment cse;
 if(idx >= *(unsigned long *)&coff386hdr.f_nsyms) return False;
 bioSeek(handle,*(unsigned long *)&coff386hdr.f_symptr + idx*sizeof(struct external_syment),SEEKF_START);
 bioReadBuffer(handle,&cse,sizeof(struct external_syment));
 if(*(unsigned long *)&cse.e.e.e_zeroes == 0L &&
    *(unsigned long *)&cse.e.e.e_offset >= 4)
 {
     coffReadLongName(handle,*(unsigned long *)&cse.e.e.e_offset,name,cb_name);
 }
 else
 {
   memcpy(name,cse.e.e_name,E_SYMNMLEN);
   name[E_SYMNMLEN] = 0;
 }
 *secnum = *(unsigned short *)&cse.e_scnum;
 *offset = *(unsigned long *)&cse.e_value;
 return True;
}

static tBool __NEAR__ __FASTCALL__ BuildReferStrCoff386(char *str,RELOC_COFF386 *rne,int flags)
{
  unsigned long val,offset,s,e;
  unsigned short secnum;
  tBool retval,val_assigned;
  int c,b;
  char name[256],pubname[256],secname[256];
  val = bmReadDWordEx(rne->offset,BM_SEEK_SET);
  /* rne->nameoff it's only pointer to name descriptor */
  retval = coffSymTabReadItemsIdx(coff_cache,rne->nameoff,name,sizeof(name),&secnum,&offset);
  val_assigned = False;
  if(retval)
  {
    if(!offset && secnum)
    {
      /*
         try resolve some relocations (like .text+1234) with names from
         public name list
      */
      if(FindPubName(pubname,sizeof(pubname),*(unsigned long *)&coff386so[secnum-1].s_scnptr + val))
      {
        strcat(str,pubname);
        val_assigned = True;
      }
      else goto def_val;
    }
    else
    {
      def_val:
      strcat(str,name);
    }    
  }
  else
    strcat(str,".?bad?");
  if(val && !val_assigned)
  {
     strcat(str,"+");
     strcat(str,Get8Digit(val));
  }
  if(rne->type == RELOC_REL32 && (flags & APREF_TRY_LABEL) != APREF_TRY_LABEL)
  {
     coff386_GetObjAttr(rne->offset,secname,sizeof(secname),&s,&e,&c,&b);
     strcat(str,"-");
     strcat(str,secname);
  }
  e = CalcEntryCoff(rne->nameoff,False);
  if(!DumpMode && !EditMode && e && (flags & APREF_TRY_LABEL) == APREF_TRY_LABEL)
                                                  GidAddGoAddress(str,e);
  return True;
}

static int __FASTCALL__ coff386_AppendRef(char *str,unsigned long ulShift,int flags,int codelen,unsigned long r_sh)
{
  RELOC_COFF386 *rcoff386,key;
  tBool ret;
  char buff[400];
  ret = False;
  if(!PubNames) coff_ReadPubNameList(bmbioHandle(),MemOutBox);
  if((*(unsigned short *)&coff386hdr.f_flags & F_RELFLG) == F_RELFLG) goto try_pub;
  if(!RelocCoff386) BuildRelocCoff386();
  key.offset = ulShift;
  __codelen = codelen;
  rcoff386 = la_Find(RelocCoff386,&key,coff386_compare_rels);
  if(!rcoff386) ret = False;
  else          ret = BuildReferStrCoff386(str,rcoff386,flags);
  try_pub:
  if(!ret && (flags & APREF_TRY_LABEL))
  {
     if(FindPubName(buff,sizeof(buff),r_sh))
     {
       strcat(str,buff);
       if(!DumpMode && !EditMode) GidAddGoAddress(str,r_sh);
       ret = True;
     }
  }
  return ret ? RAPREF_DONE : RAPREF_NONE;
}

static tBool __FASTCALL__ coff386_check_fmt( void )
{
  unsigned short id;
  id = bmReadWordEx(0,SEEKF_START);
  return !(I386BADMAG(id));
}

static void __FASTCALL__ coff386_init_fmt( void )
{
  BGLOBAL main_handle;
  unsigned long s_off = sizeof(coff386hdr);
  unsigned i;
  bmReadBufferEx(&coff386hdr,sizeof(struct external_filehdr),0,SEEKF_START);
  if(*(unsigned short *)&coff386hdr.f_opthdr) bmReadBuffer(&coff386ahdr,sizeof(AOUTHDR));
  nsections = *(unsigned long *)&coff386hdr.f_nscns;
  if(!(coff386so = PMalloc((unsigned)(sizeof(SCNHDR)*nsections))))
  {
     MemOutBox("Coff386 initialization");
     exit(EXIT_FAILURE);
  }
  main_handle = bmbioHandle();
  if((coff_cache = bioDupEx(main_handle,BBIO_SMALL_CACHE_SIZE)) == &bNull) coff_cache = main_handle;
  if(*(unsigned short *)&coff386hdr.f_opthdr) s_off += *(unsigned short *)&coff386hdr.f_opthdr;
  bioSeek(coff_cache,s_off,BIO_SEEK_SET);
  for(i = 0;i < nsections;i++)
  {
    bioReadBuffer(coff_cache,&coff386so[i],sizeof(SCNHDR));
  }
  strings_ptr = *(unsigned long *)&coff386hdr.f_symptr+*(unsigned long *)&coff386hdr.f_nsyms*sizeof(struct external_syment);
}

static void __FASTCALL__ coff386_destroy_fmt( void )
{
  BGLOBAL main_handle;
  PFREE(coff386so);
  if(PubNames) { la_Destroy(PubNames); PubNames = 0; }
  main_handle = bmbioHandle();
  if(coff_cache != &bNull && coff_cache != main_handle) bioClose(coff_cache);
}

static int __FASTCALL__ coff386_bitness(unsigned long off)
{
  UNUSED(off);
  return DAB_USE32; 
}

static tBool __FASTCALL__ coff386_AddrResolv(char *addr,unsigned long cfpos)
{
 /* Since this function is used in references resolving of disassembler
    it must be seriously optimized for speed. */
  tBool bret = True;
  uint32_t res;
  if(cfpos < sizeof(struct external_filehdr))
  {
    strcpy(addr,"coffih:");
    strcpy(&addr[7],Get2Digit(cfpos));
  }
  else 
    if((res=coff386_PA2VA(cfpos))!=0) 
    {
      addr[0] = '.';
      strcpy(&addr[1],Get8Digit(res));
    }
    else bret = False;
  return bret;
}

static unsigned long __FASTCALL__ coff386Help( void )
{
  hlpDisplay(10002);
  return BMGetCurrFilePos();
}

static void __FASTCALL__ coff_ReadPubName(BGLOBAL b_cache,const struct PubName *it,
                           char *buff,unsigned cb_buff)
{
    if(!it->addinfo)
      coffReadLongName(b_cache,it->nameoff,buff,cb_buff);
    else
    {
      bioSeek(b_cache,it->nameoff,BIO_SEEK_SET);
      bioReadBuffer(b_cache,buff,it->addinfo);
      buff[it->addinfo] = 0;
    }
}

static void __FASTCALL__ coff_ReadPubNameList(BGLOBAL handle,
                                    void (__FASTCALL__ *mem_out)(const char *))
{
 unsigned i,nnames;
 struct PubName pn;
 if(!(PubNames = la_Build(0,sizeof(struct PubName),mem_out))) return;
 bioSeek(handle,*(unsigned long *)&coff386hdr.f_symptr,SEEKF_START);
 nnames = coffSymTabNumItems(handle);
 for(i = 0;i < nnames;i++)
 {
   struct external_syment cse;
   unsigned long where;
   unsigned short sec_num;
   where = bioTell(handle);
   bioReadBuffer(handle,&cse,sizeof(struct external_syment));
   sec_num = *(unsigned short *)&cse.e_scnum;
   if(sec_num && sec_num <= *(unsigned short *)coff386hdr.f_nscns &&
     (*(unsigned char *)&cse.e_sclass == C_EXT ||
      *(unsigned char *)&cse.e_sclass == C_STAT ||
      *(unsigned char *)&cse.e_sclass == C_HIDDEN))
   {
     if(*(unsigned long *)&cse.e.e.e_zeroes == 0L &&
        *(unsigned long *)&cse.e.e.e_offset >= 4)
     {
       pn.nameoff = *(unsigned long *)&cse.e.e.e_offset;
       pn.addinfo = 0;
     }
     else
     {
       pn.nameoff = where;
       pn.addinfo = E_SYMNMLEN;
     }
     pn.pa = *(unsigned long *)&coff386so[sec_num-1].s_scnptr+*(unsigned long *)&cse.e_value;
     pn.attr = *(unsigned char *)&cse.e_sclass == C_EXT ? SC_GLOBAL : SC_LOCAL;
     if(!la_AddData(PubNames,&pn,mem_out)) break;
   }
   if(bioEOF(handle)) break;
 }
 la_Sort(PubNames,fmtComparePubNames);
}

static unsigned long __FASTCALL__ coff386_GetPubSym(char *str,unsigned cb_str,unsigned *func_class,
                          unsigned long pa,tBool as_prev)
{
  return fmtGetPubSym(coff_cache,str,cb_str,func_class,pa,as_prev,
                      coff_ReadPubNameList,
                      coff_ReadPubName);
}

static unsigned __FASTCALL__ coff386_GetObjAttr(unsigned long pa,char *name,unsigned cb_name,
                            unsigned long *start,unsigned long *end,int *class,int *bitness)
{
  unsigned i,ret;
  *start = 0;
  *end = bmGetFLength();
  *class = OC_NOOBJECT;
  *bitness = DAB_USE32;
  name[0] = 0;
  ret = 0;
  for(i = 0;i < nsections;i++)
  {
    if(pa >= *start && pa < *(unsigned long *)&coff386so[i].s_scnptr)
    {
      /** means between two objects */
      *end = *(unsigned long *)&coff386so[i].s_scnptr;
      ret = 0;
      break;
    }
    if(pa >= *(unsigned long *)&coff386so[i].s_scnptr && pa < *(unsigned long *)&coff386so[i].s_scnptr + *(unsigned long *)&coff386so[i].s_size)
    {
      *start = *(unsigned long *)&coff386so[i].s_scnptr;
      *end = *start + *(unsigned long *)&coff386so[i].s_size;
      *class = (*(unsigned long *)&coff386so[i].s_flags & STYP_TEXT) == STYP_TEXT ? OC_CODE : OC_DATA;
      strncpy(name,(const char *)coff386so[i].s_name,min(sizeof(coff386so[i].s_name),cb_name));
      name[sizeof(coff386so[i].s_name)] = 0;
      ret = i+1;
      break;
    }
    *start = *(unsigned long *)&coff386so[i].s_scnptr + *(unsigned long *)&coff386so[i].s_size;
  }
  return ret;
}

static int __FASTCALL__ coff386_platform( void ) { return DISASM_CPU_IX86; }

REGISTRY_BIN coff386Table =
{
  "coff-i386 (Common Object File Format)",
  { "CofHlp", NULL, NULL, NULL, NULL, NULL, "SymTab", NULL, NULL, "Objects" },
  { coff386Help, NULL, NULL, NULL, NULL, NULL, coffShowSymTab, NULL, NULL, coffShowObjects },
  coff386_check_fmt,
  coff386_init_fmt,
  coff386_destroy_fmt,
  ShowCoff386Header,
  coff386_AppendRef,
  fmtSetState,
  coff386_platform,
  coff386_bitness,
  coff386_AddrResolv,
  coff386_VA2PA,
  coff386_PA2VA,
  coff386_GetPubSym,
  coff386_GetObjAttr,
  NULL,
  NULL
};
