/** 
 * @namespace   biewlib
 * @file        biewlib/sysdep/generic/unix/keyboard.c
 * @brief       slang/curses/vt100 keyboard library
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Konstantin Boldyshev
 * @since       1999
 * @note        Development, fixes and improvements
**/

/* $Id: keyboard.c,v 1.1 2000/10/28 07:39:53 konst Exp $ */

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "biewlib/kbd_code.h"
#include "biewlib/biewlib.h"
#include "console.h"

#ifdef	_SLANG_
#include <slang.h>
#endif

#ifdef	_CURSES_
#include <curses.h>
MEVENT me;
#endif

#ifdef	_VT100_

#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <sys/time.h>

#ifndef STDIN_FILENO
#define STDIN_FILENO 0
#endif

static int in_fd;
static struct termios sattr;

typedef struct {
    char c;
    int key;
} seqtbl;

typedef seqtbl pseq[];
typedef seqtbl p1seq[1]; /**< dummy type to make compiler happy */

typedef struct {
    char pre1;
    char pre2;
    char suf;
    p1seq *s;
} eseq;

#define SEQ_LEN 10	/**< max sequense length */
#define SEQ_NUM 7	/**< number of sequence categories */

/**
    translatable sequences
*/

const static pseq seq0 = {
    {'A',KE_UPARROW},
    {'B',KE_DOWNARROW},
    {'C',KE_RIGHTARROW},
    {'D',KE_LEFTARROW},
    {'P',KE_F(1)},
    {'Q',KE_F(2)},
    {'R',KE_F(3)},
    {'w',KE_F(3)},
    {'y',KE_F(3)},
    {'S',KE_F(4)},
    {'x',KE_F(4)},
    {'t',KE_F(5)},
    {'v',KE_F(5)},
    {'u',KE_F(6)},
    {'l',KE_F(6)},
    {'q',KE_F(7)},
    {'s',KE_F(7)},
    {'r',KE_F(8)},
    {'p',KE_F(9)},
    {'n',KE_F(9)},
    {0,0}},
seq1 = {
    {'A',KE_UPARROW},
    {'B',KE_DOWNARROW},
    {'C',KE_RIGHTARROW},
    {'D',KE_LEFTARROW},
    {'H',KE_HOME},
    {'K',KE_END},
    {0,0}},
seq2 = {
    {'A',KE_F(1)},
    {'B',KE_F(2)},
    {'C',KE_F(3)},
    {'D',KE_F(4)},
    {'E',KE_F(5)},
    {0,0}},
seq3 = {
    {'1',KE_HOME},
    {'2',KE_INS},
    {'3',KE_DEL},
    {'4',KE_END},
    {'5',KE_PGUP},
    {'6',KE_PGDN},
    {'7',KE_HOME},
    {'8',KE_END},
    {'J',KE_CTL_PGDN},
    {0,0}},
seq4 = {
    {'1',KE_F(1)},
    {'2',KE_F(2)},
    {'3',KE_F(3)},
    {'4',KE_F(4)},
    {'5',KE_F(5)},
    {'7',KE_F(6)},
    {'8',KE_F(7)},
    {'9',KE_F(8)},
    {0,0}},
seq5 = {
    {'0',KE_F(9)},
    {'1',KE_F(10)},
    {'3',KE_SHIFT_F(1)},
    {'4',KE_SHIFT_F(2)},
    {'5',KE_SHIFT_F(3)},
    {'6',KE_SHIFT_F(4)},
    {'8',KE_SHIFT_F(5)},
    {'9',KE_SHIFT_F(6)},
    {0,0}},
seq6 = {
    {'1',KE_SHIFT_F(7)},
    {'2',KE_SHIFT_F(8)},
    {'3',KE_SHIFT_F(9)},
    {'4',KE_SHIFT_F(10)},
    {0,0}};

static eseq S[SEQ_NUM] = {
{'O', 0, 0, (p1seq *)seq0 },
{'[', 0, 0, (p1seq *)seq1 },
{'[', '[', 0, (p1seq *)seq2 },
{'[', 0, '~', (p1seq *)seq3 },
{'[', '1', '~', (p1seq *)seq4 },
{'[', '2', '~', (p1seq *)seq5 },
{'[', '3', '~',	(p1seq *)seq6 }};

mevent mouse = {0, 0, 0, 0};

#ifdef HAVE_SELECT

#ifdef	__BEOS__
#include <net/socket.h> /* select is here on BeOS */
#endif

static int __FASTCALL__ is_key_ready()
{
    struct timeval t = { 0, 0 };
    fd_set readfds;

    FD_ZERO(&readfds);
    FD_SET(in_fd, &readfds);
    return select(in_fd + 1, &readfds, NULL, NULL, &t);
}
#endif

#ifdef USE_GPM

#include <gpm.h>

static int gpmhandle;
static Gpm_Event ge;

static int __FASTCALL__ _get_ge(Gpm_Event *x)
{
    struct timeval t = { 0, 1000 };
    fd_set readfds;
    int i;

    FD_ZERO(&readfds);
    FD_SET(gpmhandle, &readfds);

    i = select(gpmhandle + 1, &readfds, NULL, NULL, &t);

    if (i) Gpm_GetEvent(x);
#define event x
    printm("mouse: %s, at %2i,%2i (delta %2i,%2i), butt %i, mod %02X\r\n",
	   event->type&GPM_DOWN ? "press  "
	   : (event->type&GPM_UP ? "release" : "motion "),
	   event->x, event->y,
	   event->dx, event->dy,
	   event->buttons, event->modifiers);
#undef event
    return i;
}
#endif


#endif

/*
    keyboard FIFO
*/

#define KBUFSIZE 64

static struct {
    int pool[KBUFSIZE];
    int current;
} keybuf;

static int	shift_status = 0;	/**< status of shift keys */
static tBool	mouse_status = True;	/**< mouse state */

static void __FASTCALL__ pushEvent(int event)
{
    if (event && keybuf.current < KBUFSIZE) {
#ifdef	_SLANG_
	if (event > 0x100 && event <0x1000) event = _2B(event);
#endif
#ifdef	_CURSES_
	if (event >= KEY_MIN && event <= KEY_MAX) event = _2B(event);
#endif
	if (keybuf.current) memmove(keybuf.pool, &keybuf.pool[1], keybuf.current);
	keybuf.pool[0] = event;
	keybuf.current++;
    }
}

/**
    ReadNextEvent is non-blocking call
*/

void __FASTCALL__ ReadNextEvent(void)
{
    int key = 0;

#define ret(x)	pushEvent(x); return;
#define set_s(x) shift_status &= (x); shift_status ^= (x); ret(KE_SHIFTKEYS);

#ifdef	_SLANG_
    if (!SLang_input_pending(0)) return;

    /* single escape */
    key = SLang_getkey();
    if (SLang_input_pending(0)) {
	SLang_ungetkey(key);
	key = SLkp_getkey();
    }

    switch(key) {
	case SL_KEY_ERR		: return;
	case KE_STATUS_RESET	: set_s(0);
	case KE_STATUS_ALT	: set_s(KS_ALT);
	case KE_STATUS_SHIFT	: set_s(KS_SHIFT);
	case KE_STATUS_CONTROL	: set_s(KS_CTRL);
	case KE_ENTER2		: ret(KE_ENTER);
	case KE_BKSPACE1	: ret(KE_BKSPACE);
	case KE_BKSPACE2	: ret(KE_BKSPACE);
	case KE_C_O		: ret(KE_CTL_(O));
    }
    goto place_key;
#endif

#ifdef	_CURSES_
    key = getch(); switch(key) {
	case ERR		: return;
	case KE_STATUS_RESET	: set_s(0);
	case KE_STATUS_ALT	: set_s(KS_ALT);
	case KE_STATUS_SHIFT	: set_s(KS_SHIFT);
	case KE_STATUS_CONTROL	: set_s(KS_CTRL);
	case KE_ENTER2		: ret(KE_ENTER);
	case KE_BKSPACE1	: ret(KE_BKSPACE);
	case KE_BKSPACE2	: ret(KE_BKSPACE);
	case KE_C_O		: ret(KE_CTL_(O));
#ifdef HAVE_MOUSE
        case KEY_MOUSE		: getmouse(&me); ret(KE_MOUSE);
#endif
    }
    goto place_key;
#endif

#ifdef	_VT100_
#define get(x) read(in_fd,&(x),1)

    int i;
    char c[SEQ_LEN];

#ifdef HAVE_SELECT
    if (!is_key_ready()) return;
#endif

    memset(c, 0, SEQ_LEN);
    get(c[0]); switch(c[0]) {
	case 0			: return; /* no key is ready */
	case KE_ESCAPE		: break;
	case KE_STATUS_RESET	: set_s(0);
	case KE_STATUS_ALT	: set_s(KS_ALT);
	case KE_STATUS_SHIFT	: set_s(KS_SHIFT);
	case KE_STATUS_CONTROL	: set_s(KS_CTRL);
	case KE_ENTER2		: ret(KE_ENTER);
	case KE_BKSPACE2	: ret(KE_BKSPACE);
	case KE_C_O		: ret(KE_CTL_(O));
	default			: key = c[0];
    }
    if (key) goto place_key;
    for (i = 1; i < SEQ_LEN - 1; i++) {
#ifdef HAVE_SELECT
	if(is_key_ready()) get(c[i]);
	else break;
#else
	get(c[i]); if(!c[i]) break;
#endif
    }
    if (i < 3) {
	key = c[0];
	goto place_key;
    }

/*
    track mouse
*/

    if (c[1] == '[' && c[2] == 'M' && i == 6) {
	mouse.pressed = 1;
	mouse.buttons = c[3] - ' ';
	mouse.x = c[4] - '!';
	mouse.y = c[5] - '!';
	key = KE_MOUSE;
	goto place_key;
    }

/*
    translate escape sequence
*/

#define S S[i]
    for (i = 0; i < SEQ_NUM && !key; i++) {
	int j, n;

	if (c[1] != S.pre1) continue;
	n = 2;
	if (S.pre2) {
	    if (c[n] != S.pre2) continue;
	    n++;
	}
	for (j = 0; S.s[j]->c; j++) if (S.s[j]->c == c[n]) {
	    if (!(S.suf && S.suf != c[n + 1])) {
		key = S.s[j]->key; break;
	    }
	}
    }
#undef S
#endif

place_key:
    if (key) ret(key);

#undef set_s
#undef ret
}

inline int __FASTCALL__ __kbdGetShiftsKey(void)
{
    return shift_status;
}

int __FASTCALL__ __kbdTestKey(unsigned long flg)
{
    if(__MsGetBtns() && flg == KBD_NONSTOP_ON_MOUSE_PRESS) return KE_MOUSE;
    ReadNextEvent();
    return keybuf.current;
}

int __FASTCALL__ __kbdGetKey(unsigned long flg)
{
    int key = 0, s = 0;

    if (__MsGetBtns() && flg == KBD_NONSTOP_ON_MOUSE_PRESS) return KE_MOUSE;

    while (!keybuf.current) { __OsYield(); ReadNextEvent(); }
    key = keybuf.pool[--keybuf.current];
    if (!(key == KE_MOUSE || key == KE_SHIFTKEYS)) {
	if ((shift_status & KS_ALT) == KS_ALT)		s |= ADD_ALT;
	if ((shift_status & KS_SHIFT) == KS_SHIFT)	s |= ADD_SHIFT;
	if ((shift_status & KS_CTRL) == KS_CTRL) {
	    s |= ADD_CONTROL;
	    if (key == 'o' || key == 'O') key = KE_CTL_(O);    /* CTRL+O */
	}
	shift_status = 0;
    }
    return key | s;
}

/*

*/

tBool __FASTCALL__ __MsGetState(void)
{
    return mouse_status;
}

void __FASTCALL__ __MsSetState(tBool ms_visible)
{
    mouse_status = ms_visible;
}

void __FASTCALL__ __MsGetPos(tAbsCoord *x, tAbsCoord *y)
{
    ReadNextEvent();
#ifdef	_CURSES_
    *x = me.x;
    *y = me.y;
#endif
#ifdef	_VT100_
    *x = mouse.x;
    *y = mouse.y;
#endif
}

int __FASTCALL__ __MsGetBtns(void)
{
    int ret = 0;
    ReadNextEvent();
#ifdef	_CURSES_
    if ((me.bstate & BUTTON1_PRESSED) || (me.bstate & BUTTON1_CLICKED))
	ret |= MS_LEFTPRESS;
    if (me.bstate & BUTTON1_RELEASED)
	ret &= ~MS_LEFTPRESS;
    if((me.bstate & BUTTON2_PRESSED) || (me.bstate & BUTTON2_CLICKED))
	ret |= MS_MIDDLEPRESS;
    if(me.bstate & BUTTON2_RELEASED)
	ret &= ~MS_MIDDLEPRESS;
    if((me.bstate & BUTTON3_PRESSED) || (me.bstate & BUTTON3_CLICKED))
	ret |= MS_RIGHTPRESS;
    if(me.bstate & BUTTON3_RELEASED)
	ret &= ~MS_RIGHTPRESS;
#endif
#ifdef	_VT100_
#ifdef USE_GPM
    if (gpmhandle & get_ge(&ge) & (ge.type & GPM_DOWN)) {
	if (ge.buttons & GPM_B_LEFT)	ret |= MS_LEFTPRESS;
	if (ge.buttons & GPM_B_MIDDLE)	ret |= MS_MIDDLEPRESS;
	if (ge.buttons & GPM_B_RIGHT)	ret |= MS_RIGHTPRESS;
    }
/*    if (ret) printm("x: %d, y: %d, ret: %d\n", ge.x, ge.y, ret); */
#else
    if (mouse.pressed) switch (mouse.buttons) {
	case 0:	ret |= MS_LEFTPRESS; break;
	case 1:	ret |= MS_MIDDLEPRESS; break;
	case 2:	ret |= MS_RIGHTPRESS; break;
	mouse.pressed = 0;
    }
#endif
#endif
    return ret;
}


void __FASTCALL__ __init_keyboard(void)
{
#ifdef	_CURSES_
    raw();
    noecho();
    intrflush(stdscr,FALSE);
    meta(stdscr,TRUE);
    keypad(stdscr, TRUE);
    nodelay(stdscr,TRUE);
/*    notimeout(stdscr,TRUE); */
#ifdef HAVE_MOUSE
    mousemask(REPORT_MOUSE_POSITION | ALL_MOUSE_EVENTS, NULL);
#endif
#endif
#ifdef	_VT100_
    struct termios tattr;

    in_fd = open(ttyname(STDIN_FILENO), O_RDONLY);
    if (in_fd < 0) in_fd = STDIN_FILENO;

    tcgetattr(in_fd, &tattr);
    sattr = tattr;
    tattr.c_lflag &= ~(ICANON | ECHO | ISIG);
    tattr.c_cc[VMIN] = 1;
    tattr.c_cc[VTIME] = 0;
    tattr.c_iflag &= ~(INPCK | ISTRIP | IXON);
    tattr.c_oflag |= OPOST | ONLCR;
    tcsetattr(in_fd, TCSANOW, &tattr);
#ifndef HAVE_SELECT
    if (fcntl(in_fd, F_SETFL, fcntl(in_fd, F_GETFL) | O_NONBLOCK) < 0) {
	printm("Can't set nonblocking mode: %s\nExiting..\n", strerror(errno));
	exit(errno);
    }
#endif
#ifdef USE_GPM
	Gpm_Connect gc = { ~0, GPM_MOVE|GPM_HARD, 0, 0};
	gpmhandle = Gpm_Open(&gc, 0);
	if (gpmhandle < 0) gpmhandle = 0;
#endif
#endif
}

void __FASTCALL__ __term_keyboard(void)
{
#ifdef	_VT100_
    tcsetattr(in_fd, TCSANOW, &sattr);
    close(in_fd);
#ifdef USE_GPM
    if (gpmhandle) Gpm_Close();
#endif
#endif
}
