/*
 * part of bidwatcher
 * Author: Jan Starzynski
 * use of this code is restricted to the terms
 * of the GNU GPL, which should have been included in this
 * distribution. If not, see www.gnu.org/copyleft/gpl.html.
 * Here is the short version:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * $Id: bidgroup.cpp,v 1.1.2.5 2004/10/25 12:11:29 sturnus Exp $
 *
 * Handling of Bidgroups
 */

#include "config.h"

#ifdef HAVE_STL

#include <algorithm>
#include <functional>

#include <gdk/gdk.h>
#include <gdk/gdkprivate.h>
#include <gtk/gtk.h>

#include "bidwatcher.h"
#include "bidgroup.h"
#include "portability.h"

using namespace std;

struct FindItem: public unary_function<BidGroup::ItemList, bool> {
  BidGroup::IdType id;
  FindItem(BidGroup::IdType id): id(id) {}
  bool operator()(const BidGroup::ItemList &list) const { return list.find(id) != list.end(); }
};

struct UselessList: public unary_function<BidGroup::ItemList, bool> {
  bool operator()(const BidGroup::ItemList &list) const { return list.size() <= 1; }
};

void BidGroup::clear()
{
  m_list.clear();
}

int BidGroup::numGroup() const
{
  return m_list.size();
}

BidGroup::ItemList *BidGroup::addItem(const IdType &id1, const IdType &id2)
{
  ItemList *item;

  item = findItem(id1);
  if(item) {
    item->insert(id2);
    return item;
  }

  item = findItem(id2);
  if(item) {
    item->insert(id1);
    return item;
  }

  if(id1 == id2) return 0;

  ItemList add;
  add.insert(id1);
  add.insert(id2);
  GroupList::iterator insert = find_if(m_list.begin(), m_list.end(), UselessList());
  if(insert == m_list.end()) {
    m_list.insert(insert, add);
  } else {
    *insert = add;
  }

  return &m_list.back();
}

int BidGroup::groupId(const IdType &id) const
{
  GroupList::const_iterator it = find_if(m_list.begin(), m_list.end(), FindItem(id));
  return it == m_list.end() ? -1 : it - m_list.begin();
}

BidGroup::ItemList *BidGroup::findItemP(const IdType &id) const
{
  return getGroupP(groupId(id));
}

BidGroup::ItemList *BidGroup::getGroupP(int gid) const
{
  return const_cast<BidGroup::ItemList*>(gid < 0 || gid >= numGroup() ? 0 : &m_list[gid]);
}

bool BidGroup::delItem(const IdType &id)
{
  bool ok = false;
  for(GroupList::iterator it = m_list.begin(); it != m_list.end(); ++it) {
    ok |= it->erase(id);
  }
  return ok;
}

vector<BidGroup::IdType> &BidGroup::findItem(vector<IdType> &vec, const IdType &id)
{
  ItemList *item = findItem(id);
  if(id) vec.insert(vec.end(), item->begin(), item->end());
  return vec;
}

bool BidGroup::read(const char *file)
{
  FILE *fp = fopen(file, "r");
  bool ret = read(fp);
  if(fp) fclose(fp);
  return ret;
}

bool BidGroup::write(const char *file) const
{
  FopenW fp(file);
  return write(fp);
}

bool BidGroup::read(FILE *fp)
{
  if(!fp) return false;

  bool ok = true;
  char buffer[30];

  ItemList group;
  while(fgets(buffer, sizeof(buffer), fp)) {
    IdType id;
    if(sscanf(buffer, LLU_FMT, &id) == 1) {
      group.insert(id);
    } else {
      if(!group.empty()) {
        m_list.push_back(group);
        group.clear();
      }
    }
  }

  if(!group.empty()) m_list.push_back(group);

  return ok;
}

bool BidGroup::write(FILE *fp) const
{
  if(!fp) return false;

  bool ok = true;
  for(GroupList::const_iterator it = m_list.begin(); it != m_list.end(); ++it) {
    if(!it->empty()) {
      for(ItemList::const_iterator jt = it->begin(); jt != it->end(); ++jt) {
        fprintf(fp, LLU_FMT"\n", *jt);
      }
      fprintf(fp, "\n");
    }
  }
  return ok;
}

void BidGroup::sweepList(bool at_end_only)
{
  if(at_end_only) {
    m_list.erase(find_if(m_list.rbegin(), m_list.rend(), not1<UselessList>(UselessList())).base(), m_list.end());
  } else {
    m_list.erase(remove_if(m_list.begin(), m_list.end(), UselessList()), m_list.end());
  }
}

#endif

