/* changelog.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/patch-logs.h"
#include "libarch/changelogs.h"
#include "commands/cmd.h"
#include "commands/changelog.h"
#include "commands/version.h"


/* __STDC__ prototypes for static functions */



static t_uchar * usage = N_("[options] [[archive]/version]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("cd to DIR first")) \
  OP (opt_no_files, 0, "no-files", 0, \
      N_("exclude file lists from ChangeLog")) \
  OP (opt_untagged, 0, "untagged", 0, \
      N_("don't implicitly tag the output file")) \
  OP (opt_new_entry, 0, "new-entry PATCH,FILE", 1, \
      N_("make FILE the first (top) entry\n" \
      "  for patch level PATCH"))

t_uchar arch_cmd_changelog_help[] = N_("generate a ChangeLog from a patch log\n"
                                     "Generate a ChangeLog for VERSION from the patch log for DIR.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_changelog (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  t_uchar * dir = 0;
  t_uchar * new_entry_arg = 0;
  int no_files = 0;
  int untagged = 0;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_changelog_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_dir:
          {
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_no_files:
          {
            no_files = 1;
            break;
          }

        case opt_untagged:
          {
            untagged = 1;
            break;
          }

        case opt_new_entry:
          {
            new_entry_arg = str_save (0, option->arg_string);
            break;
          }
        }
    }


  if (argc > 2)
    goto usage_error;

  {
    t_uchar * tree_root = 0;
    t_uchar * version_spec;
    t_uchar * new_entry_patch_lvl = 0;
    t_uchar * new_entry_file = 0;
    t_uchar * archive = 0;
    t_uchar * version = 0;

    tree_root = arch_tree_root (0, dir, 0);

    if (!tree_root)
      {
        safe_printfmt (2, "%s: not in project tree (%s)\n", argv[0], dir);
        exit (1);
      }

    if (argc == 2)
      {
        version_spec = argv[1];
      }
    else
      {
        version_spec = arch_tree_version (tree_root);

        if (!version_spec)
          {
            safe_printfmt (2, "%s: no default tree version\n  tree: %s\n",
                           argv[0], tree_root);
            exit (1);
          }
      }

    if (default_archive)
      {
        if (!arch_valid_archive_name (default_archive))
          {
            safe_printfmt (2, "%s: invalid archive name (%s)\n",
                           argv[0], default_archive);
            exit (1);
          }
      }

    if (new_entry_arg)
      {
        t_uchar * comma;

        comma = str_chr_index (new_entry_arg, ',');

        if (!comma)
          {
            safe_printfmt (2, "ill-formed argument for --new-entry (`%s')\n", new_entry_arg);
            exit (1);
          }

        new_entry_patch_lvl = str_save_n (0, new_entry_arg, comma - new_entry_arg);
        new_entry_file = str_save (0, comma + 1);

        if (!arch_valid_patch_level_name (new_entry_patch_lvl))
          {
            safe_printfmt (2, "bogus patch-level in argument for --new-entry (`%s')\n", new_entry_patch_lvl);
            exit (1);
          }
      }

    archive = arch_parse_package_name (arch_ret_archive, default_archive, version_spec);
    version = arch_parse_package_name (arch_ret_non_archive, 0, version_spec);

    arch_generate_changelog (1, tree_root, no_files, untagged, new_entry_patch_lvl, new_entry_file, archive, version);

    lim_free (0, tree_root);
    lim_free (0, new_entry_patch_lvl);
    lim_free (0, new_entry_file);
    lim_free (0, archive);
    lim_free (0, version);
  }

  lim_free (0, default_archive);
  lim_free (0, dir);
  lim_free (0, new_entry_arg);

  return 0;
}





/* tag: Tom Lord Tue May 13 14:25:54 2003 (changelog.c)
 */
