/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * ragrep - egrep expression from captured user data.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#if defined(HAVE_MMAP)
# include <sys/mman.h>
#endif
#if defined(HAVE_SETRLIMIT)
# include <sys/time.h>
# include <sys/resource.h>
#endif
#include <stdio.h>
#include "system.h"

/*
#if !defined(__FreeBSD__)
#include "getopt.h"
#endif
*/

#include "getpagesize.h"
#include "ragrep.h"

#undef MAX
#define MAX(A,B) ((A) > (B) ? (A) : (B))


struct stats
{
  struct stats *parent;
  struct stat stat;
};

/* Short options.  */
static char const short_options[] =
"0123456789A:B:C::EFGHIUVX:abcd:e:f:hiLlnqrsuvwxyZz";

/* Non-boolean long options that have no corresponding short equivalents.  */
enum
{
  BINARY_FILES_OPTION = CHAR_MAX + 1
};

/* Long options equivalences. */
/*
static struct option long_options[] =
{
  {"after-context", required_argument, NULL, 'A'},
  {"basic-regexp", no_argument, NULL, 'G'},
  {"before-context", required_argument, NULL, 'B'},
  {"binary-files", required_argument, NULL, BINARY_FILES_OPTION},
  {"byte-offset", no_argument, NULL, 'b'},
  {"context", optional_argument, NULL, 'C'},
  {"count", no_argument, NULL, 'c'},
  {"directories", required_argument, NULL, 'd'},
  {"extended-regexp", no_argument, NULL, 'E'},
  {"file", required_argument, NULL, 'f'},
  {"files-with-matches", no_argument, NULL, 'l'},
  {"files-without-match", no_argument, NULL, 'L'},
  {"fixed-regexp", no_argument, NULL, 'F'},
  {"fixed-strings", no_argument, NULL, 'F'},
  {"help", no_argument, &show_help, 1},
  {"ignore-case", no_argument, NULL, 'i'},
  {"line-number", no_argument, NULL, 'n'},
  {"line-regexp", no_argument, NULL, 'x'},
  {"mmap", no_argument, &mmap_option, 1},
  {"no-filename", no_argument, NULL, 'h'},
  {"no-messages", no_argument, NULL, 's'},
  {"null", no_argument, NULL, 'Z'},
  {"null-data", no_argument, NULL, 'z'},
  {"quiet", no_argument, NULL, 'q'},
  {"recursive", no_argument, NULL, 'r'},
  {"regexp", required_argument, NULL, 'e'},
  {"invert-match", no_argument, NULL, 'v'},
  {"silent", no_argument, NULL, 'q'},
  {"text", no_argument, NULL, 'a'},
  {"binary", no_argument, NULL, 'U'},
  {"unix-byte-offsets", no_argument, NULL, 'u'},
  {"version", no_argument, NULL, 'V'},
  {"with-filename", no_argument, NULL, 'H'},
  {"word-regexp", no_argument, NULL, 'w'},
  {0, 0, 0, 0}
};
*/

/* Define flags declared in grep.h. */
int match_icase;
int match_words;
int match_lines;
unsigned char eolbyte;

/* For error messages. */
static char *prog;
static int errseen;

/* How to handle directories.  */
/*
static enum
  {
    READ_DIRECTORIES,
    RECURSE_DIRECTORIES,
    SKIP_DIRECTORIES
  } directories;
*/

static int  install_matcher PARAMS ((char const *));
static int  grepbuf PARAMS ((char *, char *));

/* Functions we'll use to search. */
static void (*compile) PARAMS ((char *, size_t));
static char *(*execute) PARAMS ((char *, size_t, char **));

/* Flags controlling the style of output. */


static int filename_mask = 0;   /* If zero, output nulls after filenames.  */
static int out_invert = 0;   /* Print nonmatching stuff. */
static int out_before = 0;   /* Lines of leading context. */
static int out_after = 0;   /* Lines of trailing context. */

/* Internal variables to keep track of byte count, context, etc. */
static int done_on_match;      /* Stop scanning file on first match */


int RaInitialized = 0;
struct ArgusRecordStore *RaGblStore = NULL;

void
ArgusClientInit ()
{
   char *keys;
   size_t keycc = 0;
   int with_filenames;
   int default_context;
   unsigned digit_args_val;

   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaInitialized++;

      if (Hflag)
         RaHistoTimeSeries = ARGUS_TIMESERIES;

      if (nflag > 0)
         hfield = 15;

      if (nflag > 1)
         pfield =  5;

      if (!(estr))
         usage ();

      if (idflag > 0) {
         match_icase = 1;
         idflag = 0;
      } else
         match_icase = 0;

      if ((keys = (char *) ArgusCalloc (1, strlen(estr) + 2)) != NULL) {
         bcopy (estr, keys, strlen(estr));
         keycc = strlen(keys);
      } else
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusCalloc failed %s\n", strerror(errno));

      with_filenames = 0;
      eolbyte = '\n';
      filename_mask = ~0;

      SET_BINARY (desc);

      if (!install_matcher(matcher))
         abort ();

      (*compile)(keys, keycc);
  
/* The value -1 means to use DEFAULT_CONTEXT. */
      out_after = out_before = -1;

/* Default before/after context: chaged by -C/-NUM options */
      default_context = 0;

/* Accumulated value of individual digits in a -NUM option */
      digit_args_val = 0;
  
/* Internationalization. */
#if HAVE_SETLOCALE
      setlocale (LC_ALL, "");
#endif
#if ENABLE_NLS
      bindtextdomain (PACKAGE, LOCALEDIR);
      textdomain (PACKAGE);
#endif
   }
}


int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;
   }
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   fprintf (stderr, "Ragrep Version %d.%d\n", major_version, minor_version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s -e PATTERN [ra-options] [- filter-expression]\n", ArgusProgramName);

   fprintf (stderr, "options:    -e <PATTERN>   match captured user data against regular expression.\n\n");

   fprintf (stderr, "ra-options: -a             print record summaries on termination.\n");
   fprintf (stderr, "            -A             print application bytes.\n");
   fprintf (stderr, "            -b             dump packet-matching code.\n");
   fprintf (stderr, "            -c             print packet and byte counts.\n");
   fprintf (stderr, "            -C             treat the remote source as a Cisco Netflow source.\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "            -D <level>     specify debug level\n");
#endif
   fprintf (stderr, "            -E <file>      write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "            -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "            -g             print record time duration.\n");
   fprintf (stderr, "            -G             print both start and last time values.\n");
   fprintf (stderr, "            -h             print help.\n");
   fprintf (stderr, "            -i             print source probe id.\n");
   fprintf (stderr, "            -I             print transaction state and option indicators.\n");
   fprintf (stderr, "            -l             print last time values [default is start time].\n");
   fprintf (stderr, "            -m             print MAC addresses.\n");
   fprintf (stderr, "            -n             don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -q             quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R             print out response data when availabile.\n");
   fprintf (stderr, "            -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                [yyyy/]mm/dd\n");
   fprintf (stderr, "            -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>      write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z             print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


void
RaProcessRecord (struct ArgusRecord *argus)
{
   int len, retn;

   if (!(RaGblStore)) {
      if ((RaGblStore = RaNewArgusStore(argus)) != NULL) {
         if ((RaGblStore->data[0] = RaNewArgusData(argus)) == NULL)
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

   RaGblStore->data[0]->argus = argus;

   if (ArgusThisFarStatus & (ARGUS_SRCUSRDATA_DSR_STATUS | ARGUS_SRCUSRDATA_DSR_STATUS)) {
      if (ArgusThisFarStatus & ARGUS_SRCUSRDATA_DSR_STATUS) {
         struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_SRCUSRDATA_DSR_INDEX];
         char *buf = &user->data;

         if (ArgusGrepSource) {
            len = (user->length - 1) * 4;
            len = (len > argus->argus_far.src.appbytes) ? argus->argus_far.src.appbytes : len;

            if ((retn = grepbuf (buf, &buf[len]))) {
               RaGblStore->data[0]->status |= RA_MODIFIED;
               RaSendArgusRecord(RaGblStore);
               return;
            }
         }
      }
         
      if (ArgusThisFarStatus & ARGUS_DSTUSRDATA_DSR_STATUS) {
         struct ArgusUserStruct *user = (struct ArgusUserStruct *) ArgusThisFarHdrs[ARGUS_DSTUSRDATA_DSR_INDEX];
         char *buf = &user->data;

         if (ArgusGrepDestination) {
            len = (user->length - 1) * 4;
            len = (len > argus->argus_far.dst.appbytes) ? argus->argus_far.dst.appbytes : len;

            if ((retn = grepbuf (buf, &buf[len]))) {
               RaGblStore->data[0]->status |= RA_MODIFIED;
               RaSendArgusRecord(RaGblStore);
               return;
            }
         }
      }
   }
}


#include <math.h>

int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   struct ArgusAGRStruct *agr = NULL;
   int retn = 0, i;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.meanval = data->act.sumtime/data->act.n;
               data->agr.act.stdev = (unsigned int) sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = (unsigned int) sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);
            }

            bcopy ((char *) argus, buf, argus->ahdr.length);
            argus = (struct ArgusRecord *) buf;

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) == NULL) {
               if (ArgusFlowModelFile && (data->agr.count > 1)) {
                  bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                  argus->ahdr.length += data->agr.length;
                  argus->ahdr.status |= ARGUS_MERGED;
                  ArgusFree (data->argus);
                  data->argus = RaCopyArgusRecord(argus);
                  ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);

               } else {
                  argus->ahdr.status &= ~(ARGUS_MERGED);
               }
            }
      
            if ((ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusWfileList)))) {
               struct ArgusWfileStruct *wfile = NULL, *start = NULL;

               if ((wfile = ArgusFrontList(ArgusWfileList)) != NULL) {
                  start = wfile;
                  do {
                     if ((exceptfile == NULL) || strcmp(wfile->filename, exceptfile)) {
#ifdef _LITTLE_ENDIAN
                        ArgusHtoN(argus);
#endif
                        ArgusWriteNewLogfile (wfile, argus);

#ifdef _LITTLE_ENDIAN
                        ArgusNtoH(argus);
#endif
                     }
                     ArgusPopFrontList(ArgusWfileList);
                     ArgusPushBackList(ArgusWfileList, wfile);
                     wfile = ArgusFrontList(ArgusWfileList);

                  } while (wfile != start);
               }

            } else {
      
               if (argus->ahdr.type & ARGUS_MAR)
                  printf ("%s\n", get_man_string (argus));
                  
               else {
                  ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                  switch (argus->ahdr.status & 0xFFFF) {
                     case ETHERTYPE_IP:
                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP:              
                              printf ("%s", get_tcp_string (argus));
                              break;
            
                           case IPPROTO_ICMP:              
                              printf ("%s", get_icmp_string (argus));
                              break;
            
                           default:
                              printf ("%s", get_ip_string (argus));
                              break;
                        }
                        break;
            
                     case ETHERTYPE_ARP:
                     case ETHERTYPE_REVARP:
                        printf ("%s", get_arp_string (argus));
                        break;
            
                     default:
                        printf ("%s", get_nonip_string (argus));
                        break;
                  }

                  if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
                     printf ("%s\n", RaGetUserDataString(argus));
                  else
                     printf ("\n");
               }
               fflush (stdout);
            }
      
            argus = data->argus;

            if (argus->ahdr.type & ARGUS_FAR) {
               int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);
               farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
      
               while (length > 0) {
                  switch (farhdr->type) {
                     case ARGUS_FAR: {
                        struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                        far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                        far->time.last.tv_sec = 0; far->time.last.tv_usec = 0;
                        far->src.count = 0; far->src.bytes = 0;
                        far->dst.count = 0; far->dst.bytes = 0;
                        break;
                     }
      
                     case ARGUS_TCP_DSR: {
                        struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
      
                        tcp->state = 0;
                        tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                        tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                        tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                        tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
      
                        break;
                     }
      
                     case ARGUS_TIME_DSR: {
                        struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) farhdr;

                        time->src.act.n = 0;
                        time->src.act.meanval = 0;
                        time->src.act.stdev = 0;
                        time->src.act.maxval = 0;
                        time->src.act.minval = 0x7FFFFFFF;
                        time->src.idle.n = 0;
                        time->src.idle.meanval = 0;
                        time->src.idle.stdev = 0;
                        time->src.idle.maxval = 0;
                        time->src.idle.minval = 0x7FFFFFFF;

                        time->dst.act.n = 0;
                        time->dst.act.meanval = 0;
                        time->dst.act.stdev = 0;
                        time->dst.act.maxval = 0;
                        time->dst.act.minval = 0x7FFFFFFF;
                        time->dst.idle.n = 0;
                        time->dst.idle.meanval = 0;
                        time->dst.idle.stdev = 0;
                        time->dst.idle.maxval = 0;
                        time->dst.idle.minval = 0x7FFFFFFF;

                        break;
                     }

                     case ARGUS_VLAN_DSR: {
                        struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
                        vlan->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_MPLS_DSR: {
                        struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
                        mpls->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_AGR_DSR: {
                        struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
      
                        agr->count = 0;
                        agr->act.n = 0;
                        agr->act.minval = 0x7FFFFFFF;   agr->act.meanval = 0;
                        agr->act.stdev = 0;  agr->act.maxval = 0;
                        agr->idle.n = 0;
                        agr->idle.minval = 0x7FFFFFFF; agr->idle.meanval = 0;
                        agr->idle.stdev = 0; agr->idle.maxval = 0;
                        break;
                     }
                  }

                  if ((farlen = farhdr->length) == 0)
                     break;
                  if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                      (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                     farlen = farlen * 4;
                  length -= farlen;
                  farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
               }
            }

            data->agr.count = 0;
            data->agr.act.n = 0;
            data->agr.act.meanval = 0;
            data->agr.act.stdev = 0;
            data->agr.act.maxval = 0;
            data->agr.act.minval = 0x7FFFFFFF;
            data->agr.idle.n = 0;
            data->agr.idle.meanval = 0;
            data->agr.idle.stdev = 0;
            data->agr.idle.maxval = 0;
            data->agr.idle.minval = 0x7FFFFFFF;
      
            data->act.n = 0;
            data->act.sumtime = 0;
            data->act.sumsqrd = 0;
            data->idle.n = 0;
            data->idle.sumtime = 0;
            data->idle.sumtime = 0;
      
            data->argus->ahdr.status &= ~ARGUS_MERGED;
            data->status &= ~RA_MODIFIED;
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif

   return (retn);
}



/* ragrep.c - main driver file for ragrep.
 * Copyright 2000, QoSient, LLC.
 * Completely derived from grep-2.4.2.
 */

/* grep.c - main driver file for grep.
   Copyright 1992, 1997-1999, 2000 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

/* Written July 1992 by Mike Haertel.  */

static void
error (const char *mesg, int errnum)
{
   if (errnum)
      fprintf (stderr, "%s: %s: %s\n", prog, mesg, strerror (errnum));
   else
      fprintf (stderr, "%s: %s\n", prog, mesg);
   errseen = 1;
}

/* Like error (), but die horribly after printing. */

void
fatal (const char *mesg, int errnum)
{
   error (mesg, errnum);
   exit (2);
}

/* Interface to handle errors and fix library lossage. */
char *
xmalloc (size_t size)
{
   char *result;

   result = malloc (size);
   if (size && !result)
      fatal (_("memory exhausted"), 0);

   return result;
}

char * xcalloc (size_t, size_t);

char *
xcalloc (size_t num, size_t size)
{
   char *result;

   result = calloc (num, size);
   if (size && !result)
      fatal (_("memory exhausted"), 0);

   return result;
}

/* Interface to handle errors and fix some library lossage. */
char *
xrealloc (char *ptr, size_t size)
{
  char *result;

   if (ptr)
      result = realloc (ptr, size);
   else
      result = malloc (size);

   if (size && !result)
      fatal (_("memory exhausted"), 0);

   return result;
}

/* Convert STR to a positive integer, storing the result in *OUT.
   If STR is not a valid integer, return -1 (otherwise 0). */

/*
static int
ck_atoi (char const *str, int *out)
{
   char const *p;
   for (p = str; *p; p++)
      if (*p < '0' || *p > '9')
         return -1;

   *out = atoi (optarg);

   return 0;
}
*/


/* Hairy buffering mechanism for grep.  The intent is to keep
   all reads aligned on a page boundary and multiples of the
   page size. */

#define PREFERRED_SAVE_FACTOR 5   /* Preferred value of bufalloc / bufsalloc.  */

#if defined(HAVE_MMAP)
static int bufmapped;      /* True if buffer is memory-mapped.  */
static off_t initial_bufoffset;   /* Initial value of bufoffset. */
#endif

/* Return VAL aligned to the next multiple of ALIGNMENT.  VAL can be
   an integer or a pointer.  Both args must be free of side effects.  */

#define ALIGN_TO(val, alignment) \
  ((size_t) (val) % (alignment) == 0 \
   ? (val) \
   : (val) + ((alignment) - (size_t) (val) % (alignment)))

/* Return the address of a page-aligned buffer of size SIZE,
   reallocating it from *UP.  Set *UP to the newly allocated (but
   possibly unaligned) buffer used to build the aligned buffer.  To
   free the buffer, free (*UP).  */

/*
static char *
page_alloc (size_t size, char **up)
{
   size_t asize = size + pagesize - 1;
   if (size <= asize) {
      char *p = *up ? xrealloc (*up, asize) : xmalloc (asize);

      if (p) {
         *up = p;
         return ALIGN_TO (p, pagesize);
      }
   }

   return NULL;
}
*/


#if O_BINARY
#include "dosbuf.c"
#endif

/*
static void
nlscan (char *lim)
{
   char *beg;
   for (beg = lastnl;  (beg = memchr (beg, eolbyte, lim - beg));  beg++)
      totalnl++;

   lastnl = lim;
}
*/


/* Scan the specified portion of the buffer, matching lines (or
   between matching lines if OUT_INVERT is true).  Return a count of
   lines printed. */

static int
grepbuf (char *beg, char *lim)
{
   int nlines;
   register char *p, *b;
   char *endp;
   char eol = eolbyte;

   nlines = 0;
   p = beg;
   while ((b = (*execute)(p, lim - p, &endp)) != 0) {
      /* Avoid matching the empty line at the end of the buffer. */
      if (b == lim && ((b > beg && b[-1] == eol) || b == beg))
         break;
      if (!out_invert) {
         nlines += 1;
         if (done_on_match)
            return nlines;

      } else
         if (p < b) {
            nlines += 1;
         }
      p = endp;
   }

   if (out_invert && p < lim)
      nlines += 1;

   if (vflag)
      nlines = (nlines > 0) ? 0 : 1;

   return nlines;
}


/* Set the matcher to M, reporting any conflicts.  */
/*
static void
setmatcher (char const *m)
{
   if (matcher && strcmp (matcher, m) != 0)
      fatal (_("conflicting matchers specified"), 0);

   matcher = m;
}
*/

/* Go through the matchers vector and look for the specified matcher.
   If we find it, install it in compile and execute, and return 1.  */

static int
install_matcher (char const *name)
{
   int i;
#ifdef HAVE_SETRLIMIT
   struct rlimit rlim;
#endif

   for (i = 0; matchers[i].name; ++i) {
      if (strcmp (name, matchers[i].name) == 0) {
         compile = matchers[i].compile;
         execute = matchers[i].execute;

#if HAVE_SETRLIMIT && defined(RLIMIT_STACK)
         /* I think every platform needs to do this, so that regex.c
            doesn't oveflow the stack.  The default value of
            `re_max_failures' is too large for some platforms: it needs
            more than 3MB-large stack.  The test for HAVE_SETRLIMIT
            should go into `configure'.  */

         if (!getrlimit (RLIMIT_STACK, &rlim)) {
            long newlim;
            extern long int re_max_failures; /* from regex.c */

            /* Approximate the amount regex.c needs, plus some more.  */

            newlim = re_max_failures * 2 * 20 * sizeof (char *);
            if (newlim > rlim.rlim_max) {
               newlim = rlim.rlim_max;
               re_max_failures = newlim / (2 * 20 * sizeof (char *));
            }

            if (rlim.rlim_cur < newlim)
               rlim.rlim_cur = newlim;

            setrlimit (RLIMIT_STACK, &rlim);
         }
#endif
         return 1;
      }
   }
   return 0;
}


/* Find the white-space-separated options specified by OPTIONS, and
   using BUF to store copies of these options, set ARGV[0], ARGV[1],
   etc. to the option copies.  Return the number N of options found.
   Do not set ARGV[N] to NULL.  If ARGV is NULL, do not store ARGV[0]
   etc.  Backslash can be used to escape whitespace (and backslashes).  */

/*
static int
prepend_args (char const *options, char *buf, char **argv)
{
   char const *o = options;
   char *b = buf;
   int n = 0;

   for (;;) {
      while (ISSPACE ((unsigned char) *o))
         o++;

      if (!*o)
         return n;

      if (argv)
         argv[n] = b;
      n++;

      do
         if ((*b++ = *o++) == '\\' && *o)
            b[-1] = *o++;
      while (*o && ! ISSPACE ((unsigned char) *o));

      *b++ = '\0';
   }
}
*/

/* Prepend the whitespace-separated options in OPTIONS to the argument
   vector of a main program with argument count *PARGC and argument
   vector *PARGV.  */

/*
static void
prepend_default_options (char const *options, int *pargc, char ***pargv)
{
   if (options) {
      char *buf = xmalloc (strlen (options) + 1);
      int prepended = prepend_args (options, buf, (char **) NULL);
      int argc = *pargc;
      char * const *argv = *pargv;
      char **pp = (char **) xmalloc ((prepended + argc + 1) * sizeof *pp);
      *pargc = prepended + argc;
      *pargv = pp;
      *pp++ = *argv++;

      pp += prepend_args (options, buf, pp);

      while ((*pp++ = *argv++))
         continue;
    }
}
*/
