
/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (C) 2000-2001 QoSient, LLC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * ratop - top() style application for argus data.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#define ARGUS_CURSES	1

#include <signal.h>
#include <ctype.h>

#ifdef ARGUS_CURSES
#include <curses.h>
#include <term.h>
#endif

#include <argus_client.h>
#include <ratop.h>

#include <math.h>

extern int ArgusSrcUserDataLen;
extern int ArgusDstUserDataLen;

struct ArgusServiceRecord ArgusThisSrv;

void RaProcessRecord (struct ArgusRecord *argus) { }
void RaProcessSrvRecord (struct ArgusServiceRecord *);


#ifdef ARGUS_CURSES
 
WINDOW *RaWindow = NULL;
WINDOW *RaHeaderWindow = NULL;
WINDOW *RaAvailableWindow = NULL;
WINDOW *RaFailedWindow = NULL;
WINDOW *RaBeaconWindow = NULL;
WINDOW *RaRecordWindow = NULL;

int RaLines = 0;
int RaCols = 0;
 
#define FAILEDWINSIZE    15
#define HEADERWINSIZE    5
 
int RaRecordWindowNum = 0;
int RaCursesInit      = 0;
int RaServerMode      = 0;
int RaWindowStatus    = 1;
  
extern struct timeval RaClientTimeout;
int RaInitCurses (void);

#include <sys/ioctl.h>

#define MAXFLOWMODELS		22

int RaThisFlowModelIndex = 0;

char *RaSrvStatsFlowModelDesc [] = {
"Total Aggregate Traffic",
"Protocol Aggregate Traffic",
"Services Aggregate Traffic",
"Dest Class A Aggregate Traffic",
"Dest Class A Protocol Aggregate Traffic",
"Dest Class A Service Aggregate Traffic",
"Dest Class Aggregate Traffic",
"Dest Class Protocol Aggregate Traffic",
"Dest Class Service Aggregate Traffic",
"Class Matrix Aggregate Traffic",
"Class Matrix Protocol Traffic",
"Class Matrix Service Traffic",
"Destination Aggregate Traffic",
"Destination Protocol Aggregate Traffic",
"Destination Service Aggregate Traffic",
"Source Class Destination Aggregate Traffic",
"Source Class Destination Protocol Aggregate Traffic",
"Source Class Destination Services Aggregate Traffic",
"Matrix Aggregate Traffic",
"Matrix Protocol Aggregate Traffic",
"Matrix Services Aggregate Traffic",
"Aggregate Traffic",
};

char *RaSrvStatsFlowModelFile [] = {
"Flow     100       *                  *                 *       *        *         200      120",
"Flow     101       *                  *                 *       *        *         201      120",
"Flow     102       *                  *                 *       *        *         202      120",
"Flow     103       *                  *                 *       *        *         203      120",
"Flow     104       *                  *                 *       *        *         204      120",
"Flow     105       *                  *                 *       *        *         205      120",
"Flow     106       *                  *                 *       *        *         206      120",
"Flow     107       *                  *                 *       *        *         207      120",
"Flow     108       *                  *                 *       *        *         208      120",
"Flow     109       *                  *                 *       *        *         209      120",
"Flow     110       *                  *                 *       *        *         210      120",
"Flow     111       *                  *                 *       *        *         211      120",
"Flow     112       *                  *                 *       *        *         212      120",
"Flow     113       *                  *                 *       *        *         213      120",
"Flow     114       *                  *                 *       *        *         214      120",
"Flow     115       *                  *                 *       *        *         215      120",
"Flow     116       *                  *                 *       *        *         216      120",
"Flow     117       *                  *                 *       *        *         217      120",
"Flow     118       *                  *                 *       *        *         218      120",
"Flow     119       *                  *                 *       *        *         219      120",
"Flow     120       *                  *                 *       *        *         220      120",
"Flow     121       *                  *                 *       *        *         221      120",

"Model    200   0.0.0.0             0.0.0.0             no       no       no",
"Model    201   0.0.0.0             0.0.0.0             yes      no       no",
"Model    202   0.0.0.0             0.0.0.0             yes      no       yes",
"Model    203   0.0.0.0             255.0.0.0           no       no       no",
"Model    204   0.0.0.0             255.0.0.0           yes      no       no",
"Model    205   0.0.0.0             255.0.0.0           yes      no       yes",
"Model    206   0.0.0.0             class               no       no       no",
"Model    207   0.0.0.0             class               yes      no       no",
"Model    208   0.0.0.0             class               yes      no       yes",
"Model    209   class               class               no       no       no",
"Model    210   class               class               yes      no       no",
"Model    211   class               class               yes      no       yes",
"Model    212   0.0.0.0             255.255.255.255     no       no       no",
"Model    213   0.0.0.0             255.255.255.255     yes      no       no",
"Model    214   0.0.0.0             255.255.255.255     yes      no       yes",
"Model    215   class               255.255.255.255     no       no       no",
"Model    216   class               255.255.255.255     yes      no       no",
"Model    217   class               255.255.255.255     yes      no       yes",
"Model    218   255.255.255.255     255.255.255.255     no       no       no",
"Model    219   255.255.255.255     255.255.255.255     yes      no       no",
"Model    220   255.255.255.255     255.255.255.255     yes      no       yes",
"Model    221   255.255.255.255     255.255.255.255     yes      yes      yes",

NULL,
};

int 
RaInitCurses ()
{
   struct termios newtty;

   RaCursesInit++;

   RaFlowModelFile = RaReadFlowModelFile (RaSrvStatsFlowModelFile);

   if (!(RaFlowModelFile))
      usage();

   if (tcgetattr(0, &RaOrigTty) == -1)
      ArgusLog(LOG_ERR, "tcgetattr() failed %s\n", strerror(errno));

   newtty = RaOrigTty;
   newtty.c_lflag &= ~ICANON;
   newtty.c_lflag &= ~ECHO;
   newtty.c_cc[VMIN] = 1;
   newtty.c_cc[VTIME] = 0;

   if (tcsetattr(0, TCSAFLUSH, &newtty) == -1)
       ArgusLog(LOG_ERR, "cannot put tty into raw mode\n");

   RaWindow = initscr();
   clearok(RaWindow, TRUE);
   werase(RaWindow);
   wrefresh(RaWindow);
 
   RaHeaderWindow = newwin (HEADERWINSIZE, COLS, 0, 0);
   if (Vflag) {
      RaAvailableWindow = newwin (LINES - (FAILEDWINSIZE + HEADERWINSIZE + 2), COLS, HEADERWINSIZE, 0);
      RaFailedWindow = newwin (FAILEDWINSIZE, COLS, LINES - (FAILEDWINSIZE + 1), 0);
      idlok (RaAvailableWindow, TRUE);
      scrollok (RaAvailableWindow, TRUE);
      idlok (RaFailedWindow, TRUE);
      scrollok (RaFailedWindow, TRUE);

   } else {
      RaAvailableWindow = newwin ((LINES - HEADERWINSIZE), COLS, HEADERWINSIZE, 0);
      RaFailedWindow = RaAvailableWindow;
      idlok (RaAvailableWindow, TRUE);
      scrollok (RaAvailableWindow, TRUE);
   }

   refresh();
   return (1);
}
#endif

void RaTopLoop(void);
extern struct timeval ArgusGlobalTime;
struct timeval RaTopStartTime = {0, 0};

int RaInitialized = 0;

void
ArgusClientInit ()
{
   int i, x;

   if (!(RaInitialized)) {

      bzero ((char *) RaFlowArray,  sizeof(RaFlowArray));
      bzero ((char *) RaModelArray, sizeof(RaModelArray));

      if (ArgusFlowModelFile) {
         if ((RaFlowModelFile = RaReadFlowModelFile(NULL)) < 0)
            exit(0);
      } else
         ArgusFlowModelFile = " ";

      if ((RaAvailableQueue = RaNewQueue()) == NULL)
         exit(0);

      if ((RaFailedQueue = RaNewQueue()) == NULL)
         exit(0);

      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      RaPrintTransactions++;

      if (Sflag) {
         RaClientTimeout.tv_sec  = 0;
         RaClientTimeout.tv_usec = 100000;
      } else {
         RaClientTimeout.tv_sec  = 60;
         RaClientTimeout.tv_usec = 0;
      }

      bzero ((char *) RaSortAlgorithms, sizeof(RaSortAlgorithms));

      if (RaSortIndex > 0) {
         for (i = 0; i < ARGUS_MAX_SORT_ALG; i++) {
            if (RaSortAlgorithmStrings[i] != NULL) {
               for (x = 0; x < MAX_SORT_ALG_TYPES; x++) {
                  if (!strncmp (RaSortKeyWords[x], RaSortAlgorithmStrings[i], strlen(RaSortKeyWords[x]))) {
                     RaSortAlgorithms[i] = RaSortAlgorithmTable[x];
                     break;
                  }
               }
               if (x == MAX_SORT_ALG_TYPES)
                  ArgusLog (LOG_ERR, "sort syntax error. \'%s\' not supported", RaSortAlgorithmStrings[i]);
            } else
               break;
         }
      } else
         RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTBYTECOUNT];

      RaThisFlowModelIndex = Argusdflag;
      if (RaThisFlowModelIndex > (MAXFLOWMODELS - 1))
         RaThisFlowModelIndex = MAXFLOWMODELS - 1;

      RaInitialized++;

      if (nflag > 0)
         hfield = 15;

      if (nflag > 1)
         pfield = 5;

      RaCumulativeMerge = 0;

      RaTopLoop();
   }
}

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

int RaTopCompleted = 0;
int RaTasksToDo = 1;

struct ARGUS_INPUT *RaCurrentInput = NULL;

void
RaTopLoop()
{
   int retn = 0, fd = -1;

   while (!(RaTopCompleted)) {
      if (RaTasksToDo) {
         if (Sflag || Cflag) {

            mvwprintw (RaHeaderWindow, 4, 1, "RaTopLoop() Processing Remote Server");
            if ((RaCurrentInput = ArgusRemoteHostList) != NULL) {
               if ((RaCurrentInput->fd = ArgusGetServerSocket (RaCurrentInput)) >= 0) {
                  if ((ArgusReadConnection (RaCurrentInput, NULL)) >= 0) {
                     ArgusRemoteFDs[ArgusActiveServers++] = RaCurrentInput;
                     ArgusReadStream();
                  } else
                     mvwprintw (RaHeaderWindow, 4, 1, "%s", ArgusGetError());
                  
               }

               RaTasksToDo = 0;
               ArgusDeleteHostList();
            }

         } else {
            mvwprintw (RaHeaderWindow, 4, 1, "RaTopLoop() Processing Input File List");
            if ((RaCurrentInput = ArgusInputFileList) != NULL) {
               while (RaCurrentInput) {
                  if (strcmp (RaCurrentInput->filename, "-")) {
                     if ((RaCurrentInput->fd = open(RaCurrentInput->filename, O_RDONLY)) >= 0) {
                        if (((ArgusReadConnection (RaCurrentInput, RaCurrentInput->filename)) >= 0)) {
                           ArgusRemoteFDs[0] = RaCurrentInput;
                           ArgusReadStream();
                           if (RaCurrentInput)
                              close(RaCurrentInput->fd);
                        }

                     } else {
                        fprintf (stderr, "%s: open '%s': %s\n", ArgusProgramName,
                                                   RaCurrentInput->filename, strerror(errno));
                        _exit(1);
                     }

                  } else {
                     RaCurrentInput->fd = 0;
                     if (((ArgusReadConnection (RaCurrentInput, NULL)) >= 0)) {
                        ArgusRemoteFDs[0] = RaCurrentInput;
                        ArgusReadStream();
                        if (RaCurrentInput)
                           close(RaCurrentInput->fd);
                     }
                  }
                  if (RaCurrentInput)
                     RaCurrentInput = RaCurrentInput->nxt;
               }

            } else {
               struct ARGUS_INPUT addrbuf, *addr = &addrbuf;

               bzero ((char *) addr, sizeof (*addr));
               addr->fd = 0;

               if (((ArgusReadConnection (addr, NULL)) >= 0)) {
                  ArgusRemoteFDs[0] = addr;
                  ArgusReadStream();
                  if (RaCurrentInput)
                     close(RaCurrentInput->fd);
               }
            }

            ArgusClientTimeout ();
            RaTasksToDo = 0;
         }

      } else {
         usleep(50000);
         ArgusClientTimeout ();
      }
   }

   if (fd >= 0) {
      ArgusShutDown (0);
   } else
      retn = 1;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "main () exiting with %d\n", retn);
#endif

   exit (retn);
}


int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   int retn = 0;

#ifdef ARGUS_CURSES
   if (RaCursesInit) {
      if ((retn = tcsetattr(0, TCSAFLUSH, &RaOrigTty)) == ERR)
         ArgusLog (LOG_ERR, "tcsetattr error %s\n");

      mvwprintw (RaWindow, LINES - 1, 0, "\n");
      wrefresh(RaWindow);
   }

#endif

   switch (sig) {
      case SIGHUP:
      case SIGTERM:
      case SIGINT:
         RaTopCompleted = 1;
         _exit(0);
         break;
   }
}


extern char version[];
extern struct ARGUS_INPUT *ArgusInput;
struct timeval RaProbeUptime = {0, 0};

void RaRefreshDisplay(void);
void RaOutputHelpScreen(void);

int RaActiveCons = 0;
int RaClosingCons = 0;
int RaNoResponse = 0;

void RaUpdateWindow (WINDOW *, struct RaQueueStruct *);

void
RaOutputHelpScreen ()
{
   werase(RaAvailableWindow);

   mvwprintw (RaAvailableWindow, 1, 1, "RaTop Version %s\n", version);
   mvwprintw (RaAvailableWindow, 2, 1, "Key Commands: h,q,n,s");
   mvwprintw (RaAvailableWindow, 3, 1, "   h - Print help screen.");
   mvwprintw (RaAvailableWindow, 4, 1, "   q - Quit the program.");
   mvwprintw (RaAvailableWindow, 5, 1, "   n - Specify the number of entries to print.");
   mvwprintw (RaAvailableWindow, 6, 1, "   s - Sort by field (b,p,d,s,l,L).");
   mvwprintw (RaAvailableWindow, 7, 1, "         b - sort by total bytes.");
   mvwprintw (RaAvailableWindow, 8, 1, "         p - sort by total packets.");
   mvwprintw (RaAvailableWindow, 9, 1, "         d - sort by duration.");
   mvwprintw (RaAvailableWindow,10, 1, "         s - sort by service.");
   mvwprintw (RaAvailableWindow,11, 1, "         l - sort by ");
   mvwprintw (RaAvailableWindow,12, 1, "         L - sort by load.");
   mvwprintw (RaAvailableWindow,13, 1, "   F - Toggle fields on display (tDITcbpLU)");
   mvwprintw (RaAvailableWindow,14, 1, "         t - toggle time field.");
   mvwprintw (RaAvailableWindow,15, 1, "         D - toggle Duration field.");
   mvwprintw (RaAvailableWindow,16, 1, "         I - toggle Indicator field.");
   mvwprintw (RaAvailableWindow,17, 1, "         T - toggle Transaction field.");
   mvwprintw (RaAvailableWindow,18, 1, "         c - toggle counts field.");
   mvwprintw (RaAvailableWindow,19, 1, "         b - toggle bytes field.");
   mvwprintw (RaAvailableWindow,20, 1, "         p - toggle packtes field.");
   mvwprintw (RaAvailableWindow,21, 1, "         L - toggle Load field.");
   mvwprintw (RaAvailableWindow,22, 1, "         U - toggle User data field.");
   mvwprintw (RaAvailableWindow,23, 1, "   S - Toggle server display mode. When on displays entries that have multiple sources");
   mvwprintw (RaAvailableWindow,24, 1, "   d - disconnect from remote Argus");
   mvwprintw (RaAvailableWindow,25, 1, "   c - connect to remote Argus");
   mvwprintw (RaAvailableWindow,26, 1, "   o - open Argus data file");
   mvwprintw (RaAvailableWindow,27, 1, "   f - specify filter expression");
// mvwprintw (RaAvailableWindow,28, 1, "         -r <file>      read argus data <file>. '-' denotes stdin.\n");
// mvwprintw (RaAvailableWindow,29, 1, "         -R             print out response data when availabile.\n");
// mvwprintw (RaAvailableWindow,30, 1, "         -S <host>      specify remote argus <host>.\n");
// mvwprintw (RaAvailableWindow,31, 1, "         -t <timerange> specify <timerange> for reading records.\n");
// mvwprintw (RaAvailableWindow,32, 1, "               format:  timeSpecification[-timeSpecification]\n");
// mvwprintw (RaAvailableWindow,33, 1, "                        timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
// mvwprintw (RaAvailableWindow,34, 1, "                                             [yyyy/]mm/dd\n");
// mvwprintw (RaAvailableWindow,35, 1, "         -T <secs>      attach to remote server for T seconds.\n");
// mvwprintw (RaAvailableWindow,36, 1, "         -u             print time in Unix time format.\n");
// mvwprintw (RaAvailableWindow,37, 1, "         -U <user/auth> specify <user/auth> authentication information.\n");
// mvwprintw (RaAvailableWindow,38, 1, "         -w <file>      write output to <file>. '-' denotes stdout.\n");
// mvwprintw (RaAvailableWindow,39, 1, "         -z             print Argus TCP state changes.\n");
// mvwprintw (RaAvailableWindow,40, 1, "         -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
}

void
RaRefreshDisplay ()
{
   struct timeval tvp, *last, *start;
   int dsec, dusec, srcaddr;
   char timebuf[32];
   struct tm *tm;

   RaUpdateWindow(RaAvailableWindow, RaAvailableQueue);

   if (Vflag)
      RaUpdateWindow(RaFailedWindow, RaFailedQueue);

   if (RaProbeUptime.tv_sec == 0) {
      start = &ArgusInput->ArgusInitCon.argus_mar.startime;
      last  = &ArgusInput->ArgusInitCon.argus_mar.now;

      tvp.tv_sec  = ntohl(last->tv_sec)  - ntohl(start->tv_sec);
      tvp.tv_usec = ntohl(last->tv_usec) - ntohl(start->tv_usec);

      if (tvp.tv_usec < 0) {
         tvp.tv_sec--;
         tvp.tv_usec += 1000000;
      }

      if (tvp.tv_usec >= 1000000) {
         tvp.tv_sec++;
         tvp.tv_usec -= 1000000;
      }

      RaProbeUptime = tvp;
   }

   if (RaTopStartTime.tv_sec == 0)
      RaTopStartTime = ArgusGlobalTime;

   tvp = ArgusGlobalTime;

   dsec  = RaProbeUptime.tv_sec  + (tvp.tv_sec  - RaTopStartTime.tv_sec);
   dusec = RaProbeUptime.tv_usec + (tvp.tv_usec - RaTopStartTime.tv_usec);

   if (dusec < 0) {
      dsec--;
      dusec += 1000000;
   }

   if (dusec >= 1000000) {
      dsec++;
      dusec -= 1000000;
   }

   if (ArgusInput != NULL) {
      tm = localtime((time_t *) &tvp.tv_sec);
      strftime ((char *) timebuf, 32, "%d/%m/%Y %I:%M%P", tm);

#ifdef ARGUS_CURSES
      srcaddr = ntohl(ArgusInput->ArgusInitCon.argus_mar.argusid);
      mvwprintw (RaWindow, LINES - 1, 0, "\n");
      if (strlen(ArgusProgramArgs) > 60)
         ArgusProgramArgs[60] = '\0';
      mvwprintw (RaHeaderWindow, 0, 1, "%s", ArgusProgramArgs);

      if (RaTasksToDo) {
         mvwprintw (RaHeaderWindow, 1, 1, "Source %s  Version %d.%d  %s up %d days, %02d:%02d:%02d  Far %d secs Mar %d secs",
                          ipaddr_string(&srcaddr),
                          ArgusInput->ArgusInitCon.argus_mar.major_version,
                          ArgusInput->ArgusInitCon.argus_mar.minor_version,
                          timebuf, dsec / 86400, (dsec % 86400)/3600, ((dsec % 86400) % 3600)/60,
                          (((dsec % 86400) % 3600) % 60),
                          ntohs(ArgusInput->ArgusInitCon.argus_mar.reportInterval),
                          ntohs(ArgusInput->ArgusInitCon.argus_mar.argusMrInterval));

         if (Vflag)
            mvwprintw (RaHeaderWindow, 2, 1, "%4d Flows:  %3d Connected, %3d Closing, %3d Not Connected  Total Records %d",
                          RaAvailableQueue->count + RaFailedQueue->count, RaAvailableQueue->count,
                          RaClosingCons, RaFailedQueue->count, totalrecords);
         else
            mvwprintw (RaHeaderWindow, 2, 1, "%4d Flows:  %3d Active, %3d Closing, Total Records %d", 
                          RaAvailableQueue->count + RaFailedQueue->count, RaActiveCons, RaClosingCons, totalrecords);
      }
      wrefresh(RaHeaderWindow);
#endif
   }

#ifdef ARGUS_CURSES
   if (RaWindowStatus && (RaLabel != NULL)) {
      if (RaServerMode)
         mvwprintw (RaAvailableWindow, 0, 1, "Services", RaLabel);
      else 
         mvwprintw (RaAvailableWindow, 0, 1, "Flows", RaLabel);
      mvwprintw (RaAvailableWindow, 1, 1, "Rank %s", RaLabel);
#if !defined(HAVE_SOLARIS)
      mvwchgat (RaAvailableWindow, 1, 1, strlen(RaLabel) + 5, A_BOLD, COLOR_WHITE, NULL);
#endif
      wrefresh(RaAvailableWindow);

      if (Vflag) {
         if (RaServerMode)
            mvwprintw (RaFailedWindow, 0, 1, "Peer Service", RaLabel);
         else 
            mvwprintw (RaFailedWindow, 0, 1, "Not Connected", RaLabel);
         mvwprintw (RaFailedWindow, 1, 1, "Rank %s", RaLabel);
#if !defined(HAVE_SOLARIS)
         mvwchgat (RaFailedWindow, 1, 1, strlen(RaLabel) + 5, A_BOLD, COLOR_WHITE, NULL);
#endif
         wrefresh(RaFailedWindow);
      }
   }

   wrefresh(RaWindow);
#endif

};


extern void ArgusCloseInput(struct ARGUS_INPUT *);

int ArgusHourlyUpdate = 1;
int ArgusMinuteUpdate = 1;

#define RANEWCOMMAND	1
#define RAGETTINGC	2
#define RAGETTINGF	3
#define RAGETTINGo	4
#define RAGETTINGN	5
#define RAGETTINGS	6
#define RAGETTINGh	7
#define RAGETTINGf	8
#define RAREADINGSTR	9

unsigned int RaUpdateCount = 0;
unsigned int RaUpdateMod = 2;
int RaInputStatus = RANEWCOMMAND;
char RaCommandInputStr[MAXSTRLEN];
int RaCommandIndex = 0;

#define RAGETTINGCSTR		"Connect to: "
#define RAGETTINGoSTR		"Read file: "
#define RAGETTINGfSTR		"Specify filter: "
#define RAGETTINGFSTR		"Fields to display (tDITcbpLU): "
#define RAGETTINGSSTR		"Switching to Server Mode: "
#define RAGETTINGNSTR		"Connections to display: "
#define RAGETTINGhSTR		"Help menu (press any key to continue): "
#define RAGETTINGSORTSTR	"Sort By (b,p,d,s,l,L): "

#include <glob.h>

void
ArgusClientTimeout ()
{
#ifdef ARGUS_CURSES
   struct ArgusRecordStore *store = NULL;
   struct ARGUS_INPUT *addr, *tmp;
   struct timeval tv;
   fd_set in;
   char c, *ptr = NULL;
   int i;

   tv.tv_sec  = 0;
   tv.tv_usec = 0;
   FD_ZERO(&in);
   FD_SET(0, &in);

   switch (RaInputStatus) {
      case RANEWCOMMAND: {
         if (select(1, &in, 0, 0, &tv) > 0 && read(0, &c, 1) == 1) {
            mvwprintw (RaHeaderWindow, 4, 1, "%*s", COLS, "  ");
            if (RaCommandIndex != 0) {
               RaCommandIndex = 0;
               bzero (RaCommandInputStr, MAXSTRLEN);
            }
            switch (c) {
               case 'n':
                  RaInputStatus = RAGETTINGN;
                  mvwprintw (RaHeaderWindow, 4, 1, RAGETTINGNSTR);
                  break;

               case 'c':
                  if ((ArgusRemoteHostList) == NULL) {
                     RaInputStatus = RAGETTINGC;
                     mvwprintw (RaHeaderWindow, 4, 1, RAGETTINGCSTR);
                  } else
                     mvwprintw (RaHeaderWindow, 4, 1, "Connected");

                  RaRefreshDisplay();
                  break;

               case 'd':
                  if (RaCurrentInput != NULL) {
                     for (i = 0; i < ARGUS_MAX_REMOTE_CONN; i++) {
                        if (ArgusRemoteFDs[i] != NULL) {
                           ArgusCloseInput(ArgusRemoteFDs[i]);
                           ArgusRemoteFDs[i] = NULL;
                        }
                     }
                    
                     RaCurrentInput = NULL;
                  
                     if ((ArgusRemoteHostList) != NULL) {
                        ArgusDeleteHostList();
                        ArgusActiveServers = 0;
                        ArgusRemoteHostList = NULL;
                     }

                     if ((addr = ArgusInputFileList) != NULL) {
                        while (addr) {
                           tmp = addr->nxt;
                           ArgusFree(addr);
                           addr = tmp;
                        }
                        ArgusInputFileList = NULL;
                     }

                     werase(RaHeaderWindow);
                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                        RaTimeoutArgusStore(store);

                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                        RaTimeoutArgusStore(store);

                     mvwprintw (RaHeaderWindow, 4, 1, "Disconnecting");

                  } else
                     mvwprintw (RaHeaderWindow, 4, 1, "Not Connected");

                  RaRefreshDisplay();
                  break;

               case 'f':
                  RaInputStatus = RAGETTINGf;
                  mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGfSTR, RaInputFilter);
                  break;

               case 'o':
                  if (RaCurrentInput == NULL) {
                     RaInputStatus = RAGETTINGo;
                     mvwprintw (RaHeaderWindow, 4, 1, RAGETTINGoSTR);
                  } else
                     mvwprintw (RaHeaderWindow, 4, 1, "Connected");
                  break;

               case '+': {
                  if (++RaThisFlowModelIndex > (MAXFLOWMODELS - 1))
                     RaThisFlowModelIndex = MAXFLOWMODELS - 1;

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                     RaTimeoutArgusStore(store);

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                     RaTimeoutArgusStore(store);

                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  mvwprintw (RaHeaderWindow, 4, 1, "%s", RaSrvStatsFlowModelDesc[RaThisFlowModelIndex]);
                  RaRefreshDisplay();
                  break;
               }

               case '-': {
                  struct ArgusRecordStore *store = NULL;

                  if (--RaThisFlowModelIndex < 0)
                     RaThisFlowModelIndex = 0;

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                     RaTimeoutArgusStore(store);

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                     RaTimeoutArgusStore(store);

                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  mvwprintw (RaHeaderWindow, 4, 1, "%s", RaSrvStatsFlowModelDesc[RaThisFlowModelIndex]);
                  RaRefreshDisplay();
                  break;
               }

               case 'S':
                  if ((RaServerMode = (RaServerMode > 0) ? 0 : 1) > 0)
                     ptr = "on";
                  else
                     ptr = "off";

                  mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGSSTR, ptr);
                  break;

               case 's':
                  RaInputStatus = RAGETTINGS;
                  mvwprintw (RaHeaderWindow, 4, 1, RAGETTINGSORTSTR);
                  break;

               case 'F':
                  RaInputStatus = RAGETTINGF;
                  mvwprintw (RaHeaderWindow, 4, 1, RAGETTINGFSTR);
                  break;

               case 'q':
                  RaParseComplete(SIGINT);
                  break;

               case 'h':
                  RaInputStatus = RAGETTINGh;
                  mvwprintw (RaHeaderWindow, 4, 1, RAGETTINGhSTR);
                  RaWindowStatus = 0;
                  RaOutputHelpScreen();
                  break;

               default:
                  break;
            }
         }
         break;
      }

      case RAGETTINGC: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  RaCommandInputStr[RaCommandIndex] = '\0';
                  if (!(ArgusAddHostList (RaCommandInputStr))) {
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s host not found", RAGETTINGCSTR, RaCommandInputStr);
                  } else {
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGCSTR, RaCommandInputStr);
                     RaTasksToDo = 1;
                     totalrecords = 0;
                     Sflag = 1;
                     RaClientTimeout.tv_sec  = 0;
                     RaClientTimeout.tv_usec = 100000;
                  }

                  RaInputStatus = RANEWCOMMAND;

                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  break;

               } else {
                  if (iscntrl(RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGCSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwprintw (RaHeaderWindow, 4, 1, "%*s", strlen(RAGETTINGCSTR) + strlen(RaCommandInputStr), "  ");
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGCSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
         break;
      }

      case RAGETTINGf: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  RaInputStatus = RANEWCOMMAND;
                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  break;

               } else {
                  if (iscntrl(RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGfSTR, RaCommandInputStr);

                     } else {

                        RaInputStatus = RANEWCOMMAND;
                        mvwprintw (RaHeaderWindow, 4, 1, "%*s", strlen(RAGETTINGfSTR) + strlen(RaCommandInputStr), "  ");
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGfSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
      }

      case RAGETTINGo: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {

               glob_t globbuf;

               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  RaCommandInputStr[RaCommandIndex] = '\0';

                  glob(RaCommandInputStr, 0, NULL, &globbuf);

                  if (globbuf.gl_pathc > 0) {
                     for (i = 0; i < globbuf.gl_pathc; i++) {
                        if (!(ArgusAddFileList (globbuf.gl_pathv[i]))) {
                           mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr, strerror(errno));
                        } else {
                           werase(RaHeaderWindow);
                           mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr);
                           RaTasksToDo = 1;
                           totalrecords = 0;
                           RaClientTimeout.tv_sec  = 60;
                           RaClientTimeout.tv_usec = 0;
                           ArgusFree(ArgusProgramArgs);
                           if ((ArgusProgramArgs = ArgusCalloc (1, strlen(RaCommandInputStr) + 4)) != NULL)
                              sprintf (ArgusProgramArgs, "%s -r %s", ArgusProgramName, RaCommandInputStr);
                        }
                     }
                  } else
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr, " no match");

                  RaInputStatus = RANEWCOMMAND;

                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  break;

               } else {
                  if (iscntrl(RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwprintw (RaHeaderWindow, 4, 1, "%*s", strlen(RAGETTINGoSTR) + strlen(RaCommandInputStr), "  ");
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
         break;
      }

      case RAGETTINGF: {
         if (select(1, &in, 0, 0, &tv) > 0 && read(0, &c, 1) == 1) {
            switch (c) {
               case 'D':
                  RaPrintDuration = (RaPrintDuration > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'I':
                  RaPrintIndicator = (RaPrintIndicator > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 't':
                  RaPrintDate = (RaPrintDate > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'i':
                  idflag = (idflag > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'b':
                  RaPrintBytes = (RaPrintBytes > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'c':
                  RaPrintCounts = (RaPrintCounts > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'j':
                  RaPrintJitter = (RaPrintJitter > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'L':
                  RaPrintLoad = (RaPrintLoad > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'T':
                  RaPrintTransactions = (RaPrintTransactions > 0) ? 0 : 1;
                  goto RaFieldSpecified;
               case 'U':
                  RaPrintUserData = (RaPrintUserData > 0) ? 0 : 1;
                  goto RaFieldSpecified;

RaFieldSpecified:
                  RaInputStatus = RANEWCOMMAND;
                  RaLabel = NULL;
                  werase(RaHeaderWindow);
                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  mvwprintw (RaHeaderWindow, 4, 1, "%s %c", RAGETTINGFSTR, c);
                  break;

               default:
                  if (iscntrl(RaCommandInputStr[RaCommandIndex])) {
                     RaInputStatus = RANEWCOMMAND;
                     werase(RaAvailableWindow);
                     if (Vflag)
                        werase(RaFailedWindow);
                  }
                  break;
            }
         }
         break;
      }

      case RAGETTINGh: {
         if (select(1, &in, 0, 0, &tv) > 0 && read(0, &c, 1) == 1) {
            RaWindowStatus = 1;
            RaInputStatus = RANEWCOMMAND;
            werase(RaAvailableWindow);
            if (Vflag)
               werase(RaFailedWindow);
         }

         break;
      }

      case RAGETTINGS: {
         if (select(1, &in, 0, 0, &tv) > 0 && read(0, &c, 1) == 1) {
            switch (c) {
               case 'b': RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTBYTECOUNT]; goto RaSortSpecified;
               case 'p': RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTPKTSCOUNT]; goto RaSortSpecified;
               case 'd': RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTDURATION];  goto RaSortSpecified;
               case 's': RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTSTARTTIME]; goto RaSortSpecified;
               case 'l': RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTLASTTIME];  goto RaSortSpecified;
               case 'L': RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTLOAD];      goto RaSortSpecified;

RaSortSpecified:
                  mvwprintw (RaHeaderWindow, 4, 1, "%s %c", RAGETTINGSORTSTR, c);
                  RaInputStatus = RANEWCOMMAND;
                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  break;

               default:
                  if (iscntrl(RaCommandInputStr[RaCommandIndex])) {
                     RaInputStatus = RANEWCOMMAND;
                     werase(RaAvailableWindow);
                     if (Vflag)
                        werase(RaFailedWindow);
                  }
                  break;
            }
         }
         break;
      }

      case RAGETTINGN: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  Nflag = atoi(RaCommandInputStr);
                  RaInputStatus = RANEWCOMMAND;
                  werase(RaAvailableWindow);
                  if (Vflag)
                     werase(RaFailedWindow);
                  break;

               } else {
                  if (iscntrl(RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGNSTR, RaCommandInputStr);

                     } else {

                        RaInputStatus = RANEWCOMMAND;
                        mvwprintw (RaHeaderWindow, 4, 1, "%*s", strlen(RAGETTINGNSTR) + strlen(RaCommandInputStr), "  ");
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGNSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
      }
   }
#endif

   RaUpdateCount++;
   RaProcessQueue (RaAvailableQueue, ARGUS_STATUS);
   RaProcessQueue (RaFailedQueue, ARGUS_STATUS);
   
   if (!(RaUpdateCount % RaUpdateMod))
#ifdef ARGUS_CURSES
      if (RaCursesInit)
#endif
         RaRefreshDisplay();

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}



void
parse_arg (argc, argv)
int argc;
char**argv;
{}

void
usage ()
{

   fprintf (stderr, "RaTop Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -a             print record summaries on termination.\n");
   fprintf (stderr, "         -A             print application bytes.\n");
   fprintf (stderr, "         -b             dump packet-matching code.\n");
   fprintf (stderr, "         -c             print packet and byte counts.\n");
   fprintf (stderr, "         -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -d <bytes>     print number of <bytes> from user data capture buffer.\n");
   fprintf (stderr, "               format:  num | s<num> | d<num> | s<num>:d<num>\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>     specify debug level\n");
#endif
   fprintf (stderr, "         -e <encode>    convert user data using <encode> method.\n");
   fprintf (stderr, "                        Supported types are <Ascii> and <Encode64>.\n");
   fprintf (stderr, "         -E <file>      write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "         -g             print record time duration.\n");
   fprintf (stderr, "         -G             print both start and last time values.\n");
   fprintf (stderr, "         -h             print help.\n");
   fprintf (stderr, "         -i             print source probe id.\n");
   fprintf (stderr, "         -I             print transaction state and option indicators.\n");
   fprintf (stderr, "         -l             print last time values [default is start time].\n");
   fprintf (stderr, "         -m             print MAC addresses.\n");
   fprintf (stderr, "         -n             don't convert numbers to names.\n");
   fprintf (stderr, "         -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "         -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "         -q             quiet mode. don't print record outputs.\n");
   fprintf (stderr, "         -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -R             print out response data when availabile.\n");
   fprintf (stderr, "         -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "         -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                             [yyyy/]mm/dd\n");
   fprintf (stderr, "         -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>      write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "         -z             print Argus TCP state changes.\n");
   fprintf (stderr, "         -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}

#ifdef ARGUS_CURSES
int RaPrintCursesStr (char *);

int
RaPrintCursesStr (char *str)
{
   if (!(RaCursesInit))
      RaInitCurses ();
 
/*
   RaRecordWindowNum = (++RaRecordWindowNum == 10000) ? 0 : RaRecordWindowNum;
 
   bzero (buf, sizeof(buf));
   sprintf (buf, "%5d ", RaRecordWindowNum);
   strcat (buf, str);
   waddnstr (RaRecordWindow, buf, COLS - 4);
   waddch (RaRecordWindow, '\n');
   wrefresh(RaRecordWindow);
*/
   return (1);
}
#endif


void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordStore *store;
   struct ArgusRecordData *data = NULL;
   struct RaPolicyStruct *rap;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   struct RaQueueStruct *queue;
   WINDOW *window = NULL;
   int RaTimeout = -1;

   bcopy(flow, RaArgusFlow, sizeof(struct ArgusFlow));

   rap =  RaFlowArray[RaThisFlowModelIndex];
   if (RaPolicyMatch (argus, RaFlowArray[RaThisFlowModelIndex])) {
      RaModifyFlow(rap, argus);
      RaTimeout = rap->ArgusTimeout;

   } else {
      RaTimeout = 120;
      RaModifyDefaultFlow(argus);
   }

   if ((store = RaFindArgusRecord(&RaHashTable, argus)) == NULL)
      store = RaFindRevArgusRecord(&RaHashTable, argus);

   if (store) {
      RaThisArgusStore = store;

      if (RaCheckTimeout (store, argus))
         RaSendArgusRecord(store);

      if (Vflag) {
         if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
            if (store->qhdr.queue == RaAvailableQueue) {
               window = RaAvailableWindow;
               queue  = RaAvailableQueue;
            } else {
               window = RaFailedWindow;
               queue  = RaFailedQueue;
            }
   
            RaRemoveFromQueue(store->qhdr.queue, &store->qhdr);
            if (queue->count < ((Nflag > 0) ? Nflag : LINES))
               werase(window);

            RaSendArgusRecord(store);
   
            store->status &= ~RA_SVCTEST;
            store->status |= (srv->status & RA_SVCTEST);
         
            data = store->data[0];
            if (srv->status == RA_SVCPASSED)
               RaAddToQueue(RaAvailableQueue, &store->qhdr);
            else
               RaAddToQueue(RaFailedQueue, &store->qhdr);
         }
      }

      if ((data = store->data[0]) == NULL) {
         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[0] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else
         RaMergeArgusRecord(argus, data);

      RaUpdateArgusStore(argus, store);

      if (RaServerMode) {
         if ((data = store->data[0]) != NULL) {
            if ((data->status & ARGUS_FAR_SRCADDR_MODIFIED) && (store->status & RA_SVCPASSED)) {
               if (store->qhdr.queue == RaFailedQueue) {
                  RaRemoveFromQueue(RaFailedQueue, &store->qhdr);
                  RaAddToQueue(RaAvailableQueue, &store->qhdr);
               }
            } else {
               if (store->qhdr.queue == RaAvailableQueue) {
                  RaRemoveFromQueue(RaAvailableQueue, &store->qhdr);
                  RaAddToQueue(RaFailedQueue, &store->qhdr);
               }
            }
         }
      }

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         struct ArgusRecordData *data = NULL;

         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         if ((store->rahtblhdr = RaAddHashEntry (&RaHashTable, store)) != NULL) {
            if (!(RaServerMode) && (store->status & RA_SVCPASSED))
               RaAddToQueue(RaAvailableQueue, &store->qhdr);
            else
               RaAddToQueue(RaFailedQueue, &store->qhdr);
         }

         store->ArgusTimeout = RaTimeout;
         RaThisArgusStore = store;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;

/*
            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }
*/

            store->data[0] = data;

         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

         RaUpdateArgusStore(argus, store);

      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessRecord: done\n");
#endif
}


void
process_man (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_man_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }
}


struct ArgusServiceRecord ArgusThisSrv;

void
process_tcp (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_tcp_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET)
                  if (argus->argus_far.src.count && argus->argus_far.dst.count)
                     if (!(argus->argus_far.src.bytes) || !(argus->argus_far.dst.bytes))
                        ArgusThisSrv.status = RA_SVCFAILED;
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (!(argus->argus_far.src.count) || !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      } else
         if (argus->argus_far.src.count && argus->argus_far.dst.count)
            if (!(argus->argus_far.src.bytes) || !(argus->argus_far.dst.bytes))
               ArgusThisSrv.status = RA_SVCFAILED;
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_icmp (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_icmp_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (!(argus->argus_far.src.count) || !(argus->argus_far.dst.count)) 
         ArgusThisSrv.status = RA_SVCFAILED;
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_udp (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_udp_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (!(argus->argus_far.src.count) || !(argus->argus_far.dst.count)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) 
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_ip (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_ip_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (!(argus->argus_far.src.count) || !(argus->argus_far.dst.count)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

// if (argus->ahdr.status & ARGUS_CONNECTED)
      RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_arp (struct ArgusRecord *argus)
{
}


void
process_non_ip (struct ArgusRecord *argus)
{
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusAGRStruct *agr = NULL;
   int i;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.mean = data->act.sumtime/data->act.n;
               data->agr.act.stdev = (unsigned int) sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.mean = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = (unsigned int) sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL)
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);

            bcopy ((char *) argus, buf, argus->ahdr.length);
            argus = (struct ArgusRecord *) buf;

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) == NULL) {
               if (ArgusFlowModelFile && (data->agr.count > 1)) {
                  bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                  argus->ahdr.length += data->agr.length;
                  argus->ahdr.status |= ARGUS_MERGED;
                  ArgusFree (data->argus);
                  data->argus = RaCopyArgusRecord(argus);
                  ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);

               } else {
                  argus->ahdr.status &= ~(ARGUS_MERGED);
               }
            }
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}


#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  100000
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordData *data = NULL;
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
            RaTimeoutArgusStore(obj);
         break;

      default:
         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL)) {
                     RaTimeoutArgusStore(obj);
#ifdef ARGUS_CURSES
                     if ((queue->count < LINES)) {
                        if (queue == RaAvailableQueue)
                           werase(RaAvailableWindow);
                        else
                           werase(RaFailedWindow);
                     }
#endif
                  } else {
                     if ((RaServerMode) && (data = obj->data[0]) != NULL) {
                        if ((data->status & ARGUS_FAR_SRCADDR_MODIFIED) && (obj->status & RA_SVCPASSED)) {
                           RaAddToQueue(RaAvailableQueue, &obj->qhdr);
                           if ((queue != RaAvailableQueue) && (queue->count < LINES))
                              werase(RaFailedWindow);
                        } else {
                           RaAddToQueue(RaFailedQueue, &obj->qhdr);
                           if ((queue != RaFailedQueue) && (queue->count < LINES))
                              werase(RaAvailableWindow);
                        }
                     } else
                        RaAddToQueue(queue, &obj->qhdr);
                  }

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


void
RaUpdateWindow (WINDOW *window, struct RaQueueStruct *queue)
{
   struct ArgusRecord *argus = NULL;
   struct ArgusQueueHeader *qhdr;
   struct ArgusFlow *flow = NULL;
   int i, cnt, attr = A_NORMAL;
   short color = COLOR_WHITE;
   char *str = NULL;
   char buf[2048];

   if (RaWindowStatus && ((qhdr = queue->start) != NULL)) {
      RaSortQueue (queue);
      RaActiveCons = 0; RaClosingCons = 0; RaNoResponse = 0;
      cnt = ((Nflag > 0) ? Nflag : LINES);
      cnt = (cnt > queue->count) ? queue->count : cnt;

      for (i = 0; i < cnt; i++) {
         attr = A_NORMAL;
         color = COLOR_WHITE;
         if (queue->array[i] != NULL) {
            struct ArgusRecordStore *store = (struct ArgusRecordStore *) queue->array[i];
            struct ArgusRecordData *data = store->data[0];

            if (data && ((argus = data->argus) != NULL)) {
               if (RaLabel == NULL)
                  RaLabel = RaGenerateLabel(argus);

               bcopy ((char *) argus, buf, argus->ahdr.length);
               argus = (struct ArgusRecord *) buf;

               ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);
               flow = &argus->argus_far.flow;
/*
               if (!(data->status & ARGUS_FAR_SRCADDR_MODIFIED))
                  flow->ip_flow.ip_src = data->flow.ip_flow.ip_src;

               if (!(data->status & ARGUS_FAR_DSTADDR_MODIFIED))
                  flow->ip_flow.ip_dst = data->flow.ip_flow.ip_dst;

               if (!(data->status & ARGUS_FAR_PROTO_MODIFIED))
                  flow->ip_flow.ip_p = data->flow.ip_flow.ip_p;

               if (!(data->status & ARGUS_FAR_SRCPORT_MODIFIED))
                  flow->ip_flow.sport = data->flow.ip_flow.sport;

               if (!(data->status & ARGUS_FAR_DSTPORT_MODIFIED))
                  flow->ip_flow.dport = data->flow.ip_flow.dport;

               if (!(data->status & ARGUS_FAR_TPVAL_MODIFIED))
                  flow->ip_flow.tp_p = data->flow.ip_flow.tp_p;
*/

               switch (argus->ahdr.status & 0xFFFF) {
                  case ETHERTYPE_IP:
                     switch (argus->argus_far.flow.ip_flow.ip_p) {
                        case IPPROTO_TCP: {
                           int status;
                           struct ArgusTCPObject *tcp = NULL;

                           if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
                              tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

                              if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                                 if ((status & ARGUS_NORMAL_CLOSE) || (status & ARGUS_RESET) ||
                                     (status & ARGUS_FIN) || (status & ARGUS_FIN_ACK)) {
                                    RaClosingCons++;
                                 } else 
                                    if (status & ARGUS_CON_ESTABLISHED) {
                                       RaActiveCons++;
                                    }

                              } else {
                                 if (argus->argus_far.src.count && argus->argus_far.dst.count) {
                                    RaActiveCons++;
                                 } else {
                                    RaNoResponse++;
                                 }
                              }

                           str = get_tcp_string (argus);
                           break;
                        }
         
                        case IPPROTO_UDP:
                           if (argus->argus_far.src.count && argus->argus_far.dst.count) {
                              RaActiveCons++;
                           } else {
                              RaNoResponse++;
                           }
                           str = get_udp_string (argus);
                           break;

                        case IPPROTO_ICMP:
                           if (argus->argus_far.src.count && argus->argus_far.dst.count) {
                              RaActiveCons++;
                           } else {
                              RaNoResponse++;
                           }
                           str = get_icmp_string (argus);
                           break;

                        default:
                           if (argus->argus_far.src.count && argus->argus_far.dst.count) {
                              RaActiveCons++;
                           } else {
                              RaNoResponse++;
                           }
                           str = get_ip_string (argus);
                           break;
                     }
                     break;

                  case ETHERTYPE_ARP:
                  case ETHERTYPE_REVARP:
                     str = get_arp_string (argus);
                     break;

                  default:
                     str = get_nonip_string (argus);
                     break;
               }

#define RA_IDLETIMER	90
/*
               if ((queue->array[i]->qhdr.lasttime.tv_sec + RA_IDLETIMER) < ArgusGlobalTime.tv_sec)
                  attr = A_NORMAL;
               else
                  attr = A_BOLD;
*/

               attr = A_NORMAL;

#ifdef ARGUS_CURSES
               mvwprintw (window, i + 2, 2, "%3d %s", i + 1, str);
               wclrtoeol(window);
#if !defined(HAVE_SOLARIS)
               mvwchgat (window, i + 2, 1, strlen(str) + 6, attr, color, NULL);
#endif
#else
               fprintf (stdout, "%3d %s", i, str);
#endif
            }
         } else
            break;
      }
   }

#ifdef ARGUS_CURSES
   wrefresh(window);
#endif
}



