# my-id.sh - the user's id
#
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "print or change your id\\n"
		printf "usage: my-id [options] id-string\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -e --errname prog             specify program name for errors\\n"
		printf "\\n"
		printf " -u --uid                      print only the UID portion of the id\\n"
		printf " -v --verbose                  print a reassuring message\\n"
		printf "\\n"
		printf "With no argument print your arch id.\\n"
		printf "\\n"
		printf "With an argument, record ID-STRING as your id\\n"
		printf "in ~/.arch-params/=id\\n"
		printf "\\n"
		printf "Your id is recorded in various archives and log messages\\n"
		printf "as you use arch.  It must consist entirely of printable\\n"
		printf "characters and fit on one line.  By convention, it should\\n"
		printf "have the form of an email address, as in this example:\\n"
		printf "\\n"
		printf "	Jane Hacker <jane.hacker@gnu.org>\\n"
		printf "\\n"
		printf "The portion of an id string between < and > is called your\\n"
		printf "uid.  arch sometimes uses your uid as a fragment when generating\\n"
		printf "unique file names.\\n"
		printf "\\n"
		printf "The option -u (--uid) causes only the uid part of your id string\\n"
		printf "to be printed.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

verbose=0
errname=
type=

while test $# -ne 0 ; do

  case "$1" in 

    -e|--errname)	shift
			if test $# -eq 0 ; then
			  printf "my-id: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -v|--verbose)	shift
    			verbose=1
			;;

    -u|--uid)		shift
    			type=uid
			;;

    -*)			printf "my-id: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test "$type" = uid ; then

  if test $# -ne 0 ; then
    printf "usage: my-id [options] -u\\n"
    printf "try --help\\n"
    exit 1
  fi

else

  case $# in 

    0)	type=print
	;;

    1)	type=set
  	idstr="$1"
	;;

    *)	if test $# -gt 1 ; then
	  printf "usage: my-id [options] [id-string]\\n" 1>&2
	  printf "try --help\\n" 1>&2
	  exit 1
	fi
	;;
  esac

fi


################################################################
# Sanity Check
# 

if test $type = set ; then
  larch valid-id --errname "${errname:-my-id}" -- "$idstr"
fi


################################################################
# Do It
# 

case $type in

  set)		mkdir -p ~/.arch-params/
  		printf "%s\\n" "$idstr" > ~/.arch-params/=id

		if test $verbose -ne 0 ; then
		  printf "user id string set (%s)\\n" "$idstr"
		fi

		exit 0
		;;

  print|uid)	mkdir -p ~/.arch-params/

    		if test ! -e ~/.arch-params/=id ; then
		  printf "%s: no id string set!\\n" "${errname:-my-id}" 1>&2
		  printf "try \"larch my-id --help\"\\n" 1>&2
		  exit 1
		fi

		idstr="`cat ~/.arch-params/=id`"
		larch valid-id -e "${errname:-my-id}" -- "$idstr"

		if test $type = print ; then

		  printf "%s\\n" "$idstr"

		else

		  printf "%s\\n" "$idstr" | sed -e "s/.*<//" -e "s/>.*//"

		fi

		exit 0
		;;
esac
