# library-find.sh: find a revision in a revision library
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "find a revision in a revision library\\n"
                printf "usage: library-find [options] [archive/]revision\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
		printf " -s --silent                   no output (except odd errors)\\n"
		printf "\\n"
		printf " -e --errname NAME             use NAME in error messages\\n"
		printf "\\n"
		printf "Find REVISION in your revision library and print its path.\\n"
		printf "If the revision is not present, print an error message (unless\\n"
		printf "\"--silent\" is specified) and exit with non-0 status.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

silent=
errname=library-find

while test $# -ne 0 ; do

  case "$1" in 

    -e|--errname)		shift
				if test $# -eq 0 ; then
				  printf "library-find: -e and --errname require an argument\\n" 1>&2
				  printf "try --help\\n" 1>&2
				  exit 1
				fi
				errname="$1"
				shift
				;;

    -s|--silent)	shift
    			silent=--silent
			;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "library-find: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "library-find: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -*)                 printf "library-find: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: library-find [options] [archive/]revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

rvnspec="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

library="`larch my-revision-library $silent -e library-find`"

archive_revision=`larch indicated-revision -e library-find -R "$archroot" -A "$archive" "$rvnspec"`

archive=${archive_revision%%/*}
revision=${archive_revision#*/}
category="`larch parse-package-name --basename \"$revision\"`"
branch="`larch parse-package-name \"$revision\"`"
version="`larch parse-package-name --package-version \"$revision\"`"

################################################################
# Do It
# 

if test -d "$library/$archive/$category/$branch/$version/$revision" ; then

  printf "%s\\n"  "$library/$archive/$category/$branch/$version/$revision"
  exit 0
    
else

  if test -z "$silent" ; then
    printf "\\n" 1>&2
    printf "%s: revision not in library\\n" "$errname" 1>&2
    printf "  archive: %s\\n" "$archive" 1>&2
    printf "  revision: %s\\n" "$revision" 1>&2
    printf "\\n" 1>&2
  fi

  exit 1

fi  


# tag: Tom Lord Sun Jan 20 00:09:29 2002 (library/library-find.sh)
#
