# 
# tagging-method.sh - print or change the tagging method
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "print or change a project tree tagging method\\n"
                printf "usage: tagging-method [options] [method] \\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -d --dir DIR                  cd to DIR first\\n"
                printf "\\n"
                printf "Change the method by which source files are identified in DIR (or\\n"
                printf "the current directory).\\n"
                printf "\\n"
                printf "METHOD must be one of:\\n"
                printf "\\n"
                printf "	names		-- use naming conventions only\\n"
                printf "	implicit	-- use naming conventions but permit\\n"
                printf "			   for inventory tags\\n"
                printf "	explicit	-- require explicit designation of source\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

dir=.

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "tagging-method: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    --)			shift
    			break
			;;
			
    -*)                 printf "tagging-method: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: tagging-method [options] [method]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  method="$1"
  shift
else
  method=
fi


################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$dir"
dir="`pwd`"

cd "$dir"
wdroot="`larch tree-root`"

if test ! -z "$method" ; then
  set=yes
else
  set=
  if test ! -e "$wdroot/{arch}/=tagging-method" ; then
    method=names
  else
    method="`cat \"$wdroot/{arch}/=tagging-method\" | grep -E -e \"^(explicit|implicit|names)[[:space:]]*$\" | tail -n 1`"
  fi
fi


case "$method" in
  names)	;;
  implicit)	;;
  explicit)	;;
  *)		if test -z "$method" ; then
		  method=names
		else
		  printf "tagging-method: unrecognized method\\n" 1>&2
  		  printf "  method: %s\\n" "$method" 1>&2
		  printf "\\n" 1>&2
		  printf "try --help\\n" 1>&2
		  exit 1
		fi
		;;
esac


################################################################
# Record or Print
# 

if test -z "$set" ; then
  printf "%s\\n" "$method"
else

  cd "$wdroot/{arch}"
  if test ! -e "=tagging-method" ; then
    printf "%s\n" "$method" > "=tagging-method"
  else
    mv "=tagging-method" ,tagging
    cat ,tagging \
    | awk -v method="$method" \
          'BEGIN {
		   changed = 0;
		 }
	   match($0, "^(explicit|implicit|names)[ \t]*") {
							   sub("explicit|implicit|names", method, $0);
							   changed = 1;
							 }
	   { print $0; } 
	   END {
		 if (!changed)
		   print method;
	       }'\
    > "=tagging-method"
    rm ,tagging
  fi
fi

