//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008,2009,2010 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>
#ifdef __FreeBSD__
	#include <signal.h>
#endif
#include <glib.h>

#include "atsend.h"
#include "btio.h"
#include "conf.h"
#include "common.h"
#include "utils.h"
#include "executor.h"
#include "dispatcher.h"

#include "pr_frontend.h"
#ifdef USE_L2CAP
	#include "pr_l2cap.h"
#endif

#define TMPDISCONN_TIME 60

extern CONF	conf;
extern char	tmp[MAXMAXLEN];
extern GAsyncQueue *q2disp;

char callerId	    [MAXLEN];
char frontEndBuf    [MAXLEN];

gboolean dispIsJoinable = TRUE;

int  callTimer	  = 0;

int  useCallId    = 1;
int  callOn       = 0;

int  gotExitSignal  = 0;

int  serverMode   = CLIENT_AT;  // By default use AT commands and devices in /dev

int  port         = 0;
static int  initialized  = 0;
int  isConnected  = 0;

int openPort(char * portIn) 
{
	logger("INF", "[DS]: openPort()");
        
        if (portIn == NULL) return 0;
        
        char* strPort;
        int ret = 0;

        // device should be in format:
        // 1. (AT mode)     /dev/something							used with AT mode commands
        // 2. (Server mode) socket:NNNN or 							used with java client
	//                  socket:/path/to/socket						used with java client
        // 3. (Server mode) bluetooth:NN or bluetooth (port will be = DEFAULT_BT_CHANNEL) 	used with java client
        // 4. local:/some/path									used with java client, like 1, but no AT commands
        // 5. ilirc:/dev/something 								like 1, but no AT commands
        // 6. stdio								                used with java client, like 1, but no AT commands

        if ((strPort = strstr(portIn, INET_SOCKET)) != NULL) {
        	
                strPort += strlen(INET_SOCKET);

		// check is it port or unix socket
		char * ch = strPort;
		int isPort = 1;
		while (*ch != '\0') {
			if (isdigit(*ch)) {
				ch++;
			} else {
				isPort = 0;
				break;
			}
		}
		
		if (isPort) {
                	port = atoi(strPort);

                	DEBUG2("[DS]: TCP Server mode. Use port %d\n", port);

                	if ((ret = openSocketPort(SERVER_TCP, port, NULL)) < 0) {
                		return -1;
                	}

                	serverMode = SERVER_TCP;
		} else {

                	/*sprintf(tmp, "Unix socket Server mode. Use socket %s\n", strPort);
                	logger("DBG", tmp);
		
                	if ((ret = openSocketPort(SERVER_UX, -1, strPort)) < 0) {
                		return -1;
                	}
			
			serverMode = SERVER_UX;
			*/
			printf("ERROR:incorrect port\n");
                        return -1;
			
		}
        } else if ((strPort = strstr(portIn, ILIRC_SOCKET)) != NULL) {
        	
                strPort += strlen(ILIRC_SOCKET);

                DEBUG2("[DS]: Unix socket client mode. Use port %s\n", strPort);

                if (unix_open_port(strPort) < 0) {
			printf("ERROR: open inputlircd socket\n");
                	return -1;
                }

                serverMode = CLIENT_ILIRC;

        } else if ((strPort = strstr(portIn, BT_SOCKET)) != NULL) {

        	strPort += strlen(BT_SOCKET);
        
                if (strstr(strPort, ":") == NULL) { // just "bluetooth
                	port = DEFAULT_BT_CHANNEL;
                } else {
                	strPort++;
                        port = atoi(strPort);
                }
                DEBUG2("[DS]: Bluetooth Sockets Server mode. Use channel %d", port);

                if ((ret = openSocketPort(SERVER_BT, port, NULL)) < 0) {
                	return -1;
                }
                serverMode = SERVER_BT;
		
       #ifdef USE_L2CAP
       } else if ((strPort = strstr(portIn, L2CAP_SOCKET)) != NULL) {

        	strPort += strlen(L2CAP_SOCKET);
        
                if (strstr(strPort, ":") == NULL) { // just "l2cap"
                	port = DEFAULT_L2CAP_PORT;
                } else {
                	strPort++;
                        port = atoi(strPort);
                }
                DEBUG2("[DS]: L2CAP Server mode. Use port %d", port);

                if ((ret = openL2capPort(port)) < 0) {
                	return -1;
                }
                serverMode = SERVER_L2CAP;
        #endif

        } else if ((strPort = strstr(portIn, UNIX_SOCKET)) != NULL){

		strPort += strlen(UNIX_SOCKET);

        	DEBUG2("[DS]: Serial Client mode (no AT). Use device %s", strPort);

                if (bt_open_port(strPort) < 0) {
                	printf("ERROR: open device\n");
                        return -1;
                }
		
                serverMode = CLIENT_NOAT;
		
        } else if ((strPort = strstr(portIn, STDIN_STREAM)) != NULL){

                serverMode = SERVER_STDIN;
                openStdin();

        } else {	//if ((strPort = strstr(portIn, AT_DEVICE)) != NULL){

        	DEBUG2("[DS]: Serial Client mode. Use device %s", portIn);

                if (bt_open_port(portIn) < 0) {
                	logger("ERR","[DS]: can not open RFCOMM port\n");
                        return -1;
                }
                serverMode = CLIENT_AT;
        }

        return 1;
}

void createDataDir() 
{
	char dd[542];
	char *t = getenv("HOME");
	strcpy(dd, t);
	strcat(dd, "/.anyRemote");

	DIR *d = opendir(dd);
	if (d) {
        	closedir(d);
        } else {
		mkdir(dd, S_IRWXU);
		if(getuid()==0 && conf.uid) { // do not create as superuser
			chown(dd,conf.uid,conf.gid);
		}
	}
}

// Then port will be closed from a forked child use 0
void closePort(int final) 
{
	if (serverMode == SERVER_BT  || 
            serverMode == SERVER_TCP || 
            serverMode == SERVER_UX) {
         	closeSocketPort(serverMode,final);
	#ifdef USE_L2CAP	
        } else if (serverMode == SERVER_L2CAP) {
                closeL2capPort(final);
	#endif
	} else if (serverMode == CLIENT_ILIRC) {
        	if (unix_close_port() < 0) {
                	if (final) logger("ERR","[DS]: Error on closing AF_UNIX socket\n");
                }
        } else if (serverMode == CLIENT_AT || serverMode == CLIENT_NOAT) {
        	if (bt_close_port(final) < 0) {
                	if (final) logger("ERR","[DS]: Error on closing port\n");
                }
        } else if (serverMode == SERVER_STDIN && final) {
        	closeStdin();
        }

	if (getFrontEnd() > 0) {
		if (final) sendToFrontEnd("Exiting");
                // else - just close socket from forked child
		disconnectFrontEnd();
	}
}

//
// retuencodeis used in case of exit only
//
static int proceedDisconnect(void) 
{
	int canForceExit = 0;
	if (serverMode == SERVER_BT  || 
	    serverMode == SERVER_TCP || 
	    serverMode == SERVER_UX) {
	    
	        logger("INF", "[DS]: Got exit event: send disconnect message");
		if (writeSocketConn("Set(disconnect);", 16) != EXIT_OK) {
			canForceExit = 1;
		}

	} else if (serverMode == CLIENT_AT) {

		sendCMER(CMER_OFF);
		canForceExit = 1;
		
	#ifdef USE_L2CAP	
	} else if (serverMode == SERVER_L2CAP) {

		if (writeL2capConn("Set(disconnect);", 16) != EXIT_OK) {
			canForceExit = 1;
		}
	#endif
	} else {
		canForceExit = 1;
	}
	return canForceExit;
}

static int checkMsgQueue() 
{
	//logger("DBG","[DS]: checkMsgQueue");
	
	// Verify commands from queue (timeout about 1/2 sec)
	dMessage* dm = (dMessage*) g_async_queue_try_pop(q2disp);
	if (dm != NULL) {
	        logger("DBG", "[DS]: Got event");
		
		if (dm->type == DM_EVENT_INIT) {
			logger("INF", "[DS]: Got init OK event");
			free(dm);
			return EXIT_INITOK;
		} else if (dm->type == DM_EVENT_EXIT) {
		
			logger("INF", "[DS]: Got exit event");
		
			gotExitSignal = 1;
		
			// in server mode wait client to close connection 
			// and then exit
			if (proceedDisconnect()) {
				logger("INF", "[DS]: Got exit event: exit immediately");
				closePort(1);

				free(dm);
				return EXIT_ABORT;
			}
		
		} else if (dm->type == DM_EVENT_DISCONNECT) {
		
			logger("INF", "[DS]: Got disconnect event");
			proceedDisconnect();
        		closePort(1);
		
			free(dm);
			return EXIT_DISCON;
			
		} else if (isConnected) {

			char *msgIn = (char*) dm->value;
		
	        	if (dm->type == DM_SET) {
			
	        		logger("DBG", "[DS]: Set()");
				writeSocketConn(msgIn, dm->size);
			
			} else if (dm->type == DM_SENDS) {

	        		logger("DBG", "[DS]: Send(string)");
				if (writeSocketConn(msgIn, dm->size) != EXIT_OK) {
					logger("DBG","[DS]: Fails in writeSocketConn()");
				}
	        
			} else if (dm->type == DM_SENDB) {
			
	        		logger("DBG", "[DS]: Send(bytes)");
				if (writeBytesSocketConn(msgIn) != EXIT_OK) {
					logger("DBG","[DS]: Fails in writeBytesSocketConn()");
				}
			 
			} else if (dm->type == DM_CKPD) {
	        	
	        		logger("DBG", "[DS]: SendCKPD");
				sendSeq(msgIn);
	        
			} else if (dm->type == DM_CMER) {
	        	
	        		logger("DBG", "[DS]: Send CMER");
				sendCMER(dm->size);
	        
			} else if (dm->type == DM_SETL2) {
			
	        		logger("DBG", "[DS]: Set() L2CAP");
				#ifdef USE_L2CAP
				if (writeL2capConn(msgIn, dm->size) != EXIT_OK) {
					logger("DBG","[DS]: Fails in writeBytesL2capConn()");
				} 
				#endif
			} else if (dm->type == DM_SENDBL2) {

	        		logger("DBG", "[DS]: Send(bytes) L2CAP");
				#ifdef USE_L2CAP
				if (writeBytesL2capConn(msgIn) != EXIT_OK) {
					logger("DBG","[DS]: Fails in writeBytesL2capConn()");
				} 
				#endif

			
			} else {
				logger("DBG","[DS]: Invalid message type");
			}	
		} else {
			logger("DBG", "[DS]: No connection. Skip event");
		}
		
		if (dm->value != NULL) free(dm->value);
		free(dm);
	}
	return EXIT_OK;	
}

void parseCommand(char* cmd) 
{
	char *prev, *next;

	if (cmd == NULL) return ;
	
	//skip lines starting with \n and \r
	if (cmd[0] == '\r') cmd++;
	if (cmd[0] == '\n') cmd++;

	// most common case
	if (!cmd[0]) return;
        
	// if recieved multiline command - handle line by line and return
	prev = cmd;
	next = strchr(cmd, '\r');
	if (next == NULL && serverMode != CLIENT_AT) {	// Java client will send +CKEV: 1,1; +CKEV: 1,0
		next = strchr(cmd, ';');
	}
	if (next == NULL && serverMode != CLIENT_AT) {
		next = strchr(cmd, '\3');		// end-of-text marker in CommandFusion
	}
	if (next) {
		logger("DBG", "[DS]: parseCommand multiline");

		char copy[1024];
		int len;

		do {	 
			len = next-prev;
	
			if (len >= 2) {
				memcpy(copy, prev, len);
				copy[len] = 0;
			
				// use recursion
				parseCommand(copy);
			}
			prev = next+1;
			next = strchr(prev, '\r');
			if (next == NULL && serverMode != CLIENT_AT) {	// Java client will send +CKEV: 1,1; +CKEV: 1,0
				next = strchr(prev, ';');
			}
			if (next == NULL && serverMode != CLIENT_AT) {	
				next = strchr(prev, '\3');		// end-of-text marker in CommandFusion
			}
		} while (next) ;

		// now return
		return;
	}
	
	logger("DBG","[DS]: -------------------- Command read --------------------");
	DEBUG2("[DS]: parseCommand >%s<", cmd);

	if (IS_OK(cmd)) {   // OK - nothing to do
		return;
	}
	
	if (strncmp(cmd, "AT+CKPD=", 8) == 0) {        // This is echo of sent message in AT mode; nothing to do
		return;
	}
	
	if (strncmp(cmd, "Msg:_DEBUG_(", 12) == 0) {   // This is debug message from java client; nothing to do
		return;
	}
	
	// hack to make it work correctly with Bemused clients
	int hack = 0;
	char c;
	if (strncmp(cmd, "VOLM", 4) == 0 && strlen(cmd) == 4) {  // read only VOLM without value to set
        	logger("DBG","[DS]: read one more byte");
                char ret;
                if ((ret = bt_readchar(&c,500000)) < 0) {
                	DEBUG2("[DS]: parseCommand: Bemused hack: read < 0");
                } else {
                	sprintf(tmp, "[DS]: parseCommand: Bemused hack: read >%c<", c);
                	logger("DBG",tmp);
 			hack = 1;
		}
	}
	
        // This is heartbeat of iViewer; answer to it fast
	if (strncmp(cmd, "h=0", 3) == 0 && getIViewer()) {  
		logger("DBG", "[DS]: Got heartbeat message from iViewer; auto-reply to it");
		 
		// it is a trick, we do not want to waste a time to handle message 
		// through normal "shell" cycle
		sendIViewerHeartbeat();
		
		// propagate event to executor anyway 
		// (to be able to disconnect in case of client dying)
	}

 
 	eMessage* em = malloc(sizeof(eMessage));
	em->type  = EM_KEY;
	if (hack) {
		em->value = malloc(strlen(cmd)+2);
		strcpy(em->value,cmd);
		((char*)em->value)[strlen(cmd)]   = c; 
		((char*)em->value)[strlen(cmd)+1] = '\0'; 
	} else {
		em->value = strdup(cmd);
	}
	logger("DBG", "[DS]: Send message to executor thread");
	sendToExecutor(em);
	
	return;
}

static int read_command(char* buf, int l) 
{
	int ch, len = 0;
	//printf("read_command\n");
	
	if (serverMode == SERVER_STDIN) {
		/* this blocks any other operation
                char* r = fgets((char*)buf, l, (FILE*)stdin);
	        printf("read_command SERVER_STDIN %d\n", len);
		if (r == NULL) {
			buf[0] = 0;
			printf("read_command SERVER_STDIN empty\n");
                	return 0;
		} else {
		        len = strlen(buf);
			printf("read_command SERVER_STDIN %d\n", len);
		}
                */

                int ret = readStdin(buf,l);
		if (ret == 0) {
			buf[0] = 0;
                	return 0;
		} else {
		        len = strlen(buf);
                        DEBUG2("[DS]: read_command SERVER_STDIN %d %s\n", len, buf);
                }
	} else {

        	char* p = buf;

        	while (len < l) {
        		char c;
        		ch = bt_readchar(&c,100);
		
                	if (ch == EOF) {
                		buf[0] = 0;
                        	return EOF;
                	} else if (ch == EOF-1) {
                		break;
                	} else if (ch >= 0 && (c == '\r' || c == ';')) {
               			break;
                	} else  if (ch >= 0 && c != '\r' && c != ';') {
                		*p++ = c;
                        	len++;
			}
        	}
	}
        buf[len] = '\0';
        return len;
}

int initPort()
{
	char answer[1024];
        char cmd[32];

        logger("DBG","[DS]: initPort");

        if (serverMode == SERVER_BT  || 
	    serverMode == SERVER_TCP || 
	    serverMode == SERVER_UX) {	// Wait for incoming connection
                logger("INF", "[DS]: Server mode: Waiting connection");
                return listenAndAcceptSocketConn(serverMode);
        } 

	#ifdef USE_L2CAP	
        if (serverMode == SERVER_L2CAP) {
                logger("INF", "[DS]: Server mode/L2CAP: Waiting connection");
                return listenAndAcceptL2capConn();
	} 
	#endif

	if (serverMode == CLIENT_ILIRC || 
	    serverMode == CLIENT_NOAT  ||
	    serverMode == SERVER_STDIN) {
		logger("INF", "[DS]: Unix client: init OK");
		return 1;	// nothing to init
	}
	
        answer[0] = '\0';
        sendSyncCommand("ATZ\r", answer, sizeof(answer), NULL);
        //sendSyncCommand("AT\r", answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
        	printf("ERROR: AT -> %s\n", answer);
                return -1;
        }
	
        sendSyncCommand("ATE0\r", answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
        	printf("ERROR: ATE0 -> %s\n", answer);
                return -1;
        }
        char* ptr = getCharset();
        if (ptr) {
        	sprintf(cmd,"%s\"%s\"\r", DEF_AT_CSCS, ptr); 
                sendSyncCommand(cmd, answer, sizeof(answer), NULL);
                if(!IS_OK(answer)) {
                	printf("ERROR: Can't set charset to %s\n", ptr);
                }     
		free(ptr);
        }

        // Determine model
        sprintf(cmd,"%s\r", DEF_AT_CGMI); 
        sendSyncCommand(cmd, answer, sizeof(answer), NULL);

        //Set model in conf. data
        setModel(answer);
	
	if (getModel() == MODEL_MOTOROLA) {	// Motorola RIZR Z3 needs to set MODE=2 to allow AT+CLIP command
		// do not care about responce
		sendSyncCommand("AT+MODE=2\r", answer, sizeof(answer), NULL);
	}
	
        sprintf(cmd,"%s\r", DEF_AT_CLIP); 
        sendSyncCommand(cmd, answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
            printf("ERROR: Can't set CLIP ON\n");
        }


	int ret = sendCMER(CMER_ON);
        if(ret != EXIT_OK) {
        	printf("ERROR: fails in set event reporting on");
                return -1;
        }

        // Siemens S55 needs additional AT+CMEC=2 to make AT+CKPD works
        // not sure about other Siemens phones
        if (getModel() == MODEL_SIEMENS) {
        	sprintf(cmd,"%s\r", DEF_AT_CMEC); 
                sendSyncCommand(cmd, answer, sizeof(answer), NULL);
                if(!IS_OK(answer)) {
                	printf("ERROR: ON AT+CMEC\n");
                }
        }

        // Will set global callerId [MAXLEN];
        ret = getClip(callerId);
        if(ret == EXIT_NOK) {       // Got ERROR on AT+CLCC; probably phone does not supports this command
        	printf("ERROR: fails in getClip\n");
                useCallId = 0;
        }
        return 1;
}

gpointer dispatcherRoutine(gpointer thread)
{
	char answer[MAXCMDLEN];
	int ret = EXIT_OK;
	int retRead;

	logger("DBG","[DS]: start dispatcher thread");

	setCmdFiles();
	
	// wait init ok event
	while (1) {
		ret = checkMsgQueue();
		if (ret == EXIT_ABORT) {
			return NULL;
		} else if (ret == EXIT_INITOK) {
			break;
		}
		logger("DBG","[DS]: wait init OK event");
		usleep(50000);
	}
	
	int dmn = autoConnect();
	int rs  = getRetrySecs();
	strcpy(callerId, "NO CALLER ID");
	
	while (1) {
	
		char* dv = getDevice();
		
		logger("DBG","[DS]: ************ outer loop **********");
		
		// try to open port
		if(openPort(dv) == 1) {   // Open device
		
        	    if (initialized == 0) { 
		    		    
		        // setgid
		        if(conf.uid && getuid()==0){
				DEBUG2("[DS]: setuid/setgid %d,%d",conf.uid,conf.gid);
				setgid(conf.gid);	
				setuid(conf.uid);
		        }
			
                	initialized++;
        	    }
	
	  	    dispIsJoinable = FALSE;
		    
		    if (initPort() == 1 ) {  // Init modem: AT, ATE0, AT+CMER, in server mode waits for connection

 		       dispIsJoinable = TRUE;
		       
                       isConnected = 1;
                        
			if (usePassword()) {
				int PassOk = 0;
				
				char *pass = getStoredPass();
				
				if (pass) {
				
					sendToFrontEnd("Verifying password");
					
					ret = writeSocketConn("Get(password);", 14);
                			if (ret != EXIT_OK) {
                				logger("DBG","[DS]: Fails in writeSocketConn()");
	    					continue;
                			}
					
					while (1) {
						retRead = read_command(answer, sizeof(answer));
				
						if (retRead == EOF) {
							sendToFrontEnd("Disconnected");
							sendDisconnect();
							break;
						} else if (retRead > 0) {
							DEBUG2("[DS]: Password => %s\n",answer);
							DEBUG2("[DS]: Password => %s\n",answer+16);
							DEBUG2("[DS]: Compare to => %s\n",pass);
				
							if (memcmp(answer,   DEF_MSG, 4) == 0 && 
				    	                    memcmp(answer+4, "_PASSWORD_(,", 12) == 0) {	    // Got verification response from Java client
						                
								int plen = strlen(pass);
								if (memcmp(answer+16, pass, plen) == 0) {	// _PASSWORD_(,<password here>) 
									logger("INF","[DS]: Password OK");
									PassOk = 1;
								} else {
									logger("INF","[DS]: Verification failed. Incorrect password.");
									sendToFrontEnd("Verification failed");
								}
							
							} else {
								logger("INF","[DS]: Verification failed. Incorrect response.");
								sendToFrontEnd("Verification failed");
							}
							break;
						}
						
						if (checkMsgQueue() == EXIT_ABORT) {
							free(dv);
							return NULL;
						}

						usleep(50000);
					}
				} else {
					logger("DBG","[DS]: No password specified. Skip check.");
				}
				
				free(pass);
				if (PassOk == 0) {
				        writeSocketConn("Set(title,PASSWORD INVALID);", 28);
					
					sleep(1);
					
					closePort(1);
					isConnected = 0;
					
					sleep(5);
					continue;	
				}
			}
			
			sendToFrontEnd("Connected");
			sendEventToExecutor(ID_EVT_CONNECT);
		        
			logger("INF","[DS]: Start event forwarding");
		
			while (1) {	 
		    		//logger("DBG","[DS]: main loop");
		    
		    		retRead = read_command(answer, sizeof(answer));
				//printf("GOT %d\n",retRead);
				
		    		if (retRead == EOF) {
	    				logger("INF", "[DS]: Got disconnected");
					printf("Got disconnected\n");
					
					freeBtAddress();
					
					if (gotExitSignal) {
			   	 		logger("INF", "[DS]: Got signal, exiting");
						closePort(1);
						free(dv);
						return NULL;
						
					}
					sendToFrontEnd("Disconnected");
					sendDisconnect();
					sendEventToExecutor(ID_EVT_DISCONNECT);
						    		    
	    				if (serverMode == CLIENT_AT    || 
					    serverMode == CLIENT_ILIRC || 
					    serverMode == CLIENT_NOAT) {  // In server mode we will reuse already opened socket
					    
	    		    			logger("INF", "[DS]: Closing the port");
			    			closePort(1);
					}   
	    				ret = EXIT_NOK;
                                        
                                        isConnected = 0;
	    		
	    				break;
		    		} else if (retRead > 0) {
	    	    
		    			if (serverMode == CLIENT_NOAT) {	// in IR communication a lot of 'empty' chars (= -1) could be added
						char * k2 = answer;
						while (*k2 != '\0') {
							if (*k2 == -1) {
								*k2 = '\r';
							}
							k2++;
						}
					}

					parseCommand(answer);
					if (getHttp()) {
					
					 	eMessage* em = malloc(sizeof(eMessage));
						em->type  = EM_AS_IS;
						em->value = strdup("End();");
						sendToExecutor(em);
					}
	    	    		}
		    
		    		// Is call still active (timeout about 1 seconds) ?
		    		if (callOn) {
	    				if (callTimer >= 20) {
	    		   			 logger("DBG", "[DS]: Get call list again");
	    		    
	    		    			// Will set global callerId     [MAXLEN];
	    		    			ret = getClip(callerId);
	    		    			if (ret == EXIT_EXACT) {
	    						DEBUG2("[DS]: Call still active (%s)", callerId);
	    		    			} else {
	    						logger("DBG","[DS]: Call was finished");
	    						callOn = 0;
	    						strcpy(callerId, "NO CALLER ID");

							sendEventToExecutor(ID_EVT_ENDCALL);
	    		    			}

	    		   			callTimer = 0;
	    				} else {
	    		    			callTimer++;
	    				}
	    	    		}
	    	    		
				ret = checkMsgQueue();
				if (ret == EXIT_ABORT) {
					free(dv);
					return NULL;
				} else if (ret == EXIT_DISCON) {
					break;
				}
				
	    	    		// Main loop timer (1/50 of second)
		    		usleep(20000);
		 	}  
		    }	// init port
		}	// open port
		
		free(dv);
		
		printf("Connection closed or lost\n");

		// Can't open port or it closed again
		if (dmn || 
		    serverMode == SERVER_BT   || 
		    #ifdef USE_L2CAP 
		    serverMode == SERVER_L2CAP || 
		    #endif
		    serverMode == SERVER_TCP  ||
		    serverMode == SERVER_UX   || 
		    serverMode == CLIENT_NOAT ||
		    ret == EXIT_DISCON) {
		    
	    		int timeout;
	    	
	    		if (serverMode == SERVER_BT  ||  
			    #ifdef USE_L2CAP 
		            serverMode == SERVER_L2CAP ||
			    #endif
		            serverMode == SERVER_TCP ||
		            serverMode == SERVER_UX  || 
		            serverMode == CLIENT_NOAT) {
			    
				timeout = 5;	// wait only 5 seconds  
				 
			} else if (ret == EXIT_DISCON) {
			
	    	   		timeout = TMPDISCONN_TIME;
	    	    		ret = EXIT_OK;
	    		 	    		
			} else {
	    	    		timeout = rs;
	    		}

	    		INFO2("[DS]: Wait %d seconds to connect/open server socket ...", timeout);
			printf("Wait %d seconds to connect/open server socket ...\n", timeout);
			
			dispIsJoinable = FALSE;
	    		sleep(timeout);
			dispIsJoinable = TRUE;
	    	
	    	} else {	// Proceed to exit
			printf("Exiting ...\n");
		        sendAbort();
	    		break;
	    	}
	}

	// Finish all
	logger("INF", "[DS]: Stop dispatcher ...");
	closePort(1);
	
	return NULL;
}

void sendToDispatcher(dMessage *buf)
{
	if (q2disp == NULL || buf == NULL) {
		return;
	}

	DEBUG2("send to dispatcher %d", buf->type );
	
	g_async_queue_push(q2disp,buf);

	//free(buf);
}
