/*
 * Amanda, The Advanced Maryland Automatic Network Disk Archiver
 * Copyright (c) 1991-1998 University of Maryland at College Park
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: James da Silva, Systems Design and Analysis Group
 *			   Computer Science Department
 *			   University of Maryland at College Park
 */
/*
 * $Id: tapeio.c,v 1.20.4.7 2000/12/13 18:33:31 jrjackson Exp $
 *
 * implements tape I/O functions
 */
#include "amanda.h"

#include "tapeio.h"
#include "fileheader.h"
#ifndef R_OK
#define R_OK 4
#define W_OK 2
#endif

static char *errstr = NULL;

static int no_op_tapefd = -1;

#if defined(HAVE_BROKEN_FSF)
/*
 * tapefd_fsf_broken -- handle systems that have a broken fsf operation
 * and cannot do an fsf operation unless they are positioned at a tape
 * mark (or BOT).  This shows up in amrestore as I/O errors when skipping.
 */

static int
tapefd_fsf_broken(tapefd, count)
int tapefd;
int count;
{
    char buffer[TAPE_BLOCK_BYTES];
    int len = 0;

    if(tapefd == no_op_tapefd) {
	return 0;
    }
    while(--count >= 0) {
	while((len = tapefd_read(tapefd, buffer, sizeof(buffer))) > 0) {}
	if(len < 0) {
	    break;
	}
    }
    return len;
}
#endif

#ifdef UWARE_TAPEIO 

#include <sys/tape.h>

int tapefd_rewind(tapefd)
int tapefd;
{
    int st;
    return (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, T_RWD, &st);
}

int tapefd_fsf(tapefd, count)
int tapefd, count;
/*
 * fast-forwards the tape device count files.
 */
{
#if defined(HAVE_BROKEN_FSF)
    return tapefd_fsf_broken(tapefd, count);
#else
    int st;
    int c;
    int status;

    for ( c = count; c ; c--)
        if (status = (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, T_SFF, &st))
            break;

    return status;
#endif
}

int tapefd_weof(tapefd, count)
int tapefd, count;
/*
 * write <count> filemarks on the tape.
 */
{
    int st;
    int c;
    int status;

    for ( c = count; c ; c--)
        if (status = (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, T_WRFILEM, &st))
            break;

    return status;
}

#else
#ifdef AIX_TAPEIO

#include <sys/tape.h>

int tapefd_rewind(tapefd)
int tapefd;
{
    struct stop st;

    st.st_op = STREW;
    st.st_count = 1;

    return (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, STIOCTOP, &st);
}

int tapefd_fsf(tapefd, count)
int tapefd, count;
/*
 * fast-forwards the tape device count files.
 */
{
#if defined(HAVE_BROKEN_FSF)
    return tapefd_fsf_broken(tapefd, count);
#else
    struct stop st;

    st.st_op = STFSF;
    st.st_count = count;

    return (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, STIOCTOP, &st);
#endif
}

int tapefd_weof(tapefd, count)
int tapefd, count;
/*
 * write <count> filemarks on the tape.
 */
{
    struct stop st;

    st.st_op = STWEOF;
    st.st_count = count;

    return (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, STIOCTOP, &st);
}

#else /* AIX_TAPEIO */
#ifdef XENIX_TAPEIO

#include <sys/tape.h>

int tapefd_rewind(tapefd)
int tapefd;
{
    int st;
    return (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, MT_REWIND, &st);
}

int tapefd_fsf(tapefd, count)
int tapefd, count;
/*
 * fast-forwards the tape device count files.
 */
{
#if defined(HAVE_BROKEN_FSF)
    return tapefd_fsf_broken(tapefd, count);
#else
    int st;
    int c;
    int status;

    for ( c = count; c ; c--)
	if (status = (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, MT_RFM, &st))
	    break;

    return status;
#endif
}

int tapefd_weof(tapefd, count)
int tapefd, count;
/*
 * write <count> filemarks on the tape.
 */
{
    int st;
    int c;
    int status;

    for ( c = count; c ; c--)
	if (status = (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, MT_WFM, &st))
	    break;

    return status;
}


#else	/* ! AIX_TAPEIO && !XENIX_TAPEIO */


#include <sys/mtio.h>

int tapefd_rewind(tapefd)
int tapefd;
{
    struct mtop mt;
    int rc, cnt;

    mt.mt_op = MTREW;
    mt.mt_count = 1;

    /* EXB-8200 drive on FreeBSD can fail to rewind, but retrying
     * won't hurt, and it will usually even work! */
    for(cnt = 0; cnt < 10; ++cnt) {
	rc = (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, MTIOCTOP, &mt);
	if (rc == 0)
	    break;
	sleep(3);
    }
    return rc;
}

int tapefd_fsf(tapefd, count)
int tapefd, count;
/*
 * fast-forwards the tape device count files.
 */
{
#if defined(HAVE_BROKEN_FSF)
    return tapefd_fsf_broken(tapefd, count);
#else
    struct mtop mt;

    mt.mt_op = MTFSF;
    mt.mt_count = count;

    return (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, MTIOCTOP, &mt);
#endif
}

int tapefd_weof(tapefd, count)
int tapefd, count;
/*
 * write <count> filemarks on the tape.
 */
{
    struct mtop mt;

    mt.mt_op = MTWEOF;
    mt.mt_count = count;

    return (tapefd == no_op_tapefd) ? 0 : ioctl(tapefd, MTIOCTOP, &mt);
}




#endif /* !XENIX_TAPEIO */
#endif /* !AIX_TAPEIO */
#endif /* !UWARE_TAPEIO */



int tape_open(filename, mode)
     char *filename;
     int mode;
{
#ifdef HAVE_LINUX_ZFTAPE_H
    struct mtop mt;
#endif /* HAVE_LINUX_ZFTAPE_H */
    int ret = 0, delay = 2, timeout = 200;
    if (mode == 0 || mode == O_RDONLY)
	mode = O_RDONLY;
    else
	mode = O_RDWR;
#if 0
    /* Since we're no longer using a special name for no-tape, we no
       longer need this */
    if (strcmp(filename, "/dev/null") == 0) {
	filename = "/dev/null";
    }
#endif
    do {
	ret = open(filename, mode);
	/* if tape open fails with errno==EAGAIN, EBUSY or EINTR, it
	 * is worth retrying a few seconds later.  */
	if (ret >= 0 ||
	    (1
#ifdef EAGAIN
	     && errno != EAGAIN
#endif
#ifdef EBUSY
	     && errno != EBUSY
#endif
#ifdef EINTR
	     && errno != EINTR
#endif
	     ))
	    break;
	sleep(delay);
	timeout -= delay;
	if (delay < 16)
	    delay *= 2;
    } while (timeout > 0);
    if (strcmp(filename, "/dev/null") == 0) {
	no_op_tapefd = ret;
    } else {
	no_op_tapefd = -1;
    }
#ifdef HAVE_LINUX_ZFTAPE_H
    /* 
     * switch the block size for the zftape driver (3.04d) 
     * (its default is 10kb and not TAPE_BLOCK_BYTES=32kb) 
     *        A. Gebhardt <albrecht.gebhardt@uni-klu.ac.at>
     */
    if (no_op_tapefd < 0 && ret >= 0 && is_zftape(filename) == 1)
	{
	    mt.mt_op = MTSETBLK;
	    mt.mt_count = TAPE_BLOCK_BYTES;
	    ioctl(ret, MTIOCTOP, &mt);    
	}
#endif /* HAVE_LINUX_ZFTAPE_H */
    return ret;
}

int tapefd_read(tapefd, buffer, count)
int tapefd, count;
void *buffer;
{
    return read(tapefd, buffer, count);
}

int tapefd_write(tapefd, buffer, count)
int tapefd, count;
void *buffer;
{
    return write(tapefd, buffer, count);
}

int tapefd_close(tapefd)
int tapefd;
{
    if (tapefd == no_op_tapefd) {
	no_op_tapefd = -1;
    }
    return close(tapefd);
}

void tapefd_resetofs(tapefd)
int tapefd;
{
    /* 
     * this *should* be a no-op on the tape, but resets the kernel's view
     * of the file offset, preventing it from barfing should we pass the
     * filesize limit (eg OSes with 2 GB filesize limits) on a long tape.
     */
    lseek(tapefd, (off_t) 0L, SEEK_SET);
}

char *tape_rewind(devname)
char *devname;
{
    int fd;

    if((fd = tape_open(devname, O_RDONLY)) == -1) {
	errstr = newstralloc(errstr, "no tape online");
	return errstr;
    }

    if(tapefd_rewind(fd) == -1) {
	errstr = newstralloc2(errstr, "rewinding tape: ", strerror(errno));
	tapefd_close(fd);
	return errstr;
    }

    tapefd_close(fd);
    return NULL;
}


char *tape_fsf(devname, count)
char *devname;
int count;
{
    int fd;
    char count_str[NUM_STR_SIZE];

    if((fd = tape_open(devname, O_RDONLY)) == -1) {
	errstr = newstralloc(errstr, "no tape online");
	return errstr;
    }

    if(tapefd_fsf(fd, count) == -1) {
	ap_snprintf(count_str, sizeof(count_str), "%d", count);
	errstr = newvstralloc(errstr,
			      "fast-forward ", count_str, "files: ",
			      strerror(errno),
			      NULL);
	tapefd_close(fd);
	return errstr;
    }

    tapefd_close(fd);
    return NULL;
}

char *tapefd_rdlabel(tapefd, datestamp, label)
int tapefd;
char **datestamp, **label;
{
    int rc;
    char buffer[TAPE_BLOCK_BYTES];
    dumpfile_t file;

    amfree(*datestamp);
    amfree(*label);

    if(tapefd_rewind(tapefd) == -1) {
	errstr = newstralloc2(errstr, "rewinding tape: ", strerror(errno));
	return errstr;
    }

    if((rc = tapefd_read(tapefd, buffer, sizeof(buffer))) == -1) {
	errstr = newstralloc2(errstr, "reading label: ", strerror(errno));
	return errstr;
    }

    /* make sure buffer is null-terminated */
    if(rc == sizeof(buffer)) rc--;
    buffer[rc] = '\0';

    if (tapefd == no_op_tapefd) {
	strcpy(file.datestamp, "X");
	strcpy(file.name, "/dev/null");
    } else {
	parse_file_header(buffer, &file, sizeof(buffer));
	if(file.type != F_TAPESTART) {
	    errstr = newstralloc(errstr, "not an amanda tape");
	    return errstr;
	}
    }
    *datestamp = stralloc(file.datestamp);
    *label = stralloc(file.name);

    return NULL;
}


char *tape_rdlabel(devname, datestamp, label)
char *devname, **datestamp, **label;
{
    int fd;

    if((fd = tape_open(devname, O_RDONLY)) == -1) {
	errstr = newstralloc(errstr, "no tape online");
	return errstr;
    }

    if(tapefd_rdlabel(fd, datestamp, label) != NULL) {
	tapefd_close(fd);
	return errstr;
    }

    tapefd_close(fd);
    return NULL;
}


char *tapefd_wrlabel(tapefd, datestamp, label)
int tapefd;
char *datestamp, *label;
{
    int rc;
    char buffer[TAPE_BLOCK_BYTES];
    dumpfile_t file;

    if(tapefd_rewind(tapefd) == -1) {
	errstr = newstralloc2(errstr, "rewinding tape: ", strerror(errno));
	return errstr;
    }

    fh_init(&file);
    file.type=F_TAPESTART;
    strncpy(file.datestamp, datestamp, sizeof(file.datestamp)-1);
    file.datestamp[sizeof(file.datestamp)-1] = '\0';
    strncpy(file.name, label, sizeof(file.name)-1);
    file.name[sizeof(file.name)-1] = '\0';
    write_header(buffer,&file,sizeof(buffer));

    if((rc = tapefd_write(tapefd, buffer, sizeof(buffer))) != sizeof(buffer)) {
	errstr = newstralloc2(errstr, "writing label: ",
			      (rc != -1) ? "short write" : strerror(errno));
	return errstr;
    }

    return NULL;
}


char *tape_wrlabel(devname, datestamp, label)
char *devname, *datestamp, *label;
{
    int fd;

    if((fd = tape_open(devname, O_WRONLY)) == -1) {
	if(errno == EACCES) {
	    errstr = newstralloc(errstr,
				 "writing label: tape is write-protected");
	} else {
	    errstr = newstralloc2(errstr, "writing label: ", strerror(errno));
	}
	tapefd_close(fd);
	return errstr;
    }

    if(tapefd_wrlabel(fd, datestamp, label) != NULL) {
	tapefd_close(fd);
	return errstr;
    }

    tapefd_close(fd);
    return NULL;
}


char *tapefd_wrendmark(tapefd, datestamp)
int tapefd;
char *datestamp;
{
    int rc;
    char buffer[TAPE_BLOCK_BYTES];
    dumpfile_t file;

    fh_init(&file);
    file.type=F_TAPEEND;
    strncpy(file.datestamp, datestamp, sizeof(file.datestamp)-1);
    file.datestamp[sizeof(file.datestamp)-1] = '\0';
    write_header(buffer, &file,sizeof(buffer));

    if((rc = tapefd_write(tapefd, buffer, sizeof(buffer))) != sizeof(buffer)) {
	errstr = newstralloc2(errstr, "writing endmark: ",
			      (rc != -1) ? "short write" : strerror(errno));
	return errstr;
    }

    return NULL;
}


char *tape_wrendmark(devname, datestamp)
    char *devname, *datestamp;
{
    int fd;

    if((fd = tape_open(devname, O_WRONLY)) == -1) {
	errstr = newstralloc2(errstr, "writing endmark: ",
			      (errno == EACCES) ? "tape is write-protected"
						: strerror(errno));
	return errstr;
    }

    if(tapefd_wrendmark(fd, datestamp) != NULL) {
	tapefd_close(fd);
	return errstr;
    }

    tapefd_close(fd);
    return NULL;
}


char *tape_writable(devname)
char *devname;
{
    int fd;

    /* first, make sure the file exists and the permissions are right */

    if(access(devname, R_OK|W_OK) == -1) {
	errstr = newstralloc(errstr, strerror(errno));
	return errstr;
    }

    if((fd = tape_open(devname, O_WRONLY)) == -1) {
	errstr = newstralloc(errstr,
			     (errno == EACCES) ? "tape write-protected"
					       : strerror(errno));
	return errstr;
    }

    if(tapefd_close(fd) == -1) {
	errstr = newstralloc(errstr, strerror(errno));
	return errstr;
    }

    return NULL;
}

#ifdef HAVE_LINUX_ZFTAPE_H
/*
 * is_zftape(filename) checks if filename is a valid ftape device name. 
 */
int is_zftape(filename)
     const char *filename;
{
    if (strncmp(filename, "/dev/nftape", 11) == 0) return(1);
    if (strncmp(filename, "/dev/nqft",    9) == 0) return(1);
    if (strncmp(filename, "/dev/nrft",    9) == 0) return(1);
    return(0);
}
#endif /* HAVE_LINUX_ZFTAPE_H */

