//
// File:        RepositoryFactory.java
// Package:     gov.llnl.babel.repository
// Revision:    @(#) $Id: RepositoryFactory.java 4434 2005-03-17 17:05:29Z epperly $
// Description: singleton factory for creating repository interfaces
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.repository;

import gov.llnl.babel.repository.FileRepository;
import gov.llnl.babel.repository.Repository;
import gov.llnl.babel.repository.RepositoryException;
import gov.llnl.babel.repository.WebRepository;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * <code>RepositoryFactory</code> is a singleton factory that manages the
 * creation of repository interfaces based on a repository URI.  A URI that
 * starts with "http:" or "https:" specifies a web repository.  A URI that
 * starts with "file:" creates a file repository.  All other URIs are assumed
 * to be file repositories.
 */
public class RepositoryFactory {
   private static RepositoryFactory s_instance = null;

   /**
    * Create a new instance of the repository factory.  Although singleton
    * classes do not typically define a public constructor, this version does
    * so to support multiple repository factories in the same application.
    * Most implementations, however, will not directly create a factory
    * through the constructor and will instead use the singleton functions
    * <code>getInstance</code> and <code>setInstance</code>.
    */
   public RepositoryFactory() {
   }

   /**
    * Return the singleton instance of the repository factory.  If the
    * symbol table instance has not yet been created, then it will be
    * created by this call.
    */
   public static RepositoryFactory getInstance() {
      if (s_instance == null) {
         s_instance = new RepositoryFactory();
      }
      return s_instance;
   }

   /**
    * Set the singleton instance of the repository factory.  Use this
    * function only if you want to change the default factory implementation
    * or use multiple factories in the same application.  Otherwise,
    * <code>getInstance</code> will automatically create the repository
    * factory for you.
    */
   public static void setInstance(RepositoryFactory instance) {
      s_instance = instance;
   }

   /**
    * Create a new repository object based on the URI.  Web repositories
    * are created using a URL starting with "http:" or "https:" and file
    * repositories are created using a URL starting with "file:" or any
    * other prefix.  A <code>RepositoryException</code> is thrown if any
    * error is detected in creating the repository.
    */
   public Repository createRepository(String uri) throws RepositoryException {
      Repository repository = null;
      try {
         URL url = new URL(uri);
         String protocol = url.getProtocol();
         if ((protocol == null) || (protocol.equals(""))) {
            repository = new FileRepository(uri);
         } else if (protocol.equals("http") || protocol.equals("https")) {
            repository = new WebRepository(uri);
         } else if (protocol.equals("file")) {
            repository = new FileRepository(url.getFile());
         } else {
            repository = new FileRepository(uri);
         }
      } catch (MalformedURLException ex) {
         repository = new FileRepository(uri);
      }
      return repository;
   }
}
