/*
 *  misc.c:		Miscellaneous functions
 *
 *  Written by:		Ullrich Hafner
 *		
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1998/07/21 08:52:46 $
 *  $Author: hafner $
 *  $Revision: 1.9 $
 *  $State: Exp $
 */

#include "config.h"

#if HAVE_STDLIB_H
#	include <stdlib.h>
#endif /* not HAVE_STDLIB_H */
#if HAVE_STRING_H
#	include <string.h>
#else /* not HAVE_STRING_H */
#	include <strings.h>
#endif /* not HAVE_STRING_H */
#if HAVE_SYS_TYPES_H
#	include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */

#include <unistd.h>
#include <pwd.h>

#include "misc.h"

#include "error.h"

void *
Calloc (size_t n, size_t size)
/*
 *  Allocate memory like calloc ().
 *
 *  Return value: Pointer to the new block of memory on success,
 *		  else terminate the program.
 */
{
   void		*ptr;			/* pointer to the new memory block */

   if (n <= 0 || size <= 0)
      error ("Can't allocate memory for %d items of size %d",
	     (unsigned) n, (unsigned) size);

   ptr = calloc (n, size);
   if (ptr == NULL)
      error ("Out of memory!");

   return ptr;
}

void
Free (void *memory)
/*
 *  Free memory given by the pointer 'memory'
 *
 *  No return value.
 */
{
   if (memory != NULL)
      free (memory);
   else
      warning ("Can't free memory block <NULL>.");
}

#ifndef HAVE_STRDUP
char *
strdup (const char *s)
{
   assert (s);
   
   return strcpy (Calloc (strlen (s) + 1, sizeof (char)), s);
}
#endif /* not HAVE_STRDUP */

char *
get_gnustep_path (const char *domain)
/*
 *  Generate path to WindowMaker config file
 *
 *  Return value:
 *	pointer to char array containing generated path
 */
{
   char *path;
   char *gspath;

   gspath = getenv ("GNUSTEP_USER_ROOT");
   if (gspath)
   {
      path = Calloc (strlen (gspath) + strlen ("/Defaults/")
		     + strlen (domain) + 1, sizeof (char));
      
      strcpy (path, gspath);
      strcat (path, "/");
   }
   else
   {
      char *home;
	
      home = expand_tilde ("~/");
      if (streq (home, "~/"))
	 error ("Can't find HOME directory.");

      path = Calloc (strlen (home) + strlen ("/GNUstep/Defaults/")
		     + strlen (domain) + 1, sizeof (char));

      strcpy (path, home);
      strcat (path, "/GNUstep/");
      Free (home);
   }
   strcat (path, "Defaults/");
   strcat (path, domain);

   return path;
}

#ifndef HAVE_STRCASECMP
bool_t
strcaseeq (const char *s1, const char *s2)
{
   bool_t	matched;
   char		*ls1, *ls2, *ptr;

   assert (s1);
   assert (s2);
   
   ls1 = strdup (s1);
   ls2 = strdup (s2);
   
   for (ptr = ls1; *ptr; ptr++)
      *ptr = tolower (*ptr);
   for (ptr = ls2; *ptr; ptr++)
      *ptr = tolower (*ptr);

   matched = streq (ls1, ls2) ? YES : NO;

   Free (ls1);
   Free (ls2);
   
   return matched;
}
#endif /* not HAVE_STRCASECMP */

char *
expand_tilde (const char *name)
/*
 *  Try to expand tilde (~) in filename.
 *
 *  Return value:
 *	string with expanded path 
 */
{
   struct passwd *user = NULL;
   
   assert (name);

   if (name [0] != '~'
       && strncmp (name, "$HOME", 5)
       && strncmp (name, "$(HOME)", 7))
      return strdup (name);		/* nothing to do */
   
   if (name [1] == '/' || !name [1]	/* users home directory */
       || strncmp (name, "$HOME", 5) == 0
       || strncmp (name, "$(HOME)", 7) == 0)
   {
      char *homedir;
      
      if (name [0] == '~')
	 name += 1;			/* skip ~ */
      else if (strncmp (name, "$HOME", 5) == 0)
	 name += 5;
      else if (strncmp (name, "$(HOME)", 7) == 0)
	 name += 7;

      if ((homedir = getenv ("HOME")))
      {
	 char *path = Calloc (strlen (homedir) + strlen (name) + 1,
			      sizeof (char));

	 strcpy (path, homedir);
	 strcat (path, name);

	 return path;
      }
      user = getpwuid (getuid());
   }
   else					/* other user directory */
   {
      char *usrname = strchr (name, '/');

      if (usrname)
      {
	 char *tmp = Calloc (usrname - name, sizeof (char));
	 strncpy (tmp, name + 1, usrname - name - 1);
	 user = getpwnam (tmp);
	 Free (tmp);
	 name = usrname;		/* first slash */
      }
      else
      {
	 user = getpwnam (name + 1);
	 name += strlen (name);		/* empty string */
      }
   }

   if (!user)
      warning ("Can't find passwd entry to expand\n`~' in filename %s");
   else
   {
      char *path = Calloc (strlen (user->pw_dir) + strlen (name), sizeof (char));

      strcpy (path, user->pw_dir);
      strcat (path, name);
      
      return path;
   }

   return strdup (name);		/* no success */
}
