/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *     This program is a small utility to manipulate certain fields of
 *     the volume table of a floppy tape cartridge.  To be used with
 *     the QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

char src[] = "$Source: /homes/cvs/ftape/ftape-tools/src/vtblc/vtblc.c,v $";
char rev[] = "$Revision: 1.6 $";
char dat[] = "$Date: 1998/08/09 00:24:36 $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <linux/ftape.h>
#include <linux/ftape-vendors.h>
#include <linux/zftape.h>
#include <linux/ftape-header-segment.h>

#include "vtblc.h"
#include "version.h"

extern void *xmalloc(size_t size);
extern void *xrealloc(void *old, size_t size);

/* The name this program was run with.  */
char *program_name;

static char *tape_dev = FTAPE_DEF_TAPE;

static vtbl volumes[MAX_VOLUMES];
static u_int8_t vtbl_buffer[FT_SEGMENT_SIZE];
static int num_volumes;
static int fmt_code;
static int parsable;
static int trunc;

static const char *short_options = "f:Vdvh?pa:t::m:#:";
static const struct option long_options[] =
{
	{"file",               1, 0, 'f'}, /* full path to device */
	{"version",            0, 0, 'V'}, /* version information */
	{"debug",              0, 0, 'd'}, /* switch on debugging, unused */
	{"verbose",            0, 0, 'v'}, /* verbosity level, unused */ 
	{"usage",              0, 0, 'h'},
	{"help",               0, 0, 'h'},
	{"print",              2, 0, 'p'}, /* pretty print the vtbl */
	{"append",             1, 0, 'a'}, /* append an entry at the end */
	{"truncate",           2, 0, 't'}, /* truncate to given size */
	{"modify",             1, 0, 'm'}, /* modify given vtbl */
	{"vtbl-entry",         1, 0, '#'}, /* vtbl to print or modify */ 
	{0, 0, 0, 0}
};

const char *vtbl_str_tags[] =
{
	"label",
	"date",
	"start",
	"end",
	"tagged",
	NULL
};

static enum {
	vtbl_none  = -1,
	vtbl_label = 0,
	vtbl_date  = 1,
	vtbl_start = 2,
	vtbl_end   = 3,
	vtbl_tagged     = 4
} vtbl_tag = vtbl_none;

static enum {
	none     = 0,
	append   = 1,
	modify   = 2,
} op_mode = none;

static int vtbl_no      = -1;
static char *label      = NULL;
static char *datestr    = NULL;
static int setdate      = 0;
static int start_seg    = -1;
static int end_seg      = -1;
static int vtbl_print   = 0;
static int vtbl_maxsize = -1;
static int verbose      = 0;
static int debug        = 0;
static int taggedinput  = 0;

static void usage(FILE *file, int exit_val, char *message, ...);
static int tape_open(const char *name, int rd_only);
static int tape_close(int tape_fd);
static int read_vtbl(int tape_fd, vtbl *volumes, u_int8_t *buffer,
					 int *first_data_segment, int *last_data_segment);
static int write_vtbl(int tape_fd, const vtbl *volumes, u_int8_t *buffer,
					  int vtbl_cnt, int first_data_segment, int write);
static int read_header_segment(const int tape_fd,
							   int *first_data_segment,
							   int *last_data_segment);
static void print_vtbl(const vtbl *volumes, int maxnum,
					   int first_seg, int last_seg);
static void print_one_vtbl(const vtbl *volume, int first_seg, int last_seg);
static void print_one_parsable(const vtbl *volume,
							   int first_seg, int last_seg);
static int parse_tagged(vtbl *volumes, int *maxnum);
static char *decode_date(u_int32_t timestamp);
static int set_date(/* const */ char *date,
					vtbl *volumes, int maxnum, int vtbl_no);
static int set_label(const char *desc, vtbl *volumes, int maxnum, int vtbl_no);
static int set_start(int end, vtbl *volumes, int maxnum, int vtbl_no);
static int set_end(int end, vtbl *volumes, int maxnum, int vtbl_no);
static int set_id(vtbl *volumes, int maxnum, int vtbl_no);

int main(int argc, char *argv[])
{
	char *subopts, *value;
	int c;
	int tapefd;
	int first_seg, last_seg;

	program_name = argv[0];

	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	while (1) {
		int option_index = 0;
		
		c = getopt_long(argc, argv, short_options, long_options,&option_index);

		if (c == -1) {
            break;
		}

		switch(c) {
		case 0:
			break;
		case '#':
			vtbl_no = strtol(optarg, NULL, 0);
			break;
		case 'a':
		case 'm':
			subopts = optarg;
			while (subopts && *subopts != '\0') {
				vtbl_tag = getsubopt(&subopts, vtbl_str_tags, &value);
				if (value == NULL &&
					vtbl_tag != vtbl_date && vtbl_tag != vtbl_tagged &&
					vtbl_tag != -1) {
					usage(stderr, 1, _("The option \"%s\" needs a value!\n"),
						  vtbl_str_tags[vtbl_tag]);
				}
				switch(vtbl_tag) {
				case vtbl_label:
					label = value;
					break;
				case vtbl_date:
					datestr = value;
					setdate = 1;
					break;
				case vtbl_start:
					start_seg = strtol(value, NULL, 0);
					break;
				case vtbl_end:
					end_seg = strtol(value, NULL, 0);
					break;
				case vtbl_tagged:
					taggedinput = 1;
					break;
				default:
					usage(stderr, 1, _("Unknown option: %s\n"), value);
					break;
				}
			}
			if (c == 'a') {
				op_mode = append;
			} else {
				op_mode = modify;
			}
			break;
		case 'p':
			vtbl_print = 1;
			if (optarg && !strcmp(optarg, "parsable")) {
				parsable = 1;
			} else {
				parsable = 0;
			}
			break;
		case 't':
			if (optarg) {
				vtbl_maxsize = strtol(optarg, NULL, 0);
			}
			trunc = 1;
			break;
		case 'f':
			tape_dev = optarg;
			break;
		case 'V':
			printf(version_string);
			exit(0);
			break;
		case 'd':
			debug ++;
			break;
		case 'v':
			verbose ++;
			break;
		case 'h':
		case '?':
			usage(stdout, 0, NULL);
			break;
		default:
			fprintf (stderr, _("?? getopt returned character code 0%o ??\n"), c);
			exit(1);
			break;
		}
	}
	if (op_mode == none && !trunc) {
		vtbl_print ++;
	}
	if ((tapefd = tape_open(tape_dev, op_mode == none && !trunc)) == -1) {
		exit(1);
	}
	if ((num_volumes = read_vtbl(tapefd, volumes, vtbl_buffer,
								 &first_seg, &last_seg)) == -1 ) {
		(void)tape_close(tapefd);
		exit(1);
	}
	if (taggedinput) { /* implies modify or append */
		if (parse_tagged(volumes, &num_volumes) == -1) {
			(void)tape_close(tapefd);
			exit(1);
		}
#if 0
		print_vtbl(volumes, num_volumes, first_seg, last_seg);
		(void)tape_close(tapefd);
		exit(0);
#endif
	} else if (op_mode == modify || op_mode == append) {
		if (op_mode == append) {
			if (++num_volumes > MAX_VOLUMES) {
				fprintf(stderr, _("Too many volumes: %d (%d max)\n"), 
						num_volumes, MAX_VOLUMES);
				(void)tape_close(tapefd);
				exit(1);
			}
			if (end_seg == -1 || start_seg == -1) {
				fprintf(stderr, _("Missing start- or ending segment\n"));
			}
			op_mode = modify;
			volumes[num_volumes - 1].fmt_code = fmt_code;
			vtbl_no = -1;
		}
		if (setdate && set_date(datestr, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (label && set_label(label, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (start_seg != -1 &&
			set_start(start_seg, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (end_seg != -1 &&
			set_end(end_seg, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (set_id(volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
	}
	if (trunc) {
		if (vtbl_maxsize == -1) {
			vtbl_maxsize = num_volumes - 1;
		}
		if (vtbl_maxsize < 0 || vtbl_maxsize > num_volumes) {
			fprintf(stderr, _("Volume number too big or negative: %d\n"), 
					vtbl_no);
			(void)tape_close(tapefd);
			exit(1);
		}
		num_volumes = vtbl_maxsize;
	}
	if (vtbl_print) {
		print_vtbl(volumes, num_volumes, first_seg, last_seg);
	}
	if (write_vtbl(tapefd, volumes, vtbl_buffer, num_volumes, first_seg,
				   trunc)) {
		(void)tape_close(tapefd);
		exit(1);
	}
	if (tape_close(tapefd)) {
		exit(1);
	} else {
		exit(0);
	}
	exit(0);
}

static void usage(FILE *file, int exit_val, char *message, ...)
{
	va_list ap;
	if (message) {
		va_start(ap, message);
		vfprintf(file, message, ap);
		va_end(ap);
	}
	fprintf(file, version_string);
	fprintf(file, "\n");
	fprintf(file,
_("Usage: %s [OPTIONS]\n"
"Manipulate the volume table of a floppy tape cartridge, for use with\n"
"%s, %s\n"
"\n"
"Mandatory or optional arguments to long options are mandatory or optional\n"
"for short options too. Unique abbreviations for long options are "
"accepted.\n"
"\n"
"  -f, --file=FILE       Tape device to use. Default is "
"\"%s\".\n"
"  -h, --help            Print this help.\n"
"  -?                    Same as \'-h\'.\n"
"      --usage           Same as \'-h\'.\n"
"  -V, --version         Print version information.\n"
"  -d, --debug           Unused yet.\n"
"  -v, --verbose         Unused yet.\n"
"  -#, --vtbl-entry=NR   Specify the volume number for \"--print\" and\n"
"                        \"--modify\".\n"
"  -p, --print[=parsable] Print the volume table entry given by\n"
"                        \"--vtbl-entry\" or the entire table if\n"
"                        \"--vtbl-entry\" has been omitted.\n"
"                        If no other action is specified then \"--print\"\n"
"                        is the default.\n"
"                        If the argument \"parsable\" is supplied, print\n"
"                        the volume table in a format that may in turn\n"
"                        serve as input for \"--modify=tagged\" (see below).\n"
"  -t, --truncate[=SIZE] Truncate the volume table to SIZE entries by\n"
"                        filling the remainder with zero bytes. If SIZE\n"
"                        is omitted then the last entry is deleted.\n"
"  -a, --append          Append an entry to the end of the volume table.\n"
"  -m, --modify          Modify the volume table entry given by\n"
"                        \"--vtbl-entry\". If \"--vtbl-entry\" has been\n"
"                        omitted then the last entry is modified.\n"
"\n"
"\"--apend\" and \"--modify\" understand the following sub-options:\n"
"\n"
"      label=LABEL       The description for this volume table entry.\n"
"                        LABEL will be truncated to 44 bytes.\n"
"      date[=DATE]       The time stamp for this entry. DATE is parsed\n"
"                        by strptime(3) using the \"%%T %%D\" format (same\n"
"                        as \"%%H:%%M:%%S %%m/%%d/%%y\").\n"
"                        If DATE is omitted the current local time is used\n"
"                        instead.\n"
"      start=SEG_ID      The starting segment for this volume table entry.\n"
"      end=SEG_ID        The final segment for this volume table entry.\n"
"      tagged            Read the complete volume table entry from stdin\n"
"                        in a tagged format. The input stream consists of\n"
"                        pairs of keywords and values, one pair per line.\n"
"                        The keywords are separated from the values by\n"
"                        space characters (SPACE or TAB).\n"
"      `vtblc` understands the following keywords and values:\n"
"\n"
"      ENTRY NUM         Starts the description for the volume NUM. This may\n"
"                        be ommitted if the \"--vtbl-entry=NUM\" has been\n"
"                        given.\n"
"      ENTRY END         Here END is not a placeholder, but means the\n"
"                        word \"END\". Ends the description for the volume\n"
"                        table entry previously started by \"ENTRY NUM\".\n"
"                        Everything between \"ENTRY NUM\" and \"ENTRY END\"\n"
"                        modifies the volume number NUM. This provides means\n"
"                        to modify the entire volume table in a single run.\n"
"\n"
"      SIGNATURE \"SIG\"   Valid signature string. Only \"VTBL\" is\n"
"                        for now.\n"
"      START STARTSEG    First segment of this volume.\n"
"      END ENDSEG        Last segment of this volume.\n"
"      DESCRIPTION \"DESC\" Description for this volume table. Will be\n"
"                        truncated to 44 characters.\n"
"      DATE \"DATESTR\"   Date for the volume. If \"DATESTR\" is ommitted,\n"
"                        then the current local time is used.\n"
"      FLAG_VENDOR_SPECIFIC VAL\n"
"      FLAG_MULTI_CARTRIDGE VAL\n"
"      FLAG_NOT_VERIFIED VAL\n"
"      FLAG_REDIRECTION_INHIBIT VAL\n"
"      FLAG_SEGMENT_SPANNING VAL\n"
"      FLAG_DIRECTORY_LAST VAL\n"
"      FLAG_RESERVED_6 VAL\n"
"      FLAG_RESERVED_7 VAL\n"
"      MULTI_CARTRIDGE_COUNT VAL\n"
"      VENDOR_EXTENSION \"HEX\" Vendor extension data. \"HEX\" is a string\n"
"                        of hexadecimal byte values, i.e. \"0x01 0x4f ...\"\n"
"      PASSWORD \"HEX\"  Password. \"HEX\" as above.\n"
"      DIRECTORY_SIZE VAL\n"
"      DATA_SIZE VAL64   64 bit value (a decimal count)\n"
"      OS_VERSION HEX    Here \"HEX\" is a hexadecimal 4 bytes value.\n"
"      SOURCE_DRIVE DRVSTR Source drive. \"DRVSTR\" is a 16 byte string.\n"
"      DEVICE HEXBYTE    A single byte in hexadecimal notation.\n"
"      RESERVED_1 HEXBYTE\n"
"      COMPRESSION_FLAGS HEXBYTE\n"
"      FORMAT HEYBYTE\n"
"      RESERVED_2 HEXBYTE\n"
"      RESERVED_3 HEXBYTE\n"
"\n"
"Strings with spaces have to be surrounded by double quotes \"...\"\n"
"\"VAL\" is some decimal number.\n"
"\n"),
			program_name, FTAPE_VERSION, ZFTAPE_VERSION, FTAPE_DEF_TAPE);
	exit(exit_val);
}

/*
 *  open the tape device, try to determine whether is is a floppy tape,
 *  and store the hardware status in drive_config, resp. tape_config.
 *  mmap() the dma buffers if op_mode != PROBING
 *  
 *  Returns the tape fd on success, -1 otherwise
 *
 */

static int tape_open(const char *name, int rd_only)
{
	int tape_fd;
	struct mtget mtget;
	static vendor_struct vendors[] = QIC117_VENDORS;
	vendor_struct *vendor;

	/* open the tape device
	 */
	if ((tape_fd = open(name, rd_only ? O_RDONLY : O_RDWR)) == -1) {
		perror(_("Error opening tape dev"));
		return -1;
	}
	/* get its status
	 */
	if (ioctl(tape_fd, MTIOCGET, &mtget) == -1) {
		perror(_("Error getting tape drive status"));
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_DR_OPEN(mtget.mt_gstat)) {
		fprintf(stderr, _("Error: No tape cartridge present!\n"));
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_WR_PROT(mtget.mt_gstat)) {
		fprintf(stderr, _("Warning: Write protected cartridge!\n"));
	}
	if (!GMT_ONLINE(mtget.mt_gstat)) {
		fprintf(stderr, _("Error: Tape drive is offline!\n"));
		(void)close(tape_fd);
		return -1;
	}
	if ((mtget.mt_type & MT_ISFTAPE_FLAG) == 0) {
		fprintf(stderr, _("Error: This is not a floppy tape drive!\n"));
		(void)close(tape_fd);
		return -1;
	}
	mtget.mt_type &= ~MT_ISFTAPE_FLAG; /* clear the flag bit */
	if (verbose > 0) {
		vendor = &vendors[0];
		while (vendor->vendor_id != UNKNOWN_VENDOR &&
			   vendor->vendor_id != mtget.mt_type) {
			vendor++;
		}
		printf(_("Tape drive type: %s (0x%04lx)\n"), vendor->name, mtget.mt_type);
	}
	return tape_fd;
}

static int tape_close(int tape_fd)
{
	const struct mtop rewind = { MTREW, 1 };
	int result = 0;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		fprintf(stderr, _("Ioctl error rewinding tape: %s\n"), strerror(errno));
		result = -1;
	}
	if (close(tape_fd) == -1) {
		fprintf(stderr, _("Error closing tape device: %s\n"), strerror(errno));
		result = -1;
	}
	return result;
}

static int read_vtbl(int tape_fd, vtbl *volumes, u_int8_t *buffer,
					 int *first_data_segment, int *last_data_segment)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int vtbl_cnt; 
	int end_seg = 0;
	const char *ids[] = VTBL_IDS;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		fprintf(stderr, _("Ioctl error rewinding tape: %s\n"), strerror(errno));
		return -1;
	}
	if ((fmt_code = read_header_segment(tape_fd,
											first_data_segment,
											last_data_segment)) == -1) {
		return -1;
	}
	if (verbose) {
		printf(_("Reading volume table segment ... "));
		fflush(stdout);
	}
	memset(&ft_seg, 0, sizeof(ft_seg));
	ft_seg.mt_data  = buffer;
	ft_seg.mt_segno = *first_data_segment;
	ft_seg.mt_mode  = MT_FT_RD_SINGLE;
	if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
		fprintf(stderr, _("Ioctl error reading volume table: %s\n"),
				strerror(errno));
		return -1;
	}
	if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
		fprintf(stderr, _("Short read() reading volume table: %d\n"),
				ft_seg.mt_result);
		return -1;
	}
	if (verbose) {
		printf(_("done.\n"));
	}
	vtbl_cnt = 0;
	if (fmt_code == fmt_big) {
		end_seg = *first_data_segment;
	}
	while (!memcmp(buffer, ids[0], 4) ||
		   !memcmp(buffer, ids[1], 4) ||
		   !memcmp(buffer, ids[2], 4) ||
		   !memcmp(buffer, ids[3], 4)) {

		memcpy(&volumes[vtbl_cnt].entry, buffer, 128);
		if (!memcmp(&volumes[vtbl_cnt].vt_sig, "VTBL", 4)) {
			volumes[vtbl_cnt].fmt_code = fmt_code;
			if (fmt_code == fmt_big) {
				volumes[vtbl_cnt].start = end_seg + 1;
				volumes[vtbl_cnt].end = (end_seg + volumes[vtbl_cnt].vt_space);
				end_seg = volumes[vtbl_cnt].end;
			} else {
				volumes[vtbl_cnt].start = volumes[vtbl_cnt].vt_start;
				volumes[vtbl_cnt].end = volumes[vtbl_cnt].vt_end;
			}
			volumes[vtbl_cnt].fmt_code = fmt_code;
		}
		volumes[vtbl_cnt].num = vtbl_cnt;
		vtbl_cnt ++;
		buffer += VTBL_SIZE;
	}
	return vtbl_cnt;
}

/*  returns the max. number of segments
 */
static int read_header_segment(int tape_fd,
							   int *first_data_segment, int *last_data_segment)
{
		struct mtftseg ft_seg;
		int i;
		const unsigned long hseg_magic = FT_HSEG_MAGIC;
		u_int8_t hseg[FT_SEGMENT_SIZE];
		
		if (verbose) {
			printf(_("Reading header segment ... "));
			fflush(stdout);
		}
		for (i = 0; i < 64; i++) {
			memset(&ft_seg, 0, sizeof(ft_seg));
			ft_seg.mt_data  = hseg;
			ft_seg.mt_segno = i;
			ft_seg.mt_mode  = MT_FT_RD_AHEAD;
			if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
				fprintf(stderr, _("Ioctl error reading header segment: %s\n"),
						strerror(errno));
				return -1;
			}
			if (ft_seg.mt_result == FT_SEGMENT_SIZE) {
				break;
			}
		}
		if (memcmp(&hseg_magic, hseg, 4)) {
			fprintf(stderr, _("Unable to read the header segment\n"));
			return -1;
		}
		if (verbose) {
			printf(_("done.\n"));
		}
		if (hseg[FT_FMT_CODE] == fmt_big) {
			memcpy(first_data_segment, &hseg[FT_6_FRST_SEG], 4);
			memcpy(last_data_segment,  &hseg[FT_6_LAST_SEG], 4);
		} else {
			memcpy(first_data_segment, &hseg[FT_FRST_SEG], 2);
			memcpy(last_data_segment,  &hseg[FT_LAST_SEG], 2);
			*first_data_segment &= 0x0000ffff;
			*last_data_segment &= 0x0000ffff;
		}
		return (int)hseg[FT_FMT_CODE];
}

static void print_vtbl(const vtbl *volumes, int maxnum,
					   int first_seg, int last_seg)
{
	int i;

	if (parsable) {
		printf("VTBL START %d %d\n", first_seg, last_seg);
		for (i = 0; i < maxnum; i++) {
			printf("ENTRY %d\n", i);
			print_one_parsable(&volumes[i], first_seg, last_seg);
			printf("ENTRY END\n");
		}
		printf("VTBL END\n");
		return;
	}
	printf("%3s %3s %*s %*s %*s %8s %8s\n",
		   "Nr", "Id", 14, "Label", 22, "Date", 15, "Start", "End", "Space");
	for (i = 0; i < 80; i++) {
		printf("-");
	}
	printf("\n");
	for (i = 0; i < maxnum; i++) {
		print_one_vtbl(&volumes[i], first_seg, last_seg);
	}
}

static void print_one_vtbl(const vtbl *volume, int first_seg, int last_seg)
{
	char label[45];
	char sig[5];
	double usage;

	memcpy(label, volume->vt_desc, 44);
	label[22] = '\0';
	memcpy(sig, volume->vt_sig, 4);
	sig[4] = '\0';
	if (!strcmp(sig, "VTBL")) {
		usage = (double)(volume->end - volume->start + 1);
		usage = usage/(double)(last_seg - first_seg)*100.0;
		printf("%3d %4s \"%-*s\" %*s %8d %8d    %2.2f%%\n",
			   volume->num, sig, 22, label, 18, decode_date(volume->vt_date),
			   volume->start, volume->end, usage);
	} else {
		printf("%4d %4s", vtbl_no, sig);
	}
}

static void print_one_parsable(const vtbl *volume, int first_seg, int last_seg)
{
	char sig[5];
	char label[45];
	int i;
	double usage;

	memcpy(sig, volume->vt_sig, 4);
	sig[4] = '\0';
	memcpy(label, volume->vt_desc, 44);
	label[44] = '\0';
	printf("SIGNATURE \"%s\"\n", sig);
	if (strcmp(sig, "VTBL") == 0) {
		printf("START %d\n", volume->start);
		printf("END %d\n", volume->end);
		printf("DESCRIPTION \"%s\"\n", label);
		printf("DATE \"%s\"\n", decode_date(volume->vt_date));
		printf("FLAG_VENDOR_SPECIFIC %d\n", volume->entry.vendor_specific);
		printf("FLAG_MULTI_CARTRIDGE %d\n", volume->entry.multi_cartridge);
		printf("FLAG_NOT_VERIFIED %d\n", volume->entry.not_verified);
		printf("FLAG_REDIRECTION_INHIBIT %d\n",
			   volume->entry.inhibit_redirection);
		printf("FLAG_SEGMENT_SPANNING %d\n", volume->entry.segment_spanning);
		printf("FLAG_DIRECTORY_LAST %d\n", volume->entry.directory_last);
		printf("FLAG_RESERVED_6 %d\n", volume->entry.fl_reserved_6);
		printf("FLAG_RESERVED_7 %d\n", volume->entry.fl_reserved_7);
		printf("MULTI_CARTRIDGE_COUNT %d\n", volume->entry.m_no);
		printf("VENDOR_EXTENSION \"");
		for (i = 0; i < sizeof(volume->entry.ext)-1; i ++) {
			printf("0x%02x ", volume->entry.ext[i]);
		}
		printf("0x%02x\"\n", volume->entry.ext[i]);
		printf("PASSWORD \"");
		for (i = 0; i < sizeof(volume->entry.pwd)-1; i ++) {
			printf("0x%02x ", volume->entry.pwd[i]);
		}
		printf("0x%02x\"\n", volume->entry.pwd[i]);
		printf("DIRECTORY_SIZE %d\n", volume->entry.dir_size);
		printf("DATA_SIZE %Ld\n", volume->entry.data_size);
		printf("OS_VERSION 0x%04x\n", volume->entry.os_version);
		printf("SOURCE_DRIVE %.16s\n", volume->entry.source_drive);
		printf("DEVICE 0x%02x\n", volume->entry.device);
		printf("RESERVED_1 0x%02x\n", volume->entry.reserved_1);
		printf("COMPRESSION_FLAGS 0x%02x\n", volume->entry.cmpr);
		printf("FORMAT 0x%02x\n", volume->entry.format);
		printf("RESERVED_2 0x%02x\n", volume->entry.reserved_1);
		printf("RESERVED_3 0x%02x\n", volume->entry.reserved_1);
	}
}

char *mygets(FILE *fp, char *buffer, int *len)
{
	char *p = buffer;
	int remaining = *len;
	int c;

	for (;;) {
		errno = 0;
		switch (c = getc(fp)) {
		case EOF: {
			*p = '\0';
			if (ferror(fp)) {
				perror("getc():");
			}
			return NULL;
		}		
		case '\n': {
			*p = '\0';
			return buffer;				
		}
		default: {
			if (remaining <= 1) {
				buffer = xrealloc(buffer, 1024 + *len);
				p = buffer + *len - remaining;
				(*len) += 1024;
				remaining += 1024;
			}
			*(p++) = (char)c;
			break;
		}
		}
	}
}

#define GET_SINGLE_VALUE(tag, conv, dest)						\
{																\
	int value;													\
																\
	if (strncmp(#tag, buffer, sizeof(#tag)-1) == 0) {			\
		if (sscanf(buffer, #tag" %"#conv, &value) == 1) {		\
			(dest) = value;										\
		} else {												\
			fprintf(stderr, "Corrupt volume input data: %s\n",	\
					buffer);									\
			return -1;											\
		}														\
		continue;												\
	}															\
}

#define GET_STRING_VALUE(tag, dest, max, action)		\
{														\
	char *value;										\
														\
	if (strncmp(#tag, buffer, sizeof(#tag)-1) == 0) {	\
		value = buffer + sizeof(#tag);					\
		if (strchr(value, '\"')) {						\
			value = strchr(value, '\"') + 1;			\
		}												\
		if (strrchr(value, '\"')) {						\
			*strrchr(value, '\"') = '\0';				\
		}												\
		if (dest) memcpy((dest), value, max);			\
		action;											\
		continue;										\
	}													\
}

static int parse_tagged(vtbl *volumes, int *maxnum)
{
	char *line;
	char *buffer = xmalloc(1024);
	char *new;
	int len = 1024;
	int in_volume = 0;
	int num;
	vtbl *volume = NULL;
	int get_data = 0;
	char dummy[6];

	while (new = mygets(stdin, buffer, &len)) {
		buffer = new;
		if (*new == '#') {
			continue; /* comment */
		}
		if (sscanf(buffer, "VTBL %5s", dummy) == 1) {
			if (strcmp(dummy, "START") == 0) {
				get_data = 1; /* start of volume table */
				continue;
			} else if (strcmp(dummy, "END") == 0) {
				if (in_volume) {
					fprintf(stderr, "Corrupt volume input data: %s\n", buffer);
					return -1;
				}
				return 0;
			}
		}
		if (!get_data) {
			continue;
		}
		if (strncmp("ENTRY", buffer, sizeof("ENTRY")-1) == 0) {
			if (strstr(buffer, "END")) {
				if (!in_volume) {
					fprintf(stderr, "Corrupt volume input data: %s\n", buffer);
					return -1;
				}
				in_volume = 0;
			} else if (sscanf(buffer, "ENTRY %d", &num) == 1) {
				if (num == *maxnum) {
					if (++(*maxnum) > MAX_VOLUMES) {
						fprintf(stderr, _("Too many volumes: %d (%d max)\n"), 
								*maxnum, MAX_VOLUMES);
						return -1;
					}
					memset(&volumes[num], 0, sizeof(volumes[num]));
					volumes[num].fmt_code = fmt_code;
				} else if (num > *maxnum) {
					fprintf(stderr, "Corrupt volume input data: %s\n", buffer);
					return -1;
				}
				in_volume = 1;
				volume = &volumes[num];
				volume->modified = 1;
				volume->num = num;
				memcpy(volume->vt_sig, "VTBL", 4);
			} else {
				fprintf(stderr, "Corrupt volume input data: %s\n", buffer);
				return -1;
			}
			continue;
		}
		if (!in_volume) { /* need the "ENTRY" tag */
			fprintf(stderr, "Corrupt volume input data\n");
			return -1;
		}
		if (strncmp("SIGNATURE", buffer, sizeof("SIGNATURE")-1) == 0) {
			if (!strstr(buffer, "VTBL")) {
				fprintf(stderr, "Corrupt volume input data: %s\n", buffer);
				return -1;
			}
			memcpy(volume->vt_sig, "VTBL", 4);
			continue;
		}
		GET_SINGLE_VALUE(START, i, volume->start);
		GET_SINGLE_VALUE(END, i, volume->end);
		GET_SINGLE_VALUE(FLAG_VENDOR_SPECIFIC, i,
						 volume->entry.vendor_specific);
		GET_SINGLE_VALUE(FLAG_MULTI_CARTRIDGE, i,
						 volume->entry.multi_cartridge);
		GET_SINGLE_VALUE(FLAG_NOT_VERIFIED, i,
						 volume->entry.not_verified);
		GET_SINGLE_VALUE(FLAG_REDIRECTION_INHIBIT, i,
						 volume->entry.inhibit_redirection);
		GET_SINGLE_VALUE(FLAG_SEGMENT_SPANNING, i,
						 volume->entry.segment_spanning);
		GET_SINGLE_VALUE(FLAG_DIRECTORY_LAST, i,
						 volume->entry.directory_last);
		GET_SINGLE_VALUE(FLAG_RESERVED_6, i,
						 volume->entry.fl_reserved_6);
		GET_SINGLE_VALUE(FLAG_RESERVED_7, i,
						 volume->entry.fl_reserved_7);
		GET_SINGLE_VALUE(MULTI_CARTRIDGE_COUNT, i,
						 volume->entry.m_no);
		GET_SINGLE_VALUE(DIRECTORY_SIZE, i,
						 volume->entry.dir_size);
		GET_SINGLE_VALUE(DEVICE, i,
						 volume->entry.device);
		GET_SINGLE_VALUE(RESERVED_1, i,
						 volume->entry.reserved_1);
		GET_SINGLE_VALUE(COMPRESSION_FLAGS, i,
						 volume->entry.cmpr);
		GET_SINGLE_VALUE(FORMAT, i,
						 volume->entry.format);
		GET_SINGLE_VALUE(RESERVED_2, i,
						 volume->entry.reserved_1);
		GET_SINGLE_VALUE(RESERVED_3, i,
						 volume->entry.reserved_1);
		GET_SINGLE_VALUE(DATA_SIZE, Li,
						 volume->entry.data_size);
		GET_SINGLE_VALUE(OS_VERSION, i,
						 volume->entry.os_version);
		GET_STRING_VALUE(DESCRIPTION, volume->vt_desc, 44,
						 memset(volume->vt_desc + strlen(value), ' ',
								44 - strlen(value)));
		GET_STRING_VALUE(DATE, NULL, 1024,
						 { if (set_date(value, volumes, *maxnum, num)) {
							 return -1;
						 } });
		GET_STRING_VALUE(VENDOR_EXTENSION, NULL, 1024,
						 { char *p = value; int i; int val;
						 for (i = 0; i < sizeof(volume->entry.ext)-1; i ++) {
							 if (sscanf(p, "%i", &val) != 1) {
								 fprintf(stderr,
								 "Corrupt volume input data: %s\n", buffer);
								 return -1;
							 }
							 volume->entry.ext[i] = val;
							 while (isspace(*p)) p++;
							 while (*p != '\0' && !isspace(*p)) p++;
						 } });
		GET_STRING_VALUE(PASSWORD, NULL, 1024,
						 { char *p = value; int i; int val;
						 for (i = 0; i < sizeof(volume->entry.pwd)-1; i ++) {
							 if (sscanf(p, "%i", &val) != 1) {
								 fprintf(stderr,
								 "Corrupt volume input data: %s\n", buffer);
								 return -1;
							 }
							 volume->entry.pwd[i] = val;
							 while (isspace(*p)) p++;
							 while (*p != '\0' && !isspace(*p)) p++;
						 } });
		GET_STRING_VALUE(SOURCE_DRIVE, volume->entry.source_drive, 16,
						 /**/);
	}
}

static char *decode_date(u_int32_t timestamp)
{
	struct tm tapetm;
	time_t tapetime;
	static char date[18];

	memset(&tapetm, 0, sizeof(tapetm));
	
	tapetm.tm_year  = timestamp >> FT_YEAR_SHIFT;
	tapetm.tm_year += FT_YEAR_0 - 1900;
	timestamp      &= FT_TIME_MASK;

	tapetm.tm_sec   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_min   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_hour  = timestamp % 24;
	timestamp      /= 24;
	tapetm.tm_mday  = timestamp % 31 + 1;
	timestamp      /= 31;
	tapetm.tm_mon   = timestamp % 12;
	tapetm.tm_isdst = -1;
	if ((tapetime = mktime(&tapetm)) == (time_t)(-1)) {
		return _("invalid");
	} else {
		(void)strftime(date, 18, "%T %D", &tapetm);
		return date;
	}
}

static int write_vtbl(int tape_fd, const vtbl *volumes, u_int8_t *buffer,
					  int vtbl_cnt, int first_data_segment, int modified)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int i;
	u_int8_t *p = buffer;

	for (i = 0; i < vtbl_cnt; i++) {
		if (volumes[i].modified) {
			vtblentry *vt = (vtblentry *)p;

			memcpy(p, &volumes[i].entry, 128);
			if (volumes[i].fmt_code == fmt_big) {
				vt->size.space = (volumes[i].end - volumes[i].start + 1);
			} else {
				vt->size.se.start = volumes[i].start;
				vt->size.se.end   = volumes[i].end;
			}
			modified ++;
		}
		p += VTBL_SIZE;
	}
	memset(p, 0, VTBL_SIZE * (MAX_VOLUMES - vtbl_cnt));
	if (modified) {
		if (verbose) {
			printf(_("Writing volume table segment ... "));
			fflush(stdout);
		}
		memset(&ft_seg, 0, sizeof(ft_seg));
		ft_seg.mt_data  = buffer;
		ft_seg.mt_segno = first_data_segment;
		ft_seg.mt_mode  = MT_FT_WR_SINGLE;
		if (ioctl(tape_fd, MTIOCWRFTSEG, &ft_seg) == -1) {
			fprintf(stderr, _("Ioctl error writing volume table: %s\n"),
					strerror(errno));
			return -1;
		}
		if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
			fprintf(stderr, _("Short write() writing volume table: %d\n"),
					ft_seg.mt_result);
			return -1;
		}
		if (verbose) {
			printf(_("done.\n"));
		}
	}
	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		fprintf(stderr, _("Ioctl error rewinding tape: %s\n"), strerror(errno));
		return -1;
	}
	return 0;
}

static int set_date(/*const*/ char *date,
					vtbl *volumes, int maxnum, int vtbl_no)
{
	time_t raw_time;
	struct tm tapetime;
	struct tm *tapetimep = &tapetime;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, _("Volume number too big or negative: %d\n"), 
				vtbl_no);
		return -1;
	}
	if (date == NULL) {
		time(&raw_time);
		tapetimep = localtime(&raw_time);
	} else {
		(void)strptime(date, "%T %D", tapetimep);
	}
	volumes[vtbl_no].vt_date = FT_TIME_STAMP(tapetimep->tm_year + 1900, 
										  tapetimep->tm_mon, 
										  tapetimep->tm_mday-1,
										  tapetimep->tm_hour,
										  tapetimep->tm_min,
										  tapetimep->tm_sec);
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_label(const char *desc, vtbl *volumes, int maxnum, int vtbl_no)
{
	int len;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, _("Volume number too big or negative: %d\n"), 
				vtbl_no);
		return -1;
	}
	strncpy(volumes[vtbl_no].vt_desc, desc, 44);
	if ((len = strlen(desc)) < 44) {
		memset(volumes[vtbl_no].vt_desc + len, ' ', 44 - len);
	}
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_start(int start, vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, _("Volume number too big or negative: %d\n"), 
				vtbl_no);
		return -1;
	}
	volumes[vtbl_no].start = start;
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_end(int end, vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, _("Volume number too big or negative: %d\n"), 
				vtbl_no);
		return -1;
	}
	volumes[vtbl_no].end = end;
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_id(vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, _("Volume number too big or negative: %d\n"), 
				vtbl_no);
		return -1;
	}
	memcpy(volumes[vtbl_no].vt_sig, "VTBL", 4);
	volumes[vtbl_no].modified = 1;
	return 0;
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
