#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <fcntl.h>
#include <termios.h>
#include <sysexits.h>
#include <linux/kd.h>
#include <linux/keyboard.h>
#include <sys/ioctl.h>

#include <lct/local.h>
#include <lct/utils.h>
#include <lct/console.h>

int tmp;	/* for debugging */
int fd;
int oldkbmode;
struct termios old;
char* progname;
int timeout = 10;

/*
 * version 0.81 of showkey would restore kbmode unconditially to XLATE,
 * thus making the console unusable when it was called under X.
 */
void get_mode(void)
{
  char *m;

  if (ioctl(fd, KDGKBMODE, &oldkbmode))
    {
      perror("KDGKBMODE");
      exit(1);
    }
  switch(oldkbmode)
    {
    case K_RAW:
      m = "RAW"; break;
    case K_XLATE:
      m = "XLATE"; break;
    case K_MEDIUMRAW:
      m = "MEDIUMRAW"; break;
    case K_UNICODE:
      m = "UNICODE"; break;
    default:
      m = "?UNKNOWN?"; break;
    }
  printf("kb mode was %s\n", m);
  if (oldkbmode != K_XLATE)
    {
      printf("[ if you are trying this under X, it might not work\n");
      printf("since the X server is also reading /dev/console ]\n");
    }
  printf("\n");
}

void clean_up(void)
{
  if (ioctl(fd, KDSKBMODE, oldkbmode))
    {
      perror("KDSKBMODE");
      exit(1);
    }
  tmp = tcsetattr(fd, 0, &old);
  if (tmp)
    printf("tcsetattr = %d\n", tmp);
  close(fd);
}

void die(int x)
{
  printf("caught signal %d, cleaning up...\n", x);
  clean_up();
  exit(1);
}

void watch_dog(int x)
{
  clean_up();
  exit(0);
}

void usage()
{
  version();
  
  fprintf(stderr,
	  "usage: %s [options...] <command>\n"
	  "\n"
	  "valid commands are:\n"
	  "\n"
	  "        -h --help       print this help text.\n"
	  "        -V --version    print version number.\n"
	  "        -s --scancodes  display only the raw scan-codes.\n"
	  "        -k --keycodes   display only the interpreted keycodes (default).\n"
	  "        -m --keymap     display only keymap-translated chars.\n"
	  "\n"
	  "valid options are:\n"
	  "        -t --timeout=N  set the timeout to N seconds.\n"
	  "\n", progname);
}

int
main (int argc, char *argv[])
{
  const char *short_opts = "hskmt:V";
  const struct option long_opts[] =
  {
    { "help",	no_argument, NULL, 'h' },
    { "scancodes",	no_argument, NULL, 's' },
    { "keycodes",	no_argument, NULL, 'k' },
    { "keymap",	no_argument, NULL, 'm' },
    { "timeout",	required_argument, NULL, 't' },
    { "version",	no_argument, NULL, 'V' },
    { NULL, 0, NULL, 0 }
  };
  int c;
  enum {cmd_none, cmd_scancodes, cmd_keycodes, cmd_keymap} command = cmd_none;
  
  struct termios new;
  unsigned char buf[128];	       /* to decently grasp STRINGS with cmd_keymap */
  int i, n;

  progname = strip_path(argv[0]);
  
#define BAD_CMD() badusage("only one commands is allowed");
  
  while ((c = getopt_long(argc, argv,
			  short_opts, long_opts, NULL)) != -1)
    {
      switch (c)
	{
	case 'm':
	  if (command == cmd_none)
	    command = cmd_keymap;
	  else
	    BAD_CMD();
	  break;
	case 's':
	  if (command == cmd_none)
	    command = cmd_scancodes;
	  else
	    BAD_CMD();
	  break;
	case 'k':
	  if (command == cmd_none)
	    command = cmd_keycodes;
	  else
	    BAD_CMD();
	  break;
	case 't':
	  if (optarg && (optarg[0] >= '0') && (optarg[0] <= '9'))
	    timeout = atoi(optarg);
	  else
	    badusage("timeout value must be numeric");
	  break;
	case 'V':
	  version();
	  exit(0);
	case 'h':
	case '?':
	  usage();
	  exit(0);
	}
    }

  if (command == cmd_none)
    command = cmd_scancodes;
  
  if (optind < argc)
    badusage("no non-option arguments allowed");

  if (-1 == (fd = get_console_fd(NULL)))
    exit (1);

  /* the program terminates when there is no input for <timeout> secs */
  signal(SIGALRM, watch_dog);

  /*
    if we receive a signal, we want to exit nicely, in
    order not to leave the keyboard in an unusable mode
    */
  signal(SIGHUP, die);
  signal(SIGINT, die);
  signal(SIGQUIT, die);
  signal(SIGILL, die);
  signal(SIGTRAP, die);
  signal(SIGABRT, die);
  signal(SIGIOT, die);
  signal(SIGFPE, die);
  signal(SIGKILL, die);
  signal(SIGUSR1, die);
  signal(SIGSEGV, die);
  signal(SIGUSR2, die);
  signal(SIGPIPE, die);
  signal(SIGTERM, die);
#ifdef SIGSTKFLT
  signal(SIGSTKFLT, die);
#endif
  signal(SIGCHLD, die);
  signal(SIGCONT, die);
  signal(SIGSTOP, die);
  signal(SIGTSTP, die);
  signal(SIGTTIN, die);
  signal(SIGTTOU, die);

  get_mode();
  tmp = tcgetattr(fd, &old);
  if (tmp)
    printf("tcgetattr = %d\n", tmp);
  tmp = tcgetattr(fd, &new);
  if (tmp)
    printf("tcgetattr = %d\n", tmp);

  new.c_lflag &= ~ (ICANON | ECHO | ISIG);
  new.c_iflag = 0;
  new.c_cc[VMIN] = sizeof(buf);
  new.c_cc[VTIME] = 1;	/* 0.1 sec intercharacter timeout */

  tmp = tcsetattr(fd, TCSAFLUSH, &new);
  if (tmp)
    printf("tcsetattr = %d\n", tmp);
  if ((command != cmd_keymap) && ioctl(fd, KDSKBMODE,
				       (command == cmd_keycodes) ? K_MEDIUMRAW : 
				       (command == cmd_keymap ? K_XLATE : K_RAW)))
    {
      perror("KDSKBMODE");
      exit(1);
    }

  printf("press any key (program terminates after %us of last keypress)...\n", timeout);
  while (1)
    {
      alarm(timeout);
      n = read(fd, buf, sizeof(buf));
	  
      if (command == cmd_keymap)
	{
	  for (i = 0; i < n; i++)
	    if (buf[i] >= 32)
	      putchar(buf[i]);
	    else
	      putchar(' ');
	  fprintf(stdout, " ( ");
	}
	      
      for (i = 0; i < n; i++) 
	{
	  if (command == cmd_keycodes)
	    printf("keycode %3d %s\n",
		   buf[i] & 0x7f,
		   buf[i] & 0x80 ? "release" : "press");
	  else /* scancode or keymap */
	    printf("0x%02x ", buf[i]);
	}

      switch (command)
	{
	case cmd_scancodes:
	  putchar('\n');
	  break;
	case cmd_keymap:
	  printf(")\n");
	default:
	}
    }

  clean_up();
  exit(0);
}
