/*
 * Copyright (c) 1994,1996,1997,1998  Kazushi (Jam) Marukawa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice in the documentation and/or other materials provided with 
 *    the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * Routines to manipulate a buffer to hold string of multi bytes character.
 * Detect a character set from input string and convert them to internal
 * codes.  And convert it to other codes to display them.
 */

#include "defines.h"
#include "multi.h"

#include <stdio.h>
#include <assert.h>


#if ISO

static void rebuffering_multi();


#if JAPANESE

/*
 * Kanji convetion
 */
#define ISJIS(c)		(0x21 <= (c) && (c) <= 0x7e)
#define ISUJIS(c)		(0xa1 <= (c) && (c) <= 0xfe)
#define ISUJISSS(c)		((c) == 0x8e || (c) == 0x8f)
#define ISUJISKANJI(c1,c2)	(ISUJIS(c1) && ISUJIS(c2))
#define ISUJISKANA(c1,c2)	((c1) == 0x8e && ISUJIS(c2))
#define ISUJISKANA1(c)		((c) == 0x8e)
#define ISUJISKANJISUP(c1,c2,c3) ((c1) == 0x8f && ISUJIS(c2) && ISUJIS(c3))
#define ISSJISKANJI(c1,c2)	(((0x81 <= (c1) && (c1) <= 0x9f) || \
				  (0xe0 <= (c1) && (c1) <= 0xfc)) && \
				 (0x40 <= (c2) && (c2) <= 0xfc && (c2) != 0x7f))
#define ISSJISKANA(c)		(0xa1 <= (c) && (c) <= 0xdf)
#endif


/*
 * Definitions for understanding the escape sequence.
 * Following escape sequences which be understood by less:
 *  ESC 2/4 2/8,2/9,2/10,2/11,2/13,2/14,2/15 F
 *  ESC 2/4 4/0,4/1,4/2
 *  ESC 2/6 F
 *  ESC 2/8,2/9,2/10,2/11,2/13,2/14,2/15 F
 *  ESC 2/12 F		This is used in MULE.  Less support this as input.
 *  0/14,0/15
 *  ESC 4/14,4/15,6/14,6/15,7/12,7/13,7/14
 *  8/14,8/15
 */
enum escape_sequence {
	NOESC,		/* No */	ESC_,		/* ^[ */
	ESC_2_4,	/* ^[$ */	ESC_2_4_8,	/* ^[$( */
	ESC_2_4_9,	/* ^[$) */	ESC_2_4_10,	/* ^[$* */
	ESC_2_4_11,	/* ^[$+ */	ESC_2_4_13,	/* ^[$- */
	ESC_2_4_14,	/* ^[$. */	ESC_2_4_15,	/* ^[$/ */
	ESC_2_6,	/* ^[& */	ESC_2_8,	/* ^[( */
	ESC_2_9,	/* ^[) */	ESC_2_10,	/* ^[* */
	ESC_2_11,	/* ^[+ */	ESC_2_12,	/* ^[, */
	ESC_2_13,	/* ^[- */	ESC_2_14,	/* ^[. */
	ESC_2_15	/* ^[/ */
};


static CODESET def_left = iso7;		/* Default code set of left plane */
static CODESET def_right = iso8;	/* Default code set of right plane */
static int def_g0 = ASCII;		/* Default g0 plane status */
static int def_g1 = WRONGCS;		/* Default g1 plane status */
static int def_g2 = WRONGCS;		/* Default g2 plane status */
static int def_g3 = WRONGCS;		/* Default g3 plane status */

static CODESET output = iso8;		/* Code set for output */
#if JAPANESE
static CODESET def_priority = ujis;	/* Which code was given priority. */
#endif

struct multibuf {
	struct {
		CODESET left;
		CODESET right;
	} io;

	CODESET orig_io_right;
	int rotation_io_right;

	enum escape_sequence eseq;
	/*
	 * Variables to control of escape sequences as output.
	 */
	int cs;			/* Current character set */
	int g0;			/* Current g0 plane status */
	int g1;			/* Current g1 plane status */
	int g2;			/* Current g2 plane status */
	int g3;			/* Current g3 plane status */
	int *gl;		/* Current gl plane status */
	int *gr;		/* Current gr plane status */
	int *sg;		/* Current status of single-shifted plane */
	int irr;		/* Identify revised registration number */
#if JAPANESE
	CODESET priority;	/* Which code was given priority. */
	int sequence_counter;	/* Special counter for detect UJIS KANJI. */
#endif

	int icharset;		/* Last non ASCII character set of input */

	/*
	 * Buffers to keep all bytes of a multi-bytes character until it is
	 * proved to be right sequence.
	 */
	unsigned char multibuf[10]; 	/* Buffer for multi bytes data */
	int multiindex;			/* Index of multibuf */
	unsigned char multiint[10]; 	/* Buffer for translate data in */
					/* multibuf into internal data */
	CHARSET multics[10];		/* Buffer for CHARSET of translated */
					/* data */
	int intindex;			/* Index of multiint */
	int bufbytes;		/* The length of multi bytes character */
				/* before converting, and use this to adjust */
				/* a current position if a character doesn't */
				/* fit a screen. */
};


static int code_length(cs)
int cs;
{
	if (CSISWRONG(cs))
		return (1);
	switch (CS2TYPE(cs))
	{
	case TYPE_94_CHARSET:
	case TYPE_96_CHARSET:
		return (1);
	case TYPE_94N_CHARSET:
	case TYPE_96N_CHARSET:
		switch (CS2FT(cs) & 0x70)
		{
		case 0x30:		/* for private use */
			return (2);
		case 0x40:
		case 0x50:
			return (2);
		case 0x60:
			return (3);
		case 0x70:
			return (4);	/* or more bytes */
		}
	}
	assert(0);
}

/*
 * Check and normalize all codes which don't need conversion.
 */
static void internalize_noconv(mp)
MULBUF *mp;
{
	if (mp->multiindex != 1)
	{
		rebuffering_multi(mp);
	} else
	{
		mp->multiint[mp->intindex] = mp->multibuf[0];
		mp->multics[mp->intindex] = ASCII;
		mp->intindex += 1;
		mp->multiindex = 0;
	}
}

static void internalize_wrong(mp)
MULBUF *mp;
{
	register int i;

	assert(mp->intindex + mp->multiindex <= sizeof(mp->multiint));
	for (i = 0; i < mp->multiindex; i++)
	{
		mp->multiint[mp->intindex + i] = mp->multibuf[i];
		mp->multics[mp->intindex + i] = WRONGCS;
	}
	mp->intindex += mp->multiindex;
	mp->multiindex = 0;
}

/*
 * Check and normalize all ISO codes
 */
static void internalize_iso(mp)
MULBUF *mp;
{
	register int i;

	/*
	 * Reject empty character set.
	 */
	if (CSISWRONG(mp->cs))
	{
		rebuffering_multi(mp);
		return;
	}

	/*
	 * Reject DEL and SPACE codes if currently 94 character set is
	 * selected.
	 */
	if (CS2TYPE(mp->cs) == TYPE_94_CHARSET ||
	    CS2TYPE(mp->cs) == TYPE_94N_CHARSET)
	{
		if ((mp->multibuf[mp->multiindex - 1] & 0x7f) == 0x7f)
		{
			rebuffering_multi(mp);
			return;
		} else if ((mp->multibuf[mp->multiindex - 1] & 0x7f) == 0x20)
		{
			internalize_noconv(mp);
			return;
		}
	}

	/*
	 * Buffering all ISO 2022 coding.  If multi bytes code
	 * finished, flush them.
	 */
	if (mp->multiindex < code_length(mp->cs))
		return;

	/*
	 * Normalize all codes
	 */
	mp->multiint[mp->intindex] = mp->multibuf[0] & ~0x80;
	mp->multics[mp->intindex] = mp->cs;
	for (i = 1; i < mp->multiindex; i++)
	{
		mp->multiint[mp->intindex + i] = mp->multibuf[i] & ~0x80;
		mp->multics[mp->intindex + i] = REST_MASK | mp->cs;
	}
	mp->intindex += mp->multiindex;
	mp->multiindex = 0;
}

#if JAPANESE
/*
 * Check and normalize all UJIS codes
 */
static void internalize_ujis(mp)
MULBUF *mp;
{
	if (mp->multiindex == 1)
	{
	} else if (mp->multiindex == 2)
	{
		if (ISUJISKANA(mp->multibuf[0], mp->multibuf[1]))
		{
			mp->multiint[mp->intindex] = mp->multibuf[1] & ~0x80;
			mp->multics[mp->intindex] = mp->cs;
			mp->intindex += 1;
			mp->multiindex = 0;
		} else if (ISUJISKANJI(mp->multibuf[0], mp->multibuf[1]))
		{
			if (mp->multibuf[0] < 85 + 0xa0) {
				mp->multiint[mp->intindex] =
					mp->multibuf[0] & ~0x80;
				mp->multics[mp->intindex] = mp->cs;
				mp->multiint[mp->intindex + 1] =
					mp->multibuf[1] & ~0x80;
				mp->multics[mp->intindex + 1] =
					REST_MASK | mp->cs;
				mp->intindex += 2;
				mp->multiindex = 0;
			} else
			{
				/*
				 * less ignore the user defined area
				 */
				internalize_wrong(mp);
			}
		}
	} else if (mp->multiindex == 3 &&
		   ISUJISKANJISUP(mp->multibuf[0], mp->multibuf[1],
				  mp->multibuf[2]))
	{
		if (output == sjis)
		{
			rebuffering_multi(mp);
		} else
		{
			if (mp->multibuf[0] < 78 + 0xa0) {
				mp->multiint[mp->intindex] =
					mp->multibuf[1] & ~0x80;
				mp->multics[mp->intindex] = mp->cs;
				mp->multiint[mp->intindex + 1] =
					mp->multibuf[2] & ~0x80;
				mp->multics[mp->intindex + 1] =
					REST_MASK | mp->cs;
				mp->intindex += 2;
				mp->multiindex = 0;
			} else
			{
				/*
				 * less ignore the user defined area
				 */
				internalize_wrong(mp);
			}
		}
	} else
	{
		rebuffering_multi(mp);
	}
}

/*
 * Check and normalize all SJIS codes
 */
static void internalize_sjis(mp)
MULBUF *mp;
{
	if (mp->multiindex == 1)
	{
		if (!ISSJISKANA(mp->multibuf[0]))
		{
			rebuffering_multi(mp);
		} else
		{
			mp->multiint[mp->intindex] = mp->multibuf[0] & ~0x80;
			mp->multics[mp->intindex] = mp->cs;
			mp->intindex += 1;
			mp->multiindex = 0;
		}
	} else if (mp->multiindex == 2 &&
		   ISSJISKANJI(mp->multibuf[0], mp->multibuf[1]))
	{
		if (mp->multibuf[0] < 0xeb) {
			register int c1, c2;
			c1 = mp->multibuf[0] - 0x81 -
				(mp->multibuf[0] >= 0xe0 ? 0xe0 - 0xa0 : 0x0);
			c2 = mp->multibuf[1] -
				(mp->multibuf[1] >= 0x80 ? 1 : 0);
			mp->multiint[mp->intindex] =
				(c1 * 2 + (c2 >= 0x9e ? 1 + 0x21 : 0x21));
			mp->multics[mp->intindex] = mp->cs;
			mp->multiint[mp->intindex + 1] =
				(c2 - (c2 >= 0x9e ? 0x9e - 0x21 : 0x40 - 0x21));
			mp->multics[mp->intindex + 1] = REST_MASK | mp->cs;
			mp->intindex += 2;
			mp->multiindex = 0;
		} else
		{
			/*
			 * less ignore the user defined area
			 */
			internalize_wrong(mp);
		}
	} else
	{
		rebuffering_multi(mp);
	}
}
#endif

static void internalize(mp)
MULBUF *mp;
{
	int c = mp->multibuf[mp->multiindex - 1];

	if (mp->multiindex == 1)
	{
		if ((c <= 0x7f && mp->io.left == noconv) ||
		    (c >= 0x80 && mp->io.right == noconv))
		{
#if JAPANESE
			mp->sequence_counter = 0;
#endif
			if (control_char(c))
				internalize_wrong(mp);
			else
				internalize_noconv(mp);
			return;
		} else if (c >= 0x80 && mp->io.right == none)
		{
#if JAPANESE
			mp->sequence_counter = 0;
#endif
			rebuffering_multi(mp);
			return;
		}

		mp->cs = ASCII;
		if (c < 0x20)
		{
#if JAPANESE
			mp->sequence_counter = 0;
#endif
			internalize_wrong(mp);
			return;
		} else if (c <= 0x7f ||
			   (mp->io.right == iso8 && (0xa0 <= c && c <= 0xff)))
		{
#if JAPANESE
			mp->sequence_counter = 0;
#endif
			/*
			 * Decide current character set.
			 */
			mp->cs = (mp->sg ? *mp->sg :
					   (c & 0x80) ? *mp->gr : *mp->gl);
			/*
			 * Check cs that fit for output code set.
			 */
			if (output == ujis && mp->cs != ASCII &&
			    mp->cs != JISX0201KANA &&
			    mp->cs != JISX0201ROMAN &&
			    mp->cs != JISX0208_78KANJI &&
			    mp->cs != JISX0208KANJI &&
			    mp->cs != JISX0208_90KANJI &&
			    mp->cs != JISX0212KANJISUP)
			{
				rebuffering_multi(mp);
				return;
			}
			if (output == sjis && mp->cs != ASCII &&
			    mp->cs != JISX0201KANA &&
			    mp->cs != JISX0201ROMAN &&
			    mp->cs != JISX0208_78KANJI &&
			    mp->cs != JISX0208KANJI &&
			    mp->cs != JISX0208_90KANJI)
			{
				rebuffering_multi(mp);
				return;
			}

			if (mp->cs != ASCII)
				mp->icharset = mp->cs;
			internalize_iso(mp);
			return;
		} else if (control_char(c))
		{
#if JAPANESE
			mp->sequence_counter = 0;
#endif
			internalize_wrong(mp);
			return;
		}
#if JAPANESE
		if (mp->priority == sjis && ISSJISKANA(c))
		{
			if (mp->io.right == japanese)
			{
				mp->sequence_counter++;
				if (mp->sequence_counter % 2 == 1 &&
				    mp->multibuf[0] != 0xa4)
					mp->sequence_counter = 0;
				if (mp->sequence_counter >= 6)
					/*
					 * It looks like a sequence of UJIS
					 * hiragana.  Thus we give priority
					 * to not sjis.
					 */
					mp->priority = ujis;
			}
			mp->cs = JISX0201KANA;
			mp->icharset = SJIS;
			internalize_sjis(mp);
			return;
		} else if (mp->io.right == ujis || mp->io.right == sjis ||
			   mp->io.right == japanese)
		{
			mp->sequence_counter = 0;
			return;
		}
		mp->sequence_counter = 0;
#endif
		internalize_wrong(mp);
		return;
	}

#if JAPANESE
	assert(mp->sequence_counter == 0);
#endif
	if (c < 0x20)
	{
		rebuffering_multi(mp);
		return;
	} else if (mp->cs != ASCII &&
		   (c <= 0x7f ||
		    (mp->io.right == iso8 && 0xa0 <= c && c <= 0xff)))
	{
		if (mp->cs != (mp->sg ? *mp->sg :
					(c & 0x80) ? *mp->gr : *mp->gl))
			rebuffering_multi(mp);
		else
			internalize_iso(mp);
		return;
	} else if (control_char(c))
	{
		rebuffering_multi(mp);
		return;
	}
#if JAPANESE
	if (mp->multiindex == 2)
	{
		if (mp->priority == sjis && ISSJISKANJI(mp->multibuf[0], c))
		{
			mp->cs = JISX0208KANJI;
			mp->icharset = SJIS;
			internalize_sjis(mp);
			return;
		} else if (mp->priority == ujis)
		{
			if (ISUJISKANA(mp->multibuf[0], c))
			{
				mp->cs = JISX0201KANA;
				mp->icharset = UJIS;
				internalize_ujis(mp);
				return;
			} else if (ISUJISKANJI(mp->multibuf[0], c))
			{
				mp->cs = JISX0208KANJI;
				mp->icharset = UJIS;
				internalize_ujis(mp);
				return;
			} else if (ISUJISKANJISUP(mp->multibuf[0], c, 0xa1))
			{
				return;
			}
		}

		if ((mp->io.right == sjis || mp->io.right == japanese) &&
		    ISSJISKANJI(mp->multibuf[0], c))
		{
			mp->cs = JISX0208KANJI;
			mp->priority = sjis;
			mp->icharset = SJIS;
			internalize_sjis(mp);
			return;
		} else if ((mp->io.right == ujis || mp->io.right == japanese))
		{
			if (ISUJISKANA(mp->multibuf[0], c))
			{
				mp->cs = JISX0201KANA;
				mp->priority = ujis;
				mp->icharset = UJIS;
				internalize_ujis(mp);
				return;
			} else if (ISUJISKANJI(mp->multibuf[0], c))
			{
				mp->cs = JISX0208KANJI;
				mp->priority = ujis;
				mp->icharset = UJIS;
				internalize_ujis(mp);
				return;
			} else if (ISUJISKANJISUP(mp->multibuf[0], c, 0xa1))
			{
				return;
			}
		}
	} else if (mp->multiindex == 3 &&
		   (mp->priority == ujis ||
		    mp->io.right == ujis || mp->io.right == japanese) &&
		   ISUJISKANJISUP(mp->multibuf[0], mp->multibuf[1], c))
	{
		mp->cs = JISX0212KANJISUP;
		mp->priority = ujis;
		mp->icharset = UJIS;
		internalize_ujis(mp);
		return;
	}
#endif
	rebuffering_multi(mp);
}

/*
 * Check routines 
 */
static int check_ft(mp, c, type, plane)
MULBUF *mp;
register int c;
int type;
int *plane;
{
	if (mp->io.left == jis)
	{
		if ((type == TYPE_94N_CHARSET &&
		     (c == '@' || c == 'B' || c == 'D')) ||
		    (type == TYPE_94_CHARSET &&
		     (c == 'B' || c == 'I' || c == 'J')))
		{
			*plane = (mp->irr ? IRR2CS(mp->irr) : 0) |
				TYPE2CS(type) | FT2CS(c);
			mp->irr = 0;
			mp->eseq = NOESC;
			return (0);
		}
	} else if (0x30 <= c && c <= 0x7e)
	{
		*plane = (mp->irr ? IRR2CS(mp->irr) : 0) |
			TYPE2CS(type) | FT2CS(c);
		mp->irr = 0;
		mp->eseq = NOESC;
		return (0);
	}
	return (-1);
}

static int check_irr(mp, c)
MULBUF *mp;
register int c;
{
	if (0x40 <= c && c <= 0x7e)
	{
		mp->irr = CODE2IRR(c);
		mp->eseq = NOESC;
		return (0);
	}
	return (-1);
}

static void fix_status_for_escape_sequence(mp)
MULBUF *mp;
{
	if (mp->eseq == NOESC)
	{
		switch (CS2TYPE(mp->sg ? *mp->sg : *mp->gl))
		{
		case TYPE_96_CHARSET:
		case TYPE_96N_CHARSET:
			change_control_char(0177, 0);
			break;
		case TYPE_94_CHARSET:
		case TYPE_94N_CHARSET:
			change_control_char(0177, 1);
			break;
		}
		switch (CS2TYPE(mp->sg ? *mp->sg : *mp->gr))
		{
		case TYPE_96_CHARSET:
		case TYPE_96N_CHARSET:
			change_control_char(0377, 0);
			break;
		case TYPE_94_CHARSET:
		case TYPE_94N_CHARSET:
			change_control_char(0377, 1);
			break;
		}
	}
}

static int check_escape_sequence(mp)
MULBUF *mp;
{
	int c = mp->multibuf[mp->multiindex - 1];

	switch (mp->eseq)
	{
	case ESC_:
		switch (c)
		{
		case '$': mp->eseq = ESC_2_4; break;
		case '&': mp->eseq = ESC_2_6; break;
		case '(': mp->eseq = ESC_2_8; break;
		case ')': mp->eseq = ESC_2_9; break;
		case '*': mp->eseq = ESC_2_10; break;
		case '+': mp->eseq = ESC_2_11; break;
		case ',': mp->eseq = ESC_2_12; break;
		case '-': mp->eseq = ESC_2_13; break;
		case '.': mp->eseq = ESC_2_14; break;
		case '/': mp->eseq = ESC_2_15; break;
		case 'N': mp->sg = &mp->g2; mp->eseq = NOESC; /*SS2*/break;
		case 'O': mp->sg = &mp->g3; mp->eseq = NOESC; /*SS3*/break;
		case 'n': mp->gl = &mp->g2; mp->eseq = NOESC; break;
		case 'o': mp->gl = &mp->g3; mp->eseq = NOESC; break;
		case '|': if (mp->io.right != iso8) goto wrong;
			  mp->gr = &mp->g3; mp->eseq = NOESC; break;
		case '}': if (mp->io.right != iso8) goto wrong;
			  mp->gr = &mp->g2; mp->eseq = NOESC; break;
		case '~': if (mp->io.right != iso8) goto wrong;
			  mp->gr = &mp->g1; mp->eseq = NOESC; break;
		default:  goto wrong;
		}
		break;
	case ESC_2_4:
		switch (c)
		{
		case '(': mp->eseq = ESC_2_4_8; break;
		case ')': mp->eseq = ESC_2_4_9; break;
		case '*': mp->eseq = ESC_2_4_10; break;
		case '+': mp->eseq = ESC_2_4_11; break;
		case '-': mp->eseq = ESC_2_4_13; break;
		case '.': mp->eseq = ESC_2_4_14; break;
		case '/': mp->eseq = ESC_2_4_15; break;
		case '@':
		case 'A':
		case 'B': if (check_ft(mp, c, TYPE_94N_CHARSET, &mp->g0) == 0)
				break;
		default:  goto wrong;
		}
		break;
	case ESC_2_6:
		if (check_irr(mp, c) == 0)
			break;
		goto wrong;
	case ESC_2_8:
		if (check_ft(mp, c, TYPE_94_CHARSET, &mp->g0) == 0)
			break;
		goto wrong;
	case ESC_2_9:
		if (check_ft(mp, c, TYPE_94_CHARSET, &mp->g1) == 0)
			break;
		goto wrong;
	case ESC_2_10:
		if (check_ft(mp, c, TYPE_94_CHARSET, &mp->g2) == 0)
			break;
		goto wrong;
	case ESC_2_11:
		if (check_ft(mp, c, TYPE_94_CHARSET, &mp->g3) == 0)
			break;
		goto wrong;
	case ESC_2_12:
		if (check_ft(mp, c, TYPE_96_CHARSET, &mp->g0) == 0)
			break;
		goto wrong;
	case ESC_2_13:
		if (check_ft(mp, c, TYPE_96_CHARSET, &mp->g1) == 0)
			break;
		goto wrong;
	case ESC_2_14:
		if (check_ft(mp, c, TYPE_96_CHARSET, &mp->g2) == 0)
			break;
		goto wrong;
	case ESC_2_15:
		if (check_ft(mp, c, TYPE_96_CHARSET, &mp->g3) == 0)
			break;
		goto wrong;
	case ESC_2_4_8:
		if (check_ft(mp, c, TYPE_94N_CHARSET, &mp->g0) == 0)
			break;
		goto wrong;
	case ESC_2_4_9:
		if (check_ft(mp, c, TYPE_94N_CHARSET, &mp->g1) == 0)
			break;
		goto wrong;
	case ESC_2_4_10:
		if (check_ft(mp, c, TYPE_94N_CHARSET, &mp->g2) == 0)
			break;
		goto wrong;
	case ESC_2_4_11:
		if (check_ft(mp, c, TYPE_94N_CHARSET, &mp->g3) == 0)
			break;
		goto wrong;
	case ESC_2_4_13:
		if (check_ft(mp, c, TYPE_96N_CHARSET, &mp->g1) == 0)
			break;
		goto wrong;
	case ESC_2_4_14:
		if (check_ft(mp, c, TYPE_96N_CHARSET, &mp->g2) == 0)
			break;
		goto wrong;
	case ESC_2_4_15:
		if (check_ft(mp, c, TYPE_96N_CHARSET, &mp->g3) == 0)
			break;
		goto wrong;
	case NOESC:
		/*
		 * If this sequences are wrong if currently does buffering.
		 */
		if (mp->multiindex != 1)
		{
			switch (c)
			{
			case 0033:
			case 0016:
			case 0017:
			case 0031: goto wrong;
			case 0216:
			case 0217: if (mp->io.right == iso8) goto wrong;
			default:   goto wrongone;
			}
		}
		switch (c)
		{
		case 0033: mp->eseq = ESC_; break;
		case 0016: mp->gl = &mp->g1; mp->eseq = NOESC; break;
		case 0017: mp->gl = &mp->g0; mp->eseq = NOESC; break;
		case 0031: mp->sg = &mp->g2; mp->eseq = NOESC; /*SS2*/ break;
		case 0216: if (mp->io.right != iso8) goto wrongone;
			   mp->sg = &mp->g2; mp->eseq = NOESC; /*SS2*/ break;
		case 0217: if (mp->io.right != iso8) goto wrongone;
			   mp->sg = &mp->g3; mp->eseq = NOESC; /*SS3*/ break;
		default:   goto wrongone;
		}
		break;
	default:
		assert(0);
	}
	if (mp->eseq == NOESC)
	{
		fix_status_for_escape_sequence(mp);
		mp->multiindex = 0;
		return (0);
	}
	return (0);
wrong:
	if (mp->eseq != NOESC)
	{
		mp->eseq = NOESC;
		fix_status_for_escape_sequence(mp);
	}
	assert(mp->multiindex != 1);
	rebuffering_multi(mp);
	return (0);
wrongone:
	assert(mp->eseq == NOESC);
	return (-1);
}

struct planeset {
	char *name;
	char *planeset;
} planesets[] = {
	{ "ascii",		""	},
	{ "ctext",		"\\e-A"	},
	{ "latin1",		"\\e-A"	},
	{ "latin2",		"\\e-B"	},
	{ "latin3",		"\\e-C"	},
	{ "latin4",		"\\e-D"	},
	{ "greek",		"\\e-F"	},
	{ "alabic",		"\\e-G"	},
	{ "hebrew",		"\\e-H"	},
	{ "cyrillic",		"\\e-L"	},
	{ "latin5",		"\\e-M"	},
	{ "japanese",		"\\e$)B\\e*I\\e$+D"	},
	{ "ujis",		"\\e$)B\\e*I\\e$+D"	},
	{ "euc",		"\\e$)B\\e*I\\e$+D"	},
	{ NULL }
};

int set_planeset(name)
register char *name;
{
	register struct planeset *p;
	MULBUF *mp;
	int ret;

	if (name == NULL)
		return;
	for (p = planesets; p->name != NULL; p++)
		if (strcmp(name, p->name) == 0)
		{
			name = p->planeset;
			break;
		}
	mp = new_multi();
	init_priority(mp);
	while (*name)
	{
		if (*name == '\\' &&
		    (*(name + 1) == 'e' || *(name + 1) == 'E'))
		{
			mp->multibuf[mp->multiindex++] = '\033';
			ret = check_escape_sequence(mp);
			name += 2;
		} else
		{
			mp->multibuf[mp->multiindex++] = *name++;
			ret = check_escape_sequence(mp);
		}
		if (ret < 0 || mp->intindex > 0)
		{
			free(mp);
			return -1;
		}
	}
	def_g0 = mp->g0;
	def_g1 = mp->g1;
	def_g2 = mp->g2;
	def_g3 = mp->g3;
	free(mp);
	return 0;
}

void init_def_codesets(left, right, out)
CODESET left;
CODESET right;
CODESET out;
{
	def_left = left;
	def_right = right;
	output = out;
}

void init_def_priority(pri)
CODESET pri;
{
#if JAPANESE
	assert(pri == sjis || pri == ujis);
	def_priority = pri;
#endif
}

void init_priority(mp)
MULBUF *mp;
{
#if JAPANESE
	if (mp->io.right == sjis)
		mp->priority = sjis;
	else if (mp->io.right == ujis)
		mp->priority = ujis;
	else if (mp->io.right == japanese)
		mp->priority = def_priority;
	else
		mp->priority = noconv;
	mp->sequence_counter = 0;
#endif
}

CODESET get_priority(mp)
MULBUF *mp;
{
#if JAPANESE
	return (mp->priority);
#else
	return (noconv);
#endif
}

void set_priority(mp, pri)
MULBUF *mp;
CODESET pri;
{
#if JAPANESE
	assert(pri == sjis || pri == ujis || pri == noconv);
	mp->priority = pri;
#endif
}

MULBUF *new_multi()
{
	MULBUF *mp = (MULBUF*) ecalloc(1, sizeof(MULBUF));
	mp->io.left = def_left;
	mp->io.right = def_right;
	mp->orig_io_right = def_right;
	mp->rotation_io_right = 0;
	mp->eseq = NOESC;
	init_multi(mp);
	return (mp);
}

void clear_multi(mp)
MULBUF *mp;
{
	mp->multiindex = 0;
	mp->intindex = 0;
	mp->bufbytes = 0;
}

void init_multi(mp)
MULBUF *mp;
{
	if (mp->eseq != NOESC)
	{
		mp->eseq = NOESC;
		fix_status_for_escape_sequence(mp);
	}
	mp->cs = ASCII;
	mp->g0 = def_g0;
	mp->g1 = def_g1;
	mp->g2 = def_g2;
	mp->g3 = def_g3;
	mp->gl = &mp->g0;
	mp->gr = &mp->g1;
	mp->sg = NULL;
	mp->irr = 0;
#if JAPANESE
	mp->sequence_counter = 0;
#endif
	mp->icharset = ASCII;
	clear_multi(mp);
}

/*
 * Buffering characters untile get a guarantee that it is right sequence.
 */
static void buffering_multi_internal(mp, c)
MULBUF *mp;
int c;
{
	mp->multibuf[mp->multiindex] = c;
	mp->multiindex++;
	mp->bufbytes++;

	if (mp->io.left == jis || mp->io.left == iso7 ||
	    mp->io.right == iso8)
		if (check_escape_sequence(mp) == 0)
			return;		/* going process well */

	/* it is not a escape sequence, try to use it as character */
	internalize(mp);

	/*
	 * If a character was detected in internalize(),
	 * clean sg since single shift affect only one character.
	 */
	if (mp->multiindex == 0 && mp->sg)
	{
		mp->sg = NULL;
		fix_status_for_escape_sequence(mp);
	}
}

static void rebuffering_multi(mp)
MULBUF *mp;
{
	int i;
	int oldindex;

	/*
	 * Buffering character has some problem, so I clear single-shifted
	 * character set because it effect only one character which
	 * is detected wrong.
	 */
	if (mp->sg)
	{
		mp->sg = NULL;
		fix_status_for_escape_sequence(mp);
	}

	/*
	 * Convert first buffered data as wrong multi bytes data.
	 */
	assert(mp->intindex + 1 <= sizeof(mp->multiint));
	mp->multiint[mp->intindex] = mp->multibuf[0];
	mp->multics[mp->intindex] = WRONGCS;
	mp->intindex++;

#if JAPANESE
	/*
	 * Quick japanese code hack.
	 * Check whether character is SJIS KANA.  If so, less has just
	 * detected the prediction is failed.  Try recognize it well.
	 */
	if ((mp->priority == sjis ||
	     mp->io.right == sjis || mp->io.right == japanese) && 
	    ISSJISKANA(mp->multiint[mp->intindex - 1]))
	{
		mp->cs = JISX0201KANA;
		mp->priority = sjis;
		mp->icharset = SJIS;
		mp->multiint[mp->intindex - 1] &= ~0x80;
		mp->multics[mp->intindex - 1] = mp->cs;
	}
#endif

	/*
	 * Retry to parse rest of buffered data.
	 */
	oldindex = mp->multiindex;
	mp->bufbytes -= mp->multiindex - 1;
	mp->multiindex = 0;
	for (i = 1; i < oldindex; i++)
		buffering_multi_internal(mp, mp->multibuf[i]);
}

/*
 * Buffering characters untile get a guarantee that it is right sequence.
 */
void buffering_multi(mp, c, strbuf, csbuf, length)
MULBUF *mp;
int c;
unsigned char **strbuf;
CHARSET **csbuf;
unsigned int *length;
{
	assert(mp->intindex == 0);
	mp->bufbytes = mp->multiindex;

	if (c < 0)
	{
		/*
		 * Force to flush out buffered characters.
		 */
		if (mp->eseq != NOESC)
		{
			mp->eseq = NOESC;
			fix_status_for_escape_sequence(mp);
		}
		if (mp->multiindex)
			rebuffering_multi(mp);
	} else
	{
		/*
		 * Put it into buffer and parse it.
		 */
		buffering_multi_internal(mp, c);
	}

	*strbuf = mp->multiint;
	*csbuf = mp->multics;
	*length = mp->intindex;
	mp->intindex = 0;
}

/*
 * Parse and discard characters.  This routine is used for chopping line.
 */
void parsing_multi(mp, c)
MULBUF *mp;
int c;
{
	unsigned char *strbuf;
	CHARSET *csbuf;
	unsigned int length;

	buffering_multi(mp, c, &strbuf, &csbuf, &length);
}

void set_codesets(mp, left, right)
MULBUF *mp;
CODESET left;
CODESET right;
{
	mp->io.left = left;
	mp->io.right = right;
}

/*
 * Return the number of buffering characters.
 */
int get_bufbytes(mp)
MULBUF *mp;
{
	return (mp->bufbytes);
}

/*
 * Adjust the number of buffering characters.
 */
void set_bufbytes(mp, bufbytes)
MULBUF *mp;
int bufbytes;
{
	mp->bufbytes = bufbytes;
}

/*
 * Return string representation about multi bytes character
 * which was buffered.
 */
char *get_icharset_string(mp)
MULBUF *mp;
{
	static char buf[10];

	switch (mp->icharset)
	{
#if JAPANESE
	/*
	 * Code set
	 */
	case SJIS:		return ("SJIS");
	case UJIS:		return ("UJIS");
#endif
	/*
	 * Character set
	 */
	case ASCII:		return ("ASCII");
	case JISX0201KANA:	return ("JIS-KANA");
	case JISX0201ROMAN:	return ("JIS-ROMAN");
	case LATIN1:		return ("LATIN1");
	case LATIN2:		return ("LATIN2");
	case LATIN3:		return ("LATIN3");
	case LATIN4:		return ("LATIN4");
	case GREEK:		return ("GREEK");
	case ARABIC:		return ("ARABIC");
	case HEBREW:		return ("HEBREW");
	case CYRILLIC:		return ("CYRILLIC");
	case LATIN5:		return ("LATIN5");
	case JISX0208_78KANJI:	return ("JIS-78KANJI");
	case GB2312:		return ("GB2312");
	case JISX0208KANJI:	return ("JIS-83KANJI");
	case JISX0208_90KANJI:	return ("JIS-90KANJI");
	case KSC5601:		return ("KSC5601");
	case JISX0212KANJISUP:	return ("JIS-KANJISUP");
	}
	switch (CS2TYPE(mp->icharset))
	{
	case TYPE_94_CHARSET:
		strcpy(buf, "94( )");
		buf[3] = CS2FT(mp->icharset);
		break;
	case TYPE_96_CHARSET:
		strcpy(buf, "96( )");
		buf[3] = CS2FT(mp->icharset);
		break;
	case TYPE_94N_CHARSET:
		strcpy(buf, "94N( )");
		buf[4] = CS2FT(mp->icharset);
		break;
	case TYPE_96N_CHARSET:
		strcpy(buf, "96N( )");
		buf[4] = CS2FT(mp->icharset);
		break;
	default:
		assert(0);
	}
	if (CS2IRR(mp->icharset) > 0)
	{
		char num[3];
		sprintf(num, "%d", CS2IRR(mp->icharset));
		strcat(buf, num);
	}
	return (buf);
}

static int old_output_charset = ASCII;	/* Last displayed character set */

static unsigned char *make_escape_sequence(charset)
int charset;
{
	static unsigned char p[9];
	int len;

	p[0] = '\033';
	len = 1;
	if ((output == iso7 || output == iso8) && CS2IRR(charset) > 0)
	{
		p[len] = '&';
		p[len + 1] = IRR2CODE(CS2IRR(charset));
		p[len + 2] = '\033';
		len += 3;
	}
	switch (CS2TYPE(charset))
	{
	case TYPE_94_CHARSET:
		p[len] = '(';
		p[len + 1] = CS2FT(charset);
		len += 2;
		break;
	case TYPE_94N_CHARSET:
		switch (CS2FT(charset))
		{
		case '@':
		case 'A':
		case 'B':
			p[len] = '$';
			p[len + 1] = CS2FT(charset);
			len += 2;
			break;
		default:
			p[len] = '$';
			p[len + 1] = '(';
			p[len + 2] = CS2FT(charset);
			len += 3;
			break;
		}
		break;
	case TYPE_96_CHARSET:
		p[len] = '-';
		p[len + 1] = CS2FT(charset);
		len += 2;
		break;
	case TYPE_96N_CHARSET:
		p[len] = '$';
		p[len + 1] = '-';
		p[len + 2] = CS2FT(charset);
		len += 3;
		break;
	}
	if (output != iso8)
	{
		switch (CS2TYPE(charset))
		{
		case TYPE_94_CHARSET:
		case TYPE_94N_CHARSET:
			switch (CS2TYPE(old_output_charset))
			{
			case TYPE_96_CHARSET:
			case TYPE_96N_CHARSET:
				p[len] = '\017';
				len++;
			}
			break;
		case TYPE_96_CHARSET:
		case TYPE_96N_CHARSET:
			switch (CS2TYPE(old_output_charset))
			{
			case TYPE_94_CHARSET:
			case TYPE_94N_CHARSET:
				p[len] = '\016';
				len++;
			}
			break;
		}
	}
	p[len] = '\0';
	return (p);
}

static char cvbuffer[32];
static int cvindex = 0;
static char *nullcvbuffer = "";


static char *convert_to_iso(c, cs)
int c;
int cs;
{
	register unsigned char *p;
	static char buffer2[2];

	if (output == iso8 && c != 0 &&
	    (CS2TYPE(cs) == TYPE_96_CHARSET ||
	     CS2TYPE(cs) == TYPE_96N_CHARSET))
		c |= 0x80;

	buffer2[0] = c;
	buffer2[1] = '\0';

	if (CSISREST(cs))
	{
	    return (buffer2);
	}

	cs = CS2CHARSET(cs);

	if (cs == old_output_charset)
	{
		return (buffer2);
	}
	else
	{
		p = make_escape_sequence(cs);
		old_output_charset = cs;
		strcpy(cvbuffer, p);
		strcat(cvbuffer, buffer2);
		return (cvbuffer);
	}
}

#if JAPANESE
static char *convert_to_ujis(c, cs)
int c;
int cs;
{
	if (c == 0)
	{
		cvindex = 0;
		return (nullcvbuffer);
	}

	cvbuffer[cvindex++] = c;
	cvbuffer[cvindex] = '\0';

	cs = CS2CHARSET(cs);

	if (cs == ASCII || cs == JISX0201ROMAN)
	{
		assert(cvindex == 1);
		cvindex = 0;
		return (cvbuffer);
	} else if (cs == JISX0201KANA)
	{
		assert(cvindex == 1);
		cvbuffer[2] = '\0';
		cvbuffer[1] = cvbuffer[0] | 0x80;
		cvbuffer[0] = 0x8e;
		cvindex = 0;
		return (cvbuffer);
	} else if (cs == JISX0208_78KANJI || cs == JISX0208KANJI ||
		   cs == JISX0208_90KANJI)
	{
		if (cvindex == 1)
			return (nullcvbuffer);
		assert(cvindex == 2);
		cvbuffer[0] |= 0x80;
		cvbuffer[1] |= 0x80;
		cvindex = 0;
		return (cvbuffer);
	} else if (cs == JISX0212KANJISUP)
	{
		if (cvindex == 1)
			return (nullcvbuffer);
		assert(cvindex == 2);
		cvbuffer[2] = cvbuffer[1] | 0x80;
		cvbuffer[1] = cvbuffer[0] | 0x80;
		cvbuffer[0] = 0x8f;
		cvbuffer[3] = '\0';
		cvindex = 0;
		return (cvbuffer);
	}
	assert(CSISWRONG(cs));
	assert(cvindex == 1);
	cvindex = 0;
	return (cvbuffer);
}

static char *convert_to_sjis(c, cs)
int c;
int cs;
{
	if (c == 0)
	{
		cvindex = 0;
		return (nullcvbuffer);
	}

	cvbuffer[cvindex++] = c;
	cvbuffer[cvindex] = '\0';

	cs = CS2CHARSET(cs);

	if (cs == ASCII || cs == JISX0201ROMAN)
	{
		assert(cvindex == 1);
		cvindex = 0;
		return (cvbuffer);
	} else if (cs == JISX0201KANA)
	{
		assert(cvindex == 1);
		cvbuffer[0] |= 0x80;
		cvindex = 0;
		return (cvbuffer);
	} else if (cs == JISX0208_78KANJI || cs == JISX0208KANJI ||
		   cs == JISX0208_90KANJI)
	{
		register int c1, c2;
		if (cvindex == 1)
			return (nullcvbuffer);
		assert(cvindex == 2);
		c1 = ((cvbuffer[0] & ~0x80) - 0x21) / 2 + 0x81;
		c2 = (cvbuffer[1] & ~0x80) +
		     ((cvbuffer[0] & 1) ? 0x40 - 0x21 : 0x9e - 0x21);
		cvbuffer[0] = c1 + (c1 >= 0xa0 ? 0x40 : 0);
		cvbuffer[1] = c2 + (c2 >= 0x7f ? 1 : 0);
		cvindex = 0;
		return (cvbuffer);
	}
	assert(CSISWRONG(cs));
	assert(cvindex == 1);
	cvindex = 0;
	return (cvbuffer);
}
#endif

char *outchar(c, cs)
int c;
CHARSET cs;
{
	if (c < 0)
	{
		c = 0;
		cs = ASCII;
	}

	if (output == jis || output == iso7 || output == iso8)
		return (convert_to_iso(c, cs));
#if JAPANESE
	if (output == ujis)
		return (convert_to_ujis(c, cs));
	if (output == sjis)
		return (convert_to_sjis(c, cs));
#endif
	cvbuffer[0] = c;
	cvbuffer[1] = '\0';
	return (cvbuffer);
}

char *outbuf(p, cs)
unsigned char *p;
CHARSET cs;
{
	static char buffer[1024];
	char *s;
	int i = 0;

	while (*p != '\0')
	{
		s = outchar(*p++, cs);
		while (*s != '\0')
			buffer[i++] = *s++;
		assert(i < sizeof(buffer));
	}
	buffer[i] = '\0';
	return (buffer);
}

int mwidth(c, cs)
int c;
CHARSET cs;
{
	if (CSISREST(cs))
		return (0);
	switch (CS2TYPE(cs))
	{
	case TYPE_94_CHARSET:
	case TYPE_96_CHARSET:
		return (1);
	case TYPE_94N_CHARSET:
	case TYPE_96N_CHARSET:
		return (2);
	default:
		assert(0);
	}
}

char *rotate_right_codeset(mp)
MULBUF *mp;
{
	char *p;

	mp->rotation_io_right++;
	mp->rotation_io_right %= 7;
	switch (mp->rotation_io_right) {
	case 0: p = "original"; mp->io.right = mp->orig_io_right; break;
	case 1: p = "japanese"; mp->io.right = japanese; break;
	case 2: p = "ujis"; mp->io.right = ujis; break;
	case 3: p = "sjis"; mp->io.right = sjis; break;
	case 4: p = "iso8"; mp->io.right = iso8; break;
	case 5: p = "noconv"; mp->io.right = noconv; break;
	case 6: p = "none"; mp->io.right = none; break;
	default: assert(0); break;
	}
	init_priority(mp);
	return (p);
}

#endif

int strlen_cs(str, cs)
char* str;
CHARSET* cs;
{
	int i = 0;
	if (cs == NULL)
		return strlen(str);
	while (*str != NULCH || !CSISNULLCS(*cs)) {
		str++;
		cs++;
		i++;
	}
	return i;
}

int chlen_cs(chstr, cs)
char* chstr;
CHARSET* cs;
{
	int i;
	if (cs == NULL)
	{
		if (chstr == NULL || *chstr == NULCH)
			return 0;
		else
			return 1;
	}
	if (*chstr == NULCH && CSISNULLCS(*cs))
		return 0;
	i = 0;
	do {
		i++;
		cs++;
	} while (CSISREST(*cs));
	return i;
}

char* strdup_cs(str, cs, csout)
char* str;
CHARSET* cs;
CHARSET** csout;
{
	int len = strlen_cs(str, cs);
	char* save_str = (char *)ecalloc(len + 1, 1);
	CHARSET* save_cs = (CHARSET *)ecalloc(len + 1, sizeof(CHARSET));
	memcpy(save_str, str, sizeof(char) * (len + 1));
	if (cs)
		memcpy(save_cs, cs, sizeof(CHARSET) * (len + 1));
	else {
		cs = save_cs;
		while (--len >= 0)
			*cs++ = ASCII;
		*cs = NULLCS;
	}
	*csout = save_cs;
	return save_str;
}
