/*
  $Id: sock.c,v 1.3 1996/08/18 16:09:17 luik Exp $

  sock.c - socket interface for omirrd to connect to other servers.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: sock.c,v 1.3 1996/08/18 16:09:17 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <errno.h>
#include <fcntl.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "libomirr.h"
#include "error.h"
#include "sock.h"



/* XXX TODO: make the following configuration parameters configurable,
   e.g. using config file variables. */
#define BIND_RETRIES	    3	/* number of retries for bind */
#define BIND_INITIALTIMO    1	/* seconds to sleep for first retry */

#define LISTEN_RETRIES	    3	/* number of retries for listen */
#define LISTEN_INITIALTIMO  1	/* seconds to sleep for first retry */

#define CONNECT_RETRIES	    2	/* number of retries for connect */
#define CONNECT_INITIALTIMO 1	/* seconds to sleep for first retry */


/* sockOpen - open a TCP/IP socket and bind it to the specified host
   and port. If the host is NULL, the local host (INADDR_ANY) is
   used. Otherwise it specifies either a dotted-decimal address or a
   hostname of the remote host. The port is determined from the
   `service' argument, unless NULL, or may be specified explicitly in
   `port'. If `res_port' is true, the socket is bound to a reserved
   port on the local machine using `rresvport(3N)'. Returns the socket
   and the sockaddr_in structure.  */
static int sockOpen(char *host, char *service, int port, int res_port,
		    struct sockaddr_in *sock_addrp)
{
    int s;			/* socket fd */
    int resvport;		/* argument for rresvport(3N) */
    struct servent *sp;		/* from getservbyname(3N) */
    struct hostent *hp;		/* from gethostbyname(3N) */
    unsigned long inaddr;	/* from inet_addr(3N) */

    /* First clear the sockaddr_in structure, then fill in the port
       number, probably from the service database.  */
    memzero(sock_addrp, sizeof(*sock_addrp));
    sock_addrp->sin_family = AF_INET;
    sock_addrp->sin_port = htons(port);
    if (service) {
	if ((sp = getservbyname(service, "tcp")))
	    sock_addrp->sin_port = sp->s_port;
	else warning("unknown service %s/tcp, using port %d\n", service, port);
    }

    /* Now fill in the IP address of the host if specified. First try
       the name as a dotted-decimal number, if that fails, try
       gethostbyname.  */
    if (host) {
	if (((inaddr = inet_addr(host))) != -1)
	    memcpy(&sock_addrp->sin_addr, &inaddr, sizeof(inaddr));
	else {
	    if ((hp = gethostbyname(host)))
		memcpy(&sock_addrp->sin_addr, hp->h_addr, hp->h_length);
	    else return error("unknown host %s\n", host), -1;
	}
    }
    else {
	sock_addrp->sin_addr.s_addr = htonl(INADDR_ANY);
    }

    if (res_port) {
	resvport = IPPORT_RESERVED - 1;
	if (((s = rresvport(&resvport))) == -1)
	    return error("open reserved socket failed: %s\n", xstrerror(errno)), -1;
    }
    else {
	if (((s = socket(PF_INET, SOCK_STREAM, 0))) == -1)
	    return error("open socket failed: %s\n", xstrerror(errno)), -1;
    }
    return (s);
}


/* sockServer - opens a socket suitable for a TCP/IP server process
   and binds it to the specified service/port.  Since opening the
   server socket is essential, we try hard to get it done using
   several retries for the bind and listen calls.  Returns the socket
   or -1.  */

int sockServer(char *service, int port)
{
    int s;
    int retries;
    int timo;
    const int one = 1;
    struct sockaddr_in sock_addr; /* server sockaddr_in */

    if (((s = sockOpen(NULL, service, port, 0, &sock_addr))) == -1) {
	/* error() already called from sockOpen */
	return -1;
    }

#ifdef SO_REUSEADDR
    (void) setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof one);
#endif

    timo = BIND_INITIALTIMO;
    for (retries = BIND_RETRIES; retries; timo *= 2, retries--) {
	if (bind(s, (struct sockaddr *) &sock_addr, sizeof(sock_addr)) != -1)
	    break;
	if (retries > 1)	/* no need to sleep for last repetition */
	    sleep(timo);
    }
    if (retries == 0) {
	error("bind tcp socket (port %d) failed: %s\n",
	      ntohs(sock_addr.sin_port), xstrerror(errno));
	close(s);
	return -1;
    }

    timo = LISTEN_INITIALTIMO;
    for (retries = LISTEN_RETRIES; retries; timo *= 2, retries--) {
	if (listen(s, 5) != -1)
	    break;
	if (retries > 1)	/* no need to sleep for last repetition */
	    sleep(timo);
    }
    if (retries == 0) {
	error("listen on tcp socket failed: %s\n", xstrerror(errno));
	close(s);
	return -1;
    }
    return (s);
}


/* sock_client - opens a socket suitable for a client process. Flag
   `non_blocking' specifies whether socket should be put into
   non-blocking mode.  Non-blocking mode affects the connect(3N) call,
   too.  Returns the socket or -1.  */
static int sock_client(char *host, char *service, int port,
		       struct sockaddr_in *sock_addrp, int non_blocking)
{
    int s;

    if (((s = sockOpen(host, service, port, 1, sock_addrp))) == -1) {
	/* error() already called from sockOpen */
	return -1;
    }

    if (non_blocking) {
	int flags;
	if (((flags = fcntl(s, F_GETFL))) != -1)
	    if ((flags & O_NONBLOCK) == 0)
		flags = fcntl(s, F_SETFL, flags | O_NONBLOCK);
	if (flags == -1)
	    error("can't set connection to %s to non-blocking: %s\n",
		  host, xstrerror(errno));
    }
    return s;
}


/* sockClient - opens a socket suitable for a client process and
   builds the sockaddr_in structure to be used by connect using the
   specified host and port.  Returns the socket or -1.  */
int sockClient(char *host, char *service, int port,
	       struct sockaddr_in *sock_addrp)
{
    return sock_client(host, service, port, sock_addrp, 0);
}


/* sockClientNonBlocking - opens a non blocking socket suitable for a
   client process and builds the sockaddr_in structure to be used by
   connect.  Returns the socket or -1.  */
int sockClientNonBlocking(char *host, char *service, int port,
			  struct sockaddr_in *sock_addrp)
{
    return sock_client(host, service, port, sock_addrp, 1);
}



/* sockConnect - connects an open socket to a server port specified in
   `*sock_addrp'. Retries the connect CONNECT_RETRIES times or until
   the connect succeeds or an errno not equal to EALREADY, EINPROGRESS
   or EISCONN is returned.  Returns the return value of the
   connect(3N) call.  */
int sockConnect(int s, struct sockaddr_in *sock_addrp)
{
    int retries;
    int timo;
    int result;

    timo = CONNECT_INITIALTIMO;
    for (retries = CONNECT_RETRIES; retries; timo *= 2, retries--) {
      again:
	result = connect(s, (struct sockaddr *) sock_addrp, sizeof(*sock_addrp));
	if (result == -1 && errno == EINTR)
	    goto again;
	if (result != -1)
	    break;
	if (errno != EALREADY && errno != EINPROGRESS && errno != EISCONN)
	    break;
	if (retries > 1)	/* no need to sleep for last repetition */
	    sleep(timo);
    }
    return result;
}



/* sockAccept - accept a new client from `sockfd'. If `sock_addrp' is
   not NULL, returns client sockaddr_in in `*sock_addrp' and address
   length in `*sock_addr_lenp'.  Returns new socket fd or -1 on
   failure.  */
int sockAccept(int sockfd, struct sockaddr_in *sock_addrp, int *sock_addr_lenp)
{
    int s;
    struct sockaddr_in sock_addr; /* client sockaddr_in */
    int sock_addr_len;

    if (!sock_addrp) {		/* caller not interested in sock_addr data */
	sock_addrp = &sock_addr;
	sock_addr_lenp = &sock_addr_len;
    }

  again:
    *sock_addr_lenp = sizeof(sock_addr);
    s = accept(sockfd, (struct sockaddr *) sock_addrp, sock_addr_lenp);
    if (s == -1 && errno == EINTR) /* interrupted system call */
	goto again;

    return (s);
}


/* sockReservedPort - returns 1 if the port in the `sock_addrp' is in
   the range of reserved ports (512..1023) and 0 otherwise.  */
int sockReservedPort(struct sockaddr_in *sock_addrp)
{
    int port = ntohs(sock_addrp->sin_port); /* convert to host byte order */

    return (port >= IPPORT_RESERVED / 2 && port < IPPORT_RESERVED);
}


/* sockHostname - tries to determine the hostname from the
   `sock_addrp' using gethostbyaddr(3N).  Returns the official
   hostname, if successful, and NULL otherwise.  */
const char *sockHostname(struct sockaddr_in *sock_addrp)
{
    struct hostent *hp;

    hp = gethostbyaddr((char *) &sock_addrp->sin_addr, sizeof(struct in_addr),
		       sock_addrp->sin_family);
    if (hp)
	return hp->h_name;
    return NULL;
}

