/*
  $Id: bio.c,v 1.9 1997/01/12 22:50:55 luik Exp $

  bio.c - buffered input/output routines for omirrd.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: bio.c,v 1.9 1997/01/12 22:50:55 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <stdlib.h>
#include <string.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#include <fcntl.h>
#include "bio.h"


BioContext bioAlloc(void)
{
    BioContext context;
    if ((context = calloc(sizeof(BioContextRec), 1))) {
    }
    return context;
}


void bioFree(BioContext context)
{
    free(context);
}


BioContext bioAddWriteBuffer(BioContext context, int fd, MpContext mp_context)
{
    BioBuffer bb;

    if (fd >= FD_SETSIZE || (bb = &context->writebufs[fd])->buf)
	return NULL /* XXX ERROR */;

    bb->buf_size = BIO_BUF_SIZE;
    bb->buf = malloc(bb->buf_size);	/* XXX ERROR */
    bb->buf_start = 0;
    bb->buf_end = 0;

    bb->fd_flags = fcntl(fd, F_GETFL);
    if (bb->fd_flags != -1 && (bb->fd_flags & O_NONBLOCK) == 0)
	fcntl(fd, F_SETFL, bb->fd_flags | O_NONBLOCK);

    bb->mp_context = mp_context;

    return (context);
}


BioContext bioRemoveWriteBuffer(BioContext context, int fd)
{
    BioBuffer bb;

    if (fd >= FD_SETSIZE || (bb = &context->writebufs[fd])->buf == NULL)
	return NULL /* XXX ERROR */;

    if (bb->fd_flags != -1 && (bb->fd_flags & O_NONBLOCK) == 0)
	fcntl(fd, F_SETFL, bb->fd_flags);

    free(bb->buf);
    bb->buf = NULL;

    return (context);
}


/* bioMpWriteFunc - write callback function for mp module. Called if
   data may be written to fd, calls write in non-blocking mode and
   tries to write some data from the buffer to the file descriptor
   `fd'. Returns number of bytes written or -1 on error.  */

int bioMpWriteFunc(int fd, void *data)
{
    BioContext context = (BioContext) data;
    BioBuffer bb;
    int result;

    if (fd >= FD_SETSIZE || (bb = &context->writebufs[fd])->buf == NULL)
	return (errno = EBADF, -1);

    /* try to write as many bytes of buffer as possible */
    result = write(fd, bb->buf + bb->buf_start, bb->buf_end - bb->buf_start);
    if (result > 0) {
	bb->buf_start += result;

	if (bb->buf_start == bb->buf_end)
	    mpRemoveWriteCallback(bb->mp_context, fd);
    }    
    else if (result == -1 && errno == EINTR) {
	result = 0;		/* interrupted system call is not an error */
    }
    else if (result == -1 && errno == EAGAIN) {
	result = 0;		/* blocking condition is not an error */
    }
#ifdef EWOULDBLOCK
    else if (result == -1 && errno == EWOULDBLOCK) {
	result = 0;		/* blocking condition is not an error */
    }
#endif

    return (result);
}

void bioWrite(BioContext context, int fd, const void *buf, size_t len)
{
    BioBuffer bb;

    if (fd >= FD_SETSIZE || (bb = &context->writebufs[fd])->buf == NULL)
	return /* XXX ERROR */;

    /* Move buffer data to beginning of buffer if new data does not fit.  */
    if (len > bb->buf_size - bb->buf_end) {
	memmove(bb->buf, bb->buf + bb->buf_start, bb->buf_end - bb->buf_start);
	bb->buf_end -= bb->buf_start;
	bb->buf_start -= bb->buf_start;	/* bb->buf_start = 0 */
    }

    /* If data still does not fit, grow buffer.  */
    if (len > bb->buf_size - bb->buf_end) {
	bb->buf_size *= 2;
	if (len > bb->buf_size - bb->buf_end)
	    bb->buf_size = len + bb->buf_end + BIO_BUF_SIZE;
	bb->buf = realloc(bb->buf, bb->buf_size); /* XXX ERROR */
    }

    /* Add data to buffer.  */
    memcpy(bb->buf + bb->buf_end, buf, len);
    bb->buf_end += len;

    /* Add `bioMpWriteFunc' as write callback. This function will
       flush the buffered data written with `bioWrite' to `fd'.  */
    mpAddWriteCallback(bb->mp_context, fd,
		       bioMpWriteFunc, (void *) context);

    /* If current buffer size is alreay three times greater than
       original buffer size, try to flush some output data.  */
    if (bb->buf_size > 3 * BIO_BUF_SIZE)
	mpFlush(bb->mp_context, fd);
}

