/*
 * mknbi.c  -  MaKe NetBoot Image for Linux
 *
 * Copyright (C) 1995,1998 Gero Kuhlmann   <gero@gkminix.han.de>
 * Copyright (C) 1996,1997 Gero Kuhlmann   <gero@gkminix.han.de>
 *                and Markus Gutschke <gutschk@math.uni-muenster.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"
#ifdef HAVE_INET
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif
#include "nblib.h"
#include "mknbi.h"

#ifndef _MKNBI_H_LINUX_
#error Included wrong header file
#endif


static char *batchname = NULL;		/* name of system to batch process */
static char *outname = NULL;		/* Name of output file */
static char *rdname = NULL;		/* Name of ramdisk image file */
static char *nfsdir = NULL;		/* Directory to mount as root via NFS */
static char *rootdev = NULL;		/* Name of root device */
static char *kname = NULL;		/* Name of kernel image file */
static char *append = NULL;		/* String to append to end of command line */
static char *addrs = NULL;		/* String containing various addresses */
static char *rdmodebuf = NULL;		/* Buffer for ramdisk mode string */
static int rdmode = RD_AUTO;		/* Ramdisk load mode */
static int cmdl_msize = CMDLMSIZE;	/* Maximum size of command line */
static int kimage;			/* File handle for kernel image */
static int outfile;			/* File handle for output file */
static int rdimage = -1;		/* File handle for ramdisk image */
static int rdlocation = -1;		/* Memory location of ramdisk image */

static int cur_rec_num = -1;		/* Number of current load record */
static struct load_header header;	/* Load header */
static struct load_record *cur_rec;	/* Pointer to current load record */

static unsigned char copyrec_buf[SECTSIZE];



/*
 * Command line options and arguments
 */
static struct cmdopt opts[] = {
	{ "batch-sys", 'b', strval, {(char **)&batchname}, NULL,
	  "name of system to process", "SYSTEM"				},
	{ "outfile", 'o', strval, {(char **)&outname}, NULL,
	  "name of boot image output file", "FILE"			},
	{ "ramdisk-image", 'r', strval, {(char **)&rdname}, NULL,
	  "ramdisk image source file or directory", "FILE|DIR"		},
	{ "append", 'a', strval, {(char **)&append}, NULL,
	  "append string to end of kernel command line", "STRING"	},
	{ "root-dir", 'd', strval, {(char **)&nfsdir}, NULL,
	  "NFS mounted root dir:\n"
	  "  STRING syntax: rom|ram|kernel|<dir>", "STRING"				},
	{ "ip-addrs", 'i', strval, {(char **)&addrs}, NULL,
	  "IP addresses for Linux kernel:\n"
	  "  STRING syntax: rom|kernel"
#ifdef HAVE_INET
	  "|<IP-addrs>"
#endif
							, "STRING"	},
	{ "kernel", 'k', strval, {(char **)&kname}, NULL,
	  "path to Linux kernel image", "FILE"				},
	{ "ramdisk-mode", 'l', strval, {(char **)&rdmodebuf}, NULL,
	  "loading mode for ramdisk:\n"
	  "  STRING syntax: auto|eom|<fixed address>", "STRING"		},
	{ "cmdl-size", 's', intval, {(char **)&cmdl_msize}, NULL,
	  "maximum command line size", "SIZE"				},
	{ "kernel", 0, nonopt, {(char **)&kname}, NULL,
	  "Linux kernel image (if not given as option)", NULL		},
	{ "outfile", 0, nonopt, {(char **)&outname}, NULL,
	  "output file (if not given as option)", NULL			},
	{ NULL, 0, noval, {NULL}, NULL, NULL, NULL			}
};



/*
 * Parameters in each section of database file
 */
static struct paramdef dbparams[] = {
  { "outfile",	 	par_string,	NULL,	{&outname}},
  { "kernel",		par_string,	NULL,	{&kname}},
  { "ramdisk-image", 	par_string,	NULL,	{&rdname}},
  { "ramdisk-mode",	par_string,	NULL,	{&rdmodebuf}},
  { "append",		par_string,	NULL,	{&append}},
  { "root-dir",		par_string,	NULL,	{&nfsdir}},
  { "ip-addrs",		par_string,	NULL,	{&addrs}},
  { "cmdl-size",	par_int,	NULL,	{(char **)&cmdl_msize}},
  { NULL,	 	par_null,	NULL,	{NULL}}
};



/*
 * Write a buffer into the output file and update the load record
 */
static void putrec(recnum, src, size)
int recnum;
char *src;
int size;
{
  unsigned long l;
  size_t isize;
  unsigned char *buf;

  if (cur_rec_num != recnum) {
    fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
	    progname);
    exit(EXIT_LINUX_MISORDER);
  }
  isize = ((size / (SECTSIZE + 1)) + 1) * SECTSIZE;
  buf = (unsigned char *)nbmalloc(isize);
  memcpy(buf, src, size);
  (void)nbwrite(buf, isize, outfile);
  free(buf);
  l = get_long(cur_rec->ilength) + isize;
  assign(cur_rec->ilength.low, htot(low_word(l)));
  assign(cur_rec->ilength.high, htot(high_word(l)));
  l = get_long(cur_rec->mlength) + isize;
  assign(cur_rec->mlength.low, htot(low_word(l)));
  assign(cur_rec->mlength.high, htot(high_word(l)));
}



/*
 * Copy a certain number of bytes from the kernel image file into the
 * boot file
 */
static int copyrec(recnum, remaining, image, name)
int recnum;
int remaining;
int image;
char *name;
{
  unsigned int size, bytes_read;
  int i = 1;

  if (cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
								progname);
	exit(EXIT_LINUX_MISORDER);
  }
  for (bytes_read = 0; remaining > 0 && i > 0;) {
	size = (remaining > SECTSIZE) ? SECTSIZE : remaining;
	if ((i = nbread(copyrec_buf, size, image))) {
		putrec(recnum, (char *)copyrec_buf, i);
		bytes_read += i;
		remaining -= i;
	}
  }
  return(bytes_read);
}



/*
 * Initialize a load record
 */
static void initrec(recnum, segment, flags, vendor_size)
int recnum;
int segment;
int flags;
int vendor_size;
{
  if (++cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
		progname);
	exit(EXIT_LINUX_MISORDER);
  }

  if (cur_rec_num > 0)
	cur_rec = (struct load_record *)((unsigned char *)cur_rec +
					((cur_rec->rlength << 2) & 0x3c) +
					((cur_rec->rlength >> 2) & 0x3c));
  cur_rec->rlength      = (sizeof(struct load_record) -
		           sizeof(union vendor_data)) >> 2;
  cur_rec->rlength     |= ((vendor_size + 3) & 0x3c) << 2;
  cur_rec->rtag1        = recnum + VENDOR_OFF;
  cur_rec->rflags       = flags;
  assign(cur_rec->address.low, htot(low_word((unsigned long) segment << 4)));
  assign(cur_rec->address.high, htot(high_word((unsigned long) segment << 4)));
}



/*
 * Process kernel image file
 */
static void do_kernel()
{
  int flags, setup_size, bigkernel;
  struct setup_header *setup_header;
  unsigned long l;

  /* Process the floppy boot loader code */
  initrec(INITNUM, DEF_INITSEG, 0, 0);
  if (copyrec(INITNUM, INITLSIZE, kimage, kname) != INITLSIZE) {
	fprintf(stderr, "%s: Unexpected end of kernel image file\n", progname);
	exit(EXIT_LINUX_KERNEOF);
  }
  if (getval(*((__u16 *)&(((__u8 *)copyrec_buf)[BOOT_SIG_OFF]))) !=
							htot(BOOT_SIGNATURE)){
	fprintf(stderr, "%s: Could not find magic number in kernel image\n",
		progname);
	exit(EXIT_LINUX_INVKERN);
  }
  assign(cur_rec->mlength.low, htot(low_word(INITMSIZE)));
  assign(cur_rec->mlength.high, htot(high_word(INITMSIZE)));

  /* Determine size of setup code */
  setup_size = (int)(((__u8 *)copyrec_buf)[SETUP_SIZE_OFF]) * SECTSIZE;
  if (setup_size > SETUPLSIZE) {
	fprintf(stderr, "%s: Kernel setup code too large\n", progname);
	exit(EXIT_LINUX_INVSETUP);
  }

  /* Read setup header and check if this is a "big" kernel image */
  initrec(SETUPNUM, DEF_SETUPSEG, 0, 0);
  if (copyrec(SETUPNUM, SECTSIZE, kimage, kname) != SECTSIZE) {
	fprintf(stderr, "%s: Unexpected end of kernel image file\n", progname);
	exit(EXIT_LINUX_KERNEOF);
  }
  setup_header = (struct setup_header *)copyrec_buf;
  bigkernel = !memcmp(setup_header->magic, SETUP_MAGIC, strlen(SETUP_MAGIC)) &&
               (ttoh(getval(setup_header->version)) >= SETUP_VERSION) &&
               (setup_header->loadflags & SETUP_HIGH);

  /* Process the setup code */
  if ((setup_size -= SECTSIZE) > 0 &&
      copyrec(SETUPNUM, setup_size, kimage, kname) != setup_size) {
	fprintf(stderr, "%s: Unexpected end of kernel image file\n", progname);
	exit(EXIT_LINUX_KERNEOF);
  }
  assign(cur_rec->mlength.low, htot(low_word(SETUPMSIZE)));
  assign(cur_rec->mlength.high, htot(high_word(SETUPMSIZE)));

  /* Process the kernel */
  flags = rdimage >= 0 ? 0 : FLAG_EOF;
  if (bigkernel) {
	initrec(KERNELNUM, BIG_SYSSEG, flags, 0);
	while (copyrec(KERNELNUM, SECTSIZE, kimage, kname) == SECTSIZE) ;
	if (get_long(cur_rec->ilength) > BIG_SYSLSIZE) {
		fprintf(stderr, "%s: Kernel image too large\n", progname);
		exit(EXIT_LINUX_KERNSIZE);
	}
	assign(cur_rec->mlength.low, htot(low_word(BIG_SYSMSIZE)));
	assign(cur_rec->mlength.high, htot(high_word(BIG_SYSMSIZE)));
  } else {
	initrec(KERNELNUM, DEF_SYSSEG, flags, 0);
	while (copyrec(KERNELNUM, SECTSIZE, kimage, kname) == SECTSIZE) ;
	if (get_long(cur_rec->ilength) > SYSLSIZE) {
		fprintf(stderr, "%s: Kernel image too large\n", progname);
		exit(EXIT_LINUX_KERNSIZE);
	}
	assign(cur_rec->mlength.low, htot(low_word(SYSMSIZE)));
	assign(cur_rec->mlength.high, htot(high_word(SYSMSIZE)));
  }

  /* Process the ramdisk image */
  if (rdimage >= 0) {
	unsigned long imagestart = get_long(cur_rec->address) +
						get_long(cur_rec->ilength);

	if (imagestart < 0x100000L)
		imagestart = 0x100000L;
	initrec(RAMDISKNUM, 0, (rdmode == RD_EOM ? FLAG_B1 : 0) | FLAG_EOF,
					sizeof(cur_rec->vendor_data.rdflags));
	if (rdmode == RD_AUTO)
		rdlocation = imagestart;
	else if (rdmode == RD_FIXED)
		/* always align to 4kB page boundary */
		rdlocation &= ~0xfff;
	cur_rec->vendor_data.rdflags = (__u8)rdmode;
	while (copyrec(RAMDISKNUM, SECTSIZE, rdimage, rdname) == SECTSIZE) ;
	/* memory length has to be a multiple of 4kB */
	l = (get_long(cur_rec->ilength) + 0xfff) & (unsigned long)~0xfff;
	assign(cur_rec->mlength.low, htot(low_word(l)));
	assign(cur_rec->mlength.high, htot(high_word(l)));
	l = rdmode == RD_EOM ? get_long(cur_rec->mlength) : rdlocation;
	assign(cur_rec->address.low, htot(low_word(l)));
	assign(cur_rec->address.high, htot(high_word(l)));
  }
}



/*
 * Remove duplicate and contradictory entries from the commandline
 */
static void cleanup_cmdline(cmdline)
char *cmdline;
{
  static char *entries[] = { " root=", " nfsroot=", " nfsaddrs=", NULL };
  char   **cur_entry;
  char   *ptr1,*ptr2;

  /*
   * This code assumes, that the first entry in the command line is a word
   * followed by a space character. This word must not contain the letter 'r'.
   * This assumptions holds, because the standard command line adds the word
   * "auto " to the beginning of the command line!
   */
  if (strncmp(cmdline,"auto ",5)) {
	fprintf(stderr, "%s: Internal error; command line mangled\n",progname);
	exit(EXIT_INTERNAL);
  }

  for (cur_entry = entries; *cur_entry; cur_entry++) {
	while ((ptr1 = strstr(cmdline, *cur_entry)) != NULL &&
	       strstr(ptr1+1, *cur_entry) != NULL) {
		ptr2 = strchr(ptr1+1, ' ');
		memmove(ptr1+1, ptr2+1, strlen(ptr2));
	}
  }

again: /* Remove multiple occurrences of "ro" and "rw" */
  for (ptr1 = strchr(cmdline,'r'); ptr1 != NULL; ptr1 = strchr(ptr1+1, 'r'))
	if (ptr1[-1] == ' ' && (ptr1[1] == 'o' || ptr1[1] == 'w') && ptr1[2] == ' ')
		for (ptr2 = strchr(ptr1+3,'r'); ptr2 != NULL; ptr2 = strchr(ptr2+1, 'r'))
			if (ptr2[-1] == ' ' &&
			    (ptr2[1] == 'o' || ptr2[1] == 'w') &&
			    (ptr2[2] == ' ' || ptr2[2] == '\000')) {
				memmove(ptr1, ptr1+3, strlen(ptr1+2));
				goto again;
			}
  return;
}



/*
 * Dump the load record information to stderr
 */
static void dump_header(lh)
struct load_header *lh;
{
  static char *s_tags[] = { /* BOOTLNUM */   "primary boot loader",
			    /* CMDLNUM */    "command line",
			    /* INITNUM */    "floppy boot sector",
			    /* SETUPNUM */   "kernel setup",
			    /* KERNELNUM */  "kernel image",
			    /* RAMDISKNUM */ "ramdisk image"};
  static char *s_flags[]= { "absolute address", "after previous segment",
			    "at end of memory", "before previos segment"};
  static char *s_rd[]   = { /* RD_AUTO */  "RD: auto positioning",
			    /* RD_EOM */   "RD: positioned by Boot-Rom",
			    /* RD_FIXED */ "RD: fixed memory location" };
  struct load_record *lr;
  char *vendstr = NULL;
  int i, num = 0;

  i = (lh->hlength >> 2) & 0x3c;
  vendstr = nbmalloc(i + 2);
  memcpy(vendstr, lh->dummy, i);

  fprintf(stderr,"\n"
	  "Load record information:\n"
	  "  Magic number:     0x%08lX\n"
	  "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	  "  Flags:            0x%08lX\n"
	  "  Location address: %04X:%04X\n"
	  "  Execute address:  %04X:%04X\n"
	  "  Vendor data:      %s\n"
	  "\n",
	  get_long(lh->magic),
	  (lh->hlength << 2) & 0x3c,
	  (lh->hlength >> 2) & 0x3c,
	  (unsigned long)lh->hflags1 +
		((unsigned long)lh->hflags2 << 8) +
		((unsigned long)lh->hflags3 << 16),
	  ttoh(getval(lh->locn.segment)), ttoh(getval(lh->locn.offset)),
	  ttoh(getval(lh->execute.segment)), ttoh(getval(lh->execute.offset)),
	  vendstr);

  i  = ((lh->hlength >> 2) & 0x3c) + ((lh->hlength << 2) & 0x3c);
  lr = (struct load_record *)&(((__u8 *)lh)[i]);

  for (;;) {
  	fprintf(stderr,
	    "Record #%d:\n"
	    "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	    "  Vendor tag:       0x%02X (%s)\n"
	    "  Reserved flags:   0x%02X\n"
	    "  Flags:            0x%02X (%s%s)\n"
	    "  Load address:     0x%08lX%s\n"
	    "  Image length:     0x%08lX bytes\n"
	    "  Memory length:    0x%08lX bytes\n"
	    "  Vendor data:      %s\n"
	    "\n",
	    ++num,
	    (lr->rlength << 2) & 0x3c,
	    (lr->rlength >> 2) & 0x3c,
	    (int)lr->rtag1,
	    lr->rtag1 < 16 || lr->rtag1-16 >= NUM_RECORDS ? "unknown" : s_tags[lr->rtag1-16],
	    (int)lr->rtag2,
	    (int)lr->rflags, s_flags[lr->rflags & 0x03],
	    lr->rflags & FLAG_EOF ? ", last record" : "",
	    get_long(lr->address),
	    get_long(lr->address) >= 0x100000L ? " (high memory)" : "",
	    get_long(lr->ilength),
	    get_long(lr->mlength),
	    lr->rlength & 0xf0 ? lr->rtag1-16 == RAMDISKNUM ?
	    lr->vendor_data.rdflags < 3 ? s_rd[lr->vendor_data.rdflags] :
	    "illegal" : "unknown" : "none");

	if (lr->rflags & FLAG_EOF)
		break;

	i  = ((lr->rlength >> 2) & 0x3c) + ((lr->rlength << 2) & 0x3c);
	lr = (struct load_record *)&(((__u8 *)lr)[i]);
  }
  free(vendstr);
}



/*
 * Read system database
 */
static void getdb(name)
char *name;
{
  struct sectdef sect;
  char *namebuf;
  size_t len;

  /* Read one entry from database file */
  len = strlen(name) + 11;
  namebuf = nbmalloc(len);
  sprintf(namebuf, "%s:mknbi-linux", name);
  sect.name = namebuf;
  sect.params = dbparams;
  sect.startsect = NULL;
  sect.endsect = NULL;
  readdb(&sect);

  /* Check that parameters are correct */
  if (cmdl_msize < CMDLMSIZE) {
	fprintf(stderr, "%s: command line size must be >= %d bytes in section <%s>\n",
						progname, CMDLMSIZE, namebuf);
	exit(EXIT_DB);
  }
  if (kname == NULL) {
	fprintf(stderr, "%s: Need kernel image file name in section <%s>\n",
							progname, namebuf);
	exit(EXIT_DB);
  }
  if (outname == NULL) {
	fprintf(stderr, "%s: Need output file name in section <%s>\n",
							progname, namebuf);
	exit(EXIT_DB);
  }
  free(namebuf);
}



/*
 * Main program
 */
void main(argc, argv)
int argc;
char **argv;
{
  size_t len;
  char *cmdline;
  char *cp, *ip;
  int vendor_size;
  int i;

  /* Initialize option argments */
  copystr(&nfsdir, DFLT_DIR);
  copystr(&rootdev, DFLT_DEV);
  copystr(&kname, DFLT_IMAGE);

  /* Parse options and read configuration file */
  nbsetup(argc, argv, opts, NULL);
  if (batchname != NULL)
	getdb(batchname);
  if (cmdl_msize < CMDLMSIZE) {
	fprintf(stderr, "%s: command line size must be >= %d bytes\n",
						        progname, CMDLMSIZE-1);
	exit(EXIT_USAGE);
  }
  if (kname == NULL) {
	fprintf(stderr, "%s: Need kernel image file name\n", progname);
	exit(EXIT_USAGE);
  }
  if (outname == NULL) {
	fprintf(stderr, "%s: Need output file name\n", progname);
	exit(EXIT_USAGE);
  }

  /* Decode ramdisk loading mode */
  if (rdmodebuf == NULL || !strcmp("auto", rdmodebuf))
	rdmode = RD_AUTO;
  else if (!strcmp("eom", rdmodebuf))
	rdmode = RD_EOM;
  else {
	rdmode = RD_FIXED;
	if ((rdlocation = strtol(rdmodebuf, &cp, 0)) < 0x100000L) {
		fprintf(stderr, "%s: Ramdisk image load location must be above 1M\n",
								progname);
		exit(EXIT_LINUX_RDLOC);
	}
	if (*cp) {
		fprintf(stderr, "%s: invalid ramdisk location\n", progname);
		exit(EXIT_LINUX_RDLOC);
	}
  }

  /* Parse the IP address option */
#ifdef HAVE_INET
  if (addrs != NULL && *addrs && strcmp(addrs, "rom") &&
      strcmp(addrs, "kernel")) {
	struct hostent *hp;
	char *buf, *bp;

	/* Allocate memory for 6 address strings, each with max 15 chars */
	i = 0;
	ip = addrs;
	bp = buf = (char *)nbmalloc((MAX_ADDR_SIZE + 1) * 6);
	while (ip != NULL && *ip) {
		if ((cp = strchr(ip, ':')) != NULL) *cp++ = '\0';
		if (strlen(ip) > 0) {
			if ((hp = gethostbyname(ip)) == NULL) {
				fprintf(stderr, "%s: invalid hostname %s\n",
					progname, ip);
				exit(EXIT_HOSTNAME);
			}
			if (hp->h_length != sizeof(struct in_addr)) {
				fprintf(stderr, "%s: invalid host address "
					"type\n", progname);
				exit(EXIT_HOSTADDR);
			}
			/* Result from ntoa is smaller than MAX_ADDR_SIZE */
			strcpy(bp, inet_ntoa(*(struct in_addr *) hp->h_addr));
			bp += strlen(bp);
		}
		ip = cp;
		if (i < 3) *bp++ = ':';
		if (i >= 3) break;
		i++;
	}
	for (; i < 4; i++)
		*bp++ = ':';
	/* Finally copy host and network card name */
	if (ip != NULL) {
		if ((cp = strchr(ip, ':')) != NULL) *cp++ = '\0';
		if ((i = strlen(ip)) > MAX_ADDR_SIZE)
			i = MAX_ADDR_SIZE;
		strncpy(bp, ip, i);
		bp += i;
		if (cp != NULL) {
			i = strlen(cp);
			if (strncmp(cp, "eth", 3) || i != 4) {
				fprintf(stderr, "%s: Invalid ethernet device "
						"specification\n", progname);
				exit(EXIT_LINUX_INVETH);
			}
			*bp++ = ':';
			strncpy(bp, cp, i);
			bp += i;
		}
	}
	*bp = '\0';
	free(addrs);
	addrs = buf;
  }
#else
  if (addrs != NULL && *addrs && strcmp(addrs, "rom") &&
      strcmp(addrs, "kernel")) {
	fprintf(stderr, "%s: No INET support for -i option\n", progname);
	exit(EXIT_INET);
  }
#endif
  if (addrs == NULL)
	copystr(&addrs, DFLT_ADDRS);

  /* Determine root device for kernel */
  if (!strcmp(nfsdir, "ram")) {
	if (strlen(rdname) == 0) {
		fprintf(stderr, "%s: Boot from ramdisk but no image specified\n",
								progname);
		exit(EXIT_LINUX_NORD);
	}
	if (rootdev != NULL)
		free(rootdev);
	rootdev = (char *)nbmalloc(strlen(nfsdir) + 12);
	sprintf(rootdev, "root=/dev/%s", nfsdir);
	copystr(&nfsdir, DFLT_DIR);
  } else if (!strncmp(nfsdir, "/dev/", 5)) {
	if (rootdev != NULL)
		free(rootdev);
	rootdev = (char *)nbmalloc(strlen(nfsdir) + 7);
	sprintf(rootdev, "root=%s", nfsdir);
	copystr(&nfsdir, DFLT_DIR);
  }

  /* Open the input and output files */
  if ((kimage = open(kname, O_RDONLY | O_BINARY)) < 0) {
	perror(kname);
	exit(EXIT_LINUX_KERNOPEN);
  }
  if ((outfile = creat(outname, 0644)) < 0) {
	perror(outname);
	exit(EXIT_LINUX_IMGCREATE);
  }
  if (rdname != NULL && (rdimage = open(rdname, O_RDONLY | O_BINARY)) < 0) {
	perror(rdname);
	exit(EXIT_LINUX_RDOPEN);
  }
  if (verbose > 0) {
	fprintf(stderr, "Kernel image file name  = %s\n", kname);
	fprintf(stderr, "Output file name        = %s\n", outname);
	if (rdname != NULL)
		fprintf(stderr, "Ramdisk image file name = %s\n", rdname);
  }

  /* Construct command line to pass to the kernel */
  len = strlen(DFLT_CMDL) + strlen(rootdev) + 2;
  if (append != NULL && *append)
	len += strlen(append) + 1;
  if (nfsdir != NULL && *nfsdir)
	len += strlen(NFS_ROOT) + strlen(nfsdir) + 1;
  if (addrs != NULL && *addrs)
	len += strlen(NFS_ADDRS) + strlen(addrs) + 1;
  if (len > CMDLLSIZE) {
	fprintf(stderr, "%s: Command line too long\n", progname);
	exit(EXIT_LINUX_CMDLSIZE);
  }
  cmdline = (char *)nbmalloc(len + 2);

  sprintf(cmdline, "%s %s", DFLT_CMDL, rootdev);
  if (nfsdir != NULL && *nfsdir) {
	cp = cmdline + strlen(cmdline);
	sprintf(cp, " %s%s", NFS_ROOT, nfsdir);
  }
  if (addrs != NULL && *addrs) {
	cp = cmdline + strlen(cmdline);
	sprintf(cp, " %s%s", NFS_ADDRS, addrs);
  }
  if (append != NULL && *append) {
	cp = cmdline + strlen(cmdline);
	sprintf(cp, " %s", append);
  }
  cleanup_cmdline(cmdline);
  if (strlen(cmdline) > cmdl_msize)
	cmdline[cmdl_msize] = '\0';
  if (verbose > 0)
	fprintf(stderr, "Kernel command line     = \"%s\"\n", cmdline);

  /* Initialize the boot header */
  vendor_size = (sizeof(VENDOR_ID) / sizeof(__u32) + 1) * sizeof(__u32);
  memset(&header, 0, sizeof(header));
  assign(header.magic.low,       htot(low_word(HEADER_MAGIC)));
  assign(header.magic.high,      htot(high_word(HEADER_MAGIC)));
  assign(header.locn.segment,    htot(DEF_HEADERSEG));
  assign(header.locn.offset,     htot(0));
  assign(header.execute.segment, htot(DEF_BOOTLSEG));
  assign(header.execute.offset,  htot(0));
  assign(header.bootsig,         htot(BOOT_SIGNATURE));
  header.hlength         = (__u8)(((int)header.dummy - (int)&header)
                           / sizeof(__u32)) & 0x0f;
  header.hlength        |= (__u8)((vendor_size/sizeof(__u32)) << 4) & 0xf0;
  memcpy(header.dummy, VENDOR_ID, sizeof(VENDOR_ID));
  (void)nbwrite((unsigned char *)&header, sizeof(header), outfile);

  /* Initialize pointer to first load record */
  cur_rec = (struct load_record *)&(header.dummy[vendor_size]);

  /* Process the boot loader record */
  if (first_data_size > BOOTLLSIZE) {
	fprintf(stderr, "%s: Boot loader too large\n", progname);
	exit(EXIT_LINUX_BOOTLSIZE);
  }
  initrec(BOOTLNUM, DEF_BOOTLSEG, 0, 0);
  putrec(BOOTLNUM, (char *)first_data, first_data_size);
  assign(cur_rec->mlength.low, htot(low_word(BOOTLMSIZE)));
  assign(cur_rec->mlength.high, htot(high_word(BOOTLMSIZE)));

  /* Process the command line */
  initrec(CMDLNUM, DEF_CMDLSEG, 0, 0);
  putrec(CMDLNUM, cmdline, strlen(cmdline) + 1);
  assign(cur_rec->mlength.low, htot(low_word(cmdl_msize)));
  assign(cur_rec->mlength.high, htot(high_word(cmdl_msize)));

  /* Process the kernel image */
  do_kernel();

  /* After writing out all these stuff, finally update the boot header */
  if (lseek(outfile, 0, 0) != 0) {
	perror(outname);
	exit(EXIT_SEEK);
  }
  (void)nbwrite((unsigned char *)&header, sizeof(header), outfile);

  /* If user asked for detailed output, parse the header and output all of */
  /* the load record information */
  if (verbose > 1)
	dump_header(&header);

  exit(EXIT_SUCCESS);
}
