/*
 * tc_filter.c		"tc filter".
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <linux/if_ether.h>

#include "utils.h"
#include "tc_util.h"

static void usage(void)
{
	fprintf(stderr, "Usage: tc filter [ add | del | change | get ] dev STRING\n");
	fprintf(stderr, "       [ prio PRIO ] [ protocol PROTO ]\n");
	fprintf(stderr, "       [ root | classid CLASSID ]\n");
	fprintf(stderr, "       [ [ FILTER_TYPE ] [ help | OPTIONS ] ]\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "       tc filter show [ dev STRING ] [ root | parent CLASSID ]\n");
	fprintf(stderr, "Where:\n");
	fprintf(stderr, "FILTER_TYPE := { rsvp | u32 | fw | route | etc. }\n");
	fprintf(stderr, "OPTIONS := ... try tc filter add <desired FILTER_KIND> help\n");
	exit(-1);
}

int tc_filter_modify(int cmd, unsigned flags, int argc, char **argv)
{
	struct rtnl_handle rth;
	struct {
		struct nlmsghdr 	n;
		struct tcmsg 		t;
		char   			buf[4096];
	} req;
	struct filter_util *q = NULL;
	__u32 prio = 0;
	__u32 protocol = 0;
	char *fhandle = NULL;
	char  d[16];
	char  k[16];

	memset(&req, 0, sizeof(req));
	d[0] = 0;
	k[0] = 0;

	memset(&req, 0, sizeof(req));
	req.n.nlmsg_len = NLMSG_LENGTH(sizeof(struct tcmsg));
	req.n.nlmsg_flags = NLM_F_REQUEST|flags;
	req.n.nlmsg_type = cmd;
	req.t.tcm_family = AF_UNSPEC;

	while (argc > 0) {
		if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			if (d[0]) {
				fprintf(stderr, "Duplicate device specification.\n");
				usage();
			}
			strcpy(d, *argv);
		} else if (strcmp(*argv, "root") == 0) {
			if (req.t.tcm_parent) {
				fprintf(stderr, "Duplicate parent ID\n");
				usage();
			}
			req.t.tcm_parent = TC_H_ROOT;
		} else if (strcmp(*argv, "parent") == 0) {
			__u32 handle;
			if (req.t.tcm_parent) {
				fprintf(stderr, "Duplicate parent ID\n");
				usage();
			}
			NEXT_ARG();
			if (get_tc_classid(&handle, *argv)) {
				fprintf(stderr, "Invalid parent ID\n");
				usage();
			}
			req.t.tcm_parent = handle;
		} else if (strcmp(*argv, "handle") == 0) {
			if (fhandle) {
				fprintf(stderr, "Duplicate filter handle\n");
				usage();
			}
			NEXT_ARG();
			fhandle = *argv;
		} else if (matches(*argv, "priority") == 0) {
			if (prio) {
				fprintf(stderr, "Duplicate prio.\n");
				usage();
			}
			NEXT_ARG();
			if (scan_number(*argv, &prio)) {
				fprintf(stderr, "Invalid priority \"%s\".\n", *argv);
				usage();
			}
		} else if (matches(*argv, "protocol") == 0) {
			if (protocol) {
				fprintf(stderr, "Duplicate protocol.\n");
				usage();
			}
			NEXT_ARG();
			if (strcmp(*argv, "all") == 0)
				protocol = htons(ETH_P_ALL);
			else if (strcmp(*argv, "ip") == 0 ||
				 strcmp(*argv, "ipv4") == 0)
				protocol = htons(ETH_P_IP);
			else if (strcmp(*argv, "ipv6") == 0)
				protocol = htons(ETH_P_IPV6);
			else {
				if (scan_number(*argv, &protocol)) {
					fprintf(stderr, "Invalid protocol \"%s\".\n", *argv);
					usage();
				}
			}
		} else {
			strcpy(k, *argv);

			q = get_filter_kind(k);
			argc--; argv++;
			break;
		}

		argc--; argv++;
	}

	req.t.tcm_info = TC_H_MAKE(prio<<16, protocol);

	if (k[0])
		addattr_l(&req.n, sizeof(req), TCA_KIND, k, strlen(k)+1);

	if (q) {
		if (q->parse_fopt(q, fhandle, argc, argv, &req.n))
			exit(1);
	} else {
		if (argc) {
			fprintf(stderr, "Garbage instead of arguments \"%s ...\"\n", *argv);
			usage();
		}
	}

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	if (d[0])  {
		ll_init_map(&rth);

		if ((req.t.tcm_ifindex = ll_name_to_index(d)) == 0) {
			fprintf(stderr, "Unknown device \"%s\"\n", d);
			usage();
		}
	}

	if (rtnl_talk(&rth, &req.n, 0, 0, NULL, NULL, NULL) < 0)
		exit(2);

	return 0;
}

static __u32 filter_parent;
static int filter_ifindex;
static __u32 filter_prio;
static __u32 filter_protocol;

int print_filter(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	FILE *fp = (FILE*)arg;
	struct tcmsg *t = NLMSG_DATA(n);
	int len = n->nlmsg_len;
	struct rtattr * tb[TCA_MAX+1];
	struct filter_util *q;
	char abuf[256];

	if (n->nlmsg_type != RTM_NEWTFILTER && n->nlmsg_type != RTM_DELTFILTER) {
		fprintf(stderr, "Not a filter\n");
		return 0;
	}
	len -= NLMSG_LENGTH(sizeof(*t));
	if (len < 0) {
		fprintf(stderr, "Wrong len %d\n", len);
		return -1;
	}

	memset(tb, 0, sizeof(tb));
	parse_rtattr(tb, TCA_MAX, TCA_RTA(t), len);

	if (tb[TCA_KIND] == NULL) {
		fprintf(stderr, "NULL kind\n");
		return -1;
	}

	if (n->nlmsg_type == RTM_DELTFILTER)
		fprintf(fp, "deleted ");

	fprintf(fp, "filter ");
	if (!filter_ifindex || filter_ifindex != t->tcm_ifindex)
		fprintf(fp, "dev %s ", ll_index_to_name(t->tcm_ifindex));

	if (!filter_parent || filter_parent != t->tcm_parent) {
		if (t->tcm_parent == TC_H_ROOT)
			fprintf(fp, "root ");
		else {
			print_tc_classid(abuf, sizeof(abuf), t->tcm_parent);
			fprintf(fp, "parent %s ", abuf);
		}
	}
	if (t->tcm_info) {
		__u32 protocol = TC_H_MIN(t->tcm_info);
		__u32 prio = TC_H_MAJ(t->tcm_info)>>16;
		if (!filter_protocol || filter_protocol != protocol) {
			if (protocol) {
				fprintf(fp, "protocol ");
				if (protocol == htons(ETH_P_ALL))
					fprintf(fp, "all ");
				else if (protocol == htons(ETH_P_IP))
					fprintf(fp, "ip ");
				else if (protocol == htons(ETH_P_IPV6))
					fprintf(fp, "ipv6 ");
				else 
					fprintf(fp, "%04x ", protocol);
			}
		}
		if (!filter_prio || filter_prio != prio) {
			if (prio)
				fprintf(fp, "prio %u ", prio);
		}
	}
	fprintf(fp, "%s ", (char*)RTA_DATA(tb[TCA_KIND]));
	if ((q = get_filter_kind(RTA_DATA(tb[TCA_KIND]))) != NULL)
		q->print_fopt(q, fp, tb[TCA_OPTIONS], t->tcm_handle);
	else
		fprintf(fp, "[UNKNOWN]");
	fprintf(fp, "\n");
	fflush(fp);
	return 0;
}


int tc_filter_list(int argc, char **argv)
{
	struct tcmsg t;
	struct rtnl_handle rth;
	char d[16];
	__u32 prio = 0;
	__u32 protocol = 0;
	char *fhandle = NULL;

	memset(&t, 0, sizeof(t));
	t.tcm_family = AF_UNSPEC;
	d[0] = 0;

	while (argc > 0) {
		if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			if (d[0]) {
				fprintf(stderr, "Duplicate device specification.\n");
				usage();
			}
			strcpy(d, *argv);
		} else if (strcmp(*argv, "root") == 0) {
			if (t.tcm_parent) {
				fprintf(stderr, "Duplicate parent ID\n");
				usage();
			}
			filter_parent = t.tcm_parent = TC_H_ROOT;
		} else if (strcmp(*argv, "parent") == 0) {
			__u32 handle;
			if (t.tcm_parent) {
				fprintf(stderr, "Duplicate parent ID\n");
				usage();
			}
			NEXT_ARG();
			if (get_tc_classid(&handle, *argv)) {
				fprintf(stderr, "Invalid parent ID\n");
				usage();
			}
			filter_parent = t.tcm_parent = handle;
		} else if (strcmp(*argv, "handle") == 0) {
			if (fhandle) {
				fprintf(stderr, "Duplicate filter handle\n");
				usage();
			}
			NEXT_ARG();
			fhandle = *argv;
		} else if (matches(*argv, "priority") == 0) {
			if (prio) {
				fprintf(stderr, "Duplicate prio.\n");
				usage();
			}
			NEXT_ARG();
			if (scan_number(*argv, &prio)) {
				fprintf(stderr, "Invalid priority \"%s\".\n", *argv);
				usage();
			}
			filter_prio = prio;
		} else if (matches(*argv, "protocol") == 0) {
			if (protocol) {
				fprintf(stderr, "Duplicate protocol.\n");
				usage();
			}
			NEXT_ARG();
			if (strcmp(*argv, "all") == 0)
				protocol = htons(ETH_P_ALL);
			else if (strcmp(*argv, "ip") == 0 ||
				 strcmp(*argv, "ipv4") == 0)
				protocol = htons(ETH_P_IP);
			else if (strcmp(*argv, "ipv6") == 0)
				protocol = htons(ETH_P_IPV6);
			else {
				if (scan_number(*argv, &protocol)) {
					fprintf(stderr, "Invalid protocol \"%s\".\n", *argv);
					usage();
				}
			}
			filter_protocol = protocol;
		} else {
			fprintf(stderr, "What is \"%s\"?\n", *argv);
			usage();
		}

		argc--; argv++;
	}

	t.tcm_info = TC_H_MAKE(prio<<16, protocol);

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	ll_init_map(&rth);

	if (d[0]) {
		if ((t.tcm_ifindex = ll_name_to_index(d)) == 0) {
			fprintf(stderr, "Cannot find device \"%s\"\n", d);
			exit(1);
		}
		filter_ifindex = t.tcm_ifindex;
	}

	if (rtnl_dump_request(&rth, RTM_GETTFILTER, &t, sizeof(t)) < 0) {
		perror("cannot send dump request");
		exit(1);
	}

	if (rtnl_dump_filter(&rth, print_filter, stdout, NULL, NULL) < 0) {
		fprintf(stderr, "dump terminated\n");
		exit(1);
	}

	exit(0);
}

int do_filter(int argc, char **argv)
{
	if (argc < 1)
		return tc_filter_list(0, NULL);
	if (matches(*argv, "add") == 0)
		return tc_filter_modify(RTM_NEWTFILTER, NLM_F_EXCL|NLM_F_CREATE, argc-1, argv+1);
	if (matches(*argv, "change") == 0)
		return tc_filter_modify(RTM_NEWTFILTER, 0, argc-1, argv+1);
	if (matches(*argv, "replace") == 0)
		return tc_filter_modify(RTM_NEWTFILTER, NLM_F_CREATE, argc-1, argv+1);
	if (matches(*argv, "delete") == 0)
		return tc_filter_modify(RTM_DELTFILTER, 0,  argc-1, argv+1);
#if 0
	if (matches(*argv, "get") == 0)
		return tc_filter_get(RTM_GETTFILTER, 0,  argc-1, argv+1);
#endif
	if (matches(*argv, "list") == 0 || matches(*argv, "show") == 0
	    || matches(*argv, "lst") == 0)
		return tc_filter_list(argc-1, argv+1);
	usage();
}
