/*
 * iproute.c		"ip route".
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <sys/time.h>
#include <netinet/ip.h>
#include <linux/in_route.h>
#include <resolv.h>

#include "utils.h"

static void usage(void)
{
	fprintf(stderr, "Usage: ip route list SELECTOR\n");
	fprintf(stderr, "       ip route { change | del | add | append | replace | monitor } ROUTE\n");
	fprintf(stderr, "SELECTOR := [ root PREFIX ] [ match PREFIX ] [ exact PREFIX ]\n");
	fprintf(stderr, "            [ table TABLE_ID ] [ proto RTPROTO ]\n");
	fprintf(stderr, "            [ type TYPE ] [ scope SCOPE ]\n");
	fprintf(stderr, "ROUTE := NODE_SPEC [ INFO_SPEC ]\n");
	fprintf(stderr, "NODE_SPEC := [ TYPE ] PREFIX [ tos TOS ]\n");
	fprintf(stderr, "             [ table TABLE_ID ] [ proto RTPROTO ]\n");
	fprintf(stderr, "             [ type TYPE ] [ scope SCOPE ]\n");
	fprintf(stderr, "INFO_SPEC := NH OPTIONS FLAGS [ nexthop NH ]...\n");
	fprintf(stderr, "NH := [ via ADDRESS ] [ dev STRING ] [ weight NUMBER ] NHFLAGS\n");
	fprintf(stderr, "OPTIONS := FLAGS [ mtu NUMBER ] [ rtt NUMBER ] [ window NUMBER ]\n");
	fprintf(stderr, "TYPE := [ unicast | local | broadcast | multicast | throw |\n");
	fprintf(stderr, "          unreachable | prohibit | blackhole | nat ]\n");
	fprintf(stderr, "TABLE_ID := [ local | main | default | all | NUMBER ]\n");
	fprintf(stderr, "SCOPE := [ host | link | global | NUMBER ]\n");
	fprintf(stderr, "NHFLAGS := [ onlink | pervasive ]\n");
	fprintf(stderr, "RTPROTO := [ kernel | boot | static | NUMBER ]\n");
	exit(-1);
}


struct rtfilter
{
	int tb;
	int protocol;
	int scope;
	int type;
	inet_prefix rdst;
	inet_prefix mdst;
} filter;

int print_route(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	FILE *fp = (FILE*)arg;
	struct rtmsg *r = NLMSG_DATA(n);
	int len = n->nlmsg_len;
	struct rtattr * tb[RTA_MAX+1];
	char abuf[256];
	inet_prefix dst;
	int host_len = -1;
	SPRINT_BUF(b1);

	if (n->nlmsg_type != RTM_NEWROUTE && n->nlmsg_type != RTM_DELROUTE) {
		fprintf(stderr, "Not a route: %08x %08x %08x\n",
			n->nlmsg_len, n->nlmsg_type, n->nlmsg_flags);
		return 0;
	}
	len -= NLMSG_LENGTH(sizeof(*r));
	if (len < 0) {
		fprintf(stderr, "Wrong len %d\n", len);
		return -1;
	}

	if (filter.tb > 0 && filter.tb != r->rtm_table)
		return 0;
	if (filter.protocol && filter.protocol != r->rtm_protocol)
		return 0;
	if (filter.scope && filter.scope != r->rtm_scope)
		return 0;
	if (filter.type && filter.type != r->rtm_type)
		return 0;
	if (filter.rdst.family &&
	    (r->rtm_family != filter.rdst.family || filter.rdst.bitlen > r->rtm_dst_len))
		return 0;
	if (filter.mdst.family &&
	    (r->rtm_family != filter.mdst.family ||
	     (filter.mdst.bitlen >= 0 && filter.mdst.bitlen < r->rtm_dst_len)))
		return 0;

	memset(tb, 0, sizeof(tb));
	parse_rtattr(tb, RTA_MAX, RTM_RTA(r), len);

	memset(&dst, 0, sizeof(dst));
	dst.family = r->rtm_family;
	if (tb[RTA_DST])
		memcpy(&dst.data, RTA_DATA(tb[RTA_DST]), (r->rtm_dst_len+7)/8);

	if (filter.rdst.family && inet_addr_match(&dst, &filter.rdst, filter.rdst.bitlen))
		return 0;
	if (filter.mdst.family && filter.mdst.bitlen >= 0 &&
	    inet_addr_match(&dst, &filter.mdst, r->rtm_dst_len))
		return 0;

	if (n->nlmsg_type == RTM_DELROUTE)
		fprintf(fp, "deleted ");
	if (r->rtm_type != RTN_UNICAST && !filter.type)
		fprintf(fp, "%s ", rtntype_name(r->rtm_type));

	if (r->rtm_family == AF_INET6)
		host_len = 128;
	else if (r->rtm_family == AF_INET)
		host_len = 32;

	if (tb[RTA_DST]) {
		if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_DST]), abuf, sizeof(abuf))) {
			if (r->rtm_dst_len != host_len)
				fprintf(fp, "%s/%d ", abuf, r->rtm_dst_len);
			else
				fprintf(fp, "%s ", abuf);
		} else
			fprintf(fp, "0/%d ", r->rtm_dst_len);
	} else if (r->rtm_dst_len) {
		fprintf(fp, "0/%d ", r->rtm_dst_len);
	} else {
		fprintf(fp, "default ");
	}
	if (tb[RTA_SRC]) {
		if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_SRC]),
			      abuf, sizeof(abuf))) {
			if (r->rtm_src_len != host_len)
				fprintf(fp, "from %s/%d ", abuf, r->rtm_src_len);
			else
				fprintf(fp, "from %s ", abuf);
		} else
			fprintf(fp, "from 0/%d ", r->rtm_src_len);
	} else if (r->rtm_src_len) {
		fprintf(fp, "from 0/%d ", r->rtm_dst_len);
	}
	if (r->rtm_tos)
		fprintf(fp, "tos %02x ", r->rtm_tos);
	if (tb[RTA_GATEWAY]) {
		if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_GATEWAY]),
			      abuf, sizeof(abuf)))
			fprintf(fp, "via %s ", abuf);
		else
			fprintf(fp, "via ? ");
	}
	if (tb[RTA_OIF])
		fprintf(fp, "dev %s ", ll_index_to_name(*(int*)RTA_DATA(tb[RTA_OIF])));

	if (r->rtm_table != RT_TABLE_MAIN && !filter.tb && !(r->rtm_flags&RTM_F_CLONED))
		fprintf(fp, " table %s ", rttable_name(r->rtm_table));
	if (r->rtm_protocol != RTPROT_BOOT && !filter.protocol && !(r->rtm_flags&RTM_F_CLONED))
		fprintf(fp, " proto %s ", rtprot_name(r->rtm_protocol));
	if (r->rtm_scope != RT_SCOPE_UNIVERSE && !filter.scope && !(r->rtm_flags&RTM_F_CLONED))
		fprintf(fp, " scope %s ", rtscope_name(r->rtm_scope));
	if (tb[RTA_PREFSRC]) {
		if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_PREFSRC]),
			      abuf, sizeof(abuf)))
			fprintf(fp, " src %s ", abuf);
		else
			fprintf(fp, " src ? ");
	}
	if (tb[RTA_FLOW])
		fprintf(fp, "flowid %s ", sprint_tc_classid(*(__u32*)RTA_DATA(tb[RTA_FLOW]), b1));
	if (r->rtm_flags & RTNH_F_DEAD)
		fprintf(fp, "dead ");
	if (r->rtm_flags & RTNH_F_ONLINK)
		fprintf(fp, "onlink ");
	if (r->rtm_flags & RTNH_F_PERVASIVE)
		fprintf(fp, "pervasive ");
	if ((r->rtm_flags & RTM_F_CLONED) && r->rtm_family == AF_INET) {
		__u32 flags = r->rtm_flags&~0xFFFF;
		int first = 1;

		fprintf(fp, "\n    cache ");

#define PRTFL(fl,flname) if (flags&RTCF_##fl) { \
  flags &= ~RTCF_##fl; \
  fprintf(fp, "%s" flname "%s", first ? "<" : "", flags ? "," : "> "); \
  first = 0; }
		PRTFL(LOCAL, "local");
		PRTFL(REJECT, "reject");
		PRTFL(MULTICAST, "mc");
		PRTFL(BROADCAST, "brd");
		PRTFL(DNAT, "dst nat");
		PRTFL(SNAT, "src nat");
		PRTFL(MASQ, "masq");
		PRTFL(DIRECTDST, "dst direct");
		PRTFL(DIRECTSRC, "src direct");
		PRTFL(REDIRECTED, "redirected");
		PRTFL(DOREDIRECT, "redirect");
		PRTFL(FAST, "fastroute");
		PRTFL(NOTIFY, "notify");
		if (flags)
			fprintf(fp, "%s%x> ", first ? "<" : "", flags);
		if (tb[RTA_CACHEINFO]) {
			struct rta_cacheinfo *ci = RTA_DATA(tb[RTA_CACHEINFO]);
			if (show_stats) {
				if (ci->rta_clntref)
					fprintf(fp, " users %d", ci->rta_clntref);
				if (ci->rta_used != 0)
					fprintf(fp, " used %d", ci->rta_used);
				if (ci->rta_lastuse != 0)
					fprintf(fp, " age %dsec", ci->rta_lastuse/HZ);
				if (ci->rta_expires != 0)
					fprintf(fp, " expires %dsec", ci->rta_expires/HZ);
			}
			if (ci->rta_error != 0)
				fprintf(fp, " error %d", ci->rta_error);
		}
	} else if (r->rtm_family == AF_INET6) {
		struct rta_cacheinfo *ci = NULL;
		if (tb[RTA_CACHEINFO])
			ci = RTA_DATA(tb[RTA_CACHEINFO]);
		if ((r->rtm_flags & RTM_F_CLONED) || (ci && ci->rta_expires)) {
			if (r->rtm_flags & RTM_F_CLONED)
				fprintf(fp, "\n    cache ");
			if (ci->rta_expires)
				fprintf(fp, " expires %dsec", ci->rta_expires/HZ);
			if (ci->rta_clntref)
				fprintf(fp, " users %d", ci->rta_clntref);
			if (ci->rta_used != 0)
				fprintf(fp, " used %d", ci->rta_used);
			if (ci->rta_lastuse != 0)
				fprintf(fp, " age %dsec", ci->rta_lastuse/HZ);
		} else if (ci) {
			if (ci->rta_error != 0)
				fprintf(fp, " error %d", ci->rta_error);
		}
	}
	if (tb[RTA_METRICS]) {
		int i;
		unsigned mxlock = 0;
		struct rtattr *mxrta[RTAX_MAX+1];

		memset(mxrta, 0, sizeof(mxrta));

		parse_rtattr(mxrta, RTAX_MAX, RTA_DATA(tb[RTA_METRICS]),
			    RTA_PAYLOAD(tb[RTA_METRICS]));
		if (mxrta[RTAX_LOCK])
			mxlock = *(unsigned*)RTA_DATA(mxrta[RTAX_LOCK]);

		for (i=2; i<=RTAX_MAX; i++) {
			if (mxrta[i] == NULL)
				continue;
			switch (i) {
			case RTAX_MTU:
				fprintf(fp, " mtu");
				break;
			case RTAX_WINDOW:
				fprintf(fp, " window");
				break;
			case RTAX_RTT:
				fprintf(fp, " rtt");
				break;
			case RTAX_CWND:
				fprintf(fp, " cwnd");
				break;
			case RTAX_SSTHRESH:
				fprintf(fp, " ssthresh");
				break;
			case RTAX_HOPS:
				fprintf(fp, " hops");
				break;
			default:	
				fprintf(fp, " metric%d", i);
				break;
			}
			if (mxlock & (1<<i))
				fprintf(fp, " lock");
			fprintf(fp, " %u", *(unsigned*)RTA_DATA(mxrta[i]));
		}
	}
	if (tb[RTA_IIF]) {
		if (r->rtm_flags&RTM_F_CLONED && r->rtm_type == RTN_MULTICAST)
			fprintf(fp, "\n\tIif: %s\n", ll_index_to_name(*(int*)RTA_DATA(tb[RTA_IIF])));
		else
			fprintf(fp, " iif %s\n", ll_index_to_name(*(int*)RTA_DATA(tb[RTA_IIF])));
	} else
		fprintf(fp, "\n");
	if (tb[RTA_MULTIPATH]) {
		struct rtnexthop *nh = RTA_DATA(tb[RTA_MULTIPATH]);
		len = RTA_PAYLOAD(tb[RTA_MULTIPATH]);

		for (;;) {
			if (len < sizeof(*nh))
				break;
			if (nh->rtnh_len > len)
				break;
			if (r->rtm_flags&RTM_F_CLONED && r->rtm_type == RTN_MULTICAST)
				fprintf(fp, "\tOif: ");
			else
				fprintf(fp, "\tnexthop via ");
			if (nh->rtnh_len > sizeof(*nh)) {
				memset(tb, 0, sizeof(tb));
				parse_rtattr(tb, RTA_MAX, RTNH_DATA(nh), nh->rtnh_len - sizeof(*nh));
				if (tb[RTA_GATEWAY]) {
					if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_GATEWAY]),
						      abuf, sizeof(abuf)))
						fprintf(fp, "%s ", abuf);
					else
						fprintf(fp, "? ");
				}
			}
			if (r->rtm_flags&RTM_F_CLONED && r->rtm_type == RTN_MULTICAST) {
				fprintf(fp, "%s ", ll_index_to_name(nh->rtnh_ifindex));
				fprintf(fp, "ttl %d ", nh->rtnh_hops);
			} else {
				fprintf(fp, "dev %s ", ll_index_to_name(nh->rtnh_ifindex));
				fprintf(fp, "weight %d ", nh->rtnh_hops+1);
			}
			if (nh->rtnh_flags & RTNH_F_DEAD)
				fprintf(fp, "dead ");
			if (nh->rtnh_flags & RTNH_F_ONLINK)
				fprintf(fp, "onlink ");
			if (nh->rtnh_flags & RTNH_F_PERVASIVE)
				fprintf(fp, "pervasive ");
			len -= NLMSG_ALIGN(nh->rtnh_len);
			nh = RTNH_NEXT(nh);
			fprintf(fp, "\n");
		}
	}
	fflush(fp);
	return 0;
}


int parse_one_nh(struct rtattr *rta, struct rtnexthop *rtnh, int *argcp, char ***argvp)
{
	int argc = *argcp;
	char **argv = *argvp;

	while (++argv, --argc > 0) {
		if (strcmp(*argv, "via") == 0) {
			NEXT_ARG();
			rta_addattr32(rta, 4096, RTA_GATEWAY, get_addr32(*argv));
			rtnh->rtnh_len += sizeof(struct rtattr) + 4;
		} else if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			rtnh->rtnh_ifindex = ll_name_to_index(*argv);
		} else if (strcmp(*argv, "weight") == 0) {
			unsigned w;
			NEXT_ARG();
			if (scan_number(*argv, &w) || w == 0 || w > 256)
				invarg("weight is invalid\n");
			rtnh->rtnh_hops = w - 1;
		} else if (strcmp(*argv, "onlink") == 0) {
			rtnh->rtnh_flags |= RTNH_F_ONLINK;
		} else
			break;
	}
	*argcp = argc;
	*argvp = argv;
	return 0;
}

int parse_nexthops(struct nlmsghdr *n, struct rtmsg *r, int argc, char **argv)
{
	char buf[1024];
	struct rtattr *rta = (void*)buf;
	struct rtnexthop *rtnh;

	rta->rta_type = RTA_MULTIPATH;
	rta->rta_len = RTA_LENGTH(0);
	rtnh = RTA_DATA(rta);

	while (argc > 0) {
		if (strcmp(*argv, "nexthop") != 0)
			usage();
		if (argc <= 1)
			usage();
		rtnh->rtnh_len = sizeof(*rtnh);
		rta->rta_len += rtnh->rtnh_len;
		parse_one_nh(rta, rtnh, &argc, &argv);
		rtnh = RTNH_NEXT(rtnh);
	}

	if (rta->rta_len > RTA_LENGTH(0))
		addattr_l(n, 1024, RTA_MULTIPATH, RTA_DATA(rta), RTA_PAYLOAD(rta));
	return 0;
}

int iproute_modify(int cmd, unsigned flags, int argc, char **argv)
{
	struct rtnl_handle rth;
	struct {
		struct nlmsghdr 	n;
		struct rtmsg 		r;
		char   			buf[1024];
	} req;
	char  mxbuf[256];
	struct rtattr * mxrta = (void*)mxbuf;
	unsigned mxlock = 0;
	char  d[16];
	int gw_ok = 0;
	int dst_ok = 0;
	int nhs_ok = 0;
	int scope_ok = 0;
	int table_ok = 0;
	int proto_ok = 0;
	int type_ok = 0;

	memset(&req, 0, sizeof(req));
	d[0] = 0;

	req.n.nlmsg_len = NLMSG_LENGTH(sizeof(struct rtmsg));
	req.n.nlmsg_flags = NLM_F_REQUEST|flags;
	req.n.nlmsg_type = cmd;
	req.r.rtm_family = preferred_family;
	req.r.rtm_table = RT_TABLE_MAIN;

	if (cmd != RTM_DELROUTE) {
		req.r.rtm_protocol = RTPROT_BOOT;
		req.r.rtm_scope = RT_SCOPE_UNIVERSE;
		req.r.rtm_type = RTN_UNICAST;
	}

	mxrta->rta_type = RTA_METRICS;
	mxrta->rta_len = RTA_LENGTH(0);

	while (argc > 0) {
		if (strcmp(*argv, "src") == 0) {
			inet_prefix addr;
			NEXT_ARG();
			get_addr(&addr, *argv, req.r.rtm_family);
			if (req.r.rtm_family == AF_UNSPEC)
				req.r.rtm_family = addr.family;
			addattr_l(&req.n, sizeof(req), RTA_PREFSRC, &addr.data, addr.bytelen);
		} else if (strcmp(*argv, "via") == 0) {
			inet_prefix addr;
			gw_ok = 1;
			NEXT_ARG();
			get_addr(&addr, *argv, req.r.rtm_family);
			if (req.r.rtm_family == AF_UNSPEC)
				req.r.rtm_family = addr.family;
			addattr_l(&req.n, sizeof(req), RTA_GATEWAY, &addr.data, addr.bytelen);
		} else if (strcmp(*argv, "from") == 0) {
			inet_prefix addr;
			NEXT_ARG();
			get_prefix(&addr, *argv, req.r.rtm_family);
			if (req.r.rtm_family == AF_UNSPEC)
				req.r.rtm_family = addr.family;
			addattr_l(&req.n, sizeof(req), RTA_SRC, &addr.data, addr.bytelen);
			req.r.rtm_src_len = addr.bitlen;
		} else if (strcmp(*argv, "tos") == 0) {
			__u8 tos;
			NEXT_ARG();
			if (strcmp(*argv, "mincost") == 0)
				tos = IPTOS_MINCOST;
			else if (strcmp(*argv, "reliability") == 0)
				tos = IPTOS_RELIABILITY;
			else if (strcmp(*argv, "throughput") == 0)
				tos = IPTOS_THROUGHPUT;
			else if (strcmp(*argv, "lowdelay") == 0)
				tos = IPTOS_LOWDELAY;
			else if (get_u8(&tos, *argv, 16))
				invarg("tos value is invalid\n");
			req.r.rtm_tos = tos;
		} else if (strcmp(*argv, "scope") == 0) {
			NEXT_ARG();
			if (strcmp(*argv, "host") == 0)
				req.r.rtm_scope = RT_SCOPE_HOST;
			else if (strcmp(*argv, "link") == 0)
				req.r.rtm_scope = RT_SCOPE_LINK;
			else if (strcmp(*argv, "global") == 0)
				req.r.rtm_scope = RT_SCOPE_UNIVERSE;
			else if (strcmp(*argv, "universe") == 0)
				req.r.rtm_scope = RT_SCOPE_UNIVERSE;
			else {
				unsigned scope;
				if (scan_number(*argv, &scope) || scope > 255)
					invarg("scope value is invalid\n");
				req.r.rtm_scope = scope;
			}
			scope_ok = 1;
		} else if (strcmp(*argv, "mtu") == 0) {
			unsigned mtu;
			NEXT_ARG();
			if (strcmp(*argv, "lock") == 0) {
				mxlock |= (1<<RTAX_MTU);
				NEXT_ARG();
			}
			if (scan_number(*argv, &mtu))
				invarg("mtu value is invalid\n");
			rta_addattr32(mxrta, sizeof(mxbuf), RTAX_MTU, mtu);
		} else if (strcmp(*argv, "rtt") == 0) {
			unsigned rtt;
			NEXT_ARG();
			if (strcmp(*argv, "lock") == 0) {
				mxlock |= (1<<RTAX_RTT);
				NEXT_ARG();
			}
			if (scan_number(*argv, &rtt))
				invarg("rtt value is invalid\n");
			rta_addattr32(mxrta, sizeof(mxbuf), RTAX_RTT, rtt);
		} else if (matches(*argv, "window") == 0) {
			unsigned win;
			NEXT_ARG();
			if (strcmp(*argv, "lock") == 0) {
				mxlock |= (1<<RTAX_WINDOW);
				NEXT_ARG();
			}
			if (scan_number(*argv, &win))
				invarg("window clamp value is invalid\n");
			rta_addattr32(mxrta, sizeof(mxbuf), RTAX_WINDOW, win);
		} else if (matches(*argv, "classid") == 0 ||
			   matches(*argv, "flowid") == 0) {
			__u32 flowid;
			NEXT_ARG();
			if (get_tc_classid(&flowid, *argv))
				invarg("invalid flowid\n");
			addattr32(&req.n, sizeof(req), RTA_FLOW, flowid);
		} else if (strcmp(*argv, "onlink") == 0) {
			req.r.rtm_flags |= RTNH_F_ONLINK;
		} else if (strcmp(*argv, "nexthop") == 0) {
			nhs_ok = 1;
			break;
		} else if (strcmp(*argv, "proto") == 0) {
			NEXT_ARG();
			if (strcmp(*argv, "static") == 0)
				req.r.rtm_protocol = RTPROT_STATIC;
			else if (strcmp(*argv, "boot") == 0)
				req.r.rtm_protocol = RTPROT_BOOT;
			else if (strcmp(*argv, "kernel") == 0)
				req.r.rtm_protocol = RTPROT_KERNEL;
			else if (strcmp(*argv, "gated") == 0)
				req.r.rtm_protocol = RTPROT_GATED;
			else if (strcmp(*argv, "rdisc") == 0)
				req.r.rtm_protocol = RTPROT_RA;
			else if (strcmp(*argv, "mrt") == 0)
				req.r.rtm_protocol = RTPROT_MRT;
			else if (strcmp(*argv, "zebra") == 0)
				req.r.rtm_protocol = RTPROT_ZEBRA;
			else {
				unsigned protocol;
				if (scan_number(*argv, &protocol) || protocol > 255)
					invarg("protocol value is invalid\n");
				req.r.rtm_protocol = protocol;
			}
			proto_ok =1;
		} else if (matches(*argv, "table") == 0) {
			NEXT_ARG();
			if (strcmp(*argv, "main") == 0)
				req.r.rtm_table = RT_TABLE_MAIN;
			else if (strcmp(*argv, "local") == 0)
				req.r.rtm_table = RT_TABLE_LOCAL;
			else if (strcmp(*argv, "default") == 0)
				req.r.rtm_table = RT_TABLE_DEFAULT;
			else {
				unsigned tb_id;
				if (scan_number(*argv, &tb_id) || tb_id > 255)
					invarg("table id value is invalid\n");
				req.r.rtm_table = tb_id;
			}
			table_ok = 1;
		} else if (strcmp(*argv, "local") == 0 ||
			   strcmp(*argv, "nat") == 0 ||
			   matches(*argv, "broadcast") == 0 ||
			   strcmp(*argv, "brd") == 0 ||
			   matches(*argv, "anycast") == 0 ||
			   matches(*argv, "unicast") == 0 ||
			   matches(*argv, "unreachable") == 0 ||
			   strcmp(*argv, "prohibit") == 0 ||
			   strcmp(*argv, "throw") == 0 ||
			   matches(*argv, "multicast") == 0) {
			inet_prefix dst;
			char *cmd = *argv;

			if (dst_ok)
				usage();

			NEXT_ARG();

			get_prefix(&dst, *argv, req.r.rtm_family);
			if (req.r.rtm_family == AF_UNSPEC)
				req.r.rtm_family = dst.family;
			req.r.rtm_dst_len = dst.bitlen;
			dst_ok = 1;
			addattr_l(&req.n, sizeof(req), RTA_DST, &dst.data, dst.bytelen);

			if (strcmp(cmd, "local") == 0)
				req.r.rtm_type = RTN_LOCAL;
			else if (strcmp(cmd, "nat") == 0)
				req.r.rtm_type = RTN_NAT;
			else if (matches(cmd, "broadcast") == 0 ||
				 strcmp(cmd, "brd") == 0)
				req.r.rtm_type = RTN_BROADCAST;
			else if (matches(cmd, "anycast") == 0)
				req.r.rtm_type = RTN_ANYCAST;
			else if (matches(cmd, "multicast") == 0)
				req.r.rtm_type = RTN_MULTICAST;
			else if (matches(cmd, "prohibit") == 0)
				req.r.rtm_type = RTN_PROHIBIT;
			else if (matches(cmd, "unreachable") == 0)
				req.r.rtm_type = RTN_UNREACHABLE;
			else if (matches(cmd, "unicast") == 0)
				req.r.rtm_type = RTN_UNICAST;
			else if (strcmp(cmd, "throw") == 0)
				req.r.rtm_type = RTN_THROW;
			type_ok = 1;
		} else if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			strcpy(d, *argv);
		} else {
			inet_prefix addr;
			get_prefix(&addr, *argv, req.r.rtm_family);
			if (req.r.rtm_family == AF_UNSPEC)
				req.r.rtm_family = addr.family;
			if (addr.bitlen) {
				addattr_l(&req.n, sizeof(req), RTA_DST, &addr.data, addr.bytelen);
				req.r.rtm_dst_len = addr.bitlen;
			}
			dst_ok = 1;
		}
		argc--; argv++;
	}
	if (req.n.nlmsg_type == 0)
		usage();

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	if (d[0] || nhs_ok)  {
		int idx;

		ll_init_map(&rth);

		if (d[0]) {
			if ((idx = ll_name_to_index(d)) == 0)
				usage();
			addattr32(&req.n, sizeof(req), RTA_OIF, idx);
		}
	}

	if (mxrta->rta_len > RTA_LENGTH(0)) {
		if (mxlock)
			rta_addattr32(mxrta, sizeof(mxbuf), RTAX_LOCK, mxlock);
		addattr_l(&req.n, sizeof(req), RTA_METRICS, RTA_DATA(mxrta), RTA_PAYLOAD(mxrta));
	}

	if (nhs_ok)
		parse_nexthops(&req.n, &req.r, argc, argv);

	if (!table_ok) {
		if (req.r.rtm_type == RTN_LOCAL ||
		    req.r.rtm_type == RTN_BROADCAST ||
		    req.r.rtm_type == RTN_NAT ||
		    req.r.rtm_type == RTN_ANYCAST)
			req.r.rtm_table = RT_TABLE_LOCAL;
	}
	if (!scope_ok) {
		if (req.r.rtm_type == RTN_LOCAL ||
		    req.r.rtm_type == RTN_NAT)
			req.r.rtm_scope = RT_SCOPE_HOST;
		else if (req.r.rtm_type == RTN_BROADCAST ||
			 req.r.rtm_type == RTN_MULTICAST ||
			 req.r.rtm_type == RTN_ANYCAST)
			req.r.rtm_scope = RT_SCOPE_LINK;
		else if (req.r.rtm_type == RTN_UNICAST && !gw_ok && !nhs_ok)
			req.r.rtm_scope = RT_SCOPE_LINK;
	}

	if (rtnl_talk(&rth, &req.n, 0, 0, NULL, NULL, NULL) < 0)
		exit(2);

	return 0;
}

static int rtnl_rtcache_request(struct rtnl_handle *rth)
{
	struct {
		struct nlmsghdr nlh;
		struct rtmsg rtm;
	} req;
	struct sockaddr_nl nladdr;

	memset(&nladdr, 0, sizeof(nladdr));
	memset(&req, 0, sizeof(req));
	nladdr.nl_family = AF_NETLINK;

	req.nlh.nlmsg_len = sizeof(req);
	req.nlh.nlmsg_type = RTM_GETROUTE;
	req.nlh.nlmsg_flags = NLM_F_ROOT|NLM_F_MATCH|NLM_F_REQUEST;
	req.nlh.nlmsg_pid = 0;
	req.nlh.nlmsg_seq = rth->dump = ++rth->seq;
	req.rtm.rtm_family = AF_INET;
	req.rtm.rtm_flags |= RTM_F_CLONED;

	return sendto(rth->fd, (void*)&req, sizeof(req), 0, (struct sockaddr*)&nladdr, sizeof(nladdr));
}


int iproute_list(int argc, char **argv)
{
	int do_ipv6 = AF_INET;
	struct rtnl_handle rth;
	
	filter.tb = RT_TABLE_MAIN;
	filter.mdst.bitlen = -1;

	while (argc > 0) {
		if (matches(*argv, "table") == 0) {
			NEXT_ARG();
			if (strcmp(*argv, "local") == 0)
				filter.tb = RT_TABLE_LOCAL;
			else if (strcmp(*argv, "default") == 0)
				filter.tb = RT_TABLE_DEFAULT;
			else if (strcmp(*argv, "main") == 0)
				filter.tb = RT_TABLE_MAIN;
			else if (strcmp(*argv, "all") == 0) {
				filter.tb = 0;
				do_ipv6 = AF_UNSPEC;
			} else if (strcmp(*argv, "cache") == 0) {
				filter.tb = -1;
			} else if (strcmp(*argv, "main6") == 0) {
				do_ipv6 = AF_INET6;
				filter.tb = RT_TABLE_MAIN;
			} else if (strcmp(*argv, "local6") == 0) {
				do_ipv6 = AF_INET6;
				filter.tb = RT_TABLE_LOCAL;
			} else {
				unsigned tb_id;
				if (scan_number(*argv, &tb_id) || tb_id > 255)
					invarg("table id value is invalid\n");
				filter.tb = tb_id;
			}
		} else if (matches(*argv, "proto") == 0) {
			NEXT_ARG();
			if (matches(*argv, "kernel") == 0)
				filter.protocol = RTPROT_KERNEL;
			else if (matches(*argv, "boot") == 0)
				filter.protocol = RTPROT_BOOT;
			else if (matches(*argv, "ra") == 0)
				filter.protocol = RTPROT_RA;
			else if (matches(*argv, "gated") == 0)
				filter.protocol = RTPROT_GATED;
			else if (matches(*argv, "mrt") == 0)
				filter.protocol = RTPROT_MRT;
			else if (matches(*argv, "zebra") == 0)
				filter.protocol = RTPROT_ZEBRA;
			else {
				unsigned protocol;
				if (scan_number(*argv, &protocol) || protocol > 255)
					invarg("protocol value is invalid\n");
				filter.protocol = protocol;
			}
		} else if (matches(*argv, "scope") == 0) {
			NEXT_ARG();
			if (matches(*argv, "host") == 0)
				filter.scope = RT_SCOPE_HOST;
			else if (matches(*argv, "link") == 0)
				filter.scope = RT_SCOPE_LINK;
			else if (matches(*argv, "universe") == 0 ||
				 matches(*argv, "global") == 0)
				filter.scope = RT_SCOPE_UNIVERSE;
			else {
				unsigned scope;
				if (scan_number(*argv, &scope) || scope > 255)
					invarg("scope value is invalid\n");
				filter.scope = scope;
			}
		} else if (matches(*argv, "type") == 0) {
			NEXT_ARG();
			if (matches(*argv, "local") == 0)
				filter.type = RTN_LOCAL;
			else if (matches(*argv, "broadcast") == 0)
				filter.type = RTN_BROADCAST;
			else if (matches(*argv, "unicast") == 0)
				filter.type = RTN_UNICAST;
			else {
				unsigned type;
				if (scan_number(*argv, &type) || type > 255)
					invarg("node type value is invalid\n");
				filter.type = type;
			}
		} else if (matches(*argv, "root") == 0) {
			NEXT_ARG();
			get_prefix(&filter.rdst, *argv, do_ipv6);
		} else if (matches(*argv, "match") == 0) {
			NEXT_ARG();
			get_prefix(&filter.mdst, *argv, do_ipv6);
		} else if (matches(*argv, "exact") == 0) {
			NEXT_ARG();
			get_prefix(&filter.mdst, *argv, do_ipv6);
			filter.rdst = filter.mdst;
		} else
			usage();
		argc--; argv++;
	}

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	ll_init_map(&rth);

	if (filter.tb != -1) {
		if (rtnl_wilddump_request(&rth, do_ipv6, RTM_GETROUTE) < 0) {
			perror("cannot send dump request");
			exit(1);
		}
	} else {
		if (rtnl_rtcache_request(&rth) < 0) {
			perror("cannot send dump request");
			exit(1);
		}
	}

	if (rtnl_dump_filter(&rth, print_route, stdout, NULL, NULL) < 0) {
		fprintf(stderr, "dump terminated\n");
		exit(1);
	}

	exit(0);
}


int iproute_get(int argc, char **argv)
{
	struct rtnl_handle rth;
	struct {
		struct nlmsghdr 	n;
		struct rtmsg 		r;
		char   			buf[1024];
	} req;
	char  idev[16];
	char  odev[16];

	memset(&req, 0, sizeof(req));
	idev[0] = 0;
	odev[0] = 0;

	filter.tb = 0;
	filter.mdst.bitlen = -1;

	req.n.nlmsg_len = NLMSG_LENGTH(sizeof(struct rtmsg));
	req.n.nlmsg_flags = NLM_F_REQUEST;
	req.n.nlmsg_type = RTM_GETROUTE;
	req.r.rtm_family = AF_INET;
	req.r.rtm_table = 0;
	req.r.rtm_protocol = 0;
	req.r.rtm_scope = 0;
	req.r.rtm_type = 0;
	req.r.rtm_src_len = 0;
	req.r.rtm_dst_len = 0;
	
	while (argc > 0) {
		if (matches(*argv, "from") == 0) {
			NEXT_ARG();
			addattr32(&req.n, sizeof(req), RTA_SRC, get_addr32(*argv));
			req.r.rtm_src_len = 32;
		} else if (matches(*argv, "iif") == 0) {
			NEXT_ARG();
			strcpy(idev, *argv);
		} else if (matches(*argv, "oif") == 0) {
			NEXT_ARG();
			strcpy(odev, *argv);
		} else {
			addattr32(&req.n, sizeof(req), RTA_DST, get_addr32(*argv));
			req.r.rtm_dst_len = 32;
		}
		argc--; argv++;
	}

	if (req.r.rtm_dst_len == 0) {
		fprintf(stderr, "Need at least destination address\n");
		exit(1);
	}

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	ll_init_map(&rth);

	if (idev[0] || odev[0])  {
		int idx;

		if (idev[0]) {
			if ((idx = ll_name_to_index(idev)) == 0)
				usage();
			addattr32(&req.n, sizeof(req), RTA_IIF, idx);
		}
		if (odev[0]) {
			if ((idx = ll_name_to_index(odev)) == 0)
				usage();
			addattr32(&req.n, sizeof(req), RTA_OIF, idx);
		}
	}

	if (rtnl_talk(&rth, &req.n, 0, 0, &req.n, NULL, NULL) < 0)
		exit(2);

	if (print_route(NULL, &req.n, (void*)stdout) < 0) {
		fprintf(stderr, "An error :-)\n");
		exit(1);
	}

	exit(0);
}

int accept_msg(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	FILE *fp = (FILE*)arg;

	if (n->nlmsg_type == RTM_NEWROUTE || n->nlmsg_type == RTM_DELROUTE) {
		print_route(who, n, arg);
		return 0;
	}
	if (n->nlmsg_type == RTM_NEWLINK || n->nlmsg_type == RTM_DELLINK) {
		ll_remember_index(who, n, NULL);
		print_linkinfo(who, n, arg);
		return 0;
	}
	if (n->nlmsg_type == RTM_NEWADDR || n->nlmsg_type == RTM_DELADDR) {
		print_addrinfo(who, n, arg);
		return 0;
	}
	if (n->nlmsg_type == RTM_NEWNEIGH || n->nlmsg_type == RTM_DELNEIGH) {
		print_neigh(who, n, arg);
		return 0;
	}
	if (n->nlmsg_type == 15) {
		long secs = ((__u32*)NLMSG_DATA(n))[0];
		long usecs = ((__u32*)NLMSG_DATA(n))[1];
		fprintf(fp, "Timestamp: %lu secs %lu usecs\n", secs, usecs);
		return 0;
	}
	if (n->nlmsg_type != NLMSG_ERROR && n->nlmsg_type != NLMSG_NOOP &&
	    n->nlmsg_type != NLMSG_DONE) {
		fprintf(fp, "Unknown message: %08x %08x %08x\n",
			n->nlmsg_len, n->nlmsg_type, n->nlmsg_flags);
	}
	return 0;
}

int iproute_monitor(int argc, char **argv)
{
	struct rtnl_handle rth;

	filter.tb = 0;
	filter.mdst.bitlen = -1;

	if (argc > 0) {
		if (matches(*argv, "file") == 0) {
			FILE *fp;
			NEXT_ARG();
			fp = fopen(*argv, "r");
			if (fp == NULL) {
				perror("fopen");
				exit(-1);
			}
			return rtnl_from_file(fp, accept_msg, (void*)stdout);
		}
	}

	if (rtnl_open(&rth, RTMGRP_IPV6_ROUTE|RTMGRP_IPV4_ROUTE) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	ll_init_map(&rth);

	if (rtnl_listen(&rth, print_route, (void*)stdout) < 0)
		exit(2);

	exit(0);
}

int do_iproute(int argc, char **argv)
{
	if (argc < 1)
		return iproute_list(0, NULL);
	
	if (matches(*argv, "add") == 0)
		return iproute_modify(RTM_NEWROUTE, NLM_F_CREATE|NLM_F_EXCL,
				      argc-1, argv+1);
	if (matches(*argv, "change") == 0 || strcmp(*argv, "chg") == 0)
		return iproute_modify(RTM_NEWROUTE, NLM_F_REPLACE,
				      argc-1, argv+1);
	if (matches(*argv, "replace") == 0)
		return iproute_modify(RTM_NEWROUTE, NLM_F_CREATE|NLM_F_REPLACE,
				      argc-1, argv+1);
	if (matches(*argv, "append") == 0)
		return iproute_modify(RTM_NEWROUTE, NLM_F_CREATE,
				      argc-1, argv+1);
	if (matches(*argv, "delete") == 0)
		return iproute_modify(RTM_DELROUTE, 0,
				      argc-1, argv+1);
	if (matches(*argv, "monitor") == 0)
		return iproute_monitor(argc-1, argv+1);
	if (matches(*argv, "list") == 0 || matches(*argv, "show") == 0
	    || matches(*argv, "lst") == 0)
		return iproute_list(argc-1, argv+1);
	if (matches(*argv, "get") == 0)
		return iproute_get(argc-1, argv+1);
	usage();
}

