/* uvwwidget.c */

/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


/*
 * U, V, W variable widget window.
 *
 * This widget is used to specify which variables to use for the trajectory,
 * and wind slices' U, V, and W components.
 */



#include <stdlib.h>
#include <string.h>
#include "../lui5/lui.h"
#include "api.h"
#include "gui.h"



static int current_ctx;



static void read_uvw_widgets( GuiContext gtx )
{
   int index = gtx->context_index;
   char varname[20];
   int u1, v1, w1, u2, v2, w2, traju, trajv, trajw;

   LUI_FieldGetText( gtx->u1_field, varname );
   u1 = vis5d_find_var( index, varname );
   LUI_FieldGetText( gtx->v1_field, varname );
   v1 = vis5d_find_var( index, varname );
   LUI_FieldGetText( gtx->w1_field, varname );
   w1 = vis5d_find_var( index, varname );

   LUI_FieldGetText( gtx->u2_field, varname );
   u2 = vis5d_find_var( index, varname );
   LUI_FieldGetText( gtx->v2_field, varname );
   v2 = vis5d_find_var( index, varname );
   LUI_FieldGetText( gtx->w2_field, varname );
   w2 = vis5d_find_var( index, varname );

   LUI_FieldGetText( gtx->traju_field, varname );
   traju = vis5d_find_var( index, varname );
   LUI_FieldGetText( gtx->trajv_field, varname );
   trajv = vis5d_find_var( index, varname );
   LUI_FieldGetText( gtx->trajw_field, varname );
   trajw = vis5d_find_var( index, varname );


   vis5d_set_wind_vars( index, u1, v1, w1, u2, v2, w2, traju, trajv, trajw );
}




/* called when Cancel button is pressed:  just close window */
static int cancel_cb( LUI_NEWBUTTON *b, int state )
{
   int index = b->context_index;
   GuiContext gtx = get_gui_gtx( index );

   XUnmapWindow( GuiDpy, gtx->uvw_window );
   return 0;
}



/* called when apply button is pressed:  use new variables, keep window open */
static int apply_cb( LUI_NEWBUTTON *b, int state )
{
   int index = b->context_index;
   GuiContext gtx = get_gui_gtx( index );

   read_uvw_widgets( gtx );

   /* recompute graphics here??? */
   return 0;
}


/* called when ok button is pressed:  use new vars, close window */
static int ok_cb( LUI_NEWBUTTON *b, int state )
{
   int index = b->context_index;
   GuiContext gtx = get_gui_gtx( index );

   read_uvw_widgets( gtx );

   /* recompute graphics here??? */
   XUnmapWindow( GuiDpy, gtx->uvw_window );
   return 0;
}




/* called when the text in a field changes */
static int field_cb( LUI_FIELD *f, char *text )
{
#ifdef LEAVEOUT
   int index = f->context_index;
   GuiContext gtx = get_gui_gtx( index );
   int var, numvars, *varptr;
   char varname[20];

   if (f==gtx->traju_field) {
      varptr = &gtx->traju_var;
   }
   else if (f==gtx->trajv_field) {
      varptr = &gtx->trajv_var;
   }
   else if (f==gtx->trajw_field) {
      varptr = &gtx->trajw_var;
   }
   else if (f==gtx->u1_field) {
      varptr = &gtx->u1_var;
   }
   else if (f==gtx->v1_field) {
      varptr = &gtx->v1_var;
   }
   else if (f==gtx->w1_field) {
      varptr = &gtx->w1_var;
   }
   else if (f==gtx->u2_field) {
      varptr = &gtx->u2_var;
   }
   else if (f==gtx->v2_field) {
      varptr = &gtx->v2_var;
   }
   else if (f==gtx->w2_field) {
      varptr = &gtx->w2_var;
   }
   else {
      /* this should never happen! */
      abort();
   }

   /* Be sure the varname entered is valid and determine the corresponding
    * var number.
    */
   var = vis5d_find_var( index, text );
   if (var>=0) {
      *varptr = var;

      /* recompute graphics? */

      return 0;
   }

   /* If we get here, that means the user entered an invalid varname so
    * restore previous name.
    */
   if (*varptr>=0) {
      vis5d_get_var_name( index, *varptr, varname );
      LUI_FieldSetText( f, varname );
   }
   else {
      LUI_FieldSetText( f, "" );
   }

#endif
   return 0;
}




/*
 * Make the U,V,W widget window.
 */
static void make_uvw_widget( GuiContext gtx )
{
   LUI_NEWBUTTON *apply, *ok, *cancel;
   Window w;

   current_ctx = gtx->context_index;

   w = LUI_CreateWindowAt( LUI_RootWindow, 150, 400, 210, 310 );
   gtx->uvw_window = w;

   LUI_BorderWidth( 2 );

   LUI_NewLabelCreate( w, LUI_LEFT, LUI_TOP, 380, 20, "Trajectory Vars:" );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   East/West:" );
   gtx->traju_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   North/South:" );
   gtx->trajv_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   Vertical:" );
   gtx->trajw_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );

   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 380, 20, "Wind, Stream slice 1:" );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   East/West:" );
   gtx->u1_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   North/South:" );
   gtx->v1_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   Vertical:" );
   gtx->w1_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );

   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 380, 20, "Wind, Stream slice 2:" );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   East/West:" );
   gtx->u2_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   North/South:" );
   gtx->v2_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 120, 20, "   Vertical:" );
   gtx->w2_field = LUI_FieldCreate( w, LUI_NEXT_X, LUI_SAME_Y, 80, 20 );

   /* Link fields together for TABing */
   LUI_FieldLink( gtx->traju_field, gtx->trajv_field );
   LUI_FieldLink( gtx->trajv_field, gtx->trajw_field );
   LUI_FieldLink( gtx->trajw_field, gtx->u1_field );
   LUI_FieldLink( gtx->u1_field, gtx->v1_field );
   LUI_FieldLink( gtx->v1_field, gtx->w1_field );
   LUI_FieldLink( gtx->w1_field, gtx->u2_field );
   LUI_FieldLink( gtx->u2_field, gtx->v2_field );
   LUI_FieldLink( gtx->v2_field, gtx->w2_field );

   LUI_NewLabelCreate( w, LUI_LEFT, LUI_NEXT_Y, 10, 5, "" );  /* spacer */

   apply = LUI_PushButtonCreate( w, LUI_LEFT, LUI_NEXT_Y, 66, 20, "Apply" );
   LUI_ButtonCallback( apply, apply_cb );
   ok = LUI_PushButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 66, 20, "OK" );
   LUI_ButtonCallback( ok, ok_cb );
   cancel = LUI_PushButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 66, 20, "Cancel");
   LUI_ButtonCallback( cancel, cancel_cb );

   XMapWindow( GuiDpy, w );
}




/*
 * Update all the text fields with the names of the current U,V,W variables.
 */
static void load_uvw_widgets( GuiContext gtx )
{
   int index = gtx->context_index;
   char varname[20];
   int u1, v1, w1, u2, v2, w2, traju, trajv, trajw;


   vis5d_get_wind_vars( index, &u1, &v1, &w1,
                               &u2, &v2, &w2,
                               &traju, &trajv, &trajw );

   /*** Traj vars ***/
   if (traju>=0) {
      vis5d_get_var_name( index, traju, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->traju_field, varname );

   if (trajv>=0) {
      vis5d_get_var_name( index, trajv, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->trajv_field, varname );

   if (trajw>=0) {
      vis5d_get_var_name( index, trajw, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->trajw_field, varname );

   /*** First set of slices ****/
   if (u1>=0) {
      vis5d_get_var_name( index, u1, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->u1_field, varname );

   if (v1>=0) {
      vis5d_get_var_name( index, v1, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->v1_field, varname );

   if (w1>=0) {
      vis5d_get_var_name( index, w1, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->w1_field, varname );

   /*** First set of slices ****/
   if (u2>=0) {
      vis5d_get_var_name( index, u2, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->u2_field, varname );

   if (v2>=0) {
      vis5d_get_var_name( index, v2, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->v2_field, varname );

   if (w2>=0) {
      vis5d_get_var_name( index, w2, varname );
   }
   else {
      varname[0] = 0;
   }
   LUI_FieldSetText( gtx->w2_field, varname );
}



void show_uvw_widget( GuiContext gtx )
{
   int index = gtx->context_index;

   /* create the widget if needed */
   if (!gtx->uvw_window) {
      make_uvw_widget( gtx );
   }

   /* map the window */
   XMapWindow( GuiDpy, gtx->uvw_window );

   /* update the text fields */
   load_uvw_widgets( gtx );
}
