/*  map.c  */


/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* map lines module */



#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "binio.h"
#include "graphics.h"
#include "globals.h"
#include "map.h"
#include "proj.h"
#include "topo.h"

#ifndef SEEK_SET
#  define SEEK_SET  0
#endif



#define FABSF(X)  ( (float) fabs( (double) (X) ) )




#define CLIP_CODE(x,y,code)     code = 0;                                \
                                if (x<ctx->ClipMin0)       code |= 1;    \
                                else if (x>ctx->ClipMax0)  code |= 2;    \
                                if (y<ctx->ClipMin1)       code |= 4;    \
                                else if (y>ctx->ClipMax1)  code |= 8;


/*
 * Clip the line segment defined by (x0,y0) and (x1,y1) against the
 * bounds of ClipMin0,ClipMax0,ClipMin1 and ClipMax1.
 * Input:  x0,y0, x1,y1 - coordinates of endpoints of line
 *         quickclip - value to use for trival rejection
 * Return:  1 = entire or partial segment resulted
 *          0 = entire line is out of bounds
 */
static int clip( Context ctx, float *x0, float *y0, float *x1, float *y1,
                 float quickclip )
{
   int code0, code1;
   float t, dx, dy;

   if (*x0>quickclip || *x0<-quickclip || *y0>quickclip || *y0<-quickclip ||
       *x1>quickclip || *x1<-quickclip || *y1>quickclip || *y1<-quickclip) {
      /* probably a point mapped to infinity */
      return 0;
   }

   dx = *x1 - *x0;
   dy = *y1 - *y0;

   while (1) {

      /* compute codes */
      CLIP_CODE( *x0, *y0, code0 );
      CLIP_CODE( *x1, *y1, code1 );

      /* check if entirely in bounds */
      if (code0==0 && code1==0)
         return 1;

      /* check if entirely out of bounds */
      if (code0 & code1)
         return 0;

      /* eliminate line segments across the whole box */
      if ((code0 & 1 && code1 & 2) ||
          (code0 & 2 && code1 & 1) ||
          (code0 & 4 && code1 & 8) ||
          (code0 & 8 && code1 & 4))
         return 0;

      if (code0 & 1) {  /* clip against ClipMin0 */
         t = (ctx->ClipMin0 - *x1) / dx;
         *x0 = ctx->ClipMin0;
         *y0 = *y1 + t * dy;
      }
      else if (code0 & 2) {  /* clip against ClipMax0 */
         t = (ctx->ClipMax0 - *x1) / dx;
         *x0 = ctx->ClipMax0;
         *y0 = *y1 + t * dy;
      }
      else if (code0 & 4) {  /* clip against ClipMin1 */
         t = (ctx->ClipMin1 - *y1) / dy;
         *y0 = ctx->ClipMin1;
         *x0 = *x1 + t * dx;
      }
      else if (code0 & 8) {  /* clip against ClipMax1 */
         t = (ctx->ClipMax1 - *y1) / dy;
         *y0 = ctx->ClipMax1;
         *x0 = *x1 + t * dx;
      }
      else if (code1 & 1) {  /* clip against ClipMin0 */
         t = (ctx->ClipMin0 - *x0) / dx;
         *x1 = ctx->ClipMin0;
         *y1 = *y0 + t * dy;
      }
      else if (code1 & 2) {  /* clip against ClipMax0 */
         t = (ctx->ClipMax0 - *x0) / dx;
         *x1 = ctx->ClipMax0;
         *y1 = *y0 + t * dy;
      }
      else if (code1 & 4) {  /* clip against ClipMin1 */
         t = (ctx->ClipMin1 - *y0) / dy;
         *y1 = ctx->ClipMin1;
         *x1 = *x0 + t * dx;
      }
      else if (code1 & 8) {  /* clip against ClipMax1 */
         t = (ctx->ClipMax1 - *y0) / dy;
         *y1 = ctx->ClipMax1;
         *x1 = *x0 + t * dx;
      }

   }

}


#define EPS 0.1

/*
 * Given the geographic coordinates of the end points of a map line,
 * convert the coordinates to graphics coordinates, clip against the 3-D
 * box and save the line if it's visible.
 */
static void add_line( Context ctx, float lat0, float lon0, float hgt0,
                      float lat1, float lon1, float hgt1, int *new )
{
   static float height_kludge = 0.0;

   if (height_kludge==0.0) {
      height_kludge = (ctx->TopBound-ctx->BottomBound) / 25.0;
   }

   if (!ctx->CurvedBox) {
      /* Rectangular box, clip in graphics coords. */

      float x0, y0, z0, x1, y1, z1;
      float flat0, flon0, fhgt0, flat1, flon1, fhgt1;

      /* the arguments are really doubles so we must convert to float */
      flat0 = (float) lat0;
      flon0 = (float) lon0;
      fhgt0 = (float) hgt0;
      geo_to_xyz( ctx, -1, -1, 1, &flat0, &flon0, &fhgt0, &x0, &y0, &z0 );
/*      z0 -= 0.01; WLH 4-24-95 */

      flat1 = (float) lat1;
      flon1 = (float) lon1;
      fhgt1 = (float) hgt1;
      geo_to_xyz( ctx, -1, -1, 1, &flat1, &flon1, &fhgt1, &x1, &y1, &z1 );

      if (ctx->Projection == PROJ_ROTATED) {
        pandg_for(&flat0, &flon0, ctx->CentralLat, ctx->CentralLon, ctx->Rotation);
        pandg_for(&flat1, &flon1, ctx->CentralLat, ctx->CentralLon, ctx->Rotation);
      }
      if (flon0 < -150.0 && flon1 > 150.0 ||
          flon1 < -150.0 && flon0 > 150.0) {
        x0 = 200.0; /* ensure clipping failure */
      }

/* WLH 8-1-95 */
      if (ctx->Projection == PROJ_LAMBERT ||
          ctx->Projection == PROJ_STEREO) {
        float tlat0, tlon0, thgt0, tlat1, tlon1, thgt1;
        xyz_to_geo(ctx, -1, -1, x0, y0, z0, &tlat0, &tlon0, &thgt0);
        xyz_to_geo(ctx, -1, -1, x1, y1, z1, &tlat1, &tlon1, &thgt1);
        if (FABSF(flat0 - tlat0) > EPS) {
          x0 = 200.0; /* ensure clipping failure */
        }
        if (FABSF(flon0 - tlon0) > EPS &&
            FABSF(flon0 - tlon0 - 360.0) > EPS &&
            FABSF(flon0 - tlon0 + 360.0) > EPS) {
          x0 = 200.0; /* ensure clipping failure */
        }
        if (FABSF(flat1 - tlat1) > EPS) {
          x0 = 200.0; /* ensure clipping failure */
        }
        if (FABSF(flon1 - tlon1) > EPS &&
            FABSF(flon1 - tlon1 - 360.0) > EPS &&
            FABSF(flon1 - tlon1 + 360.0) > EPS) {
          x0 = 200.0; /* ensure clipping failure */
        }
      }

      /* kludge z values so they're slightly above the topo */
      z0 += 0.015;
      z1 += 0.015;

      if (clip( ctx, &x0, &y0, &x1, &y1, 100.0 )) {
         /* save the line */
         if (*new==0 && ctx->VertCount>0
                              && ctx->MapVert[ctx->VertCount-1][0]==x0
                              && ctx->MapVert[ctx->VertCount-1][1]==y0) {
            /* add to current polyline */
            ctx->MapVert[ctx->VertCount][0] = x1;
            ctx->MapVert[ctx->VertCount][1] = y1;
            ctx->MapVert[ctx->VertCount][2] = z1;
            ctx->FlatMapVert[ctx->VertCount][0] = x1;
            ctx->FlatMapVert[ctx->VertCount][1] = y1;
            ctx->FlatMapVert[ctx->VertCount][2] = ctx->Zmin+0.01;
            ctx->VertCount++;
            ctx->Len[ctx->SegCount]++;
         }
         else {
            /* begin a new polyline */
            *new = 0;
            if (ctx->Len[ctx->SegCount]>0) {
               ctx->SegCount++;
               ctx->Start[ctx->SegCount] = ctx->VertCount;
            }
            ctx->MapVert[ctx->VertCount][0] = x0;
            ctx->MapVert[ctx->VertCount][1] = y0;
            ctx->MapVert[ctx->VertCount][2] = z0;
            ctx->FlatMapVert[ctx->VertCount][0] = x0;
            ctx->FlatMapVert[ctx->VertCount][1] = y0;
            ctx->FlatMapVert[ctx->VertCount][2] = ctx->Zmin+0.01;
            ctx->VertCount++;
            ctx->MapVert[ctx->VertCount][0] = x1;
            ctx->MapVert[ctx->VertCount][1] = y1;
            ctx->MapVert[ctx->VertCount][2] = z1;
            ctx->FlatMapVert[ctx->VertCount][0] = x1;
            ctx->FlatMapVert[ctx->VertCount][1] = y1;
            ctx->FlatMapVert[ctx->VertCount][2] = ctx->Zmin+0.01;
            ctx->VertCount++;
            ctx->Len[ctx->SegCount] = 2;
         }
      }
   }
   else {
      /* Curved box, clip in lat/lon coordinates */

      float x0, y0, z0, x1, y1, z1;
      float flat0, flon0, fhgt0, flat1, flon1, fhgt1;

      /* the arguments are really doubles so we must convert to float */
      flat0 = (float) lat0;
      flon0 = (float) lon0;
      fhgt0 = (float) hgt0 + height_kludge;
      flat1 = (float) lat1;
      flon1 = (float) lon1;
      fhgt1 = (float) hgt1 + height_kludge;

      if (clip( ctx, &flat0, &flon0, &flat1, &flon1, 1000.0 )) {
         /* save the line */
         float fx0, fy0, fz0, fx1, fy1, fz1;

         /* graphics coords for raised map lines */
         geo_to_xyz( ctx, -1, -1, 1, &flat0, &flon0, &fhgt0, &x0, &y0, &z0 );
         geo_to_xyz( ctx, -1, -1, 1, &flat1, &flon1, &fhgt1, &x1, &y1, &z1 );

         /* graphics coords for 'flat' map lines */
         geo_to_xyz( ctx, -1, -1, 1, &flat0, &flon0, &ctx->BottomBound,
                     &fx0, &fy0, &fz0 );
         geo_to_xyz( ctx, -1, -1, 1, &flat1, &flon1, &ctx->BottomBound,
                     &fx1, &fy1, &fz1 );


         if (*new==0 && ctx->VertCount>0
                              && ctx->MapVert[ctx->VertCount-1][0]==x0
                              && ctx->MapVert[ctx->VertCount-1][1]==y0) {
            /* add to current polyline */
            ctx->MapVert[ctx->VertCount][0] = x1;
            ctx->MapVert[ctx->VertCount][1] = y1;
            ctx->MapVert[ctx->VertCount][2] = z1;
            ctx->FlatMapVert[ctx->VertCount][0] = fx1;
            ctx->FlatMapVert[ctx->VertCount][1] = fy1;
            ctx->FlatMapVert[ctx->VertCount][2] = fz1;
            ctx->VertCount++;
            ctx->Len[ctx->SegCount]++;
         }
         else {
            /* begin a new polyline */
            *new = 0;
            if (ctx->Len[ctx->SegCount]>0) {
               ctx->SegCount++;
               ctx->Start[ctx->SegCount] = ctx->VertCount;
            }
            ctx->MapVert[ctx->VertCount][0] = x0;
            ctx->MapVert[ctx->VertCount][1] = y0;
            ctx->MapVert[ctx->VertCount][2] = z0;
            ctx->FlatMapVert[ctx->VertCount][0] = fx0;
            ctx->FlatMapVert[ctx->VertCount][1] = fy0;
            ctx->FlatMapVert[ctx->VertCount][2] = fz0;
            ctx->VertCount++;
            ctx->MapVert[ctx->VertCount][0] = x1;
            ctx->MapVert[ctx->VertCount][1] = y1;
            ctx->MapVert[ctx->VertCount][2] = z1;
            ctx->FlatMapVert[ctx->VertCount][0] = fx1;
            ctx->FlatMapVert[ctx->VertCount][1] = fy1;
            ctx->FlatMapVert[ctx->VertCount][2] = fz1;
            ctx->VertCount++;
            ctx->Len[ctx->SegCount] = 2;
         }
      }

   }
}



/*
 * Read a map file and make the graphics.
 * Input:  mapname - name of map file
 * Return:  0 = error, non-zero = success
 */
int init_map( Context ctx, char *mapname )
{
   int mapfile;
   int nsegs;
   int *dir, *verts;
   int i, j, offset, length, new;
   float prevlat, prevlon, prevhgt;
   float lat, lon, hgt;
   float start_shift, end_shift, lon_shift;

   mapfile = open( mapname, O_RDONLY );
   if (mapfile<0) {
      printf("Map file %s not found\n", mapname );
      return 0;
   }

   /* read number of segments (number of polylines) */
   if (!read_int4( mapfile, &nsegs)) {
      close( mapfile );
      return 0;
   }

   /* allocate and read segment directory */
   dir = (int *) malloc( nsegs * 6 * sizeof(int) );
   if (!dir) {
      close( mapfile );
      return 0;
   }
   if (read_int4_array( mapfile, dir, nsegs*6) < nsegs*6) {
      free( dir );
      close( mapfile );
      return 0;
   }

   ctx->SegCount = ctx->VertCount = 0;
   ctx->Start[0] = 0;
   ctx->Len[0] = 0;
   new = 1;

   if (ctx->CurvedBox==0) {
      /* rectangular box, clip in graphics coords */
      ctx->ClipMin0 = ctx->Xmin;
      ctx->ClipMax0 = ctx->Xmax;
      ctx->ClipMin1 = ctx->Ymin;
      ctx->ClipMax1 = ctx->Ymax;
   }
   else {
      /* curved box, clip in lat/lon coords */
      ctx->ClipMin0 = ctx->SouthBound;
      ctx->ClipMax0 = ctx->NorthBound;
      ctx->ClipMin1 = ctx->EastBound;
      ctx->ClipMax1 = ctx->WestBound;
   }

   /*
    * Map vertices are in lat/lon coordinates whose ranges are [-90,90]
    * and [-180,180], respectively.  If the dataset's longitude boundaries
    * are outside the [-180,180] range, we must make multiple passes
    * through the map line list, shifting the longitudes by +/- 360 
    * degrees to make sure we get all the map lines we're supposed to see.
    */
/* WLH 8-1-95 */
   if (ctx->Projection == PROJ_LAMBERT ||
       ctx->Projection == PROJ_STEREO) {
     start_shift = -360.0;
     end_shift = 360.1;
   }
   else {
     if (ctx->EastBound<-180.0) start_shift = -360.0;
     else start_shift = 0.0;
     if (ctx->WestBound>180.0) end_shift = 360.0;
     else end_shift = 0.0;
   }

   for (lon_shift=start_shift; lon_shift<=end_shift; lon_shift+=360.0) {

      for (i=0;i<nsegs && ctx->SegCount<MAXMAPSEG;i++) {
         /* WLH 6-9-95  kludge for bad segment in OUTLCOPL */
         if (dir[6*i] == 0 && dir[6*i+1] == 0 &&
             dir[6*i+2] == 0 && dir[6*i+3] == 0) continue;
         /* get list of vertices for ith segment */
         offset = dir[i*6+4] * 4;
         if (lseek(mapfile,offset,SEEK_SET)<0) {
            free( dir );
            close( mapfile);
            return 0;
         }
         length = dir[i*6+5];
         /* read vertices from disk */
         verts = (int *) malloc( length * sizeof(int) );
         if (read_int4_array( mapfile, verts, length) !=length ) {
            free( dir );
            free( verts );
            close( mapfile );
            return 0;
         }

         for (j=0;j<length/2 && ctx->VertCount<MAXMAPVERT;j++) {
            if (j==0) {
               prevlat = (float) verts[j*2] * 0.0001;
               prevlon = (float) verts[j*2+1] * 0.0001 + lon_shift;
               prevhgt = elevation(ctx,prevlat,prevlon,NULL) / 1000.0;
               new = 1;
            }
            else {
               lat = (float) verts[j*2] * 0.0001;
               lon = (float) verts[j*2+1] * 0.0001 + lon_shift;
               hgt = elevation(ctx,lat,lon,NULL) / 1000.0;
               if (hgt!=hgt) {
                  printf("nan hgt!\n");
               }
               add_line( ctx, prevlat, prevlon, prevhgt, lat, lon, hgt, &new );
               prevlat = lat;
               prevlon = lon;
               prevhgt = hgt;
            }
         }
         if (ctx->Len[ctx->SegCount]>0) {
            ctx->SegCount++;
            ctx->Start[ctx->SegCount] = ctx->VertCount;
            ctx->Len[ctx->SegCount] = 0;
         }

         free( verts );
      }
   }

   if (ctx->Len[ctx->SegCount]>0) {
      ctx->SegCount++;
   }

   free( dir );
   close( mapfile );

/*
   printf("done (SegCount=%d, VertCount=%d)\n", ctx->SegCount, ctx->VertCount);
   for (i=0;i<ctx->SegCount;i++) {
      printf("%3d:  %5d %5d\n", i, ctx->Start[i], ctx->Len[i] );
   }
*/
   return 1;
}






/*
 * Draw the map.
 * Input:  time - time step
 *         flat - 1 = draw flat map
 *                  0 = draw raised map
 * Return:  nothing.
 */
int draw_map( Context ctx, int time, int flat )
{
   int i;

   if (flat) {
      /* draw a flat map */
      for (i=0;i<ctx->SegCount;i++) {
         polyline( ctx->FlatMapVert+ctx->Start[i], ctx->Len[i] );
      }
   }
   else {
      /* draw a map with heights */
      for (i=0;i<ctx->SegCount;i++) {
         polyline( ctx->MapVert+ctx->Start[i], ctx->Len[i] );
      }
   }
   return 0;
}


