/*  isocolor.c */

/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




/*
 * Isosurface color widget
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include "../lui5/lui.h"
#include "api.h"
#include "gui.h"
#include "graphics.h"
#include "isocolor.h"
#include "mwmborder.h"



static int current_ctx = -1;
static int current_graphic = -1;
static int current_var = -1;


/*static ColorBar isocb;*/
static int curbutton = 0;



/*
 * Called when one of the variable buttons in the isosurface color
 * widget is pressed.
 */
static int callback( LUI_BUTTON_MATRIX *bm, int row, int col, int button )
{
   float *p;
   GuiContext gtx = get_gui_gtx( current_ctx );

   LUI_ButtonMatrixSetState( bm, curbutton, 0, 0 );
   LUI_ButtonMatrixSetState( bm, row, 0, 1 );
   if (current_graphic==VIS5D_TOPO) {
      XUnmapWindow( GuiDpy, gtx->isocolor_subwin1 );
      XMapWindow( GuiDpy, gtx->isocolor_subwin2 );
   }
   else {
      /* Isosurface of trajectory color table */
      if (curbutton==0 && row>0) {
         XUnmapWindow( GuiDpy, gtx->isocolor_subwin1 );
         XMapWindow( GuiDpy, gtx->isocolor_subwin2 );
      }
      else if (curbutton>0 && row==0) {
         XMapWindow( GuiDpy, gtx->isocolor_subwin1 );
         XUnmapWindow( GuiDpy, gtx->isocolor_subwin2 );
      }
   }

   curbutton = row;

   if (row>-1 || current_graphic==VIS5D_TOPO) {
      unsigned int *table;
      char varname[20];
      float min, max;
      int cvar = row-1;
      int numtimes, curtime, time;
      vis5d_get_numtimes( current_ctx, &numtimes );
      vis5d_get_timestep( current_ctx, &curtime );
      if (current_graphic==VIS5D_ISOSURF) {
         if (cvar>=0) {
            vis5d_get_color_table_address( current_ctx, VIS5D_ISOSURF,
                                           cvar, &table );
            vis5d_get_color_table_params(current_ctx, VIS5D_ISOSURF, cvar, &p);
            vis5d_get_var_name( current_ctx, cvar, varname );
            vis5d_get_var_range( current_ctx, cvar, &min, &max );
            LUI_ColorBarChange( gtx->iso_colorbar,
                                varname, min, max, table, 255, p);
         }
         vis5d_set_isosurface_color_var( current_ctx, current_var, cvar );
         /* recompute/recolor the isosurface acording to new coloring var */
         for (time=0;time<numtimes;time++) {
            vis5d_make_isosurface( current_ctx, time, current_var,
                                   time==curtime );
         }
      }
      else if (current_graphic==VIS5D_TRAJ) {
         if (cvar>=0) {
            vis5d_get_color_table_address( current_ctx, VIS5D_TRAJ,
                                           cvar, &table );
            vis5d_get_color_table_params(current_ctx, VIS5D_TRAJ, cvar, &p);
            vis5d_get_var_name( current_ctx, cvar, varname );
            vis5d_get_var_range( current_ctx, cvar, &min, &max );
            LUI_ColorBarChange( gtx->iso_colorbar,
                                varname, min, max, table, 255, p);
         }
         /* current_var is really the trajectory set */
         vis5d_set_trajectory_color_var( current_ctx, current_var, cvar );
      }
      else if (current_graphic==VIS5D_TOPO) {
         vis5d_get_color_table_address( current_ctx, VIS5D_TOPO,
                                        cvar, &table );
         vis5d_get_color_table_params(current_ctx, VIS5D_TOPO, cvar, &p);
         if (cvar>=0) {
            vis5d_get_var_name( current_ctx, cvar, varname );
            vis5d_get_var_range( current_ctx, cvar, &min, &max );
         }
         else {
            vis5d_get_topo_range( current_ctx, &min, &max );
            strcpy( varname, "Elevation" );
         }
         LUI_ColorBarChange( gtx->iso_colorbar,
                                varname, min, max, table, 255, p);
         vis5d_set_topo_color_var( current_ctx, cvar );
      }
   }
   return 0;
}




/* called when red slider is changed */
static int red_slider_cb( LUI_NEWSLIDER *s, float value )
{
   GuiContext gtx = get_gui_gtx( current_ctx );
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   r = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   if (current_graphic==VIS5D_ISOSURF) {
      LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, current_var, 0, r, g, b );
   }
   else if (current_graphic==VIS5D_TRAJ) {
      LUI_ButtonSetColor( gtx->TrajButton[current_var], (int)(r*255),
                          (int)(g*255), (int)(b*255) );
   }
   vis5d_signal_redraw( current_ctx, 1 );
   return 0;
}



/* called when green slider is changed */
static int green_slider_cb( LUI_NEWSLIDER *s, float value )
{
   GuiContext gtx = get_gui_gtx( current_ctx );
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   g = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   if (current_graphic==VIS5D_ISOSURF) {
      LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, current_var, 0, r, g, b );
   }
   else if (current_graphic==VIS5D_TRAJ) {
      LUI_ButtonSetColor( gtx->TrajButton[current_var], (int)(r*255),
                          (int)(g*255), (int)(b*255) );
   }
   vis5d_signal_redraw( current_ctx, 1 );
   return 0;
}



/* called when blue slider is changed */
static int blue_slider_cb( LUI_NEWSLIDER *s, float value )
{
   GuiContext gtx = get_gui_gtx( current_ctx );
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   b = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   if (current_graphic==VIS5D_ISOSURF) {
      LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, current_var, 0, r, g, b );
   }
   else if (current_graphic==VIS5D_TRAJ) {
      LUI_ButtonSetColor( gtx->TrajButton[current_var], (int)(r*255),
                          (int)(g*255), (int)(b*255) );
   }
   vis5d_signal_redraw( current_ctx, 1 );
   return 0;
}



/* called when alpha slider is changed */
static int alpha_slider_cb( LUI_NEWSLIDER *s, float value )
{
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   a = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   vis5d_signal_redraw( current_ctx, 1 );
   return 0;
}



/* Called when Close button is pressed */
static int close_cb( LUI_NEWBUTTON *b, int state )
{
   GuiContext gtx = get_gui_gtx( current_ctx );

   XUnmapWindow( GuiDpy, gtx->isocolor_window );
   current_ctx = current_graphic = current_var = -1;
   return 0;
}



/*
 * Called when the colorbar's color table is changed.
 */
static int colorbar_callback( LUI_COLORBAR *cb, int action )
{
   GuiContext gtx = get_gui_gtx( current_ctx );
   float *p;

   if (action==LUI_RGB_RESET || action==LUI_ALPHA_RESET
       || action==LUI_RGB_SHAPE) {
      if (current_graphic==VIS5D_ISOSURF) {
         unsigned int *ctable;
         int cvar;
         vis5d_get_isosurface_color_var( current_ctx, current_var, &cvar );
         vis5d_get_color_table_address( current_ctx, VIS5D_ISOSURF,
                                        cvar, &ctable );
         vis5d_get_color_table_params( current_ctx, VIS5D_ISOSURF, cvar, &p);
         if (action==LUI_RGB_RESET) {
            vis5d_color_table_init_params( p, 1, 0 );
            vis5d_color_table_recompute( ctable, 256, p, 1, 0 );
         }
         else if (action==LUI_ALPHA_RESET) {
            vis5d_color_table_init_params( p, 0, 1 );
            vis5d_color_table_recompute( ctable, 256, p, 0, 1 );
         }
         else if (action==LUI_RGB_SHAPE) {
            vis5d_color_table_recompute( ctable, 256, p, 1, 0 );
         }
         else if (action==LUI_ALPHA_SHAPE) {
            vis5d_color_table_recompute( ctable, 256, p, 0, 1 );
         }
         else if (action==LUI_ALPHA_CHANGE) {
            /* recompute min/max */

         }
      }
      else if (current_graphic==VIS5D_TRAJ) {
         unsigned int *ctable;
         int cvar, trajset = current_var;
         vis5d_get_trajectory_color_var( current_ctx, trajset, &cvar );
         vis5d_get_color_table_address( current_ctx, VIS5D_TRAJ,
                                        cvar, &ctable );
         vis5d_get_color_table_params( current_ctx, VIS5D_TRAJ, cvar, &p);
         if (action==LUI_RGB_RESET) {
            vis5d_color_table_init_params( p, 1, 0 );
            vis5d_color_table_recompute( ctable, 256, p, 1, 0);
         }
         else if (action==LUI_ALPHA_RESET) {
            vis5d_color_table_init_params( p, 0, 1 );
            vis5d_color_table_recompute( ctable, 256, p, 0, 1);
         }
         else if (action==LUI_RGB_SHAPE) {
            vis5d_color_table_recompute( ctable, 256, p, 1, 0);
         }
         else if (action==LUI_ALPHA_SHAPE) {
            vis5d_color_table_recompute( ctable, 256, p, 0, 1);
         }
         else if (action==LUI_ALPHA_CHANGE) {
            /* recompute min/max? */
         }
      }
      else if (current_graphic==VIS5D_TOPO) {
         unsigned int *ctable;
         int cvar;
         vis5d_get_topo_color_var( current_ctx, &cvar );
         if (cvar>=0) {
            vis5d_get_color_table_address( current_ctx, VIS5D_TOPO,
                                           cvar, &ctable );
            vis5d_get_color_table_params( current_ctx, VIS5D_TOPO, cvar, &p);
         }
         if (action==LUI_RGB_RESET) {
            if (cvar<0) {
               vis5d_reset_topo_colors( current_ctx );
            }
            else {
               vis5d_color_table_init_params( p, 1, 0 );
               vis5d_color_table_recompute( ctable, 256, p, 1, 0);
            }
         }
         else if (action==LUI_ALPHA_RESET) {
            if (cvar<0) {
               vis5d_reset_topo_colors( current_ctx );
            }
            else {
               vis5d_color_table_init_params( p, 0, 1 );
               vis5d_color_table_recompute( ctable, 256, p, 0, 1);
            }
         }
         else if (action==LUI_RGB_SHAPE && cvar>=0) {
            vis5d_color_table_recompute( ctable, 256, p, 1, 0);
         }
         else if (action==LUI_ALPHA_SHAPE && cvar>=0) {
            vis5d_color_table_recompute( ctable, 256, p, 0, 1);
         }
         else if (action==LUI_ALPHA_CHANGE) {
            /* recompute min/max? */
         }
      }
      else {
         abort();
      }

      /* Now actually redraw the color curves */
      LUI_ColorBarRedraw( cb );
   }

   /* redraw window in any case */
   vis5d_signal_redraw( current_ctx, 1 );
   return 0;
}



/*
 * Make the isosurface color widget window.  This is only called once.
 */
void make_isocolor_window( GuiContext gtx )
{
   LUI_BUTTON_MATRIX *bm;
   LUI_NEWSLIDER *s;
   LUI_NEWBUTTON *b;
   char varname[20];
   int numvars;
   int i;
   float red=1.0, green=1.0, blue=1.0;
   int index = gtx->context_index;
   int subx, suby;
   Window w;

   vis5d_get_numvars( index, &numvars );

   current_ctx = gtx->context_index;

   w = LUI_CreateWindowAt( LUI_RootWindow, 10, 800, 380, 182 );
   gtx->isocolor_window = w;
/*   set_mwm_border( GuiDpy, w, MWM_DECOR_BORDER | MWM_DECOR_TITLE );*/

/*JUNK
   LUI_NewLabelCreate( w, LUI_LEFT, LUI_TOP, 380, 20,
                       "Isosurface of SPD:" );
   s = LUI_NewSliderCreate( w, LUI_LEFT, LUI_NEXT_Y, 340 );
   LUI_NewSliderChange( s, "Isolevel", 0.0, 78.0, 44.0 );

   LUI_PushButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 30, 36, "OK" );
   bm = LUI_ButtonMatrixCreate( w, LUI_LEFT, LUI_NEXT_Y, 100, 154, 1);
*/

   gtx->isocolor_label = LUI_NewLabelCreate( w, LUI_LEFT, LUI_TOP, 310, 20,
                                             "default label" );

   b = LUI_PushButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 60, 20, "Close" );
   LUI_ButtonCallback( b, close_cb );

   /* create the button matrix (actually 1-column list) of variable buttons */
   bm = LUI_ButtonMatrixCreate( w, LUI_LEFT, LUI_NEXT_Y, 100, 154, 1);
   gtx->iso_button_matrix = bm;
   for (i=-1;i<numvars;i++) {
      char *labels[1];
      if (i==-1) {
         labels[0] = "MonoColor";
      }
      else {
         vis5d_get_var_name( index, i, varname );
         labels[0] = varname;
      }
      LUI_ButtonMatrixAddRow( bm, labels, &red, &green, &blue );
   }
   LUI_ButtonMatrixCallback( bm, callback );
   LUI_ButtonMatrixSetState( bm, curbutton, 0, 1 );

   subx = LUI_LayoutGet( LUI_NEXT_X );
   suby = LUI_LayoutGet( LUI_SAME_Y );

   gtx->isocolor_subwin1 = LUI_CreateWindowAt( w, subx, suby, 280, 200 );

   /* red slider */
   s = LUI_NewSliderCreate( gtx->isocolor_subwin1, 0, 0, 270 );
   gtx->iso_red_slider = s;
   LUI_NewSliderChange( s, "Red", 0.0, 1.0, 1.0 );
   LUI_NewSliderCallback( s, red_slider_cb );

   /* green slider */
   s = LUI_NewSliderCreate( gtx->isocolor_subwin1, 0, LUI_NEXT_Y, 270 );
   gtx->iso_green_slider = s;
   LUI_NewSliderChange( s, "Green", 0.0, 1.0, 1.0 );
   LUI_NewSliderCallback( s, green_slider_cb );

   /* blue slider */
   s = LUI_NewSliderCreate( gtx->isocolor_subwin1, 0, LUI_NEXT_Y, 270 );
   gtx->iso_blue_slider = s;
   LUI_NewSliderChange( s, "Blue", 0.0, 1.0, 1.0 );
   LUI_NewSliderCallback( s, blue_slider_cb );

   /* alpha slider */
   s = LUI_NewSliderCreate( gtx->isocolor_subwin1, 0, LUI_NEXT_Y, 270 );
   gtx->iso_alpha_slider = s;
   LUI_NewSliderChange( s, "Opacity", 0.0, 1.0, 1.0 );
   LUI_NewSliderCallback( s, alpha_slider_cb );


   XMapWindow( GuiDpy, gtx->isocolor_subwin1 );

   LUI_FrameWidth(4);
   LUI_ColorBarPacking( PACK_COLOR('R','G','B','A') );

   gtx->isocolor_subwin2 = LUI_CreateWindowAt( w, subx, suby, 280, 200 );
   gtx->iso_colorbar = LUI_ColorBarCreate( gtx->isocolor_subwin2,
                                                 0, 0, 270, 154 );

   LUI_ColorBarCallback( gtx->iso_colorbar, colorbar_callback );

   XMapWindow( GuiDpy, bm->mainwindow );
}




/* Unmap/hide the isosurface color window */
void hide_isocolor_window( GuiContext gtx )
{
   XUnmapWindow( GuiDpy, gtx->isocolor_window );
   current_ctx = current_graphic = current_var = -1;
}



/* Map/display the isosurface color window */
void show_isocolor_window( GuiContext gtx, int graphic, int var )
{
   float r, g, b, a;
   unsigned int *color;
   int numvars;
   int colorvar;
   int row;
   unsigned int *table;
   float min, max;
   char varname[20];
   char label[100];
   float *p;

   /* unmap window in case it's already displayed */
   XUnmapWindow( GuiDpy, gtx->isocolor_window );

   if (current_ctx==gtx->context_index && current_graphic==graphic
       && current_var==var) {
      /* toggle off */
      current_ctx = current_graphic = current_var = -1;
      return;
   }

   current_ctx = gtx->context_index;
   current_graphic = graphic;
   current_var = var;

   /* set the label string */
   if (graphic==VIS5D_ISOSURF) {
      vis5d_get_var_name( gtx->context_index, var, varname );
      sprintf( label, "%s isosurface color:", varname );
   }
   else if (graphic==VIS5D_TRAJ) {
      sprintf( label, "Trajectory set %d color:", var+1 );
   }
   else if (graphic==VIS5D_TOPO) {
      sprintf( label, "Topography color:" );
   }
   else {
      abort();
   }
   LUI_NewLabelChangeText( gtx->isocolor_label, label );

   /* Set RGBA sliders to appropriate color values */
   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   LUI_NewSliderSetValue( gtx->iso_red_slider, r );
   LUI_NewSliderSetValue( gtx->iso_green_slider, g );
   LUI_NewSliderSetValue( gtx->iso_blue_slider, b );
   LUI_NewSliderSetValue( gtx->iso_alpha_slider, a );

   /* turn off previous button */
   vis5d_get_numvars( current_ctx, &numvars );
   for (row=0;row<numvars+1;row++) {
      LUI_ButtonMatrixSetState( gtx->iso_button_matrix, row, 0, 0 );
   }

   /* turn on current button */
   /* and load colorbar widget with apropriate color table */
   if (current_graphic==VIS5D_ISOSURF) {
      vis5d_get_isosurface_color_var( current_ctx, current_var, &colorvar );
      if (colorvar>=0) {
         vis5d_get_color_table_address( current_ctx, VIS5D_ISOSURF,
                                       colorvar, &table );
         vis5d_get_color_table_params(current_ctx, VIS5D_ISOSURF, colorvar, &p);
         vis5d_get_var_name( current_ctx, colorvar, varname );
         vis5d_get_var_range( current_ctx, colorvar, &min, &max );
         LUI_ColorBarChange( gtx->iso_colorbar,
                             varname, min, max, table, 255, p);
      }
   }
   else if (current_graphic==VIS5D_TRAJ) {
      vis5d_get_trajectory_color_var( current_ctx, current_var, &colorvar );
      if (colorvar>=0) {
         vis5d_get_color_table_address( current_ctx, VIS5D_TRAJ,
                                       colorvar, &table );
         vis5d_get_color_table_params(current_ctx, VIS5D_TRAJ, colorvar, &p);
         vis5d_get_var_name( current_ctx, colorvar, varname );
         vis5d_get_var_range( current_ctx, colorvar, &min, &max );
         LUI_ColorBarChange( gtx->iso_colorbar,
                             varname, min, max, table, 255, p);
      }
   }
   else if (current_graphic==VIS5D_TOPO) {
      vis5d_get_topo_color_var( current_ctx, &colorvar );
      vis5d_get_color_table_address( current_ctx, VIS5D_TOPO,
                                     colorvar, &table );
      vis5d_get_color_table_params(current_ctx, VIS5D_TOPO, colorvar, &p);
      if (colorvar>=0) {
         vis5d_get_var_name( current_ctx, colorvar, varname );
         vis5d_get_var_range( current_ctx, colorvar, &min, &max );
      }
      else {
         vis5d_get_topo_range( current_ctx, &min, &max );
         strcpy( varname, "Elevation" );
      }
      LUI_ColorBarChange( gtx->iso_colorbar,
                          varname, min, max, table, 255, p);
   }
   else {
      /* should never get here */
      abort();
   }

   if (current_graphic==VIS5D_TOPO) {
      LUI_ButtonMatrixChangeLabel( gtx->iso_button_matrix, 0, 0, "Height" );
   }
   else {
      LUI_ButtonMatrixChangeLabel( gtx->iso_button_matrix, 0, 0, "MonoColor" );
   }

   LUI_ButtonMatrixSetState( gtx->iso_button_matrix, colorvar+1, 0, 1 );
   curbutton = colorvar+1;

   /* show either color sliders or color widget */
   XUnmapWindow( GuiDpy, gtx->isocolor_subwin1 );
   XUnmapWindow( GuiDpy, gtx->isocolor_subwin2 );
   if (colorvar>=0 || current_graphic==VIS5D_TOPO) {
      XMapWindow( GuiDpy, gtx->isocolor_subwin2 );
   }
   else {
      XMapWindow( GuiDpy, gtx->isocolor_subwin1 );
   }

   XMapWindow( GuiDpy, gtx->isocolor_window );
}



/*
 * Call this function to udpate the sliders, color curves to current values.
 * This is usefull after a Vis5D RESTORE operation.
 */
void refresh_isocolor_window( GuiContext gtx )
{
   if (current_ctx>=0 && current_graphic>=0 && current_var>=0) {
      show_isocolor_window( gtx, current_graphic, current_var );
   }
}

