/*
   Siag, Scheme In A Grid
   Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <X11/cursorfont.h>
#include <X11/keysym.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>	/* for XtNtitle */
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Paned.h>
#include <X11/xpm.h>

#include "../siod/siod.h"

#include "../common/common.h"
#include "../common/cmalloc.h"
#include "../common/fonts.h"

#include "../common/bitmaps/rbm.xbm"
#include "../common/bitmaps/siag_fg.xpm"

#include "../xcommon/Table.h"
#include "../xcommon/Siag.h"
#include "../xcommon/xfonts.h"
#include "../xcommon/embed.h"
#include "../xcommon/dialogs.h"
#include "../xcommon/DragAndDrop.h"
#include "../xcommon/icon.h"
#include "../xcommon/xcommon.h"
#include "../xcommon/plugin.h"
#include "../xcommon/Textentry.h"
#include "../xcommon/Hsep.h"
#include "../xcommon/Vsep.h"

#include "../siag/types.h"
#include "../siag/calc.h"

#include "xsiag.h"

String fallback_resources[] = {
#include "app-defaults.h"
	NULL
};

typedef struct {
	Boolean plugin;
	Boolean grid_only;
} AppData;

static AppData app_data;

#define XtNplugin "plugin"
#define XtCPlugin "Plugin"
#define XtNgridOnly "gridOnly"
#define XtCGridOnly "GridOnly"

static XtResource resources[] = {
	{
		XtNplugin,
		XtCPlugin,
		XtRBoolean,
		sizeof(Boolean),
		XtOffsetOf(AppData, plugin),
		XtRImmediate,
		(XtPointer)False,
	}, {
		XtNgridOnly,
		XtCGridOnly,
		XtRBoolean,
		sizeof(Boolean),
		XtOffsetOf(AppData, grid_only),
		XtRImmediate,
		(XtPointer)False
	}
};

static XrmOptionDescRec options[] = {
	{"-plugin", "*plugin", XrmoptionNoArg, "True"},
	{"-gridonly", "*gridOnly", XrmoptionNoArg, "True"}
};

window *w_list;

int pr_scr_flag;
int grid_lines = 1;

int macro_flag;

textbuf kbd_macro = {0, 0, 0};

static XtAppContext app_context;

Widget topLevel;

static Widget topbox, gridpane,	text1, label1, label2, label3;

static Widget btnFont, mnuFont, btnSize, mnuSize, btnStyle, mnuStyle, btnColor, mnuColor;
static Widget cmdBold, cmdItalic, cmdHLeft, cmdHCenter, cmdHRight;
static Widget cmdBorders, cmdGrid, cmdUline, cmdNone;
static Widget shortcuts;

extern int recalc;	/* number of iterations */

enum {ABORT=0, DONE, WAITING, GOLEFT, GORIGHT, GOUP, GODOWN};

static int status;

static void DialogCancelAction(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	status = ABORT;
}

static void DialogDoneAction(Widget w, XEvent *event,
			String *params, Cardinal *n)
{
	status = DONE;
	if (*n == 1) {
		if (!strcmp(*params, "Left")) status = GOLEFT;
		else if (!strcmp(*params, "Right")) status = GORIGHT;
		else if (!strcmp(*params, "Up")) status = GOUP;
		else if (!strcmp(*params, "Down")) status = GODOWN;
	}
}

static XtActionsRec input_actions[] =
{
	{"celledit-done", DialogDoneAction},
	{"celledit-cancel", DialogCancelAction}
};

static void draw_input(char *text)
{
	long r, c;
	int type, intp;
	char b[1024];

	if (label1 == None) return;

	r = w_list->point_pos.row;
	c = w_list->point_pos.col;
	sprintf(b, "[%ld,%ld]", r, c);
	label_set(label1, b);
	type = ret_type(w_list->buf, r, c);
	intp = ret_interpreter(w_list->buf, r, c);
	switch (type) {
	case EXPRESSION:
	case STRING:
		sprintf(b, "%s", interpreter2name(intp));
		break;
	case LABEL:
		strcpy(b, "LABEL");
		break;
	case EMPTY:
		strcpy(b, "EMPTY");
		break;
	default:
		strcpy(b, "ERROR");
	}
	label_set(label3, b);
	XtVaSetValues(text1,
		XtNstring, text,
		(char *)0);
}

static void draw_status(char *text)
{
	if (label2 != None) {
		label_set(label2, text);
		XFlush(XtDisplay(label2));
	}
}

/* Print the string p on the bottom line of the screen.  If p is empty and
   the last string printed was also empty, the string isn't printed. */
void llpr(char *p)
{
	static int isclear = FALSE;

	if (isclear && p[0] == '\0')
		return;
	isclear = (p[0] == '\0');

	draw_status(p);
}

static void draw_vbar(window *w)
{
	double float_value;

	if (w->ui->vscroll != None) {
		float_value = w->top.row;
		float_value /= BUFFER_ROWS;

		XawScrollbarSetThumb(w->ui->vscroll, float_value, 0.0);
	}
}

static void draw_hbar(window *w)
{
	double float_value;

	if (w->ui->hscroll != None) {
		float_value = w->top.col;
		float_value /= BUFFER_COLS;

		XawScrollbarSetThumb(w->ui->hscroll, float_value, 0.0);
	}
}

static void draw_scrollbars(window *w)
{
	draw_vbar(w);
	draw_hbar(w);
}

static char *rownum_text(buffer *buf, int row, int col)
{
	static char b[80];
	sprintf(b, "%d", row);
	return b;
}

/* convert digit number to "text" number
	1 => A
	26 => Z
	27 => AA
	52 => AZ
	53 => BA
   1. decrement number by one
   2. use mod 26 for rightmost digit. A is 0.
   3. Continue with rest, if any
*/
static char *colnum_text(buffer *buf, int row, int col)
{
	static char b[80];

	if (a1_refs) {
		static char letters[] = " ABCDEFGHIJKLMNOPQRSTUVWXYZ";
		int i = 0, j = 0, c;
		while (col) {
			int digit = col % 26;
			if (!digit) digit = 26;
			b[i++] = letters[digit];
			col -= digit;
			col /= 26;
		}
		b[i--] = '\0';
		while (j < i) {
			c = b[i];
			b[i--] = b[j];
			b[j++] = c;
		}
	} else {
		sprintf(b, "%d", col);
	}
	return b;
}

static unsigned long rowcol_format(buffer *buf, int row, int col)
{
	return HELVETICA | SIZE_12 /*| BOLD */| HADJ_CENTER;
}

/* 970427: prot */
/* 970812: Table widget */
static void draw_colnums(window *w)
{
	if (w->ui->colnum != None) {
		XtVaSetValues(w->ui->colnum,
			XtNtableProtCol, w->prot.col,
			XtNtableTopCol, w->top.col,
			XtNtableData, w->buf,
			XtNtableRedisplay, True,
			(char *)NULL);
	}
}

static void draw_rownums(window *w)
{
	if (w->ui->rownum != None) {
		XtVaSetValues(w->ui->rownum,
			XtNtableProtRow, w->prot.row,
			XtNtableTopRow, w->top.row,
			XtNtableData, w->buf,
			XtNtableRedisplay, True,
			(char *)NULL);
	}
}

/*
How to make Siag always draw the top row and/or the left column.

This is desirable to keep headers on the screen when the sheet
is scrolled. To make this possible, I need to add a field in
the window structure to specify the first unprotected cell.
Everything above and to the left of this cell (called w->prot)
should always be drawed, regardless of what top is.

While this sounds easy, it profoundly changes the mapping from
cell coordinates to screen coordinates and vice versa. For instance,
answering the question "is the cursor on the screen" is no longer
as easy.

Make top be the first unprotected column. Point can never leave
the unprotected area. Leave mark and other positions for now.
*/

static int cell_type(buffer *b, int row, int col)
{

	if (ret_type(b, row, col) == EMBED)
		return TABLE_EMBED;

	return TABLE_TEXT;
}

static char *cell_text(buffer *b, int row, int col)
{
	static char s[1024];
	s[0] = '\0';
	ret_pvalue(s, b, row, col, -1);
	return s;
}

static unsigned long cell_format(buffer *b, int row, int col)
{
	return ret_format(b, row, col);
}

/* 970427: First shot at implementing this protection */
/* 970812: Use the Table widget's own drawing routines */
static void draw_cells(window *w)
{
	int top_row, top_col;

	/* this is more verbose than necessary, but I want to set
	   every resource there is just to see if they work ;-) */
	XtVaSetValues(w->ui->grid,
		XtNtableMaxRow, BUFFER_ROWS,
		XtNtableMaxCol, BUFFER_COLS,
		XtNtableProtRow, w->prot.row,
		XtNtableProtCol, w->prot.col,
		XtNtableTopRow, w->top.row,
		XtNtableTopCol, w->top.col,
		XtNtableSelectTopRow, w->blku.row,
		XtNtableSelectBottomRow, w->blkl.row,
		XtNtableSelectLeftCol, w->blku.col,
		XtNtableSelectRightCol, w->blkl.col,
		XtNtablePointRow, w->point_pos.row,
		XtNtablePointCol, w->point_pos.col,
		XtNtableRowHeight, cell_height,
		XtNtableColWidth, cell_width,
		XtNtableType, cell_type,
		XtNtableText, cell_text,
		XtNtableFormat, cell_format,
		XtNtableData, w->buf,
		XtNtableRedisplay, True,
		XtNtableGridLines, grid_lines,
		(char *)NULL);
	/* there is a chance that this moved top */
	XtVaGetValues(w->ui->grid,
		XtNtableTopRow, &top_row,
		XtNtableTopCol, &top_col,
		(char *)NULL);
	if (top_row != w->top.row && w->ui->rownum != None) {
		w->top.row = top_row;
		XtVaSetValues(w->ui->rownum,
			XtNtableTopRow, top_row,
			XtNtableRedisplay, !pr_scr_flag,
			(char *)NULL);
	}
	if (top_col != w->top.col && w->ui->colnum != None) {
		w->top.col = top_col;
		XtVaSetValues(w->ui->colnum,
			XtNtableTopCol, top_col,
			XtNtableRedisplay, !pr_scr_flag,
			(char *)NULL);
	}
}

int ask_for_str_comp(char *prompt, char *buffr, int (*comp)(char *))
{
	XtAppContext app_context = XtWidgetToApplicationContext(topLevel);
	String string;

	return dialog_input(topLevel, prompt, buffr, comp);

	if (text1 == None)
		return dialog_input(topLevel, prompt, buffr, comp);

	status = WAITING;

	XtVaSetValues(text1,
		XtNdisplayCaret, True,
		XtNstring, buffr,
		XtNinsertPosition, strlen(buffr), (char *)NULL);

	XtAddGrab(text1, True, False);

	XtSetKeyboardFocus(topLevel, text1);

	while (status == WAITING) {
		XEvent event_return;

		label_set(label1, prompt);

		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}

	XtVaGetValues(text1,
		XtNstring, &string, (char *)NULL);
	strcpy(buffr, string);	/* no check on length... */
	XtVaSetValues(text1,
		XtNdisplayCaret, False,
		XtNstring, "", (char *)NULL);

	XtRemoveGrab(text1);

	activate_window(w_list);
	return status;
}

int edit_cell(char *prompt, char *buffr)
{
	XtAppContext app_context = XtWidgetToApplicationContext(topLevel);
	String string;
	Widget edit_text;
	int x, y;
	Dimension w, h;
	buffer *buf = w_list->buf;
	int r = w_list->point_pos.row;
	int c = w_list->point_pos.col;

	x = y = 0;

	get_cell_coords(w_list, w_list->top.row, w_list->top.col, r, c, &x, &y);
	x--;
	y--;

	w = cell_width(buf, c)-1;
	h = cell_height(buf, r)-1;

	edit_text = XtVaCreateManagedWidget("edit_text",
		textentryWidgetClass, w_list->ui->grid,
		XtNx, x,
		XtNy, y,
		XtNwidth, w,
		XtNheight, h,
		(char *)0);

	status = WAITING;

	XtVaSetValues(edit_text,
		XtNx, x,
		XtNy, y,
		XtNwidth, w,
		XtNheight, h,
		(char *)0);
	XtVaSetValues(edit_text,
		XtNdisplayCaret, True,
		XtNstring, buffr,
		XtNinsertPosition, strlen(buffr), (char *)NULL);

	XtAddGrab(edit_text, False, False);
	XtSetKeyboardFocus(topLevel, edit_text);

	XtVaSetValues(edit_text,
		XtNinsertPosition, strlen(buffr), (char *)0);

	while (status == WAITING) {
		XEvent event_return;
		int n;

		label_set(label1, prompt);
		XtVaGetValues(edit_text,
			XtNstring, &string,
			XtNinsertPosition, &n,
			(char *)NULL);
		if (text1 != None) {
			XtVaSetValues(text1,
				XtNstring, string,
				XtNinsertPosition, n,
				(char *)NULL);
		}

		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}

	XtVaGetValues(edit_text,
		XtNstring, &string, (char *)NULL);
	strcpy(buffr, string);	/* no check on length... */
	if (text1 != None) {
		XtVaSetValues(text1,
			XtNdisplayCaret, False,
			XtNstring, "", (char *)NULL);
	}

	XtRemoveGrab(edit_text);
	XtDestroyWidget(edit_text);

	activate_window(w_list);
	return status;
}

/* This particular completion function doesn't complete at all, it just
   returns TRUE, making TAB equivalent to RET and LFD. */
static int nocomp(char *b)
{
	return TRUE;
}

/* Calls ask_for_str_comp with nocomp as completion function. */
/* 95-06-29: changed "buffer" to "buffr" to please gcc */
int ask_for_str(char *prompt, char *buffr)
{
	return ask_for_str_comp(prompt, buffr, nocomp);
}

int add_str_to_input_queue(textbuf buf)
{
        return FALSE;
}

static void set_block(int row1, int col1, int row2, int col2)
{
	int ur = w_list->blku.row, uc = w_list->blku.col;
	int lr = w_list->blkl.row, lc = w_list->blkl.col;

	if (row1 < 1) row1 = 1;
	if (row1 > BUFFER_ROWS) row1 = BUFFER_ROWS;
	if (col1 < 1) col1 = 1;
	if (col1 > BUFFER_COLS) col1 = BUFFER_COLS;
        if (row2 < 1) row2 = 1;
        if (row2 > BUFFER_ROWS) row2 = BUFFER_ROWS;
        if (col2 < 1) col2 = 1;
        if (col2 > BUFFER_COLS) col2 = BUFFER_COLS;

	if (row1 < row2) {
		w_list->blku.row = row1;
		w_list->blkl.row = row2;
	} else {
		w_list->blku.row = row2;
		w_list->blkl.row = row1;
	}
	if (col1 < col2) {
		w_list->blku.col = col1;
		w_list->blkl.col = col2;
	} else {
		w_list->blku.col = col2;
		w_list->blkl.col = col1;
	}


	/* Redraw if any change */
	if (ur != w_list->blku.row || uc != w_list->blku.col ||
		lr != w_list->blkl.row || lc != w_list->blkl.col) {
		pr_scr_flag = TRUE;
	}


	/* Become selection owner */
	/* this function should be integrated with the one in cmds.c */
	if (XtOwnSelection(w_list->ui->grid, XA_PRIMARY,
		CurrentTime, convert_proc,
		lose_ownership_proc, NULL) == False) {
		XtWarning("Siag: failed to become selection owner\n");
		w_list->blku.row = w_list->blku.col = -1;
		w_list->blkl.row = w_list->blkl.col = -1;
	}
}

#ifndef ABS
#define ABS(a) ((a)>0?(a):-(a))
#endif

static int colnum_grab(Widget w)
{
	int x = -1;
	int owner_events = True;
	unsigned int event_mask = ButtonReleaseMask |
				PointerMotionMask;
	int pointer_mode = GrabModeAsync;
	int keyboard_mode = GrabModeAsync;
	Window confine_to = XtWindow(w);
	static Cursor cursor = None;
	Time time = CurrentTime;
	int waiting = True;
	if (cursor == None)
		cursor = XCreateFontCursor(XtDisplay(w), XC_right_side);
	XtGrabPointer(w, owner_events, event_mask,
		pointer_mode, keyboard_mode, confine_to, cursor, time);
	while (waiting) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		if (event_return.type == ButtonRelease) {
			waiting = False;
		} else if (event_return.type == MotionNotify) {
			x = event_return.xmotion.x;
		} else {
			XtDispatchEvent(&event_return);
		}
	}
	XtUngrabPointer(w, CurrentTime);
	return x;
}

static void ColnumButtonAction(Widget w,
		XEvent *event, String *params, Cardinal *n)
{
	int col, row, x, y;

	x = event->xbutton.x;
	y = 0;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
			&row, &col, x, y);
	if (*n < 1 || !strcmp(params[0], "set")) {
		/* If we are within 4 pixels from a cell boundary, grab the
		   pointer and move the border to wherever we release it. */
		int x1, y1, w1;
		get_cell_coords(w_list, w_list->top.row, w_list->top.col,
			row, col, &x1, &y1);
		w1 = x-x1;
		if (cell_width(w_list->buf, col)-w1 < 5) {
			x = colnum_grab(w);
		w1 = x-x1;
			if (w1 > 5 && w1 < 500)
				set_width(w_list->buf, col, w1);
		} else {
		/* Otherwise do everything as before */
			set_point_col(w_list, col);
			set_block(1, col, BUFFER_ROWS, col);
		}
		pr_scr_flag = TRUE;

	}
	else if (!strcmp(params[0], "left"))
		w_list->blku.col = col;
	else if (!strcmp(params[0], "right"))
		w_list->blkl.col = col;
	else if (!strcmp(params[0], "select")) {
		int r1 = w_list->blku.row, c1 = w_list->blku.col;
		int r2 = w_list->blkl.row, c2 = w_list->blkl.col;
		/* is the selection set to something already? */
		if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1)
			set_block(1, col,
				BUFFER_ROWS,
				get_point(w_list).col);
		else {
			/* which corner is closer? */
			if (ABS(c1-col) < ABS(c2-col))
				set_block(1, col, BUFFER_ROWS, c2);
			else
				set_block(1, c1, BUFFER_ROWS, col);
		}
	}
	else if (!strcmp(params[0], "adjust")) {
		set_block(1, col,
			BUFFER_ROWS,
			get_point(w_list).col);
	}
	show_cur(w_list);
}

static int rownum_grab(Widget w)
{
	int y = -1;
	int owner_events = True;
	unsigned int event_mask = ButtonReleaseMask |
				PointerMotionMask;
	int pointer_mode = GrabModeAsync;
	int keyboard_mode = GrabModeAsync;
	Window confine_to = XtWindow(w);
	static Cursor cursor = None;
	Time time = CurrentTime;
	int waiting = True;
	if (cursor == None)
		cursor = XCreateFontCursor(XtDisplay(w), XC_bottom_side);
	XtGrabPointer(w, owner_events, event_mask,
		pointer_mode, keyboard_mode, confine_to, cursor, time);
	while (waiting) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		if (event_return.type == ButtonRelease) {
			waiting = False;
		} else if (event_return.type == MotionNotify) {
			y = event_return.xmotion.y;
		} else {
			XtDispatchEvent(&event_return);
		}
	}
	XtUngrabPointer(w, CurrentTime);
	return y;
}

static void RownumButtonAction(Widget w,
		XEvent * event, String * params, Cardinal * n)
{
	int col, row, x, y;

	x = 0;
	y = event->xbutton.y;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
			&row, &col, x, y);
	if (*n < 1 || !strcmp(params[0], "set")) {
		/* If we are within 4 pixels from a cell boundary, grab the
		   pointer and move the border to wherever we release it. */
		int x1, y1, h1;
		get_cell_coords(w_list, w_list->top.row, w_list->top.col,
			row, col, &x1, &y1);
		h1 = y-y1;
		if (cell_height(w_list->buf, row)-h1 < 5) {
			y = rownum_grab(w);
		h1 = y-y1;
			if (h1 > 5 && h1 < 500)
				set_height(w_list->buf, row, h1);
		} else {
		/* Otherwise do everything as before */
			set_point_row(w_list, row);
			set_block(row, 1, row, BUFFER_COLS);
		}
		pr_scr_flag = TRUE;

	}
	else if (!strcmp(params[0], "top"))
		w_list->blku.row = row;
	else if (!strcmp(params[0], "bottom"))
		w_list->blkl.row = row;
	else if (!strcmp(params[0], "select")) {
		int r1 = w_list->blku.row, c1 = w_list->blku.col;
		int r2 = w_list->blkl.row, c2 = w_list->blkl.col;
		/* is the selection set to something already? */
		if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1)
			set_block(row, 1,
				get_point(w_list).row,
				BUFFER_COLS);
		else {
			/* which corner is closer? */
			if (ABS(r1-row) < ABS(r2-row))
				set_block(row, 1, r2, BUFFER_COLS);
			else
				set_block(r1, 1, row, BUFFER_COLS);
		}
	}
	else if (!strcmp(params[0], "adjust")) {
		set_block(row, 1,
			get_point(w_list).row,
			BUFFER_COLS);
	}
	show_cur(w_list);
}

static void DestroyNotifyAction(Widget w,
		XEvent *event, String *params, Cardinal *n)
{
	XDestroyWindowEvent dnevent = event->xdestroywindow;
	fprintf(stderr, "Window %lx was destroyed\n",
		(unsigned long)dnevent.window);
}

#define CTRL(c) ((c)&31)
#define CONTROL_MASK 4
#define ALT(c) ((c)|0x80)
#define ALT_MASK 8

static void KeyEventAction(Widget w,
		XEvent *event, String *params, Cardinal *n)
{
	int count, bufsiz = 10;
	char buf[12];
	KeySym keysym;
	XKeyEvent kevent;
	kevent = event->xkey;
	count = XLookupString(&kevent, buf, bufsiz, &keysym, NULL);

	lastc = keysym;
	switch (lastc) {
	case XK_Home:		lastc = CTRL('a'); break;
	case XK_Left:		lastc = CTRL('b'); break;
	case XK_Delete:		lastc = CTRL('d'); break;
	case XK_Up:		lastc = CTRL('p'); break;
	case XK_Right:		lastc = CTRL('f'); break;
	case XK_Down:		lastc = CTRL('n'); break;
	case XK_Page_Up:	lastc = ALT('v'); break;
	case XK_Page_Down:	lastc = CTRL('v'); break;
	case XK_End:		lastc = CTRL('e'); break;
	case XK_Tab:		lastc = '\t'; break;
	case XK_BackSpace:	lastc = '\b'; break;
	case XK_Linefeed:	lastc = '\n'; break;
	case XK_Clear:		lastc = '\f'; break;
	case XK_Return:		lastc = '\r'; break;
	case XK_Escape:		lastc = CTRL('[');
	default:		if (lastc > 255) return;
	}
	if (kevent.state & CONTROL_MASK) lastc = CTRL(lastc);
	if (kevent.state & ALT_MASK) lastc = ALT(lastc);
	do_cmd(lastc);
}

#define SCROLL_INTERVAL 100

static int scroll_timeout = None;

static void scroll_oneline(XtPointer client_data, XtIntervalId *id)
{
	Widget w = (Widget)client_data;
	XEvent event;
	scroll_timeout = None;
	event.xclient.type = ClientMessage;
	event.xclient.format = 8;
	event.xclient.window = XtWindow(w);

	XSendEvent(XtDisplay(w), XtWindow(w), True, NoEventMask, &event);
}

static void grid_grab(Widget w)
{
	int x, y, row, col, startrow, startcol, oldrow, oldcol;
	int owner_events = False;
	unsigned int event_mask = ButtonReleaseMask | PointerMotionMask |
			EnterWindowMask | LeaveWindowMask;
	int pointer_mode = GrabModeAsync;
	int keyboard_mode = GrabModeAsync;
	static Cursor cursor = None;
	Time time = CurrentTime;
	enum {START = 0, OUTSIDE, STOP} state = START;
	if (cursor == None)
		cursor = XCreateFontCursor(XtDisplay(w), XC_dot/*XC_sizing*/);
	XtGrabPointer(w, owner_events, event_mask,
		pointer_mode, keyboard_mode, None, cursor, time);
	startrow = oldrow = get_point(w_list).row;
	startcol = oldcol = get_point(w_list).col;
	while (state != STOP) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		switch (state) {
		case START:
			if (event_return.type == ButtonRelease) {
				state = STOP;
			} else if (event_return.type == MotionNotify) {
				/* figure out where we are an set selection */
				x = event_return.xmotion.x;
				y = event_return.xmotion.y;
				get_coords_cell(w_list, w_list->top.row,
					w_list->top.col, &row, &col, x, y);
				set_point_row(w_list, row);
				set_point_col(w_list, col);
				set_block(startrow, startcol, row, col);
				show_cur(w_list);
			} else if (event_return.type == LeaveNotify) {
				state = OUTSIDE;
				scroll_timeout = XtAppAddTimeOut(
					XtWidgetToApplicationContext(w),
					SCROLL_INTERVAL,
					scroll_oneline, (XtPointer)w);
			} else {
				XtDispatchEvent(&event_return);
			}
			break;
		case OUTSIDE:
			if (event_return.type == ButtonRelease) {
				state = STOP;
			} else if (event_return.type == EnterNotify) {
				state = START;
			} else if (event_return.type == ClientMessage &&
					scroll_timeout == None) {
				Window root, child;
				int rx, ry;
				unsigned int keys_buttons;
				Dimension width, height;
				XtVaGetValues(w,
					XtNwidth, &width,
					XtNheight, &height,
					(char *)0);
				XQueryPointer(XtDisplay(w), XtWindow(w),
					&root, &child, &rx, &ry, &x, &y,
					&keys_buttons);
#if 1
				if (x < 0) col--;
				if (x > width) col++;
				if (y < 0) row--;
				if (y > height) row++;
				if (row < 1) row = 1;
				if (row > BUFFER_ROWS) row = BUFFER_ROWS;
				if (col < 1) col = 1;
				if (col > BUFFER_COLS) col = BUFFER_COLS;
#else
				get_coords_cell(w_list, w_list->top.row,
					w_list->top.col, &row, &col, x, y);
#endif
				set_point_row(w_list, row);
				set_point_col(w_list, col);
				set_block(startrow, startcol, row, col);
				show_cur(w_list);
				scroll_timeout = XtAppAddTimeOut(
                                        XtWidgetToApplicationContext(w),
                                        SCROLL_INTERVAL,
                                        scroll_oneline, (XtPointer)w);
			} else {
				XtDispatchEvent(&event_return);
			}
			break;
		default:
			fprintf(stderr, "grid_grab: You shouldn't be here\n");
		}
	}
	if (scroll_timeout != None)
		XtRemoveTimeOut(scroll_timeout);
	scroll_timeout = None;
	XtUngrabPointer(w, CurrentTime);
}

static void GridButtonAction(Widget w,
		XEvent * event, String * params, Cardinal * n)
{
	int col, row;
	int x, y;

	x = event->xbutton.x;
	y = event->xbutton.y;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
			&row, &col, x, y);
	if (*n < 1 || !strcmp(params[0], "point")) {
		set_point_row(w_list, row);
		set_point_col(w_list, col);
		grid_grab(w);
	} else if (!strcmp(params[0], "mark")) {
		set_mark_row(w_list, row);
		set_mark_col(w_list, col);
	} else if (!strcmp(params[0], "block")) {
		set_block(row, col,
			get_mark(w_list).row, get_mark(w_list).col);
		pr_scr_flag = TRUE;
	}
	else if (!strcmp(params[0], "paste")) {
		set_point_row(w_list, row);
		set_point_col(w_list, col);
		XtGetSelectionValue(w, XA_PRIMARY, target_atom,
			requestor_callback, event, event->xbutton.time);
		pr_scr_flag = TRUE;
	}
	else if (!strcmp(params[0], "select")) {
		int r1 = w_list->blku.row, c1 = w_list->blku.col;
		int r2 = w_list->blkl.row, c2 = w_list->blkl.col;
		/* is the selection set to something already? */
		if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1)
			set_block(row, col,
				get_point(w_list).row,
				get_point(w_list).col);
		else {
			/* which corner is closer? */
			if (ABS(r1-row) < ABS(r2-row)) r1 = r2;
			if (ABS(c1-col) < ABS(c2-col)) c1 = c2;
			set_block(r1, c1, row, col);
		}
	}
	show_cur(w_list);
}

static LISP ltooltip_mode(LISP newmode)
{
	tooltip_mode_set(get_c_long(newmode), label2);
	return NIL;
}

static void siaghelp_action(Widget w, XEvent *event,
	String *params, Cardinal *num_params)
{
	char b[256];

	sprintf(b, "file:%s/siag/docs/%s", siagdocs, params[0]);
	if (!fork()) {
		execlp(siaghelp, "Siaghelp", b, (char *)NULL);
		exit(0);
	}
}

static void execute_callback(Widget w,
		XtPointer client_data, XtPointer call_data)
{
	execute((char *)client_data);
}

static void vscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
	float top;
	int gridtop;
	position p;

	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNtopOfThumb, &top, (char *)NULL);
	gridtop = top*BUFFER_ROWS;
	if (gridtop < 1) gridtop = 1;
	/* make sure protection is respected */
	/* is it necessary to move point here? */
	p.row = gridtop;
	p.col = w_list->point_pos.col;
	set_point(w_list, p);
	p.col = w_list->top.col;
	set_top(w_list, p);
	pr_scr_flag = TRUE;
	show_cur(w_list);
}

static void vscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
	int i = (int) call_data;
	Dimension length;

	if (i == 0) return;	/* no div by zero */
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNlength, &length, (char *)NULL);
	if (i < 0) {
		if ((length / -i) > 15)
			execute("(scroll-cell-down)");
		else
			execute("(scroll-down)");
	} else {
		if ((length / i) > 15)
			execute("(scroll-cell-up)");
		else
			execute("(scroll-up)");
	}
}

static void hscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
	float top;
	int gridtop;
	position p;

	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNtopOfThumb, &top, (char *)NULL);
	gridtop = top*BUFFER_COLS;
	if (gridtop < 1) gridtop = 1;
	p.col = gridtop;
	p.row = w_list->point_pos.row;
	set_point(w_list, p);
	p.row = w_list->top.row;
	set_top(w_list, p);
	pr_scr_flag = TRUE;
	show_cur(w_list);
}

static void hscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
	int i = (int) call_data;
	Dimension length;

	if (i == 0) return;	/* no div by zero */
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNlength, &length, (char *)NULL);
	if (i < 0) {
		if ((length / -i) > 15)
			execute("(scroll-cell-left)");
		else
			execute("(scroll-left)");
	} else {
		if ((length / i) > 15)
			execute("(scroll-cell-right)");
		else
			execute("(scroll-right)");
	}
}

static struct {
	char *label;
	Widget button, menu;
} *menubar;

static int menucount = 0;

static void make_menu(char *label)
{
	char button_name[80];

	if (topbox == None) return;

	sprintf(button_name, "btn%s", label);
	menubar = crealloc(menubar, (menucount+1)*(sizeof *menubar));
	menubar[menucount].label = cstrdup(label);
	menubar[menucount].button = XtVaCreateManagedWidget(button_name,
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop, (char *)NULL);
	label_set(menubar[menucount].button, label);
	if (menucount) {
		XtVaSetValues(menubar[menucount].button,
			XtNfromHoriz, menubar[menucount-1].button, (char *)NULL);
	}
	menubar[menucount].menu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, menubar[menucount].button, (char *)NULL);

	menucount++;
}

static void init_menu(void)
{
	make_menu("File");
	make_menu("Edit");
	make_menu("Block");
	make_menu("Format");
	make_menu("Data");
	make_menu("Plugin");
	make_menu("Window");
	make_menu("Plot");
	make_menu("Tools");
	make_menu("Help");
}

static Widget find_menu_by_name(char *label)
{
	int i;

	if (!cstrcasecmp("Shortcuts", label))
		return shortcuts;

	for (i = 0; i < menucount; i++) {
		if (!cstrcasecmp(menubar[i].label, label))
			return menubar[i].menu;
	}
	return NULL;
}

static LISP add_menu_entry(LISP menu, LISP label, LISP function)
{
	Widget entry, menuw;

	menuw = find_menu_by_name(get_c_string(menu));
	if (!menuw) {
#if 0
		fprintf(stderr, "add_menu_entry: No menu! Bailing out.\n");
#endif
		return NIL;
	}

	if (!strcmp(get_c_string(label), "-")) {	/* line pane */
		entry = XtCreateManagedWidget("-",
			smeLineObjectClass,
			menuw,
			NULL, 0);
	} else {
		entry = XtVaCreateManagedWidget(get_c_string(function),
			smeBSBObjectClass, menuw,
			XtNlabel, translate(get_c_string(label)),
			(char *)NULL);
		XtAddCallback(entry,
			XtNcallback, execute_callback,
			cstrdup(get_c_string(function)));
	}
	return NIL;
}

static struct {
	char *label;
	char *sublabel;
	Widget entry, menu;
} *submenus;

static int submenucount = 0;

static Widget find_submenu_by_name(char *label, char *sublabel)
{
	int i;

	for (i = 0; i < submenucount; i++) {
		if (!cstrcasecmp(submenus[i].label, label) &&
				!cstrcasecmp(submenus[i].sublabel, sublabel))
			return submenus[i].menu;
	}
	return None;
}

static Widget find_submenu_by_entry(Widget entry)
{
	int i;

	for (i = 0; i < submenucount; i++) {
		if (entry == submenus[i].entry)
			return submenus[i].menu;
	}
	return None;
}

static LISP add_submenu(LISP label, LISP sublabel)
{
	Widget menuw;
	static Pixmap rbm = None;

	menuw = find_menu_by_name(get_c_string(label));

	if (!menuw) {
#if 0
		fprintf(stderr, "add_submenu: No menu! Bailing out.\n");
#endif
		return NIL;
	}
	if (rbm == None) {
		rbm = XCreatePixmapFromBitmapData(XtDisplay(menuw),
			XDefaultRootWindow(XtDisplay(menuw)),
			rbm_bits, rbm_width, rbm_height, 1, 0, 1);
	}
	submenus = crealloc(submenus, (submenucount+1)*(sizeof *submenus));
	submenus[submenucount].label = cstrdup(get_c_string(label));
	submenus[submenucount].sublabel = cstrdup(get_c_string(sublabel));
	submenus[submenucount].entry =
		XtVaCreateManagedWidget(get_c_string(sublabel),
			smeBSBObjectClass, menuw,
			XtNlabel, translate(get_c_string(sublabel)),
			XtNrightBitmap, rbm,
			(char *)0);
	submenus[submenucount].menu = XtVaCreatePopupShell("submenu",
		simpleMenuWidgetClass, menuw, (char *)0);
	submenucount++;
	return NIL;
}

static LISP add_submenu_entry(LISP menu, LISP submenu,
				LISP label, LISP function)
{
	Widget entry, menuw;

#if 0
	if (app_data.grid_only) return NIL;
#endif

	menuw = find_submenu_by_name(get_c_string(menu),
				get_c_string(submenu));
	if (!menuw) {
#if 0
		fprintf(stderr, "add_submenu_entry: No menu! Bailing out.\n");
#endif
		return NIL;
	}

	if (!strcmp(get_c_string(label), "-")) {	/* line pane */
		entry = XtCreateManagedWidget("-",
			smeLineObjectClass,
			menuw,
			NULL, 0);
	} else {
		entry = XtVaCreateManagedWidget(get_c_string(function),
			smeBSBObjectClass, menuw,
			XtNlabel, translate(get_c_string(label)),
			(char *)NULL);
		XtAddCallback(entry,
			XtNcallback, execute_callback,
			cstrdup(get_c_string(function)));
	}
	return NIL;
}

static Widget tbNew, tbCopy;	/* needed as fromVert and fromHoriz */

static Widget make_toggle(char *name, char *cmd, Widget pw, 
	Widget below, Widget after, char *pm)
{
	Widget w;
	XpmAttributes xa;
	Pixmap pm_return;
	int result;
	XpmColorSymbol symbol;
	Pixel color;
	char fn[1024];

	if (pm[0] == '/') strcpy(fn, pm);
	else sprintf(fn, "%s/common/bitmaps/%s", siaghome, pm);

	XtVaGetValues(pw, XtNbackground, &color, (char *)0);
	xa.closeness = 40000;
	xa.exactColors = False;
	xa.valuemask = XpmCloseness | XpmExactColors;

	if (pw == None) return None;

	w = XtVaCreateManagedWidget(name,
		toggleWidgetClass, pw, (char *)NULL);
	symbol.name = NULL;
	symbol.value = "none";
	symbol.pixel = color;
	xa.colorsymbols = &symbol;
	xa.numsymbols = 1;
	xa.valuemask |= XpmColorSymbols;

	result = XpmReadFileToPixmap(XtDisplay(w),
		XRootWindowOfScreen(XtScreen(w)),
		fn, &pm_return, NULL, &xa);

	if (result != XpmSuccess) {
		fprintf(stderr, "XpmReadFileToPixmap returns %s\n",
                        XpmGetErrorString(result));
                return NULL;
        }

	XtVaSetValues(w,
		XtNfromVert, below,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNbitmap, pm_return,
		(char *)0);
	if (after) {
		XtVaSetValues(w,
			XtNfromHoriz, after, (char *)NULL);
	}
	XtAddCallback(w,
		XtNcallback, execute_callback, (XtPointer)cmd);
	return w;
}

static Widget make_command(char *name, char *cmd, Widget pw, 
	Widget below, Widget after, char *pm)
{
	Widget w;
	XpmAttributes xa;
	Pixmap pm_return;
	int result;
	XpmColorSymbol symbol;
	Pixel color;
	char fn[1024];

	if (pm[0] == '/') strcpy(fn, pm);
	else sprintf(fn, "%s/common/bitmaps/%s", siaghome, pm);

	XtVaGetValues(pw, XtNbackground, &color, (char *)0);
	xa.closeness = 40000;
	xa.exactColors = False;
	xa.valuemask = XpmCloseness | XpmExactColors;
	if (pw == None) return None;

	w = XtVaCreateManagedWidget(name,
		commandWidgetClass, pw, (char *)NULL);
	symbol.name = NULL;
	symbol.value = "none";
	symbol.pixel = color;
	xa.colorsymbols = &symbol;
	xa.numsymbols = 1;
	xa.valuemask |= XpmColorSymbols;

	result = XpmReadFileToPixmap(XtDisplay(w),
		XRootWindowOfScreen(XtScreen(w)),
		fn, &pm_return, NULL, &xa);

	if (result != XpmSuccess) {
		fprintf(stderr, "XpmReadFileToPixmap returns %s\n",
                        XpmGetErrorString(result));
                return NULL;
        }

	XtVaSetValues(w,
		XtNfromVert, below,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNbitmap, pm_return,
		(char *)0);

	if (after) {
		XtVaSetValues(w,
			XtNfromHoriz, after, (char *)NULL);
	}
	XtAddCallback(w,
		XtNcallback, execute_callback, (XtPointer)cmd);
	return w;
}

/* The toolbar */

#define BELOW_WIDGET1 hsep1
#define BELOW_WIDGET2 hsep2

static Widget hsep1, hsep2, hsep3;

static void init_toolbar(void)
{
	Widget tbOpen, tbSave, tbView, tbPrint, tbPlot, tbSum, tbHelp;

#if 1
	hsep1 = XtVaCreateManagedWidget("hsep1",
		hsepWidgetClass, topbox,
		XtNfromVert, menubar[0].button,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainTop,
		(char *)0);
#else	/* this actually works, but only for "real" Xaw3d */
	hsep1 = XtVaCreateManagedWidget("hsep1",
		labelWidgetClass, topbox,
		XtNfromVert, menubar[0].button,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainTop,
		(char *)0);
#endif

	tbNew = make_command("tbNew", "(new-siag)", topbox,
		BELOW_WIDGET1, NULL, "new.xpm");
	hsep2 = XtVaCreateManagedWidget("hsep2",
		hsepWidgetClass, topbox,
		XtNfromVert, tbNew,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainTop,
		(char *)0);

	tbOpen = make_command("tbOpen", "(load-buffer)", topbox,
		BELOW_WIDGET1, tbNew, "fld_open.xpm");
	tbSave = make_command("tbSave", "(save-buffer-as)", topbox,
		BELOW_WIDGET1, tbOpen, "floppy3.xpm");
	tbView = make_command("tbView", "(preview)", topbox,
		BELOW_WIDGET1, tbSave, "preview.xpm");
	tbPrint = make_command("tbPrint", "(print)", topbox,
		BELOW_WIDGET1, tbView, "printer.xpm");
	tbPlot = make_command("tbPlot", "(plot \"lines\")", topbox,
		BELOW_WIDGET1, tbPrint, "plotter.xpm");
	tbSum = make_command("tbSum", "(block-sum)", topbox,
		BELOW_WIDGET1, tbPlot, "sigma.xpm");
	tbHelp = make_command("tbHelp", "(help-contents)", topbox,
		BELOW_WIDGET1, tbSum, "info.xpm");
	tbCopy = make_command("tbCopy", "(help-copyright)", topbox,
		BELOW_WIDGET1, tbHelp, "copyright.xpm");
}

static void init_toggle(void)
{
	cmdBold = make_toggle("cmdBold", "(toggle-fontflag BOLD)", topbox,
		BELOW_WIDGET2, btnColor, "bold.xpm");
	cmdItalic = make_toggle("cmdItalic", "(toggle-fontflag ITALIC)", topbox,
		BELOW_WIDGET2, cmdBold, "italic.xpm");
	cmdHLeft = make_toggle("cmdHLeft", "(change-font HADJ_LEFT HADJ_MASK)",
		topbox, BELOW_WIDGET2, cmdItalic, "hleft.xpm");
	cmdHCenter = make_toggle("cmdHCenter", "(change-font HADJ_CENTER HADJ_MASK)",
		topbox, BELOW_WIDGET2, cmdHLeft, "hcenter.xpm");
	cmdHRight = make_toggle("cmdHRight", "(change-font HADJ_RIGHT HADJ_MASK)",
		topbox, BELOW_WIDGET2, cmdHCenter, "hright.xpm");
	cmdBorders = make_command("cmdBorders", "(block-borders 1)", topbox,
		BELOW_WIDGET1, tbCopy, "borders.xpm");
	cmdGrid = make_command("cmdGrid", "(block-borders 2)", topbox,
		BELOW_WIDGET1, cmdBorders, "grid.xpm");
	cmdUline = make_command("cmdUline", "(block-borders 3)", topbox,
		BELOW_WIDGET1, cmdGrid, "uline.xpm");
	cmdNone = make_command("cmdNone", "(block-borders 0)", topbox,
		BELOW_WIDGET1, cmdUline, "none.xpm");
}

static void font_menu(Widget w, char *label, char *cmd)
{
	Widget entry;
	if (w == None) return;

	entry = XtVaCreateManagedWidget(cmd,
			smeBSBObjectClass, w,
			(char *)NULL);
	XtAddCallback(entry,
		XtNcallback, execute_callback, cmd);
	label_set(entry, label);
}

static void setup_buttons(void)
{
	if (topbox == None) return;

	btnFont = XtVaCreateManagedWidget("btnFont",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, BELOW_WIDGET2,
		XtNwidth, 160,
		XtNlabel, "Font", (char *)NULL);

	mnuFont = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnFont, (char *)NULL);

	font_menu(mnuFont, "Courier",
		"(change-font COURIER FONT_MASK)");
	font_menu(mnuFont, "Helvetica",
		"(change-font HELVETICA FONT_MASK)");
	font_menu(mnuFont, "New Century Schoolbook",
		"(change-font NEW_CENTURY FONT_MASK)");
	font_menu(mnuFont, "Times",
		"(change-font TIMES FONT_MASK)");

	btnSize = XtVaCreateManagedWidget("btnSize",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, BELOW_WIDGET2,
		XtNfromHoriz, btnFont,
		XtNwidth, 40,
		XtNlabel, "Size", (char *)NULL);

	mnuSize = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnSize, (char *)NULL);

	font_menu(mnuSize, "8",
		"(change-font SIZE_8 SIZE_MASK)");
	font_menu(mnuSize, "10",
		"(change-font SIZE_10 SIZE_MASK)");
	font_menu(mnuSize, "12",
		"(change-font SIZE_12 SIZE_MASK)");
	font_menu(mnuSize, "14",
		"(change-font SIZE_14 SIZE_MASK)");
	font_menu(mnuSize, "18",
		"(change-font SIZE_18 SIZE_MASK)");
	font_menu(mnuSize, "20",
		"(change-font SIZE_20 SIZE_MASK)");
	font_menu(mnuSize, "24",
		"(change-font SIZE_24 SIZE_MASK)");
	font_menu(mnuSize, "30",
		"(change-font SIZE_30 SIZE_MASK)");

	btnStyle = XtVaCreateManagedWidget("btnStyle",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, BELOW_WIDGET2,
		XtNfromHoriz, btnSize,
		XtNwidth, 80,
		XtNlabel, "Style", (char *)NULL);

	mnuStyle = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnStyle, (char *)NULL);

	font_menu(mnuStyle, "Default",
		"(change-font FMT_DEFAULT FMT_MASK)");
	font_menu(mnuStyle, "Invisible",
		"(change-font FMT_INVISIBLE FMT_MASK)");
	font_menu(mnuStyle, "Integer",
		"(change-font FMT_INTEGER FMT_MASK)");
	font_menu(mnuStyle, "Scientific",
		"(change-font FMT_SCIENTIFIC FMT_MASK)");
	font_menu(mnuStyle, "Fixed",
		"(change-font FMT_FIXED FMT_MASK)");
	font_menu(mnuStyle, "Date",
		"(change-font FMT_DATE FMT_MASK)");
	font_menu(mnuStyle, "Time",
		"(change-font FMT_TIME FMT_MASK)");
	font_menu(mnuStyle, "Time Difference",
		"(change-font FMT_TIMEDIFF FMT_MASK)");
	font_menu(mnuStyle, "Percent",
		"(change-font FMT_PERCENT FMT_MASK)");
	font_menu(mnuStyle, "Hex",
		"(change-font FMT_HEX FMT_MASK)");
	font_menu(mnuStyle, "Currency",
		"(change-font FMT_CURRENCY FMT_MASK)");
	font_menu(mnuStyle, "User 1",
		"(change-font FMT_USER1 FMT_MASK)");
	font_menu(mnuStyle, "User 2",
		"(change-font FMT_USER2 FMT_MASK)");
	font_menu(mnuStyle, "User 3",
		"(change-font FMT_USER3 FMT_MASK)");
	font_menu(mnuStyle, "User 4",
		"(change-font FMT_USER4 FMT_MASK)");
	font_menu(mnuStyle, "User 5",
		"(change-font FMT_USER5 FMT_MASK)");

	btnColor = XtVaCreateManagedWidget("btnColor",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, BELOW_WIDGET2,
		XtNfromHoriz, btnStyle,
		XtNwidth, 80,
		XtNlabel, "Color", (char *)NULL);

	mnuColor = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnColor, (char *)NULL);

	font_menu(mnuColor, "Black",
		"(change-font COLOR_BLACK COLOR_MASK)");
	font_menu(mnuColor, "Red",
		"(change-font COLOR_RED COLOR_MASK)");
	font_menu(mnuColor, "Green",
		"(change-font COLOR_GREEN COLOR_MASK)");
	font_menu(mnuColor, "Blue",
		"(change-font COLOR_BLUE COLOR_MASK)");
	font_menu(mnuColor, "Yellow",
		"(change-font COLOR_YELLOW COLOR_MASK)");
	font_menu(mnuColor, "Magenta",
		"(change-font COLOR_MAGENTA COLOR_MASK)");
	font_menu(mnuColor, "Cyan",
		"(change-font COLOR_CYAN COLOR_MASK)");
	font_menu(mnuColor, "White",
		"(change-font COLOR_WHITE COLOR_MASK)");
}

static void drop_handler(Widget w, XtPointer data,
		XEvent *event, Boolean *b)
{
	unsigned char *Data;
	unsigned char *filename;
	unsigned long DataSize;
	int DataType = DndDataType(event);

	if (DataType == DndNotDnd) return;
	DndGetData(&Data, &DataSize);
	switch (DataType) {
	case DndRawData:
		/* This should allow dnd within siag(s) */
		break;
	case DndFile:
		/* load file */
		filename = Data;
		if (!fork()) {
			execlp("siag", "Siag", filename, (char *)NULL);
			exit(1);
		}
		break;
	case DndFiles:
		/* load several files */
		filename = Data;
		while (filename[0] != '\0') {
			if (!fork()) {
				execlp("siag", "Siag", filename, (char *)NULL);
				exit(1);
			}
			filename = filename+strlen((char *)filename)+1;
		}
		break;
	case DndText:
		/* insert the text at the drop point */
		break;
	case DndDir:
		/* chdir to Dir and start another Siag */
		filename = Data;
		if (!fork()) {
			chdir((char *)filename);
			execlp("siag", "Siag", (char *)NULL);
			exit(1);
		}
		break;
	case DndLink:
		/* why would anyone want this?! a stale link to nowhere */
		break;
	case DndExe:
		/* possible use: run the program and load the output */
		break;
	case DndURL:
		/* load the document using ftpget or httpget */
		break;
	case DndMIME:
		/* some more conversion needed */
		break;
	case DndUnknown:
		/* according to the docs, this should be treated as Raw */
		/* I think it's better to just bail out */
		break;
	default:
		/* something's wrong */
		;
	}
}

static int abort_siag(Display *foo, XErrorEvent *bar)
{
	abort();
	return 0;
}

static void place_shortcuts(Widget w, XEvent *event,
		String *p, Cardinal *n)
{
	XButtonEvent *bev = (XButtonEvent *)event;
	int col, row;
	int x, y;

	activate_window(find_window_by_widget(w));
	x = event->xbutton.x;
	y = event->xbutton.y;
	get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
			&row, &col, x, y);
	XtVaSetValues(shortcuts,
		XtNx, bev->x_root-10,
		XtNy, bev->y_root-10,
		(char *)0);
}

static void menu_motion(Widget w, XEvent *event,
		String *p, Cardinal *n)
{
	int fromright;
	Dimension width;
	XMotionEvent *mev = (XMotionEvent *)event;
	Widget menu, submenu;

	menu = XawSimpleMenuGetActiveEntry(w);
	if (menu == None) return;
	submenu = find_submenu_by_entry(menu);
	if (submenu == None) return;
	XtVaGetValues(w, XtNwidth, &width, (char *)0);
	fromright = width-mev->x;
	if (fromright > 0 && fromright < 10) {
		Position x, y, x_root, y_root;
		XtVaGetValues(menu,
			XtNx, &x,
			XtNy, &y,
			(char *)0);
		XtTranslateCoords(w,
			x+width-20, y, &x_root, &y_root);
		XtVaSetValues(submenu,
			XtNx, x_root,
			XtNy, y_root,
			(char *)0);
		XtPopup(submenu, XtGrabNonexclusive);
	}
}

static void popup_menu(Widget w, XEvent *event,
		String *p, Cardinal *n)
{
	Widget menu = find_menu_by_name(p[0]);
	XtPopupSpringLoaded(menu);
}

static void popdown_submenu(Widget w, XEvent *event,
		String *p, Cardinal *n)
{
	XtPopdown(w);
	XtPopdown(XtParent(w));
}

static XtActionsRec actions[] =
{
	{"destroy-notify", DestroyNotifyAction},
	{"key-event", KeyEventAction},
	{"grid-button", GridButtonAction},
	{"colnum-button", ColnumButtonAction},
	{"rownum-button", RownumButtonAction},
	{"siaghelp", siaghelp_action},
	{"place-shortcuts", place_shortcuts},
	{"menu-motion", menu_motion},
	{"popup-menu", popup_menu},
	{"popdown-submenu", popdown_submenu}
};

void init_windows1(int *argc, char **argv)
{
	unsigned long highlight_color, unhighlight_color;

	XtSetLanguageProc(NULL, (XtLanguageProc) NULL, NULL);

	topLevel = XtVaAppInitialize(
		    &app_context,	/* application context */
		    "Siag",		/* application class */
		    options,		/* command line options list */
		    XtNumber(options),
		    argc, argv,		/* command line args */
		    fallback_resources,	/* for missing app-defaults file */
		    (char *)NULL);	/* terminate varargs list */

	XtGetApplicationResources(topLevel, &app_data, resources,
			XtNumber(resources), NULL, 0);

	XtAppAddActions(app_context, actions, XtNumber(actions));

	shortcuts = XtVaCreatePopupShell("shortcuts",
		simpleMenuWidgetClass, topLevel, (char *)0);
	XtVaCreateManagedWidget("-",
		smeLineObjectClass, shortcuts, (char *)0);

	if (!app_data.grid_only) {
		topbox = XtCreateManagedWidget("topbox",
			formWidgetClass, topLevel, NULL, 0);
	}

	highlight_color = WhitePixel(XtDisplay(topLevel),
        	DefaultScreen(XtDisplay(topLevel)));
	XtVaGetValues(topLevel,   /* or any widget with grey bg */
	        XtNbackground, &unhighlight_color,
	        (char *)0);
	tooltip_init(topLevel, highlight_color, unhighlight_color);

	if (!app_data.grid_only) {
		init_menu();

		init_toolbar();
		setup_buttons();
		init_toggle();

		hsep3 = XtVaCreateManagedWidget("hsep3",
			hsepWidgetClass, topbox,
			XtNtop, XawChainTop,
                        XtNleft, XawChainLeft,
                        XtNright, XawChainRight,
                        XtNbottom, XawChainTop,
                        XtNfromVert, btnFont,
			(char *)0);

		label1 = XtVaCreateManagedWidget("label1",
			labelWidgetClass, topbox,
			XtNtop, XawChainTop,
			XtNleft, XawChainLeft,
			XtNright, XawChainLeft,
			XtNbottom, XawChainTop,
			XtNfromVert, hsep3,
			XtNwidth, 100, (char *)NULL);
	}

	/* should move this line to somewhere after the conditionals */
	XtAppAddActions(app_context, input_actions, XtNumber(input_actions));

	if (!app_data.grid_only) {
		text1 = XtVaCreateManagedWidget("text1",
			textentryWidgetClass, topbox,
			XtNdisplayCaret, False,
			XtNtop, XawChainTop,
			XtNleft, XawChainLeft,
			XtNright, XawChainRight,
			XtNbottom, XawChainTop,
			XtNfromHoriz, label1,
			XtNfromVert, hsep3,
			XtNwidth, 496, (char *)NULL);

		gridpane = XtVaCreateManagedWidget("gridpane",
			panedWidgetClass, topbox,
			XtNtop, XawChainTop,
			XtNleft, XawChainLeft,
			XtNright, XawChainRight,
			XtNbottom, XawChainBottom,
			XtNfromVert, text1,
			XtNallowResize, True,
			XtNmin, 40,
			XtNwidth, 600,
			XtNheight, 200, (char *)NULL);

		label2 = XtVaCreateManagedWidget("label2",
			labelWidgetClass, topbox,
			XtNtop, XawChainBottom,
			XtNleft, XawChainLeft,
			XtNright, XawChainRight,
			XtNbottom, XawChainBottom,
			XtNfromVert, gridpane,
			XtNwidth, 496, (char *)NULL);

		label3 = XtVaCreateManagedWidget("label3",
			labelWidgetClass, topbox,
			XtNtop, XawChainBottom,
			XtNleft, XawChainRight,
			XtNright, XawChainRight,
			XtNbottom, XawChainBottom,
			XtNfromVert, gridpane,
			XtNfromHoriz, label2,
			XtNwidth, 100,
			(char *)0);
	}
}

void activate_window(window *w)
{
	char b[256];

	if (w_list)
		XtVaSetValues(w_list->ui->grid,
			XtNtableVisibleCursor, False,
			(char *)NULL);
	w_list = w;

	strcpy(b, "Siag: ");
	strncat(b, w->buf->name, 200);
	XtVaSetValues(topLevel, XtNtitle, b, (char *)NULL);
	XtVaSetValues(w->ui->grid, XtNtableVisibleCursor, True, (char *)NULL);
	XtSetKeyboardFocus(topLevel, w->ui->grid);
}

Pixmap draw_snapshot(void)
{
	int x, y;
	unsigned int width, height, border_width, depth;
	Window cell_win, root;
	Pixmap bitmap;
	GC gc;
	unsigned long valuemask = 0;
	XGCValues values;
	Widget grid = w_list->ui->grid;

	cell_win = XtWindow(grid);
	XGetGeometry(XtDisplay(grid), cell_win, &root,
		&x, &y, &width, &height, &border_width, &depth);
	bitmap = XCreatePixmap(XtDisplay(topLevel), cell_win, width, height, 1);
	gc = XCreateGC(XtDisplay(grid), bitmap, valuemask, &values);
	XCopyPlane(XtDisplay(grid), cell_win, bitmap, gc, 0, 0, width, height,
			0, 0, 1);
	XFreeGC(XtDisplay(grid), gc);

	return bitmap;
}

window *find_window_by_widget(Widget wdg)
{
	window *w = w_list;
	do {
		if (w->ui->viewport == wdg || w->ui->selectall == wdg ||
			w->ui->colnum == wdg || w->ui->rownum == wdg ||
			w->ui->grid == wdg || w->ui->vscroll == wdg ||
			w->ui->hscroll == wdg)
			return w;
		w = w->next;
	} while (w != w_list);
	return NULL;
}

void free_window(window *w)
{
	window *pw;
	int i;

	for (pw = w_list; pw->next != w && pw->next != pw; pw = pw->next);
	pw->next = w->next;

	if (w_list == w) w_list = w_list->next;
	if (w_list == w) w_list = NULL;
	if (w->ui->viewport != None)
		XtDestroyWidget(w->ui->viewport);

	if (b_list) {	/* don't do this if there are no buffers left */
		for (i = 0; i < w->buf->nplugin; i++) {
			if (w->buf->plugin[i].displayed) {
				plugin_hide(w->buf->plugin[i].ph);
				w->buf->plugin[i].displayed = 0;
			}
		}
	}
	cfree(w->ui);
	cfree(w);
}

/* used for the XtNtablePluginCoords resource */
static void plugin_coordinates(Widget w, XtPointer p, long *x, long *y)
{
	buffer *b = (buffer *)p;
	int n, ph;
	*x = *y = 0;
	ph = plugin_find_by_widget(w);
	if (ph == -1) {
		/* not a plugin; must be editing in place */
		if (w_list == NULL) *x = *y = 0;
		else buffer_global_coords(b, w_list->point_pos.row,
				w_list->point_pos.col, x, y);
		(*x)--;
		(*y)--;
		return;
	}
	n = buffer_plugin2index(b, ph);
	if (n == -1) return;
	buffer_global_coords(b, b->plugin[n].row, b->plugin[n].col, x, y);
}

static Cursor lr_cursor, ud_cursor;

static void colnum_cursor(Widget w, XtPointer p, XEvent *event, Boolean *n)
{
	int col, row, x, y, x1, y1, w1;
	window *win = find_window_by_widget(w);

	x = event->xmotion.x;
	y = 0;
	get_coords_cell(win, win->top.row, win->top.col,
			&row, &col, x, y);
	get_cell_coords(win, win->top.row, win->top.col,
			row, col, &x1, &y1);
	w1 = x-x1;
	if (cell_width(win->buf, col)-w1 < 5) {
		XDefineCursor(XtDisplay(w), XtWindow(w), lr_cursor);
	} else {
		XDefineCursor(XtDisplay(w), XtWindow(w), None);
	}
}

static void rownum_cursor(Widget w, XtPointer p, XEvent *event, Boolean *n)
{
        int col, row, x, y, x1, y1, h1;
        window *win = find_window_by_widget(w);

        y = event->xmotion.y;
        x = 0;
        get_coords_cell(win, win->top.row, win->top.col,
                        &row, &col, x, y);
        get_cell_coords(win, win->top.row, win->top.col,
                        row, col, &x1, &y1);
        h1 = y-y1;
        if (cell_height(win->buf, row)-h1 < 5) {
                XDefineCursor(XtDisplay(w), XtWindow(w), ud_cursor);
        } else {
                XDefineCursor(XtDisplay(w), XtWindow(w), None);
        }
}

window *new_window(buffer *b, window *prev)
{
	Dimension totalwidth, formheight, w1, h1;
	int d;	/* distance between widgets in form */
	window *w;

	if (w_list && app_data.grid_only)
		return NULL;	/* there can be only one */

	w = (window *)cmalloc(sizeof(window));

	if (w == NULL) return NULL;

	w->ui = (siag_ui *)cmalloc(sizeof(siag_ui));
	if (w->ui == NULL) {
		cfree(w);
		return NULL;
	}

	w->buf = b;
	w->point_pos.row = 1;
	w->point_pos.col = 1;
	w->blku.row = w->blku.col = 2;
	w->blkl.row = w->blkl.col = 1;
	w->prot = w->top = w->point_pos;

	if (prev == NULL) prev = w;
	else w->next = prev->next;
	prev->next = w;

	if (app_data.grid_only) {
		w->ui->viewport = w->ui->selectall =
			w->ui->colnum = w->ui->rownum =
			w->ui->vscroll = w->ui->hscroll = None;
		w->ui->grid = XtVaCreateManagedWidget("grid",
			tableWidgetClass, topLevel,
			XtNtableData, b,
			XtNtablePluginCoords, plugin_coordinates,
			(char *)NULL);
		return w;
	}

	/* the rest is for running with the full set of widgets */
			
	/* Figure out how big the new form should be.			*/
	/* The total width must be the width of the gridpane.		*/
	XtVaGetValues(gridpane,
		XtNwidth, &totalwidth, (char *)NULL);

	/* The form height is whatever we get, but if it is too small
	we cannot create the new window.				*/
	formheight = 100;
	w->ui->viewport = XtVaCreateManagedWidget("viewport",
		siagWidgetClass, gridpane,
		XtNwidth, totalwidth,
		(char *)0);
	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, XtNdefaultDistance, &d, (char *)NULL);

	/* The selectall button should get its size from the ad */
	w->ui->selectall = XtVaCreateManagedWidget("selectall",
		commandWidgetClass, w->ui->viewport, (char *)NULL);
	XtAddCallback(w->ui->selectall, XtNcallback,
			execute_callback, "(select-all)");

	/* The colnum should take up all the space between the button
	and vscroll							*/
	w1 = totalwidth-width_get(w->ui->selectall)-20-4*d;	/* just guessing */
	h1 = height_get(w->ui->selectall);

	w->ui->colnum = XtVaCreateManagedWidget("colnum",
		tableWidgetClass, w->ui->viewport,
		/*XtNwidth, w1, XtNheight, h1,*/
		XtNtableDefaultHeight, h1,
		XtNtableColWidth, cell_width,
		XtNtableFormat, rowcol_format,
		XtNtableText, colnum_text,
		XtNtableMaxRow, 1,
		XtNtableMaxCol, BUFFER_COLS,
		XtNtable3D, True,
		/*XtNfromHoriz, w->ui->selectall,*/ (char *)NULL);
	XtAddEventHandler(w->ui->colnum,
		PointerMotionMask, False, colnum_cursor, NULL);
	w1 = width_get(w->ui->selectall);
	h1 = formheight-height_get(w->ui->selectall)-20-4*d;

	w->ui->rownum = XtVaCreateManagedWidget("rownum",
		tableWidgetClass, w->ui->viewport,
		/*XtNwidth, w1, XtNheight, h1,*/
		XtNtableDefaultWidth, w1,
		XtNtableRowHeight, cell_height,
		XtNtableFormat, rowcol_format,
		XtNtableText, rownum_text,
		XtNtableMaxRow, BUFFER_ROWS,
		XtNtableMaxCol, 1,
		XtNtable3D, True,
		/*XtNfromVert, w->ui->selectall,*/ (char *)NULL);
	XtAddEventHandler(w->ui->rownum,
		PointerMotionMask, False, rownum_cursor, NULL);
	w1 = width_get(w->ui->colnum);
	h1 = height_get(w->ui->rownum);

	w->ui->grid = XtVaCreateManagedWidget("grid",
		tableWidgetClass, w->ui->viewport,
		XtNtableData, b,
		XtNtablePluginCoords, plugin_coordinates,
		(char *)NULL);
	h1 = height_get(w->ui->colnum)+height_get(w->ui->grid)+d;
	w->ui->vscroll = XtVaCreateManagedWidget("vscroll",
		scrollbarWidgetClass, w->ui->viewport,
		(char *)NULL);
	w1 = width_get(w->ui->rownum)+width_get(w->ui->grid)+d;
	w->ui->hscroll = XtVaCreateManagedWidget("hscroll",
		scrollbarWidgetClass, w->ui->viewport,
		(char *)NULL);
	XtAddCallback(w->ui->vscroll, XtNjumpProc, vscroll_jump, NULL);
	XtAddCallback(w->ui->vscroll, XtNscrollProc, vscroll_scroll, NULL);

	XtAddCallback(w->ui->hscroll, XtNjumpProc, hscroll_jump, NULL);
	XtAddCallback(w->ui->hscroll, XtNscrollProc, hscroll_scroll, NULL);

	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, (char *)NULL);

	return w;
}

int remove_window(window *w)
{
	
	if (w == w->next) return FALSE;
	free_window(w);
	return TRUE;
}

int split_window(window *w)
{
	window *w2 = new_window(w->buf, w);

	if (w2 == NULL) return FALSE;
	w2->point_pos = w->point_pos;
	w2->top = w->top;
	w2->prot = w->prot;
	return TRUE;
}

/* this does probably not belong here in window.c,
   because it doesn't depend on X */
static void save_plugin(char *p)
{
	if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
	else {
		if (savematrix(p, w_list->buf, NULL)) {
			printf("501 Can't save %s\n", p);
		} else {
			printf("250 Saved %s\n", p);
		}
	}
}

static void load_plugin(char *p)
{
	if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
	else {
		if (loadmatrix(p, w_list->buf, NULL)) {
			printf("501 Can't load %s\n", p);
		} else {
			printf("250 Loaded %s\n", p);
		}
	}
}

static void exec_plugin(char *p)
{
	if (*p++ != ' ' || *p == '\0') printf("501 Command missing\n");
	else {
		execute(p);
		printf("250 OK\n");
	}
}

static void help_plugin(char *p)
{
	printf("214 SAVE LOAD EXEC HELP NOOP QUIT PRNT\n");
}

static void noop_plugin(char *p)
{
        printf("250 OK\n");
}

static void win_plugin(char *p)
{
	printf("250 %lx\n", (unsigned long)XtWindow(topLevel));
}

static void quit_plugin(char *p)
{
	printf("221 Over and out\n");
	execute("(quit-siag)");
}

static void prnt_plugin(char *p)
{
        printf("502 Can't print yet\n");
}

static struct {
        char *verb;
        void (*cb)(char *);
} plugin_cmds[] = {
        {"SAVE", save_plugin},
        {"LOAD", load_plugin},
        {"EXEC", exec_plugin},
        {"HELP", help_plugin},
        {"NOOP", noop_plugin},
	{"WIN", win_plugin},
        {"QUIT", quit_plugin},
        {"PRNT", prnt_plugin},
        {NULL, NULL}
};

static void read_plugin_cmd(XtPointer client_data, int *fid, XtInputId *id)
{
	char b[1024], *p;
	int i, n;

	if ((n = read(*fid, b, 1020)) == -1) return;

	b[n] = '\0';
	if ((p = strchr(b, '\n')) == NULL) {
		printf("501 Incomplete command\n");
		fflush(stdout);
		return;
	}

	*p = '\0';
	for (i = 0; plugin_cmds[i].verb; i++) {
		if (!strncmp(b, plugin_cmds[i].verb,
				strlen(plugin_cmds[i].verb)))
			break;
	}
	if (plugin_cmds[i].verb)
		(*plugin_cmds[i].cb)(b+strlen(plugin_cmds[i].verb));
	else
		printf("500 What are you talking about\n");
	fflush(stdout);
}

void mainloop(void)
{
	if (app_data.plugin) {
		/* control plugin from stdin */
		XtAppAddInput(XtWidgetToApplicationContext(topLevel),
			fileno(stdin), (XtPointer)XtInputReadMask,
			read_plugin_cmd, NULL);
		printf("220 %s\n", VERSION);
		fflush(stdout);
	}

	/* keep X from terminating gracefully upon fatal error */
	XSetErrorHandler(abort_siag);

	XtAppMainLoop(app_context);

	exit(0);
}

/* handle spontaneous exit of plugins */
static void handle_plugin_exit(int ph)
{
	buffer *b = b_list;

	do {
		int n = buffer_plugin2index(b, ph);
		if (n != -1) {
			cfree(b->plugin[n].name);
			b->nplugin--;
			for (; n < b->nplugin; n++)
				b->plugin[n] = b->plugin[n+1];
			b->change = pr_scr_flag = TRUE;
		}
		b = b->next;
	} while (b != b_list);
}

/*
   Set up the whole initial window structure and initialize scrupd.
   The window list w_list is set to a list with a single window with the
   buffer b.
*/
/*
 * This function is by no means as transparent as I wold like it to be.
 * There are some very complex interactions going on, many of which
 * span across functions and even libraries. And I frankly don't have
 * a firm grip on them myself.
 * 
 * Here is how things *should* work
 * 1. Create toplevel window, menu et al (init_windows1)
 * 2. Initialize input (req. topLevel). Sets up dialog actions
 * 3. Create text widget and the rest of the main widgets (init_windows2)
 * 4. Initialize parsers? Is it necessary to do that here?
 * 5. Create a dummy buffer
 * 6. Call init_windows (req. a buffer)
 */
void init_windows(buffer * b, int argc, char **argv)
{
	char *p;
	Cursor rownum_cursor;
	Cursor colnum_cursor;
	unsigned long foreground;	/* pixel value */
	Window cell_win;
	Atom wm_delete_window;	/* Atom sent to destroy a window */

	XtRealizeWidget(topLevel);

	plugin_init(topLevel, handle_plugin_exit);

	lr_cursor = XCreateFontCursor(XtDisplay(topLevel),
		XC_sb_h_double_arrow);
	ud_cursor = XCreateFontCursor(XtDisplay(topLevel),
		XC_sb_v_double_arrow);


	rownum_cursor = XCreateFontCursor(XtDisplay(topLevel),
					XC_sb_right_arrow);
	colnum_cursor = XCreateFontCursor(XtDisplay(topLevel),
					XC_sb_down_arrow);

	activate_window(new_window(b, NULL));

	wm_delete_window = XInternAtom(XtDisplay(topLevel),
					"WM_DELETE_WINDOW", False);
	XtOverrideTranslations(topLevel,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: execute(quit-siag)"));

	XSetWMProtocols(XtDisplay(topLevel),
				XtWindow(topLevel), &wm_delete_window, 1);

	init_color(XtDisplay(topLevel));

	foreground = BlackPixelOfScreen(XtScreen(topLevel));
	cell_win = XtWindow(w_list->ui->grid);

	draw_scrollbars(w_list);
	p = ret_text(w_list->buf,
		      w_list->point_pos.row, w_list->point_pos.col);
	if (p == NULL)
		p = "";
	draw_input(p);
	draw_status("");
	draw_cells(w_list);
	draw_colnums(w_list);
	draw_rownums(w_list);

	/* Set up selection */
	target_atom = XInternAtom(XtDisplay(topLevel), "SIAG_BLOCK", False);

        embed_init(topLevel);

        init_subr_3("add-menu-entry", add_menu_entry);
	init_subr_2("add-submenu", add_submenu);
	init_subr_4("add-submenu-entry", add_submenu_entry);
        init_subr_1("tooltip-mode", ltooltip_mode);

	init_calc_cmds();

        init_form(topLevel);

        icon_set(topLevel, siag_fg_xpm);
        DndInitialize(topLevel);        /* initialize drag and drop */
        DndRegisterOtherDrop(drop_handler);     /* in any window */

        font_init(topLevel);
}

/* Clean up before exit.  All buffers and windows are freed. */
void exit_windows(void)
{
	/* free all buffers */
	while (b_list != NULL)
		free_buffer(b_list);
	while (w_list != NULL)
		free_window(w_list);
	/* remove all temp files */
	deletia_mark(0);
	deletia_reap();
}

/*
   Prints and refreshes all the windows.
   Sets pr_scr_flag to FALSE.
   970422: also recalculate if needed
   980717: check for unparented plugins
*/
static void pr_scr(void)
{
	window *w;
	buffer *b;
	int i;

	draw_status("");
	b = b_list;
	do {
		if (b->recalc) {
			int i;
			b->recalc = 0;
			for (i = 0; i < recalc; i++)
				calc_matrix(b);
		}
		b = b->next;
	} while (b != b_list);

	w = w_list;
	do {
		draw_cells(w);
		draw_colnums(w);
		draw_rownums(w);
		draw_scrollbars(w);

		/* check for nondisplayed plugins */
		/* should also *remove* plugins that can't be displayed */
		for (i = 0; i < w->buf->nplugin; i++) {
			if (!w->buf->plugin[i].displayed) {
				plugin_show(w->buf->plugin[i].ph,
						w->ui->grid);
				w->buf->plugin[i].displayed = 1;
			}
		}
		w = w->next;
	} while (w != w_list);
	pr_scr_flag = FALSE;
	deletia_reap();
}	/* pr_scr */

/* 970427: the following four functions facilitate navigating in
	the cell area with protection implemented */

static int cell_next_row(window *w, int row)
{
	if (row+1 == w->prot.row) return w->top.row;
	return row+1;
}

static int cell_next_col(window *w, int col)
{
	if (col+1 == w->prot.col) return w->top.col;
	return col+1;
}

static int cell_prev_row(window *w, int row)
{
	if (row == w->top.row) return w->prot.row-1;
	return row-1;
}

static int cell_prev_col(window *w, int col)
{
	if (col == w->top.col) return w->prot.col-1;
	return col-1;
}

/* From (row, col), calculate (x, y) coordinates. This is a little tricker
	now. Rather than starting the calculation in (0, 0) we must
	take the protected cells into account. */
void get_cell_coords(window *w, int top_row, int top_col,
		     int cell_row, int cell_col,
		     int *cell_x, int *cell_y)
{
	int i;

	*cell_y = 0;
	for (i = 1; i < w->prot.row; i++)
		*cell_y += cell_height(w->buf, i);

	while (cell_row < top_row) {
		cell_row = cell_next_row(w, cell_row);
		*cell_y -= cell_height(w->buf, cell_row);
	}
	while (cell_row > top_row) {
		cell_row = cell_prev_row(w, cell_row);
		*cell_y += cell_height(w->buf, cell_row);
	}
	*cell_x = 0;
	for (i = 1; i < w->prot.col; i++)
		*cell_x += cell_width(w->buf, i);

	while (cell_col < top_col) {
		cell_col = cell_next_col(w, cell_col);
		*cell_x -= cell_width(w->buf, cell_col);
	}
	while (cell_col > top_col) {
		cell_col = cell_prev_col(w, cell_col);
		*cell_x += cell_width(w->buf, cell_col);
	}
}

void get_coords_cell(window *w, int top_row, int top_col,
		     int *cur_row, int *cur_col,
		     int cur_x, int cur_y)
{
	int prot_x = 0, prot_y = 0, i;

	for (i = 1; i < w->prot.col; i++)
		cur_x -= cell_width(w->buf, i);
	for (i = 1; i < w->prot.row; i++)
		cur_y -= cell_height(w->buf, i);

	*cur_row = top_row;
	*cur_col = top_col;
	while (cur_y < prot_y && *cur_row > 1) {
		cur_y += cell_height(w->buf, *cur_row);
		(*cur_row) = cell_prev_row(w, *cur_row);
	}
	while (cur_y > cell_height(w->buf, *cur_row) && *cur_row < BUFFER_ROWS) {
		cur_y -= cell_height(w->buf, *cur_row);
		(*cur_row) = cell_next_row(w, *cur_row);
	}
	while (cur_x < prot_x && *cur_col > 1) {
		cur_x += cell_width(w->buf, *cur_col);
		(*cur_col) = cell_prev_col(w, *cur_col);
	}
	while (cur_x > cell_width(w->buf, *cur_col) && *cur_col < BUFFER_COLS) {
		cur_x -= cell_width(w->buf, *cur_col);
		(*cur_col) = cell_next_col(w, *cur_col);
	}
}

void show_format(void)
{
	static int last_fmt = -1;
	int fmt = ret_format(w_list->buf,
				w_list->point_pos.row,
				w_list->point_pos.col);
	int bold = fmt & BOLD;
	int italic = fmt & ITALIC;
	int hadj = fmt & HADJ_MASK;
	int hadj_left = (hadj == HADJ_LEFT);
	int hadj_center = (hadj == HADJ_CENTER);
	int hadj_right = (hadj == HADJ_RIGHT);

	if (fmt != last_fmt) {
		/* menus */
		label_set(btnFont, family2name(fmt));
		label_set(btnSize, size2name(fmt));
		label_set(btnStyle, format2name(fmt));
		label_set(btnColor, color2name(fmt));
	}

	last_fmt = fmt;

	/* toggle buttons */
	state_set(cmdBold, (bold?1:0));
	state_set(cmdItalic, (italic?1:0));
	state_set(cmdHLeft, (hadj_left?1:0));
	state_set(cmdHCenter, (hadj_center?1:0));
	state_set(cmdHRight, (hadj_right?1:0));
}

int cursor_visible = FALSE;

/*
   void show_cur(window *w)
   Moves the cursor to reflect the position of point in w.
   If point is not visible, the window is moved so that point is in
   the middle of the screen.
*/
void show_cur(window *w)
{
	char *p;
	int top_row, top_col;

	if (!w) return;

	top_row = w->top.row;
	top_col = w->top.col;

	XtVaSetValues(w->ui->grid,
		XtNtablePointRow, w->point_pos.row,
		XtNtablePointCol, w->point_pos.col,
		XtNtableVisibleCursor, True,
		(char *)NULL);

	if (pr_scr_flag) {
		pr_scr();
	} else {
		/* this may have moved the top, so we must check that */
		XtVaGetValues(w->ui->grid,
			XtNtableTopRow, &(w->top.row),
			XtNtableTopCol, &(w->top.col),
			(char *)NULL);

		if (top_row != w->top.row) {
			draw_rownums(w);
			draw_vbar(w);
		}
		if (top_col != w->top.col) {
			draw_colnums(w);
			draw_hbar(w);
		}
	}
	p = ret_text(w->buf, w->point_pos.row,
		      w->point_pos.col);
	if (p == NULL)
		p = "";
	draw_input(p);
	show_format();
	cursor_visible = TRUE;
}	/* show_cur */

void hide_cur(window *w)
{
	;
}

