/*
   Copyright (C) 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the Licence, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include "../common/cmalloc.h"
#include "xfonts.h"	/* from common */
#include <X11/xpm.h>
#include "embed.h"

#include "../common/richchar.h"
#include "../common/traceme.h"

#include "RichtextP.h"

static void plugin_coords(Widget, XtPointer, long *, long *);

#define offset(field) XtOffsetOf(RichtextRec, richtext.field)
static XtResource resources[] = {
	{
		XtNrichtextTopRow,	/* name */
		XtCRichtextTopRow,	/* class */
		XtRInt,			/* type */
		sizeof(int),		/* size */
		offset(top_row),	/* offset */
		XtRImmediate,		/* default_type */
		(XtPointer)1		/* default_addr */
	}, {
		XtNrichtextTopCol,
		XtCRichtextTopCol,
		XtRInt,
		sizeof(int),
		offset(top_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNrichtextSelectTopRow,
		XtCRichtextSelectTopRow,
		XtRInt,
		sizeof(int),
		offset(sel_top_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNrichtextSelectTopCol,
		XtCRichtextSelectTopCol,
		XtRInt,
		sizeof(int),
		offset(sel_top_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNrichtextSelectBottomRow,
		XtCRichtextSelectBottomRow,
		XtRInt,
		sizeof(int),
		offset(sel_bottom_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNrichtextSelectBottomCol,
		XtCRichtextSelectBottomCol,
		XtRInt,
		sizeof(int),
		offset(sel_bottom_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNrichtextPointRow,
		XtCRichtextPointRow,
		XtRInt,
		sizeof(int),
		offset(point_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNrichtextPointCol,
		XtCRichtextPointCol,
		XtRInt,
		sizeof(int),
		offset(point_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNrichtextRowHeight,
		XtCRichtextRowHeight,
		XtRPointer,
		sizeof(XtPointer),
		offset(row_height),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNrichtextAdjHoriz,
		XtCRichtextAdjHoriz,
		XtRPointer,
		sizeof(XtPointer),
		offset(adj_horiz),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNrichtextStyle,
		XtCRichtextStyle,
		XtRPointer,
		sizeof(XtPointer),
		offset(style),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNrichtextText,
		XtCRichtextText,
		XtRPointer,
		sizeof(XtPointer),
		offset(text),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNrichtextData,
		XtCRichtextData,
		XtRPointer,
		sizeof(XtPointer),
		offset(data),
		XtRImmediate,
		(XtPointer) NULL
	}, {
		XtNrichtextRedisplay,
		XtCRichtextRedisplay,
		XtRBoolean,
		sizeof(Boolean),
		offset(redisplay),
		XtRImmediate,
		(XtPointer)False
	}, {
		XtNrichtextVisibleCursor,
		XtCRichtextVisibleCursor,
		XtRBoolean,
		sizeof(Boolean),
		offset(visible_cursor),
		XtRImmediate,
		(XtPointer)False
	}, {
		XtNrichtextPaperWidth,
		XtCRichtextPaperWidth,
		XtRInt,
		sizeof(int),
		offset(paper_width),
		XtRImmediate,
		(XtPointer)595
	}, {
		XtNrichtextLeftMargin,
		XtCRichtextLeftMargin,
		XtRInt,
		sizeof(int),
		offset(left_margin),
		XtRImmediate,
		(XtPointer)72
	}, {
		XtNrichtextRightMargin,
		XtCRichtextRightMargin,
		XtRInt,
		sizeof(int),
		offset(right_margin),
		XtRImmediate,
		(XtPointer)72
	}, {
		XtNrichtextPluginCoords,
		XtCRichtextPluginCoords,
		XtRPointer,
		sizeof(XtPointer),
		offset(plugin_coords),
		XtRImmediate,
		(XtPointer)plugin_coords
	}
};
#undef offset

/* methods */
static void DoLayout();
static void Resize();
static XtGeometryResult GeometryManager();
static void ChangeManaged();
static void Redisplay(Widget, XEvent *, Region);
static void Realize(Widget, XtValueMask *, XSetWindowAttributes *);
static void Destroy(Widget);
static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal *);

static void plugin_coords(Widget w, XtPointer p, long *x, long *y)
{
	*x = *y = 0;
}

/* actions */
static void RichtextAction(Widget, XEvent *, String *, Cardinal *);

static XtActionsRec actions[] =
{
	{"richtext", RichtextAction},
};

/* translations */
static char translations[] =
"<Key>:		richtext()\n";

RichtextClassRec richtextClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &compositeClassRec,
    /* class_name		*/	"Richtext",
    /* widget_size		*/	sizeof(RichtextRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	NULL,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },{
/* composite_class fields */
    /* geometry_manager   */    GeometryManager,
    /* change_managed     */    ChangeManaged,
    /* insert_child       */    XtInheritInsertChild,
    /* delete_child       */    XtInheritDeleteChild,
    /* extension          */    NULL
  }, { /* richtext fields */
    /* empty			*/	0
  }
};

WidgetClass richtextWidgetClass = (WidgetClass)&richtextClassRec;


/* supporting code copied directly from window.c */

static void RichtextAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
	;
}

static GC get_gc(Widget w, unsigned long fg, unsigned long bg, Font font)
{
        unsigned long valuemask = 0;
        XGCValues values;
        GC gc = XCreateGC(XtDisplay(w), XtWindow(w), valuemask, &values);

        XSetForeground(XtDisplay(w), gc, fg);
        XSetBackground(XtDisplay(w), gc, bg);
        if (font != -1)
                XSetFont(XtDisplay(w), gc, font);
        return gc;
}

#define superclass (&coreClassRec)
static void Realize(Widget w, XtValueMask *valueMask,
		XSetWindowAttributes *attributes)
{
	RichtextWidget rtw = (RichtextWidget)w;
	unsigned long fg, bg, blockbg;
	XColor screen_color, exact_color;

	(*superclass->core_class.realize)(w, valueMask, attributes);
	fg = BlackPixelOfScreen(XtScreen(w));
	bg = rtw->core.background_pixel;
	XAllocNamedColor(XtDisplay(w),
		DefaultColormap(XtDisplay(w), DefaultScreen(XtDisplay(w))),
		"grey", &screen_color, &exact_color);
	blockbg = screen_color.pixel;
	rtw->richtext.clear_gc = get_gc(w, bg, fg, -1);
	rtw->richtext.cell_gc = get_gc(w, fg, blockbg,
			get_font(XtDisplay(w), CELL_FONT));
	rtw->richtext.cursor_gc = get_gc(w, fg^bg, 0, -1);
	XSetFunction(XtDisplay(w), rtw->richtext.cursor_gc, GXxor);
        XSetLineAttributes(XtDisplay(w), rtw->richtext.cursor_gc,
			1, LineSolid, CapButt, JoinMiter);
}

static void Destroy(Widget w)
{
	RichtextWidget rtw = (RichtextWidget)w;

	XFreeGC(XtDisplay(w), rtw->richtext.clear_gc);
	XFreeGC(XtDisplay(w), rtw->richtext.cell_gc);
	XFreeGC(XtDisplay(w), rtw->richtext.cursor_gc);
}

static Dimension row_height(RichtextWidget rtw, int row)
{
	if (rtw->richtext.row_height)
		return (*rtw->richtext.row_height)(rtw->richtext.data, row);
	return 20;
}

/* forget about x for now */
void richtext_global_coords(RichtextWidget tw,
		long row, long col, long *x, long *y)
{
	long i;

	*x = *y = 0;

	for (i = 1; i < row; i++)
		*y += row_height(tw, i);
}

int inblock(RichtextWidget rtw, int r, int c)
{
	return ((r > rtw->richtext.sel_top_row ||
		 (r == rtw->richtext.sel_top_row &&
		  c >= rtw->richtext.sel_top_col)) &&
		 (r < rtw->richtext.sel_bottom_row ||
		  (r == rtw->richtext.sel_bottom_row &&
		   c <= rtw->richtext.sel_bottom_col)));
}

static int ret_hadj(RichtextWidget rtw, long row)
{
	if (rtw->richtext.adj_horiz)
		return (*rtw->richtext.adj_horiz)(rtw->richtext.data, row);
	return HADJ_LEFT;
}

static int ret_style(RichtextWidget rtw, long row)
{
	if (rtw->richtext.style)
		return (*rtw->richtext.style)(rtw->richtext.data, row);
	return STY_DEFAULT;
}

static rich_char *ret_text(RichtextWidget rtw, long row)
{
	if (rtw->richtext.text)
		return (*rtw->richtext.text)(rtw->richtext.data, row);
	return NULL;
}

static int line_width(RichtextWidget rtw, int row, int col)
{
	rich_char *line;
        char *p;
        int i, l;
        unsigned int width;
        XFontStruct *font_info;
	line = ret_text(rtw, row);
	if (line == NULL) return 0;

	p = (char *)rc_makeplain(line);

        if (ret_style(rtw, row) == STY_EMBED) {
                if (col == 0) return 0;
                embed_size(p, &width, NULL);
                return width;
        }

        l = rc_strlen(line);
        if (col > l) col = l;
        width = 0;      /* accumulated width */
        for (i = 0; i < col && line[i].c; i++) {
                if (line[i].c == '\t') {
                        width += 36;
                        width -= (width % 36);
                } else {
                        font_info = font_struct(XtDisplay(rtw), line[i].fmt & 127);
                        width += XTextWidth(font_info, p+i, 1);
                }
        }
        cfree(p);
        return width;
}

static int line_length(RichtextWidget rtw, int row)
{
	rich_char *p;
	int len;
	p = ret_text(rtw, row);
	if (p == NULL) return 0;
	if (ret_style(rtw, row) == STY_EMBED) return 1;
	len = rc_strlen(p);
	return len;
}

static int line_start(RichtextWidget rtw, int row)
{
        int x = 0;
        int w = rtw->richtext.paper_width-rtw->richtext.left_margin
		-rtw->richtext.right_margin;
        int hadj;
        hadj = ret_hadj(rtw, row);
        if (hadj == HADJ_CENTER)
                x = (w-line_width(rtw, row, line_length(rtw, row))) / 2;
        else if (hadj == HADJ_RIGHT)
                x = w-line_width(rtw, row, line_length(rtw, row));

        return x+rtw->richtext.left_margin;
}

void richtext_char2coords(RichtextWidget rtw,
		int cell_row, int cell_col,
		int *cell_x, int *cell_y)
{
	int r;
        *cell_y = 0;
        for (r = rtw->richtext.top_row; r < cell_row; r++)
                *cell_y += row_height(rtw, r);
        *cell_x = line_start(rtw, cell_row)
                        +line_width(rtw, cell_row, cell_col);
}

void richtext_coords2char(RichtextWidget rtw,
		int *cur_row, int *cur_col,
		int cur_x, int cur_y)
{
        int r, c, h;

        r = rtw->richtext.top_row;
        h = row_height(rtw, r);
        while (cur_y > h) {
                r++;
                h += row_height(rtw, r);
        }
        *cur_row = r;

        c = 0;
        cur_x -= line_start(rtw, r);
        cur_x += rtw->richtext.top_col;       /* it's in pixels, y'know */

        while (c < line_length(rtw, r) &&
                        cur_x > line_width(rtw, r, c)) {
                c++;
        }
        *cur_col = c;
}

static Boolean move_top(RichtextWidget rtw)
{
	Boolean pr_scr_flag = False;
	int cur_x, cur_y;
	unsigned int width, height;
	if (rtw->richtext.point_row < rtw->richtext.top_row) {
		rtw->richtext.top_row = rtw->richtext.point_row;
		pr_scr_flag = True;
	}
	/* Figure out how big the window is */
	width = rtw->core.width;
	height = rtw->core.height;
	richtext_char2coords(rtw,
		rtw->richtext.point_row, rtw->richtext.point_col,
		&cur_x, &cur_y);
	/* this isn't efficient, but it will work */
        while (cur_y < 0) {     /* this should never happen */
                rtw->richtext.top_row--;
                cur_y += row_height(rtw, rtw->richtext.top_row);
                pr_scr_flag = TRUE;
        }
        while (cur_y + row_height(rtw, rtw->richtext.point_row) > height) {
                cur_y -= row_height(rtw, rtw->richtext.top_row);
                rtw->richtext.top_row++;
                pr_scr_flag = TRUE;
        }
	return pr_scr_flag;
}

static void toggle_cursor(RichtextWidget rtw)
{
	int cur_x, cur_y;

	richtext_char2coords(rtw,
		rtw->richtext.point_row, rtw->richtext.point_col,
		&cur_x, &cur_y);
	XDrawLine(XtDisplay((Widget)rtw), XtWindow((Widget)rtw),
			rtw->richtext.cursor_gc, cur_x, cur_y,
                        cur_x, row_height(rtw, rtw->richtext.point_row)+cur_y);
}

static void draw_line(RichtextWidget rtw, Drawable cell_win,
		int y_base, int row, int clr)
{
        int font_index, color_index;
        int i;
        Font font;
        unsigned long color;
        int height = row_height(rtw, row);
        int x_base = 0, old_x_base;
	Display *display = XtDisplay(rtw);
        rich_char *line = ret_text(rtw, row);

        if (clr) {
                /* the 4 is from trial and error; should be from font metrics */
                XClearArea(display, cell_win, 0, y_base+4,
                                rtw->core.width, height, FALSE);
        }

        if (!line) return;
        x_base = line_start(rtw, row);

        if (ret_style(rtw, row) == STY_EMBED) {
                char *p = (char *)rc_makeplain(line);
                embed_draw(cell_win, x_base, y_base, p);
                cfree(p);
                return;         /* done */
        }

        for (i = 0; line[i].c; i++) {
                int width, vadj;
                char b[2];
                b[0] = line[i].c;
                b[1] = '\0';
                font_index = (line[i].fmt) & 127;
                font = get_font(display, font_index);
                XSetFont(display, rtw->richtext.cell_gc, font);

                switch (line[i].fmt & VADJ_MASK) {
                case VADJ_TOP:
                        vadj = -6;
                        break;
                case VADJ_BOTTOM:
                        vadj = 6;
                        break;
                default:
                        vadj = 0;
                }
                color_index = ((line[i].fmt) & COLOR_MASK) >> COLOR_SHIFT;
                color = get_color(display, color_index);
                old_x_base = x_base;
                XSetForeground(display, rtw->richtext.cell_gc, color);
                if (inblock(rtw, row, i))
                        XDrawImageString(display, cell_win,
				rtw->richtext.cell_gc,
                                x_base, y_base+height+vadj, b, 1);
                else
                        XDrawString(display, cell_win,
				rtw->richtext.cell_gc,
                                x_base, y_base+height+vadj, b, 1);
                if (b[0] == '\t') {     /* hard coded tabs */
                        x_base += 36;   /* one inch */
                        x_base -= (x_base % 36);
                } else {
                        width = XTextWidth(font_struct(display, font_index),
                                        b, 1);
                        x_base += width;
                }
                if (line[i].fmt & ULINE) {
                        XDrawLine(display, cell_win,
				rtw->richtext.cell_gc,
                                old_x_base, y_base+height+1,
                                x_base, y_base+height+1);
                }
        }
}

void rt_pr_line(Widget w, int row)
{
        Window cell_win = XtWindow(w);
        RichtextWidget rtw = (RichtextWidget)w;
        int i;
        int y_base = 0;

        for (i = rtw->richtext.top_row; i < row; i++)
                y_base += row_height(rtw, i);
        /* here i == row+1 and y_base is just right */
	if (rtw->richtext.visible_cursor)
		toggle_cursor(rtw);
        draw_line(rtw, cell_win, y_base, row, TRUE);
	if (rtw->richtext.visible_cursor)
		toggle_cursor(rtw);
}

/* This is the equivalent of the Table widget's draw_table function. */
void draw_richtext(RichtextWidget rtw, Drawable d)
{
        int i, y_base;
        unsigned int height;

	height = rtw->core.height;
        y_base = 0;
        for (i = rtw->richtext.top_row;
			/*i <= w->buf->used_lines &&*/
			y_base < height; i++) {
                draw_line(rtw, d, y_base, i, FALSE);
                y_base += row_height(rtw, i);
                /* the rest moved to draw_line() */
        }
}

Pixmap richtext_pixmap(RichtextWidget rtw)
{
        unsigned int width, height, depth;
        Pixmap scribble;
        Widget w = (Widget)rtw;

	width = rtw->core.width;
	height = rtw->core.height;
	depth = rtw->core.depth;
	if (width > 2000 || height > 2000) return None;

        scribble = XCreatePixmap(XtDisplay(w), XtWindow(w),
                width, height, depth);
        XFillRectangle(XtDisplay(w), scribble, rtw->richtext.clear_gc,
                0, 0, width, height);
        draw_richtext(rtw, scribble);
        return scribble;
}

static void Redisplay(Widget w, XEvent *xevent, Region r)
{
        Pixmap scribble;
        GC gc;
        unsigned long valuemask = 0;
        XGCValues values;
	RichtextWidget rtw = (RichtextWidget)w;

	scribble = richtext_pixmap(rtw);
        if (scribble == None) return;
        gc = XCreateGC(XtDisplay(w), XtWindow(w),
                        valuemask, &values);
        XCopyArea(XtDisplay(w), scribble, XtWindow(w),
                gc, 0, 0, rtw->core.width, rtw->core.height, 0, 0);
        XFreePixmap(XtDisplay(w), scribble);
        XFreeGC(XtDisplay(w), gc);
	if (rtw->richtext.visible_cursor) {
		toggle_cursor(rtw);
	}
	/* update plugin positions */
	DoLayout(rtw);
}

static Boolean SetValues(Widget current, Widget request, Widget new,
		ArgList args, Cardinal *nargs)
{
	RichtextWidget currtw = (RichtextWidget)current;
	RichtextWidget newrtw = (RichtextWidget)new;
	Boolean do_redisplay = False;

	do_redisplay = (currtw->richtext.sel_top_row != newrtw->richtext.sel_top_row
		|| currtw->richtext.sel_top_col != newrtw->richtext.sel_top_col
		|| currtw->richtext.sel_bottom_row != newrtw->richtext.sel_bottom_row
		|| currtw->richtext.sel_bottom_col != newrtw->richtext.sel_bottom_col);
	if (newrtw->richtext.visible_cursor)
		do_redisplay |= move_top(newrtw);

	if (newrtw->richtext.redisplay) {
		do_redisplay = True;
		newrtw->richtext.redisplay = False;
	}

	/* can't let Xt handle this because it will flicker */
	if (do_redisplay) {
		Redisplay((Widget)newrtw, NULL, None);
		do_redisplay = False;
	} else {
		if (currtw->richtext.visible_cursor)
			toggle_cursor(currtw);
		if (newrtw->richtext.visible_cursor)
			toggle_cursor(newrtw);
	}

	return do_redisplay;
}

/*
 * Do a layout, actually assigning positions.
 */

static void DoLayout(RichtextWidget sw)
{
        int i;
        long x, y, top_x, top_y;

        if (sw->composite.num_children) {
                richtext_global_coords(sw,
                        sw->richtext.top_row, sw->richtext.top_col,
                        &top_x, &top_y);
        }
        for (i = 0; i < sw->composite.num_children; i++) {
                (*sw->richtext.plugin_coords)(sw->composite.children[i],
                        sw->richtext.data, &x, &y);
		x = (sw->richtext.left_margin
			+sw->richtext.paper_width
			-sw->richtext.right_margin
			-sw->composite.children[i]->core.width)/2;
                XtMoveWidget(sw->composite.children[i],
                        x-top_x, y-top_y);
        }
}

/*
 * Actually layout the table
 */

static void Resize(Widget w)
{
        DoLayout((RichtextWidget)w);
} /* Resize */

/*
 * Geometry Manager
 *
 * 'reply' is unused; we say only yeay or nay, never almost.
 */

static XtGeometryResult GeometryManager(Widget w,
                XtWidgetGeometry request, XtWidgetGeometry reply)
{
        return XtGeometryYes;
}

static void ChangeManaged(Widget w)
{
    DoLayout((RichtextWidget)w);
}

