/*
   Siag, Scheme In A Grid
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/*
   fileio_ps.c

   Produces data suitable for printing on a Postscript device. I have made
   several assumptions to simplify things:
   - paper size is A4 = 595x842 points
   - margin on all sides = 72 points = 1 inch
   - 1 pixel = 1 point means no scaling is necessary
   - PS font has same geometry as X font
   - no background pattern
   - don't draw the grid
   - The lpr command is used for printing

   This works for me, but will probably break (more or less) for anyone
   not using my printer (a NEC S62P) with A4 paper.

   971101: use properties to customize the output.
	ps_paper_width		paper width in points
	ps_paper_height		paper height in points
	ps_top_margin		top margin in points
	ps_left_margin		left margin in points
	ps_right_margin		right margin in points
	ps_bottom_margin	bottom margin in points
	ps_page_header		page header in special format
	ps_page_footer		page footer in special format

   The header/footer format is as follows:
	%n	expands to buffer name
	%p	expands to page number
	%%	expands to %
	any other character is simply copied as it is

   980802: print plugins. Before terminating each page, print every plugin
	that is completely or partially displayed on that page.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>

#include <unistd.h>

#include "../common/fonts.h"
#include "../common/common.h"

#include "types.h"
#include "calc.h"

int paper_height = 842;
int paper_width = 595;

int margin = 72;
int lastfont = 0;

char *psformat;

static int ps_paper_width;
static int ps_paper_height;
static int ps_top_margin;
static int ps_left_margin;
static int ps_right_margin;
static int ps_bottom_margin;

static void read_properties(buffer *b)
{
	char *p;

	if ((p = get_property(b, "ps_paper_width")))
		ps_paper_width = strtol(p, NULL, 10);
	else
		ps_paper_width = paper_width;
	if ((p = get_property(b, "ps_paper_height")))
		ps_paper_height = strtol(p, NULL, 10);
	else
		ps_paper_height = paper_height;
	if ((p = get_property(b, "ps_top_margin")))
		ps_top_margin = strtol(p, NULL, 10);
	else
		ps_top_margin = margin;
	if ((p = get_property(b, "ps_left_margin")))
		ps_left_margin = strtol(p, NULL, 10);
	else
		ps_left_margin = margin;
	if ((p = get_property(b, "ps_right_margin")))
		ps_right_margin = strtol(p, NULL, 10);
	else
		ps_right_margin = margin;
	if ((p = get_property(b, "ps_bottom_margin")))
		ps_bottom_margin = strtol(p, NULL, 10);
	else
		ps_bottom_margin = margin;
}

static void makefonts(FILE *fp)
{
	int i;
	for (i = 0; i < 128; i += 8) {
		fprintf(fp, "/%s findfont\n", ps_fontname(i));

		fprintf(fp, "dup length dict begin\n");
		fprintf(fp, "  {1 index /FID ne {def} {pop pop} ifelse} forall\n");
		fprintf(fp, "  /Encoding ISOLatin1Encoding def\n");
		fprintf(fp, "  currentdict\n");
		fprintf(fp, "end\n");
		fprintf(fp, "/%s-ISOLatin1 exch definefont pop\n",
			ps_fontname(i));
	}
}

static void set_font(FILE *fp, int newfont)
{
	if (newfont == lastfont)
		return;
	lastfont = newfont;
	fprintf(fp, "/%s-ISOLatin1 findfont\n", ps_fontname(newfont));
	fprintf(fp, "%d scalefont\n", ps_font_size(newfont & SIZE_MASK));
	fprintf(fp, "setfont\n");
}

static void print_cell(FILE *fp, buffer *buf,
		int row, int col, int x_base, int y_base,
		int text_height)
{
	char b[1024], *p;
	int x_pos, y_pos, text_width, font_index;
	int f = ret_format(buf, row, col);

	if (f & BORDER_MASK) {
		fprintf(fp, "newpath\n");
		if (f & BORDER_BOTTOM) {
			fprintf(fp, "%d %d moveto\n", x_base, y_base);
			fprintf(fp, "%d %d lineto\n",
				x_base + cell_width(buf, col), y_base);
		}
		if (f & BORDER_RIGHT) {
			if (!(f & BORDER_BOTTOM))
				fprintf(fp, "%d %d moveto\n",
				  x_base + cell_width(buf, col), y_base);
			fprintf(fp, "%d %d lineto\n",
				x_base + cell_width(buf, col),
				y_base + cell_height(buf, row));
		}
		if (f & BORDER_TOP) {
			if (!(f & BORDER_RIGHT))
				fprintf(fp, "%d %d moveto\n",
					x_base + cell_width(buf, col),
					y_base + cell_height(buf, row));
			fprintf(fp, "%d %d lineto\n",
				x_base, y_base + cell_height(buf, row));
		}
		if (f & BORDER_LEFT) {
			if (!(f & BORDER_TOP))
				fprintf(fp, "%d %d moveto\n",
				 x_base, y_base + cell_height(buf, row));
			fprintf(fp, "%d %d lineto\n",
				x_base, y_base);
		}
		fprintf(fp, "stroke\n");
	}

	b[0] = '\0';
	if (ret_type(buf, row, col) != EMPTY)
		set_font(fp, ret_font(buf, row, col));
	ret_pvalue(b, buf, row, col, -1);

	font_index = ret_font(buf, row, col);
	text_width = ps_text_width(font_index, b);

	switch (f & HADJ_MASK) {
	case HADJ_CENTER:
		x_pos = (cell_width(buf, col) - text_width) / 2;
		break;
	case HADJ_RIGHT:
		x_pos = cell_width(buf, col) - text_width - 5;
		break;
	default:
		x_pos = 5;
	}

	switch (f & VADJ_MASK) {
	case VADJ_BOTTOM:
		y_pos = 5;
		break;
	case VADJ_TOP:
		y_pos = cell_height(buf, row) - text_height;
		break;
	default:	/* VADJ_CENTER */
		y_pos = (cell_height(buf, row) - text_height) / 2
			+ ps_font_descent(font_index);
	}

	if (b[0]) {	/* don't print empty cells */
		fprintf(fp, "newpath\n");
		fprintf(fp, "%d %d moveto\n", x_base + x_pos, y_base + y_pos);

	/* print letters and digits as is, the rest in octal */
		fprintf(fp, "(");
		for (p = b; *p; p++) {
			int c = *p & 0xFF;
			if (isalnum(c))
				putc(c, fp);
			else
				fprintf(fp, "\\%03o", c);
		}
		fprintf(fp, ")\n");
		fprintf(fp, "show\n");
	}
}

/* How to print the whole document without wasting paper

	The document can in theory contain up to 1000 rows by 1000 columns,
	but clearly in any real-world spreadsheet most of that area
	is unused and we would be printing hundreds of blank pages.
	The easy way is to check all the used rows, find the longest one
	and print the resulting square. This calls for a function print_area
	which takes the top left and bottom right coordinates as arguments
	and prints everything inbetween. The same function can
	also be used to print the block.
*/
static int page_height(buffer *buf, int r)
{
	int y_base, i;

	y_base = ps_paper_height - ps_top_margin;
	for (i = r; y_base > ps_bottom_margin; i++)
		y_base -= cell_height(buf, i);
	return (i-r);
}

static int page_width(buffer *buf, int c)
{
	int x_base, j;

	x_base = ps_left_margin;
	for (j = c; x_base+cell_width(buf, j) < ps_paper_width - ps_right_margin; j++)
		x_base += cell_width(buf, j);
	return (j-c);
}

static void expand(FILE *fp, int x, int y, char *from, char *name, int number)
{
	char b[256];
	int i;

	unsigned int text_height, text_width;

	/* set the font */
	set_font(fp, HELVETICA | SIZE_10);

	/* first make the real header/footer string */
	i = 0;
	while (*from && (i < 250)) {
		if (*from == '%') {
			from++;
			if (*from == '\0') break;
			else if (*from == 'n') {
				strcpy(b+i, name);
				i += strlen(b+i);
			} else if (*from == 'p') {
				sprintf(b+i, "%d", number);
				i += strlen(b+i);
			} else b[i++] = *from;
		} else b[i++] = *from;
		from++;
	}

	/* next center it around (x,y) */

	text_height = ps_font_height(HELVETICA | SIZE_10);
	text_width = ps_text_width(HELVETICA | SIZE_10, b);
	fprintf(fp, "newpath\n");
	fprintf(fp, "%d %d moveto\n", x - text_width/2, y + text_height/2);

	/* finally print the string */
	fprintf(fp, "(");
	for (i = 0; b[i]; i++) {
		int c = b[i] & 0xFF;
		if (isalnum(c))
			putc(c, fp);
		else
			fprintf(fp, "\\%03o", c);
	}
	fprintf(fp, ")\n");
	fprintf(fp, "show\n");
}

static void page_header(FILE *fp, buffer *b, int number)
{
	char *ps_page_header = get_property(b, "ps_page_header");
	if (!ps_page_header) ps_page_header = "%n";	/* buffer name */
	expand(fp, ps_paper_width/2, ps_paper_height-ps_top_margin/2,
		ps_page_header, b->name, number);
}

static void page_footer(FILE *fp, buffer *b, int number)
{
	char *ps_page_footer = get_property(b, "ps_page_footer");
	if (!ps_page_footer) ps_page_footer = "%p";	/* page number */
	expand(fp, ps_paper_width/2, ps_bottom_margin/2,
		ps_page_footer, b->name, number);
}

static int print_page(FILE *fp, buffer *buf, int number,
		int fromr, int fromc, int tor, int toc)
{
	int i, j=0, x_base, y_base;
	unsigned int text_height;
	int x, y;
	unsigned long width, height;

	page_header(fp, buf, number);
	y_base = ps_paper_height - ps_top_margin;
	for (i = fromr; y_base > ps_bottom_margin; i++) {
		y_base -= cell_height(buf, i);
		x_base = ps_left_margin;
		for (j = fromc;
		     x_base + cell_width(buf, j) <= ps_paper_width - ps_right_margin;
		     j++) {


			if (ret_type(buf, i, j) == EMBED) {
				ps_embed_print(fp, ret_text(buf, i, j),
					x_base, y_base);
				continue;
			}



			text_height = ps_font_height(ret_font(buf, i, j));

			print_cell(fp, buf,
				   i, j, x_base, y_base, text_height);
			x_base += cell_width(buf, j);
		}
	}

	/* print plugins here! */
	for (i = 0; i < buf->nplugin; i++) {
		long topx, topy, botx, boty, plx, ply;
		buffer_global_coords(buf, fromr, fromc, &topx, &topy);
		botx = topx+ps_paper_width-ps_left_margin-ps_right_margin;
		boty = topy+ps_paper_height-ps_top_margin-ps_bottom_margin;
		buffer_global_coords(buf, buf->plugin[i].row,
			buf->plugin[i].col, &plx, &ply);
		siag_plugin_size_get(buf->plugin[i].ph, &width, &height);
		if (plx+width < topx || plx > botx) continue;
		if (ply+height < topy || ply > boty) continue;
		x = ps_left_margin+plx-topx;
		y = ps_paper_height-ps_top_margin-(ply-topy)-height;
		/* move to the right place */
		fprintf(fp, "%d %d translate\n", x, y);
		/* and let the plugin do its thing */
		siag_plugin_print(buf->plugin[i].ph, fp);
		fprintf(fp, "%d %d translate\n", -x, -y);

	}

	page_footer(fp, buf, number);
	fprintf(fp, "showpage\n");
	return 0;
}

static int print_area(char *fn, buffer *buf,
		int fromr, int fromc, int tor, int toc)
{
	FILE *fp;
	int r, c;
	int pages;
	time_t t;

	read_properties(buf);

	fp = fopen(fn, "w");

	fprintf(fp, "%%!PS-Adobe-3.0 EPSF-3.0\n");
	fprintf(fp, "%%%%Creator: %s\n", VERSION);
	fprintf(fp, "%%%%Title: Siag\n");	/* todo: print better title */
	t = time(NULL);
	fprintf(fp, "%%%%CreationDate: %s\n", ctime(&t));
	fprintf(fp, "%%%%Pages: (atend)\n");
	fprintf(fp, "%%%%PageOrder: Ascend\n");
	fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n",
		0, 0, paper_width, paper_height);
	fprintf(fp, "%%%%Orientation: Portrait\n");
	fprintf(fp, "%%%%EndComments\n");

	/* Use ISO-Latin1 encoding */
	fprintf(fp, "%%%%BeginProlog\n");
	makefonts(fp);
	fprintf(fp, "%%%%EndProlog\n");

	pages = 0;
	for (r = fromr; r <= tor; r += page_height(buf, r)) {
		for (c = fromc; c <= toc; c += page_width(buf, c)) {
			pages++;
			lastfont = 0;	/* force setfont on top of page */
			fprintf(fp, "%%%%Page: %d %d\n", pages, pages);

			print_page(fp, buf, pages, r, c, tor, toc);
		}
	}
	fprintf(fp, "%%%%Trailer:\n");
	fprintf(fp, "%%%%Pages: %d\n", pages);
	fprintf(fp, "%%%%EOF\n");
	fclose(fp);
	return 0;
}

#define MAX(x,y) ((x)<(y)?(y):(x))

static int save(char *fn, buffer *buf)
{
	int fromr, fromc, tor, toc;
	int i, clu;

	fromr = 1;
	fromc = 1;
	tor = line_last_used(buf);
	toc = 1;
	for (i = 1; i <= tor; i++) {
		clu = col_last_used(buf, i);
		if (clu > toc) toc = clu;
	}
	for (i = 0; i < buf->nplugin; i++) {
		tor = MAX(tor, buf->plugin[i].row);
		toc = MAX(toc, buf->plugin[i].col);
	}

	return print_area(fn, buf, fromr, fromc, tor, toc);
}

/* conservative file format guessing:
   1. extension .ps
   2. first line starts with "%!PS"
*/
#define PS_MAGIC "%!PS"
static int myformat(char *fn)
{
	char *ext;
	FILE *fp;
	char b[100];

	ext = strrchr(fn, '.');
	if (!ext) return 0;	/* no extension */
	if (cstrcasecmp(ext, ".ps"))
		return 0;	/* wrong extension */
	if ((fp = fopen(fn, "r")) == NULL)
		return 0;	/* can't open */
	if (fgets(b, sizeof b, fp) && !strncmp(b, PS_MAGIC, strlen(PS_MAGIC))) {
		fclose(fp);
		return 1;
	}
	fclose(fp);
	return 0;
}

void fileio_ps_init(void)
{
	register_format(NULL, save, myformat, psformat = "Postscript (*.ps)");
}

