/*
   Siag, Scheme In A Grid
   Copyright (C) 1996  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/*
 * buffer.c
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "../common/cmalloc.h"
#include "../common/fonts.h"
#include "calc.h"

/*X
   Module name: buffer.c

   This module handles creating and deleting buffers.  It also contains
   a function for finding a certain buffer, given its name.
   X */

buffer *b_list;

static int
unique_name(char *p)
{
	buffer *b = b_list;

	do {
		b = b->next;
		if (!strcmp(p, b->name)) return 0;
	} while (b != b_list);
	return 1;
}

char *buffer_name(char *fn)
{
	static char name[256];
	int i;

	char *p = strrchr(fn, '/');
	if (p != NULL) p++;
	if (p == NULL || *p == '\0') p = fn;
	if (unique_name(p)) {
		strncpy(name, p, 255);
		return name;
	}
	for (i = 2; i < 100; i++) {
		sprintf(name, "%s<%d>", p, i);
		if (unique_name(name))
			return name;
	}
	return name;	/* unlikely and no problem */
}


/* Add string property to a buffer.
   If value is NULL, remove property.
   Returns value if successful, otherwise NULL. */
char *put_property(buffer *b, char *key, char *value)
{
	property_list *p, *q;

	if ((!b) || (!key)) return NULL;

	/* remove old value, if there is one */
	for (p = b->p_list; p; p = p->next) {
		if (!strcmp(key, p->key)) break;
	}
	if (p) {
		if (p == b->p_list) {
			b->p_list = p->next;
		} else {
			for (q = b->p_list; q->next != p; q = q->next);
			q->next = p->next;
		}
		cfree(p->key);
		cfree(p->value);
		cfree(p);
	}
	/* add new value, if there is one */
	if (value) {
		p = (property_list *)cmalloc(sizeof(property_list));
		p->key = cstrdup(key);
		p->value = cstrdup(value);
		p->next = b->p_list;
		b->p_list = p;
		return p->value;
	}
	return NULL;
}

char *get_property(buffer *b, char *key)
{
	property_list *p;

	if ((!b) || (!key)) return NULL;

	for (p = b->p_list; p; p = p->next) {
		if (!strcmp(p->key, key)) return p->value;
	}
	return NULL;
}

/*X
   buffer *new_buffer(spread **matrix, char *name, char *path)
   Creates a new buffer with name as name.  It is the responsability of
   the caller to make sure that the name is unique.  The new buffer is
   inserted in the buffer list.
   X */
buffer *new_buffer(char *name, char *path)
{
	buffer *b;

	b = (buffer *) cmalloc(sizeof(buffer));

	if (b_list == NULL)
		b_list = b;	/* first buffer ever */
	else
		b->next = b_list->next;		/* add to list */
	b_list->next = b;

	strncpy(b->name, name, 1000);
	strncpy(b->path, path, 1000);
	b->mark_pos.row = 1;
	b->mark_pos.col = 1;
	b->change = FALSE;
	b->recalc = FALSE;
	b->alloc_lines = 0;
	b->alloc_cols = NULL;
	b->used_lines = 0;
	b->used_cols = 0;
	b->sh = 20;
	b->sw = 80;
	b->sf = SIZE_10 | HELVETICA;
	b->height = NULL;
	b->width = NULL;
	b->matrix = NULL;
	b->p_list = NULL;
	b->nplugin = 0;
	b->plugin = NULL;
	return b;
}

/*X
   buffer *free_buffer(buffer *b)
   Removes b from the buffer list and frees the memory.
   If b_list points to b, it is set to point to the next buffer.
   If the buffer list is empty, b_list is set to NULL.
   Returns the next buffer in the buffer list, or NULL if it is empty.
   X */
buffer *free_buffer(buffer * b)
{
	int i;
	buffer *pb, *next;

	next = b->next;
	if (next == b)
		next = NULL;

	/* unlink from buffer list */
	for (pb = b_list;
	     pb->next != b && pb->next != pb;
	     pb = pb->next);
	pb->next = b->next;

	/* make sure b_list does not point to a deleted buffer */
	if (b_list == b)
		b_list = b_list->next;

	/* no buffers in the list => b_list = NULL */
	if (b_list == b)
		b_list = NULL;

	/* stop any plugins */
	for (i = 0; i < b->nplugin; i++) {
		siag_plugin_stop(b->plugin[i].ph);
		b->plugin[i].displayed = 0;
	}

	cfree((char *) b);

	return next;
}

static int plugin_name_taken(char *name)
{
	buffer *b = b_list;
	int i;

	do {
		for (i = 0; i < b->nplugin; i++)
			if (!strcmp(b->plugin[i].name, name)) return 1;
		b = b->next;
	} while (b != b_list);
	return 0;
}

/* make a unique plugin name */
void plugin_unique_name(char *from, char *to)
{
	int i;

	strcpy(to, from);
	for (i = 2; i < 100 && plugin_name_taken(to); i++)
		sprintf(to, "%d_%s", i, from);
}

/* figure out where to put the plugins */
char *plugin_basedir(buffer *buf, char *dir)
{
	int i;
	static char b[1024];
	pid_t pid = getpid();

	if (!dir) dir = b;
	sprintf(dir, "%s/%ld/%s", siag_basedir, (long)pid, buf->name);
	/* now, we don't want to keep any ugly characters */
	for (i = strlen(dir)-strlen(buf->name); dir[i]; i++)
		if (!isalnum(dir[i])) dir[i] = '_';
	return dir;
}

/*X
   buffer *find_buffer_by_name(char *name)
   Returns the buffer with the name name, or NULL if it can't be found.
   X */
buffer *find_buffer_by_name(char *name)
{
	buffer *b;

	if ((b = b_list) == NULL)
		return NULL;

	do {
		if (!strcmp(name, b->name))
			return b;
		b = b->next;
	} while (b != b_list);

	return NULL;
}

int cell_width(buffer * b, long col)
{
	if (b->width == NULL || col > b->used_cols) return b->sw;
	return b->width[col];
}

int cell_height(buffer * b, long row)
{
	if (b->height == NULL || row > b->used_lines) return b->sh;
	return b->height[row];
}

void set_width(buffer *b, long col, int width)
{
	long i;

	if (b->width == NULL) {
		b->width = (int *)cmalloc((col+1) * sizeof(int));
		for (i = 0; i < col; i++)
			b->width[i] = b->sw;
		b->used_cols = col;
	} else if (col > b->used_cols) {
		b->width = (int *)crealloc((void *)b->width, (col+1)*sizeof(int));
		for (i = b->used_cols+1; i < col; i++)
			b->width[i] = b->sw;
		b->used_cols = col;
	}
	b->width[col] = width;
}

void set_height(buffer *b, long row, int height)
{
	long i;

	if (b->height == NULL) {
		b->height = (int *)cmalloc((row+1)*sizeof(int));
		for (i = 0; i < row; i++)
			b->height[i] = b->sh;
		b->used_lines = row;
	} else if (row > b->used_lines) {
		b->height = (int *)crealloc((void *)b->height, (row+1)*sizeof(int));
		for (i = b->used_lines+1; i < row; i++)
			b->height[i] = b->sh;
		b->used_lines = row;
	}
	b->height[row] = height;
}

void buffer_global_coords(buffer *buf, long row, long col,
        long *x, long *y)
{
        long i;

        *x = *y = 0;

        for (i = 1; i < col; i++)
                *x += cell_width(buf, i);

        for (i = 1; i < row; i++)
                *y += cell_height(buf, i);
}

int buffer_plugin2index(buffer *buf, int ph)
{
	int n;

	if (!buf) return -1;

	for (n = 0; n < buf->nplugin; n++)
		if (ph == buf->plugin[n].ph) return n;

	return -1;
}

