
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>

#include "../common/fonts.h"
#include "sgrid.h"

static void gtk_sgrid_class_init    (GtkSGridClass *klass);
static void gtk_sgrid_init          (GtkSGrid      *sgrid);
static void gtk_sgrid_realize       (GtkWidget           *widget);
static void gtk_sgrid_size_allocate (GtkWidget           *widget,
				     GtkAllocation       *allocation);
static void gtk_sgrid_size_request  (GtkWidget      *widget,
				     GtkRequisition *requisition);
static gint gtk_sgrid_expose        (GtkWidget      *widget,
				     GdkEventExpose *event);

guint
gtk_sgrid_get_type ()
{
  static guint sgrid_type = 0;

  if (!sgrid_type)
    {
      GtkTypeInfo sgrid_info =
      {
	"GtkSGrid",
	sizeof (GtkSGrid),
	sizeof (GtkSGridClass),
	(GtkClassInitFunc) gtk_sgrid_class_init,
	(GtkObjectInitFunc) gtk_sgrid_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL
      };
      sgrid_type = gtk_type_unique (gtk_widget_get_type (), &sgrid_info);
    }
  return sgrid_type;
}

static void
gtk_sgrid_class_init (GtkSGridClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;

  widget_class->realize = gtk_sgrid_realize;
  widget_class->size_allocate = gtk_sgrid_size_allocate;
  widget_class->size_request = gtk_sgrid_size_request;
  widget_class->expose_event = gtk_sgrid_expose;
/*   widget_class->button_press_event = gtk_sgrid_button_press; */
}

static void
gtk_sgrid_init (GtkSGrid *sgrid)
{
  GTK_WIDGET_SET_FLAGS (sgrid, GTK_BASIC);
  GTK_WIDGET_SET_FLAGS (sgrid, GTK_CAN_FOCUS);
  GTK_WIDGET_UNSET_FLAGS (sgrid, GTK_HAS_FOCUS);
  /* set all the stff in a sgrid */
}

GtkWidget*
gtk_sgrid_new(gint rows, 
	      gint cols)
{
  GtkSGrid *sgrid;

  sgrid = gtk_type_new (gtk_sgrid_get_type ());
  sgrid->prot_row = 1;
  sgrid->prot_col = 1;
  sgrid->top_row = 1;
  sgrid->top_col = 1;
  sgrid->point_row = 0;
  sgrid->point_col = 0;
  sgrid->sel_top_row = 0;
  sgrid->sel_bottom_row = 0;
  sgrid->sel_left_col = 0;
  sgrid->sel_right_col = 0;
  
  sgrid->max_row = rows;
  sgrid->max_col = cols;
  sgrid->grid_lines = 1;
  sgrid->visible_cursor = 0;
  
  /* initialize rows and cols ? */
  /* maybe gc's ? */

  return GTK_WIDGET (sgrid);
}

static void 
gtk_sgrid_size_request (GtkWidget      *widget,
		       GtkRequisition *requisition)
{
  GtkSGrid *sgrid;
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SGRID (widget));
  g_return_if_fail (requisition != NULL);
  sgrid = GTK_SGRID (widget);
  requisition->width = GTK_WIDGET (sgrid)->requisition.width;
  requisition->height =   GTK_WIDGET (sgrid)->requisition.height;
}

static void
gtk_sgrid_size_allocate (GtkWidget     *widget,
			GtkAllocation *allocation)
{
  GdkEventConfigure event;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SGRID (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);

      event.type = GDK_CONFIGURE;
      event.window = widget->window;
      event.x = allocation->x;
      event.y = allocation->y;
      event.width = allocation->width;
      event.height = allocation->height;

      gtk_widget_event (widget, (GdkEvent*) &event);
    }
}

void gtk_sgrid_size (GtkSGrid *sgrid, gint width, gint height)
{
  g_return_if_fail (sgrid != NULL);
  g_return_if_fail (GTK_IS_SGRID (sgrid));

  GTK_WIDGET (sgrid)->requisition.width = width;
  GTK_WIDGET (sgrid)->requisition.height = height;
}

static int inblock(GtkSGrid *tw, int r, int c)
{
  return ((r >= tw->sel_top_row) &&
	  (r <= tw->sel_bottom_row) &&
	  (c >= tw->sel_left_col) &&
	  (c <= tw->sel_right_col));
}

/* return the type of the cell */
/* if a query function isn't defined, assume TEXT */
static int ret_type(GtkSGrid *tw, int row, int col)
{
  if (tw->type)
    return (*tw->type)(tw, tw->data, row, col);
  return SGRID_TEXT;
}

/* return the printable value for the cell */
/* if a query function isn't defined, return NULL */
static char *ret_pvalue(GtkSGrid *tw, int row, int col)
{
  if (tw->text)
    return (*tw->text)(tw, tw->data, row, col);
  else return NULL;
}

static gint cell_width(GtkSGrid *tw, int col)
{
  gint wi;
  
  if (tw->col_width) {
    wi = (*tw->col_width)(tw, tw->data, col);
    return wi;
  }
  return tw->default_width;
}

static gint cell_height(GtkSGrid *tw, int row)
{
  gint h;
  
  if (tw->row_height) {
    h = (*tw->row_height)(tw, tw->data, row);
    return h;
  }
  return tw->default_height;
}

static gint cell_format(GtkSGrid *tw, int row, int col)
{
  if (tw->format)
    return (*tw->format)(tw, tw->data, row, col);
  return HELVETICA | SIZE_10;
}

static void draw_cell(GtkSGrid *tw, GdkPixmap *scribble,
			int row, int col, int x_base, int y_base)
{
  GtkWidget *w;
  gchar *p;
  GdkFont *font;                  
  GdkGC *gc;
  gint x_pos, y_pos;
  guint text_height, text_width;
  gulong format = 0;
  guchar highlight = 0;
  gint cw, ch;

  w = GTK_WIDGET(tw);
  if (row > tw->max_row || col > tw->max_col) return;

  if (ret_type(tw, row, col) == SGRID_PIXMAP) {
    g_warning("The sgrid widget does not yet display pixmaps");
    return;
  }
 
  format = cell_format(tw, row, col);
  gc = w->style->fg_gc[GTK_STATE_NORMAL];
  font = GTK_WIDGET(tw)->style->font;

  p = ret_pvalue(tw, row, col);
  if (p == NULL) p = "";
  
#if 1	/* this seems to be necessary to keep X from crashing */
  if (strlen(p) > 1000) p[1000] = '\0';
#endif

  text_height = font->ascent;
  text_width = gdk_string_width(font, p);
  cw = cell_width(tw, col);
  ch = cell_height(tw, row);

  if (cw<=0 || ch <=0) return;

  if (highlight) {
    GdkGCValues values;
    gdk_gc_get_values(gc,&values);
    gdk_gc_set_foreground(gc,&(values.background));
    gdk_draw_rectangle(scribble, gc,
		       TRUE,
		       x_base, y_base,
		       cw, ch);
    gdk_gc_set_foreground(gc,(&values.foreground));
  }

  /* FIXME */
  switch (format & HADJ_MASK) {
  case HADJ_CENTER:
    x_pos = (cw-text_width) / 2;
    break;
  case HADJ_RIGHT:
    x_pos = cw-text_width-5;
    break;
  default:	/* HADJ_LEFT */
    x_pos = 5;
  }
  
  switch (format & VADJ_MASK) {
  case VADJ_BOTTOM:
    y_pos = ch-5;
    break;
  case VADJ_TOP:
    y_pos = text_height+font->descent;
    break;
  default:	/* VADJ_CENTER */
    y_pos = (ch+text_height)/2
      -font->descent;
  }
  
  gdk_draw_string (scribble, font, gc, x_base+x_pos, y_base+y_pos, p);
  
  if (tw->grid_lines) {
    gdk_draw_line(scribble, GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		  x_base, y_base+ch, x_base+cw, y_base+ch);
    gdk_draw_line(scribble, GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		  x_base+cw, y_base+ch, x_base+cw, y_base);
  }

  /* these lines are from the format */
  if (format & BORDER_LEFT)
    gdk_draw_line(scribble, GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		  x_base+1, y_base+1, x_base+1, y_base+ch-1);
  if (format & BORDER_RIGHT)
    gdk_draw_line(scribble, GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		  x_base+cw-1, y_base+1, x_base+cw-1, y_base+ch-1);
  if (format & BORDER_TOP)
    gdk_draw_line(scribble, GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		  x_base+1, y_base+1, x_base+cw-1, y_base+1);
  if (format & BORDER_BOTTOM)
    gdk_draw_line(scribble, GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		  x_base+1, y_base+ch-1, x_base+cw-1, y_base+ch-1);

  if (inblock(tw, row, col)) 
    gdk_draw_rectangle(scribble, tw->xor_gc, TRUE, x_base, y_base, cw, ch);
   
  /* line around grid - my bad way to do it - RW */

  if (row == (tw->top_row))
    gdk_draw_line(scribble,GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		  x_base+1, y_base, x_base+cw-1, y_base);
  if (col == (tw->top_col))
      gdk_draw_line(scribble,GTK_WIDGET(tw)->style->fg_gc[GTK_STATE_NORMAL],
		    x_base, y_base+1, x_base, y_base+ch);
} 

static void cell_row(GtkSGrid *tw, GdkPixmap *scribble,
		     int width, int y_base, int i)
{
  int j, x_base;
  x_base = 0;
  
  for (j = 1; (j < tw->prot_col) && (x_base < width); j++) {
    draw_cell(tw, scribble, i, j, x_base, y_base);
    x_base += cell_width(tw, j);
  }
  for (j = tw->top_col; x_base < width; j++) {
    draw_cell(tw, scribble, i, j, x_base, y_base);
    x_base += cell_width(tw, j);
  }
}

static void draw_sgrid(GtkSGrid *tw, /*GdkPixmap*/GdkWindow *d)
{
  gint y_base = 0;
  gint width = GTK_WIDGET(tw)->allocation.width;
  gint height = GTK_WIDGET(tw)->allocation.height;
  gint i;
  
  for (i = 1; (i < tw->prot_row) && (y_base < height); i++) {
    cell_row(tw, d, width, y_base, i);
    y_base += cell_height(tw, i);
  }
  for (i = tw->top_row; y_base < height; i++) {
    cell_row(tw, d, width, y_base, i);
    y_base += cell_height(tw, i);
  }
}

/* create, draw and return a pixmap of the stage at time now.
   Caller must free */
/* Strategy:
   1. Clear the window, or create a new empty Pixmap
   2. Paint a rectangle for the block, if there is one (coords != 0)
   3. Draw the grid
   4. Draw the data
   5. Draw the cursor using XOR, if one is desired (coords != 0)
   The purpose of using XOR is to make sure the cursor is visible
   on any background. Technicolor is a potential side effect.
*/

#if 0
static GC get_gc(Widget w, unsigned long fg, unsigned long bg)
{
  unsigned long valuemask = 0;
  XGCValues values;
  GC gc = XCreateGC(XtDisplay(w), XtWindow(w),
		    valuemask, &values);
  XSetForeground(XtDisplay(w), gc, fg);
  XSetBackground(XtDisplay(w), gc, bg);
  return gc;
}
#endif /* 0 */

static void gtk_sgrid_realize (GtkWidget *widget)
{
  GtkSGrid *sgrid;
  GdkWindowAttr attributes;
  GdkGCValues values;
  gint attributes_mask;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SGRID (widget));

  sgrid = GTK_SGRID (widget);
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_ENTER_NOTIFY_MASK |
			    GDK_LEAVE_NOTIFY_MASK |
			    GDK_KEY_PRESS_MASK |
			    GDK_KEY_RELEASE_MASK |
			    GDK_BUTTON_PRESS_MASK);

  /* FIXME: add all other wanted events */

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (widget->parent->window, &attributes, 
				   attributes_mask);
  gdk_window_set_user_data (widget->window, sgrid);

  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
  sgrid->bg = &widget->style->white;

  gdk_window_set_background (widget->window, &widget->style->white);

  sgrid->non_gr_exp_gc = gdk_gc_new (widget->window);
  gdk_gc_set_exposures (sgrid->non_gr_exp_gc, FALSE);

  values.foreground = widget->style->white;
  values.function = GDK_XOR;
  values.line_width = 2;
  values.line_style = GDK_LINE_SOLID;
  values.cap_style = GDK_CAP_BUTT;;
  values.join_style = GDK_JOIN_MITER;
  values.subwindow_mode = GDK_INCLUDE_INFERIORS;
  sgrid->xor_gc = gdk_gc_new_with_values (widget->window,
					  &values,
					  GDK_GC_LINE_WIDTH |
					  GDK_GC_LINE_STYLE |
					  GDK_GC_CAP_STYLE |
					  GDK_GC_FOREGROUND |
					  GDK_GC_JOIN_STYLE |
					  GDK_GC_FUNCTION |
					  GDK_GC_SUBWINDOW);

/*   gdk_window_set_background (grid->window, &widget->style->white); */

/*   gdk_window_show (grid->window); */
}

#if 0
static void Destroy(Widget w)
{
	TableWidget tw = (TableWidget) w;

	XFreeGC(XtDisplay(w), tw->clear_gc);
	XFreeGC(XtDisplay(w), tw->cell_gc);
	XFreeGC(XtDisplay(w), tw->grid_gc);
	XFreeGC(XtDisplay(w), tw->cursor_gc);
}
#endif /* 0 */

static gint cell_next_row(GtkSGrid *tw, int row)
{
        if (row+1 == tw->prot_row) return tw->top_row;
        return row+1;
}

static gint cell_next_col(GtkSGrid *tw, int col)
{
        if (col+1 == tw->prot_col) return tw->top_col;
        return col+1;
}

static gint cell_prev_row(GtkSGrid *tw, int row)
{
        if (row == tw->top_row) return tw->prot_row-1;
        return row-1;
}

static int cell_prev_col(GtkSGrid *tw, int col)
{
        if (col == tw->top_col) return tw->prot_col-1;
        return col-1;
}

void gtk_sgrid_rowcol_to_coord(GtkSGrid *sgrid,
			       gint cell_row, gint cell_col,
			       gint *cell_x,  gint *cell_y)
{
        int i;

        *cell_y = 0;
        for (i = 1; i < sgrid->prot_row; i++)
                *cell_y += cell_height(sgrid, i);

        while (cell_row < sgrid->top_row) {
                *cell_y -= cell_height(sgrid, cell_row);
                cell_row = cell_next_row(sgrid, cell_row);
        }
        while (cell_row > sgrid->top_row) {
                cell_row = cell_prev_row(sgrid, cell_row);
                *cell_y += cell_height(sgrid, cell_row);
        }
        *cell_x = 0;
        for (i = 1; i < sgrid->prot_col; i++)
                *cell_x += cell_width(sgrid, i);

        while (cell_col < sgrid->top_col) {
                *cell_x -= cell_width(sgrid, cell_col);
                cell_col = cell_next_col(sgrid, cell_col);
        }
        while (cell_col > sgrid->top_col) {
                cell_col = cell_prev_col(sgrid, cell_col);
                *cell_x += cell_width(sgrid, cell_col);
        }
}

void gtk_sgrid_coord_to_rowcol(GtkSGrid *tw,
			       int cur_x, int cur_y,
			       int *cur_row, int *cur_col)
{
  int prot_x = 0, prot_y = 0, i;
  
  for (i = 1; i < tw->prot_col; i++)
    /*              prot_x += cell_width(w->buf, i);
     */              cur_x -= cell_width(tw, i);
  for (i = 1; i < tw->prot_row; i++)
    /*              prot_y += cell_height(w->buf, i);
     */              cur_y -= cell_height(tw, i);
  
  *cur_row = tw->top_row;
  *cur_col = tw->top_col;
  while (cur_y < prot_y) {
    cur_y += cell_height(tw, *cur_row);
    (*cur_row) = cell_prev_row(tw, *cur_row);
  }
  while (cur_y > cell_height(tw, *cur_row)) {
    cur_y -= cell_height(tw, *cur_row);
    (*cur_row) = cell_next_row(tw, *cur_row);
  }
  while (cur_x < prot_x) {
    cur_x += cell_width(tw, *cur_col);
    (*cur_col) = cell_prev_col(tw, *cur_col);
  }
  while (cur_x > cell_width(tw, *cur_col)) {
    cur_x -= cell_width(tw, *cur_col);
    (*cur_col) = cell_next_col(tw, *cur_col);
  }
  
}

static int move_top(GtkSGrid *tw)
{
  int pr_scr_flag = FALSE;
  int cur_x, cur_y, prot_x = 0, prot_y = 0, i;
  unsigned int width, height;
  
  for (i = 1; i < tw->prot_col; i++)
    prot_x += cell_width(tw, i);
  for (i = 1; i < tw->prot_row; i++)
    prot_y += cell_height(tw, i);
  
  /* Figure out how big the window is */
  width = GTK_WIDGET(tw)->allocation.width;
  height = GTK_WIDGET(tw)->allocation.height;
  
  gtk_sgrid_rowcol_to_coord(tw,
			    tw->point_row, tw->point_col,
			    &cur_x, &cur_y);
  /* this isn't efficient, but it will work */
  while (cur_y < prot_y) {
    tw->top_row--;
    cur_y += cell_height(tw, tw->top_row);
    pr_scr_flag = TRUE;
  }
  while (cur_y + cell_height(tw, tw->point_row) > height) {
    cur_y -= cell_height(tw, tw->top_row);
    tw->top_row++;
    pr_scr_flag = TRUE;
  }
  while (cur_x < prot_x) {
    tw->top_col--;
    cur_x += cell_width(tw, tw->top_col);
    pr_scr_flag = TRUE;
  }
  while (cur_x + cell_width(tw, tw->point_col) > width) {
    cur_x -= cell_width(tw, tw->top_col);
    tw->top_col++;
    pr_scr_flag = TRUE;
  }
  return pr_scr_flag;
}

static void toggle_cursor(GtkSGrid *tw)
{
  int cur_x, cur_y, w, h, row, col;
  row = tw->point_row;
  col = tw->point_col;

  if (row < 0 || (row >tw->prot_row && row < tw->top_row)) return;
  if (col < 0 || (col >tw->prot_col && col < tw->top_col)) return;

  gtk_sgrid_rowcol_to_coord(tw, row, col,
			    &cur_x, &cur_y);

  w = cell_width(tw, tw->point_col);
  h = cell_height(tw, tw->point_row);
  if (w<4 || h<4) return;
  gdk_draw_rectangle(GTK_WIDGET(tw)->window,
		     tw->xor_gc, 
		     FALSE,
		     cur_x+3, cur_y+3,
		     w-5,
		     h-5);
}

static gint gtk_sgrid_expose (GtkWidget *widget, GdkEventExpose *event)
{
  GtkSGrid *sgrid;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SGRID (widget), FALSE);

  if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget)
	&& GTK_WIDGET_REALIZED(widget))
    {
      int do_redisplay = FALSE;

      sgrid = GTK_SGRID (widget);
      if (sgrid->visible_cursor) do_redisplay = move_top(sgrid);
      gdk_window_set_background(widget->window, sgrid->bg);
      gdk_window_clear_area (widget->window,
 			     0, 0,
 			     widget->allocation.width,
 			     widget->allocation.height);
      draw_sgrid(sgrid, widget->window);
      /* draw the cursor, perhaps? */
      if (sgrid->visible_cursor) {
	toggle_cursor(sgrid);
      }
    }
  return TRUE;
}

void gtk_sgrid_set_point (GtkSGrid *sgrid, gint row, gint col) 
{
  sgrid->point_row = row;
  sgrid->point_col = col;
  /* should apply a little sense here wrt display or not, but this is easiest */
  gtk_sgrid_expose(GTK_WIDGET(sgrid),NULL);

}  

void gtk_sgrid_get_point (GtkSGrid *sgrid, gint *row, gint *col) 
{
  *row = sgrid->point_row;
  *col = sgrid->point_col;
}  

