/*
   Egon Animator
   Copyright (C) 1997-1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * fileio_egon.c
 */

/*
   Brief description of the file format:

   This is a completely new file format, similar to the one used in
   Siag and PW. It includes support for structured files and plugins.
   Images are stored together with the rest of the document.

   #comment			A comment; ignored
   .delta t			Time between frames in ms
   .duration t			Total playing time in ms
   .geometry w h		Visible width and height
   .background fn		Name of background image file
   .object type x y w h v c f s	Initial object appearance
   .tick time x y w h v		Object appearance at time

	x y = position
	w h = size
	v = visible (1 or 0)
	c = colour (numeric code)
	f = font (numeric code)
	s = text
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/xpm.h>

#include "../xcommon/Animator.h"
#include "egon.h"
#include "types.h"
#include "user_interface.h"
#include "../common/common.h"
#include "../common/fonts.h"
#include "../common/cmalloc.h"
#include "../xcommon/embed.h"

/* returns the name of a component in a structured file. Caller must free */
static char *file_name(buffer *buf, char *n)
{
        char b[1024];
        plugin_basedir(buf, b);
        strcat(b, "/");
        strcat(b, n);
        return cstrdup(b);
}

/* use structured format to save images and plugins */
static int save_flat(char *fn, buffer * buf)
/* Returns: 0 if successful, otherwise 1 */
{
	ani_object *o;
	ani_script *s;
	char *string;
	FILE *fp = fopen(fn, "w");
	char b[1024];

	if (!fp) return 1;

	plugin_basedir(buf, b);

	fprintf(fp, "# Creator: %s\n", VERSION);
	fprintf(fp, "# %s\n", buf->name);
	fprintf(fp, ".delta %ld\n", buf->delta);
	fprintf(fp, ".duration %ld\n", buf->duration);
	fprintf(fp, ".geometry %d %d\n", buf->width, buf->height);
	if (buf->bg)
		fprintf(fp, ".background %s\n", buf->bg);
	for (o = buf->cast; o; o = o->next) {
		fprintf(fp, "# Object %s\n", o->name);
		/* strip off directory part of image files in
		   a structured file */
		if (o->type == ANI_PIXMAP &&
				!strncmp(o->string, b, strlen(b)))
			string = strrchr(o->string, '/')+1;
		else
			string = o->string;
		fprintf(fp, ".object %d %s %d %d %s\n",
			o->type, o->name, o->color, o->font,
			string?string:"");
		for (s = o->script; s; s = s->next)
			fprintf(fp, ".tick %ld %d %d %d %d %d\n",
				s->time, s->x, s->y,
				s->width, s->height, s->visible);
	}
	fclose(fp);
	return 0;
}				/* save */

static int load_flat(char *fn, buffer *buf)
/* Returns: 0 if successful, otherwise 1 */
{
	FILE *fp;
	char instring[1024], name[256], string[1024];
	ani_object *o = NULL;
	ani_script *s = NULL;
	char b[1024];

	if ((fp = fopen(fn, "r")) == NULL)
		return 1;

	plugin_basedir(buf, b);

	buf->delta = 100;
	buf->duration = 4000;
	buf->now = 0;
	buf->width = 600;
	buf->height = 400;
	buf->cast = NULL;
	buf->state = ANI_STOP;
	buf->bg = NULL;
	buf->change = FALSE;
	buf->plugin = NULL;
	buf->nplugin = 0;

	while (fgets(instring, 1024, fp) != NULL) {
		if (instring[0] == '#') {
			continue;
		} else if (!strncmp(instring, ".delta ", 7)) {
			sscanf(instring, ".delta %ld", &buf->delta);
		} else if (!strncmp(instring, ".duration ", 10)) {
			sscanf(instring, ".duration %ld", &buf->duration);
		} else if (!strncmp(instring, ".geometry ", 10)) {
			sscanf(instring, ".geometry %d %d",
				&buf->width, &buf->height);
		} else if (!strncmp(instring, ".object ", 8)) {
			if (buf->cast == NULL) {
				o = (ani_object *)cmalloc(sizeof(ani_object));
				buf->cast = o;
			} else {
				o->next = (ani_object *)cmalloc(sizeof(ani_object));
				o = o->next;
			}
			o->next = NULL;
			o->script = NULL;
			name[0] = string[0] = '\0';
			sscanf(instring,
				".object %d %s %d %d %[^\n]",
				&o->type, name, &o->color, &o->font, string);
			o->name = cstrdup(name);
			if (o->type == ANI_PIXMAP && string[0] != '/') {
				sprintf(instring, "%s/%s", b, string);
				o->string = cstrdup(instring);
			} else {
				o->string = cstrdup(string);
			}
		} else if (!strncmp(instring, ".tick ", 6)) {
			if (o->script == NULL) {
				s = (ani_script *)cmalloc(sizeof(ani_script));
				o->script = s;
			} else {
				s->next = (ani_script *)cmalloc(sizeof(ani_script));
				s = s->next;
			}
			s->next = NULL;
			sscanf(instring,
				".tick %ld %d %d %d %d %d",
				&s->time, &s->x, &s->y, &s->width, &s->height,
				&s->visible);
		}
	}

	fclose(fp);
	return 0;
} /* load_flat */

/* file format guessing:
   1. extension .egon
   2. Starts with "# Creator: Egon Animator"
*/
#define EGON_MAGIC "# Creator: Egon Animator"

static int flatfile(char *fn)
{
        char b[100];
        FILE *fp = fopen(fn, "r");
        if (!fp) return 0;
        if (fgets(b, sizeof b, fp) &&
                        !strncmp(b, EGON_MAGIC, strlen(EGON_MAGIC))) {
                fclose(fp);
                return 1;
        }
        return 0;
}

static int tryuntar(char *fn)
{
        char b[1024];
        int result;
        struct stat statbuf;

        sprintf(b,
                "mkdir -p %s/untar;"
                "cat %s|(cd %s/untar;tar xf - INDEX.egon) 2> /dev/null",
                siag_basedir, fn, siag_basedir);
        system(b);
        sprintf(b, "%s/untar/INDEX.egon", siag_basedir);
        result = !stat(b, &statbuf);
        sprintf(b, "rm -rf %s/untar", siag_basedir);
        system(b);
        return result;
}

static int save(char *fn, buffer *buf)
{
        char cmd[1024];
        char *dir, *b, *p;
        int i, result;
	ani_object *o;

	for (o = buf->cast; o; o = o->next)
		if (o->type == ANI_PIXMAP) break;

        if (o == NULL && buf->nplugin == 0) {
	        /* no images or plugins, save as usual */
                return save_flat(fn, buf);
        }

        /* save all the plugins */
        dir = file_name(buf, "");
        p = strrchr(dir, '/');
        if (p) *p = '\0';
        mkdir(dir, 0700);
        for (i = 0; i < buf->nplugin; i++) {
                b = file_name(buf, buf->plugin[i].name);
                egon_plugin_save(buf->plugin[i].ph, b);
                cfree(b);
        }
        /* save the main file */
        b = file_name(buf, "INDEX.egon");
        result = save_flat(b, buf);
        cfree(b);
        /* tar up the lot */
        sprintf(cmd, "(cd %s;tar cf - *)>%s", dir, fn);
        system(cmd);
        cfree(dir);
        return result;
}

static int load(char *fn, buffer *buf)
{
        char cmd[1024];
        char *p;
        int n;

        /* old files don't have the MAGIC but we still want to load them.
           So first try flat, then tar, then flat again. */
        if (flatfile(fn) || !tryuntar(fn))
                return load_flat(fn, buf);

        p = plugin_basedir(buf, NULL);
        sprintf(cmd,
                "mkdir -p %s;"
                "cat %s|(cd %s;tar xf -)",
                p, fn, p);
        system(cmd);
        p = file_name(buf, "INDEX.egon");
        n = load_flat(p, buf);
        cfree(p);
        return n;
}

static int myformat(char *fn)
{
	char *ext;
	FILE *fp = NULL;
	char b[250];
	int result;

	result = ((ext = strrchr(fn, '.')) &&
		!cstrcasecmp(ext, ".egon") &&
		(fp = fopen(fn, "r")) &&
		fgets(b, sizeof b, fp) &&
		!strncmp(b, EGON_MAGIC, strlen(EGON_MAGIC)));
	if (fp) fclose(fp);
	return result || tryuntar(fn);
}

void fileio_egon_init(void)
{
	register_format(load, save, myformat, "Egon Animator (*.egon)");
}

