;;
;; Runtime Library for Egon Animator
;; Copyright 1996-1998 Ulric Eriksson
;;

; Load the runtime library from SIOD
(require (string-append SIAGHOME "/siod/siod.scm"))

; These are missing
(define (caaaar l) (caar (caar l)))
(define (cdaaar l) (cdar (caar l)))
(define (cadaar l) (cadr (caar l)))
(define (cddaar l) (cddr (caar l)))
(define (caadar l) (caar (cdar l)))
(define (cdadar l) (cdar (cdar l)))
(define (caddar l) (cadr (cdar l)))
(define (cdddar l) (cddr (cdar l)))
(define (caaadr l) (caar (cadr l)))
(define (cdaadr l) (cdar (cadr l)))
(define (cadadr l) (cadr (cadr l)))
(define (cddadr l) (cddr (cadr l)))
(define (caaddr l) (caar (cddr l)))
(define (cdaddr l) (cdar (cddr l)))
(define (cadddr l) (cadr (cddr l)))
(define (cddddr l) (cddr (cddr l)))

; Set up the menu
(require (string-append SIAGHOME "/egon/menu.scm"))

; X string definitions
(require (string-append SIAGHOME "/xcommon/StringDefs.scm"))

; Form interface
(require (string-append SIAGHOME "/xcommon/form.scm"))

; Animation interface
(require (string-append SIAGHOME "/egon/animator.scm"))

(define (llpr x)
	(puts x))

;; Handle the "position"
(define (make-position row col)
	(list row col))

(define (position-row p)
	(car p))

(define (position-col p)
	(cadr p))

; Find a position not out of bounds
(define (safe-position row col)
  (if (< row 1) (set! row 1))
  (if (> row (max-lines)) (set! row (max-lines)))
  (if (< col 0) (set! col 0))
  (if (> col (col-last-used row)) (set! col (col-last-used row)))
  (make-position row col))

(define (get-point-row)
  (position-row (get-point)))

(define (get-point-col)
  (position-col (get-point)))

(define (set-point-row row)
  (set-point (safe-position row (get-point-col))))

(define (set-point-col col)
  (set-point (safe-position (get-point-row) col)))



(define (<= x y)
	(or (< x y) (= x y)))

(define (>= x y)
	(or (> x y) (= x y)))

(define (at-beginning-of-line p)
	(<= (position-col p) 0))

(define (at-end-of-line p)
	(>= (position-col p) (col-last-used (position-row p))))


(define (find-beginning-of-line p)
	(make-position (position-row p) 0))

(define (find-end-of-line p)
	(make-position (position-row p) (col-last-used (position-row p))))

(define (at-top-of-buffer p)
	(<= (position-row p) 1))

(define (at-bottom-of-buffer p)
	(>= (position-row p) (line-last-used)))

(define (at-beginning-of-buffer p)
	(and (at-top-of-buffer p) (at-beginning-of-line p)))

(define (at-end-of-buffer p)
	(and (at-bottom-of-buffer p) (at-end-of-line p)))


(define (find-beginning-of-buffer)
	(make-position 1 0))

(define (find-end-of-buffer)
	(let ((row (line-last-used)))
		(make-position row (col-last-used row))))


(define (line-forward p)
	(if (< (position-row p) (max-lines))
		(safe-position (+ (position-row p) 1) (position-col p))
		p))

(define (line-backward p)
	(if (at-top-of-buffer p)
		p
		(safe-position (- (position-row p) 1) (position-col p))))


(define (char-forward p)
	(if (at-end-of-line p)
		(find-beginning-of-line (line-forward p))
		(make-position (position-row p) (+ (position-col p) 1))))


(define (char-backward p)
	(if (at-beginning-of-line p)
		(if (at-beginning-of-buffer p)
			p
			(find-end-of-line (line-backward p)))
		(make-position (position-row p) (- (position-col p) 1))))



(define (backward-char)
	(set-point (char-backward (get-point))))

(define (forward-char)
	(set-point (char-forward (get-point))))


(define (set-mark-command)
	(let ()
		(set-mark (get-point))
		(llpr "Mark set")))

(define (exchange-point-and-mark)
	(let ((p (get-point)))
		(set-point (get-mark))
		(set-mark p)))

(define (beginning-of-buffer)
	(set-mark (get-point))
	(set-point (find-beginning-of-buffer)))

(define (end-of-buffer)
	(set-mark (get-point))
	(set-point (find-end-of-buffer)))

(define (top-of-buffer)
	(set-point-row 1))

(define (bottom-of-buffer)
	(set-point-row (line-last-used)))

(define (beginning-of-line)
	(set-point-col 0))

(define (end-of-line)
	(set-point-col (col-last-used (get-point-row))))

(define (next-line)
	(set-point (line-forward (get-point))))

(define (previous-line)
	(set-point (line-backward (get-point))))

(define BUFFER-ROWS 1000)
(define BUFFER-COLS 1000)

(define (select-all)
  (let* ((p (get-point))
	 (r (line-last-used))
	 (c (col-last-used r)))
    (set-mark (make-position 1 1))
    (set-point (make-position r c))
    (set-block)
    (set-point p)))

(define (execute-extended-command)
  (execute-interpreter-command 'SIOD))

(define (new-egon)
  (spawn "egon"))

(define (do-help helpfile)
  (spawn (string-append SIAGHELP " file://localhost"
	 SIAGDOCS "/" helpfile)))

(define (help-contents)
  (do-help "egon/docs/egon.html"))

(define (help-copyright)
  (do-help "egon/docs/COPYING"))

(define (help-for-help)
  (do-help "common/docs/siaghelp.html"))

(define (help-search)
  (do-help "common/docs/search.html"))

(define (keyboard-quit)
  (llpr "Quit"))

(define (no-op)
  (llpr "This command does nothing"))


(define (newline)
  (begin
    (split-line (position-row (get-point))
		(position-col (get-point)))
    (forward-char)
    (change-style (style-follower (get-style (position-row (get-point)))))))

(define (recenter)
  (set-pr-scr))


(define (delete-char)
	(if (at-end-of-line (get-point))
		(join-lines (position-row (get-point)))
		(remove-text 1)))


(define (delete-char-backward)
	(if (at-beginning-of-buffer (get-point))
		(bell)
		(let ()
			(backward-char)
			(delete-char))))

(define (kill-line)
  (let ((row (position-row (get-point)))
	(col (position-col (get-point))))
    (if (at-end-of-line (get-point))
      (join-lines row)
      (begin
	(split-line row col)
	(delete-lines (+ row 1) 1)))
    (set-pr-scr)))

(define (delete-block)
  (let ((ur (position-row (get-blku)))
	(uc (position-col (get-blku)))
	(lr (position-row (get-blkl)))
	(lc (position-col (get-blkl))))
    (split-line lr lc)
    (split-line ur uc)
    (delete-lines (+ ur 1) (+ (- lr ur) 1))
    (join-lines ur)
    (set-pr-scr)))

(define (set-line-format)
	(let ()
		(unite-line)
		(set-segment-format)))

(define *tooltip-mode* 2)

(define preview preview-block)

(define BLACK 0)
(define RED 1)
(define GREEN 2)
(define BLUE 3)
(define YELLOW 4)
(define MAGENTA 5)
(define CYAN 6)
(define WHITE 7)

(define *colors*
  (list (cons "black" BLACK)
	(cons "red" RED)
	(cons "green" GREEN)
	(cons "blue" BLUE)
	(cons "yellow" YELLOW)
	(cons "magenta" MAGENTA)
	(cons "cyan" CYAN)
	(cons "white" WHITE)))

(define (string->color color)
  (or (cdr (assoc color *colors*)) BLACK))

(define (ask-for-color)
  (form-begin)
  (form-label "Color:")
  (form-menu "color")
  (form-properties XtNwidth 100)
  (let ((c *colors*))
    (while c
      (form-menuentry (caar c))
      (set! c (cdr c))))
  (form-newline)
  (form-okbutton "Ok")
  (form-property XtNwidth 80)
  (form-cancelbutton "Cancel")
  (form-property XtNwidth 80)
  (cdr (assoc "color" (form-end))))

(define (cell-color)
  (set-color nil
	     (get-point)
	     (bit-and 7 (string->color (ask-for-color)))))

(define (block-color)
  (let ((blku (get-blku nil))
	(blkl (get-blkl nil))
	(color (bit-and 7 (string->color (ask-for-color))))
	(row 0)
	(col 0))
    (set! row (position-row blku))
    (while (<= row (position-row blkl))
      (set! col (position-col blku))
      (while (<= col (position-col blkl))
	(set-color nil (make-position row col) color)
	(set! col (+ col 1)))
      (set! row (+ row 1)))))

(define (search-forward)
  (require (string-append SIAGHOME "/egon/find.scm"))
  (search-forward))

(define (search-backward)
  (require (string-append SIAGHOME "/egon/find.scm"))
  (search-backward))

; (change-format format mask)
; format = new format 
; mask = which bits to change
(define (change-format format mask)
  (let ((oldfmt (get-format)))
    (set-format (bit-or (bit-and format mask)
			(bit-and oldfmt (bit-not mask))))))

(define FONT_MASK 96)
(define COURIER 0)
(define HELVETICA 32)
(define NEW_CENTURY 64)
(define TIMES 96)

(define SIZE_MASK 7)
(define SIZE_8 0)
(define SIZE_10 1)
(define SIZE_12 2)
(define SIZE_14 3)
(define SIZE_18 4)
(define SIZE_24 5)
(define SIZE_20 6)
(define SIZE_30 7)

(define FMT_SHIFT 16)
(define FMT_MASK (ash 15 FMT_SHIFT))
;(define FMT_DEFAULT (ash 0 FMT_SHIFT))
;(define FMT_HEADER1 (ash 1 FMT_SHIFT))
;(define FMT_HEADER2 (ash 2 FMT_SHIFT))
;(define FMT_HEADER3 (ash 3 FMT_SHIFT))
;(define FMT_HEADER4 (ash 4 FMT_SHIFT))
;(define FMT_HEADER5 (ash 5 FMT_SHIFT))
;(define FMT_HEADER6 (ash 6 FMT_SHIFT))
;(define FMT_ADDRESS (ash 7 FMT_SHIFT))
;(define FMT_OLIST (ash 8 FMT_SHIFT))
;(define FMT_ULIST (ash 9 FMT_SHIFT))
;(define FMT_PREFORMAT (ash 10 FMT_SHIFT))
;(define FMT_USER1 (ash 11 FMT_SHIFT))
;(define FMT_USER2 (ash 12 FMT_SHIFT))
;(define FMT_USER3 (ash 13 FMT_SHIFT))
;(define FMT_USER4 (ash 14 FMT_SHIFT))
;(define FMT_USER5 (ash 15 FMT_SHIFT))

(define STY_DEFAULT 0)
(define STY_HEADER1 1)
(define STY_HEADER2 2)
(define STY_HEADER3 3)
(define STY_HEADER4 4)
(define STY_HEADER5 5)
(define STY_HEADER6 6)
(define STY_ADDRESS 7)
(define STY_OLIST 8)
(define STY_ULIST 9)
(define STY_PREFORMAT 10)
(define STY_USER1 11)
(define STY_USER2 12)
(define STY_USER3 13)
(define STY_USER4 14)
(define STY_USER5 15)
(define STY_EMBED STY_USER5)

(define COLOR_SHIFT 20)
(define COLOR_MASK (ash 7 COLOR_SHIFT))
(define COLOR_BLACK (ash 0 COLOR_SHIFT))
(define COLOR_RED (ash 1 COLOR_SHIFT))
(define COLOR_GREEN (ash 2 COLOR_SHIFT))
(define COLOR_BLUE (ash 3 COLOR_SHIFT))
(define COLOR_YELLOW (ash 4 COLOR_SHIFT))
(define COLOR_MAGENTA (ash 5 COLOR_SHIFT))
(define COLOR_CYAN (ash 6 COLOR_SHIFT))
(define COLOR_WHITE (ash 7 COLOR_SHIFT))

(define HADJ_MASK (bit-or 4096 8192))
(define HADJ_LEFT 4096)
(define HADJ_CENTER 8192)
(define HADJ_RIGHT (bit-or 4096 8192))

(define ITALIC 8)
(define BOLD 16)

; Not needed, really
;(define (name->font name)
;  (cond ((equal? name "Times") TIMES)
;	((equal? name "Courier") COURIER)
;	((equal? name "Helvetica") HELVETICA)
;	((equal? name "New Century Schoolbook") NEW_CENTURY)
;	(t HELVETICA)))

(define (change-font font mask)
    (change-format font mask))

(define (change-style style)
  (set-style (position-row (get-point)) style))

(define (change-adjust adj)
  (set-adjust (position-row (get-point)) adj))

(define (toggle-fontflag flag)
  (let ((oldfmt (get-format nil (get-point))))
    (if (= (bit-and oldfmt flag) flag)
      (change-font 0 flag)
      (change-font flag flag))))

(define (exec-siod)
  (execute-interpreter-command 'SIOD))

(define (exec-guile)
  (execute-interpreter-command 'Guile))

(define (exec-tcl)
  (execute-interpreter-command 'Tcl))

(define (form-ask-for-number prompt value)
  (eval (read-from-string (form-ask-for-str prompt value))))

;; TODO: make this interface nicer with better forms
(define (ins-object)
  (ani-object (form-ask-for-number "Object type:" "")))

(define (ins-tick)
  (ani-time (form-ask-for-number "Time:" "")))

(define (ins-property)
  (let* ((props (form-record "Name" "Value"))
	 (name (cdr (assoc "Name" props)))
	 (value (cdr (assoc "Value" props))))
    (ani-property
      (eval (read-from-string name))
      (eval (read-from-string value)))))

(define (ins-properties)
  (let* ((props (form-record "X" "Y" "Width" "Height" "Visible" "Text"))
	 (x (cdr (assoc "X" props)))
	 (y (cdr (assoc "Y" props)))
	 (width (cdr (assoc "Width" props)))
	 (height (cdr (assoc "Height" props)))
	 (visible (cdr (assoc "Visible" props)))
	 (text (cdr (assoc "Text" props))))
    (if (not (equal? x ""))
      (ani-property ANI_X (eval (read-from-string x))))
    (if (not (equal? y ""))
      (ani-property ANI_Y (eval (read-from-string y))))
    (if (not (equal? width ""))
      (ani-property ANI_WIDTH (eval (read-from-string width))))
    (if (not (equal? height ""))
      (ani-property ANI_HEIGHT (eval (read-from-string height))))
    (if (not (equal? visible ""))
      (ani-property ANI_VISIBLE (eval (read-from-string visible))))
    (if (not (equal? text ""))
      (ani-property ANI_TEXT text))))

(define (add-object type)
  (ani-object type)
  (ani-properties ANI_X 0 ANI_Y 0 ANI_WIDTH 100 ANI_HEIGHT 100 ANI_VISIBLE 1
		ANI_COLOR BLACK ANI_FONT (+ SIZE_12 HELVETICA)))

(define (add-line)
  (add-object ANI_LINE))

(define (add-rectangle)
  (add-object ANI_RECTANGLE))

(define (add-arc)
  (add-object ANI_ARC))

(define (add-ellipse)
  (add-object ANI_ELLIPSE))

(define (add-image)
  (let ((fn (image-filename)))
    (if fn
      (begin
	(add-object ANI_PIXMAP)
	(ani-properties ANI_TEXT fn)))))

(define (add-string)
  (let ((fn (ask-for-str "String:" "")))
    (if fn
      (begin
  	(add-object ANI_STRING)
  	(ani-properties ANI_TEXT fn)))))

(define (add-point)
  (add-object ANI_POINT))

(define (add-filled-rectangle)
  (add-object ANI_FILLRECT))

(define (add-filled-arc)
  (add-object ANI_FILLARC))

(define (add-filled-ellipse)
  (add-object ANI_FILLELLIPSE))

(define (set-background)
  (ani-background (ask-for-file)))

(define (set-timing)
  (let* ((timing (form-record "Delta" "Duration"))
	 (delta (cdr (assoc "Delta" timing)))
	 (duration (cdr (assoc "Duration" timing))))
    (if (not (equal? delta ""))
      (ani-delta (eval (read-from-string delta))))
    (if (not (equal? duration ""))
      (ani-duration (eval (read-from-string duration))))))

(define (set-duration)
  (ani-duration (form-ask-for-number "Duration:" "")))

(define (set-delta)
  (ani-delta (form-ask-for-number "Delta:" "")))

(define (set-geometry)
  (let* ((geometry (form-record "Width" "Height"))
	 (width (cdr (assoc "Width" geometry)))
	 (height (cdr (assoc "Height" geometry))))
    (ani-geometry
      (eval (read-from-string width))
      (eval (read-from-string height)))))

(define (change-type)
  (set-type (form-ask-for-number "Type:" "")))

(require (string-append SIAGHOME "/plugins/plugin.scm"))

