/*
 * Copyright 1996 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: Algext.c,v 1.5 1996/09/13 16:22:18 bousch Exp $
 *
 * Algebraic extensions of a field (or a ring) K, having the form K[X]/(P)
 * where P is some polynomial of degree >= 2. The leading coefficient of P
 * should be invertible.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "saml.h"
#include "saml-errno.h"
#include "mnode.h"
#include "builtin.h"

typedef struct {
	struct mnode_header hdr;
	s_mnode *val;
	s_mnode *mod;
} aext_mnode;

static void aext_free (aext_mnode*);
static gr_string* aext_stringify (aext_mnode*);
static s_mnode* aext_make (s_mnode*);
static s_mnode* aext_add (aext_mnode*, aext_mnode*);
static s_mnode* aext_sub (aext_mnode*, aext_mnode*);
static s_mnode* aext_mul (aext_mnode*, aext_mnode*);
static int aext_notzero (aext_mnode*);
static s_mnode* aext_zero (aext_mnode*);
static s_mnode* aext_negate (aext_mnode*);
static s_mnode* aext_one (aext_mnode*);
static s_mnode* aext_invert (aext_mnode*);
static s_mnode* anytype2aext (s_mnode*, aext_mnode*);

static unsafe_s_mtype MathType_Algext = {
	"Algebraic extension",
	aext_free, NULL, aext_stringify,
	aext_make, NULL,
	aext_add, aext_sub, aext_mul, mn_std_div, mn_field_gcd,
	aext_notzero, NULL, NULL, mn_std_differ, NULL,
	aext_zero, aext_negate, aext_one, aext_invert, NULL
};

void init_MathType_Algext (void)
{
	register_mtype(ST_ALGEXT, &MathType_Algext);
	register_CV_routine(-1, ST_ALGEXT, anytype2aext);
}

static inline aext_mnode* aext_new (void)
{
	return (aext_mnode*) __mnalloc(ST_ALGEXT, sizeof(aext_mnode));
}

static void aext_free (aext_mnode* A)
{
	unlink_mnode(A->val);
	unlink_mnode(A->mod);
	free(A);
}

static gr_string* aext_stringify (aext_mnode* A)
{
	return mnode_stringify(A->val);
}

static s_mnode* aext_make (s_mnode* modulus)
{
	aext_mnode* result = aext_new();

	result->val = mnode_zero(modulus);
	result->mod = copy_mnode(modulus);
	return (s_mnode*) result;
}

static s_mnode* aext_add (aext_mnode* A, aext_mnode* B)
{
	aext_mnode* result;

	if (A->mod != B->mod)
		return mnode_error(SE_NSMOD, "aext_add");
	result = aext_new();
	result->val = mnode_add(A->val, B->val);
	result->mod = copy_mnode(A->mod);
	return (s_mnode*) result;
}

static s_mnode* aext_sub (aext_mnode* A, aext_mnode* B)
{
	aext_mnode* result;

	if (A->mod != B->mod)
		return mnode_error(SE_NSMOD, "aext_sub");
	result = aext_new();
	result->val = mnode_sub(A->val, B->val);
	result->mod = copy_mnode(A->mod);
	return (s_mnode*) result;
}

static s_mnode* aext_mul (aext_mnode* A, aext_mnode* B)
{
	aext_mnode *result;
	s_mnode *prod;

	if (A->mod != B->mod)
		return mnode_error(SE_NSMOD, "aext_mul");
	result = aext_new();
	result->mod = copy_mnode(A->mod);

	/* Maybe it's not the most efficient approach */
	prod = mnode_mul(A->val, B->val);
	result->val = mnode_mod(prod, A->mod);
	unlink_mnode(prod);
	return (s_mnode*) result;
}

static int aext_notzero (aext_mnode* A)
{
	return mnode_notzero(A->val);
}

static s_mnode* aext_zero (aext_mnode* model)
{
	aext_mnode *result = aext_new();

	result->val = mnode_zero(model->mod);
	result->mod = copy_mnode(model->mod);
	return (s_mnode*) result;
}

static s_mnode* aext_negate (aext_mnode* A)
{
	aext_mnode *result = aext_new();

	result->val = mnode_negate(A->val);
	result->mod = copy_mnode(A->mod);
	return (s_mnode*) result;
}

static s_mnode* aext_one (aext_mnode* model)
{
	aext_mnode *result = aext_new();

	result->val = mnode_one(model->mod);
	result->mod = copy_mnode(model->mod);
	return (s_mnode*) result;
}

static int bezout (s_mnode* A, s_mnode* B, s_mnode** pX, s_mnode** pY)
{
	s_mnode *q, *r, *t;

	if (mnode_notzero(B)) {
		q = mnode_div(A, B);
		t = mnode_mul(q, B);
		r = mnode_sub(A, t); unlink_mnode(t);
		/*
		 * The equation AX+BY=1 becomes B(qX+Y)+rX=1
		 */
		if (bezout(B,r,pY,pX) < 0) {
			unlink_mnode(q);
			unlink_mnode(r);
			return -1;
		}
		unlink_mnode(r);
		t = mnode_mul(q, *pX); unlink_mnode(q);
		r = mnode_sub(*pY, t); unlink_mnode(t);
		unlink_mnode(*pY); *pY = r;
		return 0;
	}
	t = mnode_invert(A);
	if (t->type == ST_VOID) {
		/* Not invertible */
		unlink_mnode(t);
		return -1;
	}
	*pX = t;
	*pY = mnode_zero(t);
	return 0;
}

static s_mnode* aext_invert (aext_mnode* A)
{
	s_mnode *X, *Y;
	aext_mnode *result;

	if (!mnode_notzero(A->val))
		return mnode_error(SE_DIVZERO, "aext_invert");
	if (bezout(A->val,A->mod,&X,&Y) < 0)
		return mnode_error(SE_SINGULAR, "aext_invert");
	unlink_mnode(Y);
	result = aext_new();
	result->val = X;
	result->mod = copy_mnode(A->mod);
	return (s_mnode*) result;
}

#if 0
static s_mnode* upoly2aext (s_mnode* A, aext_mnode* model)
{
	aext_mnode *result;
	
	if (model == NULL)
		return mnode_error(SE_ICAST, "upoly2aext");
	result = aext_new();
	result->mod = copy_mnode(model->mod);
	result->val = mnode_mod(A, result->mod);
	return (s_mnode*) result;
}
#endif

static s_mnode* anytype2aext (s_mnode* A, aext_mnode* model)
{
	s_mnode *A2;
	aext_mnode *result;

	if (model != NULL) {
		A2 = mnode_promote(A, model->mod);
		if (A2->type == ST_VOID)
			return A2;
		result = aext_new();
		result->val = A2;
		result->mod = copy_mnode(model->mod);
		return (s_mnode*) result;
	}
	/* No model */
	if (A->type != ST_UPOLY)
		return mnode_error(SE_ICAST, "anytype2aext");
	result = aext_new();
	result->val = mnode_zero(A);
	result->mod = copy_mnode(A);
	return (s_mnode*) result;
}
