"gelman.diag" <-
function (x, confidence = 0.95, transform = F) 
{
        #
        # Gelman and Rubin's code
        #
        # Adapted to work on mcmc objects. Now you can analyse
        # several variables at once.
        #
        x <- as.mcmc(x)
        #
        # We compute the following statistics:
        #
        #  xdot:  vector of sequence means
        #  s2:  vector of sequence sample variances (dividing by n-1)
        #  W = mean(s2):  within MS
        #  B = n*var(xdot):  between MS.
        #  muhat = mean(xdot):  grand mean; unbiased under strong stationarity
        #  varW = var(s2)/m:  estimated sampling var of W
        #  varB = B^2 * 2/(m-1):  estimated sampling var of B
        #  covWB = (n/m)*(cov(s2,xdot^2) - 2*muhat*cov(s^2,xdot)):
        #          estimated sampling cov(W,B)
        #  sig2hat = ((n-1)/n))*W + (1/n)*B:  estimate of sig2; unbiased under
        #            strong stationarity
        #  quantiles:  emipirical quantiles from last half of simulated
        #              sequences
        #
        if (nchain(x) == 1) 
                stop("You need at least two chains")
        if (start(x) < end(x)/2) 
                x <- window(x, start = end(x)/2 + 1)
        Niter <- niter(x)
        Nchain <- nchain(x)
        confshrink <- matrix(nrow = nvar(x), ncol = 2, dimnames = list(varnames(x), 
                c("Point est.", paste(50 * (1 + confidence), 
                        "% quantile", sep = ""))))
        for (i in 1:nvar(x)) {
                z <- x[, i, , drop = T]
                if (transform) 
                        if (min(z) > 0) 
                                z <- if (max(z) < 1) 
                                 log(z/(1 - z))
                                else log(z)
                s2 <- apply(z, 2, cov)
                W <- mean(s2)
                zbar <- apply(z, 2, mean)
                B <- Niter * var(zbar)
                sig2hat <- ((Niter - 1) * W + B)/Niter
                muhat <- mean(zbar)
                varW <- var(s2)/Nchain
                varB <- 2 * B^2/(Nchain - 1)
                covWB <- (Niter/Nchain) * (cov(s2, zbar^2) - 
                        2 * muhat * cov(s2, zbar))
                #
                # Posterior interval post.range combines all uncertainties
                # in a t interval with center muhat, scale sqrt(postvar),
                # and postvar.df degrees of freedom.
                #
                # postvar = sig2hat + B/(mn):  variance for the posterior
                #           interval. The B/(mn) term is there because of the
                #           sampling variance of muhat.
                # varpostvar:  estimated sampling variance of postvar
                #
                # 
                # Posterior interval post.range combines all uncertainties
                # in a t interval with center muhat, scale sqrt(postvar), 
                # and postvar.df degrees of freedom.
                #
                # postvar = sig2hat + B/(mn):  variance for the posterior
                #           interval. The B/(mn) term is there because of the
                #           sampling variance of muhat.
                # varpostvar:  estimated sampling variance of postvar
                #
                postvar <- sig2hat + B/(Niter * Nchain)
                varpostvar <- ((Niter - 1)^2 * varW + (1 + 1/Nchain)^2 * 
                        varB + 2 * (Niter - 1) * (1 + 1/Nchain) * 
                        covWB)/Niter^2
                post.df <- 2 * postvar^2/varpostvar
                df.adj <- (post.df + 3)/(post.df + 1)
                #
                # Estimated potential scale reduction (that would be achieved
                # by continuing simulations forever) has two components: an
                # estimate and an approx upper bound.
                #
                # confshrink = sqrt(postvar/W), 
                #     multiplied by sqrt((df+3)/(df+1)) as an adjustment for the
                #     width of the t-interval with df degrees of freedom.
                #
                # postvar/W = (n-1)/n + (1+1/m)(1/n)(B/W); we approximate
                # the sampling dist.  of (B/W) by an F distribution, with
                # degrees of freedom estimated from the approximate
                # chi-squared sampling dists for B and W.  (The F
                # approximation assumes that the sampling dists of B and W
                # are independent; if they are positively correlated, the
                # approximation is conservative.)
                #
                varlo.df <- 2 * W^2/varW
                R2.fixed <- (Niter - 1)/Niter
                R2.random <- (1 + 1/Nchain) * (1/Niter) * (B/W)
                R2.estimate <- R2.fixed + R2.random
                R2.upper <- R2.fixed + qf((1 + confidence)/2, 
                        Nchain - 1, varlo.df) * R2.random
                confshrink[i, 1] <- sqrt(df.adj * R2.estimate)
                confshrink[i, 2] <- sqrt(df.adj * R2.upper)
        }
        out <- list(confshrink = confshrink)
        class(out) <- "gelman.diag"
        out
}
