#include <urktree.h>
#include <gather.h>
#include <seqport.h>

extern TreeNodePtr TreeCenterNode (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return ptrNode;

  if (ptrNode->ptrUp == NULL)
    return ptrNode->ptrUp;

  if (ptrNode->ptrUp->ptrUp == ptrNode)
    return ptrNode;

  return TreeCenterNode (ptrNode->ptrUp);
}

extern TreeNodePtr ExploreUnrootedTree (TreeNodePtr ptrNode,
                                        TreeNodePtr ptrSkip,
                                        Int4 nodenumber)
{
  TreeNodePtr ptrTemp;

  if (ptrNode == NULL)
    return ptrNode;

  if (ptrNode->nodenumber == nodenumber)
    return ptrNode;

  if ((ptrTemp = ExploreUnrootedTree (ptrNode->ptrLeft, ptrSkip,
                                      nodenumber)) != NULL)
    return ptrTemp;

  if ((ptrTemp = ExploreUnrootedTree (ptrNode->ptrRight, ptrSkip,
                                      nodenumber)) != NULL)
    return ptrTemp;

  if (ptrNode != ptrSkip)
    return ptrTemp;

  return ExploreUnrootedTree (ptrNode->ptrUp, ptrSkip, nodenumber);
}

extern TreeNodePtr ExploreUnrootedTreeNumber (TreeNodePtr ptrNode,
                                              Int4 nodenumber)
{
  ptrNode = TreeCenterNode (ptrNode);
  return ExploreUnrootedTree (ptrNode, ptrNode, nodenumber);
}

extern Int4 GetDeepestLevelBranch (TreeNodePtr ptrNode, Int4 level)
{
  if (ptrNode == NULL)
    return level;

  if (ptrNode->ptrLeft != NULL || ptrNode->ptrRight != NULL)
    if (ptrNode->level > level)
      level = ptrNode->level;

  level = GetDeepestLevelBranch (ptrNode->ptrLeft, level);
  level = GetDeepestLevelBranch (ptrNode->ptrRight, level);

  return level;
}

extern void SetLevelNode (TreeNodePtr ptrNode, Int4 level)
{
  if (ptrNode == NULL)
    return;

  ptrNode->level = level;
  level++;
  SetLevelNode (ptrNode->ptrLeft, level);
  SetLevelNode (ptrNode->ptrRight, level);

  return;
}

extern TreeNodePtr ExploreTreeLevel (TreeNodePtr ptrNode, Int4 level,
                                     TreeNodePtr ptrSkip)
{
  TreeNodePtr ptrTemp;

  if (ptrNode == NULL)
    return ptrNode;

  if (ptrNode->ptrLeft == NULL && ptrNode->ptrRight == NULL)
    return NULL;

  if (ptrNode->level == level)
  {
    if (ptrNode != ptrSkip)
    {
      return ptrNode;
    }
  }

  if ((ptrTemp = ExploreTreeLevel (ptrNode->ptrLeft, level,
                                   ptrSkip)) != NULL)
    return ptrTemp;

  return ExploreTreeLevel (ptrNode->ptrRight, level, ptrSkip);
}

extern void SetZeroNode (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return;

  ptrNode->nodenumber = 0;
  SetZeroNode (ptrNode->ptrLeft);
  SetZeroNode (ptrNode->ptrRight);

  return;
}

extern Int4 SetNumberNode (TreeNodePtr ptrNode, Int4 nodenumber)
{
  if (ptrNode == NULL)
    return nodenumber;

  if (ptrNode->nodenumber == 0)
    ptrNode->nodenumber = nodenumber++;
  nodenumber = SetNumberNode (ptrNode->ptrLeft, nodenumber);
  nodenumber = SetNumberNode (ptrNode->ptrRight, nodenumber);

  return nodenumber;
}

extern TreeNodePtr HeadNode (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return ptrNode;

  while (ptrNode->ptrUp != NULL)
    ptrNode = ptrNode->ptrUp;

  return ptrNode;
}

extern TreeNodePtr LastNode (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return ptrNode;

  if (ptrNode->ptrRight != NULL)
    ptrNode = LastNode (ptrNode);

  return ptrNode;
}

extern TreeNodePtr ExploreTreeNumber (TreeNodePtr ptrNode,
                                      Int4 nodenumber)
{
  TreeNodePtr ptrTemp;

  if (ptrNode == NULL)
    return ptrNode;

  if (nodenumber == 0)
    return HeadNode (ptrNode);

  if (nodenumber == -1)
    return LastNode (HeadNode (ptrNode));

  if (ptrNode->nodenumber == nodenumber)
    return ptrNode;

  if ((ptrTemp = ExploreTreeNumber (ptrNode->ptrLeft,
                                    nodenumber)) != NULL)
    return ptrTemp;

  return ExploreTreeNumber (ptrNode->ptrRight, nodenumber);
}

extern TreeNodePtr FindPivotNode (TreeNodePtr ptrNode,
                                  TreeNodePtr ptrSkip)
{
  if (ptrNode == NULL)
    return ptrNode;

  if (ptrNode != ptrSkip)
  {
    if (ptrNode->flagPivot)
      return ptrNode;
  }

  if ((ptrNode = FindPivotNode (ptrNode->ptrLeft, ptrSkip)) != NULL)
    return ptrNode;

  return FindPivotNode (ptrNode->ptrRight, ptrSkip);
}

extern void UnSetPivotNode (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return;

  ptrNode->flagPivot = FALSE;;
  UnSetPivotNode (ptrNode->ptrLeft);
  UnSetPivotNode (ptrNode->ptrRight);

  return;
}

extern TreeNodePtr SetPivotNode (TreeNodePtr ptrNode, Int4 level)
{
  TreeNodePtr ptrOne, ptrTwo;

  if (ptrNode == NULL)
    return ptrNode;

  ptrOne = ExploreTreeLevel (ptrNode, level, NULL);
  ptrTwo = ExploreTreeLevel (ptrNode, level, ptrOne);

  if (ptrOne != NULL && ptrTwo == NULL)
  {
    ptrOne->flagPivot = TRUE;
    return ptrOne;
  }
  else
  {
    return NULL;
  }
}

extern TreeNodePtr SetAllPivotNodes (TreeNodePtr ptrNode)
{
  Int4       i, level;
  TreeNodePtr ptrPivot;

  ptrNode = HeadNode (ptrNode);
  level = GetDeepestLevelBranch (ptrNode, -1);

  for (i = 0; i < level; i++)
  {
    ptrPivot = SetPivotNode (ptrNode, i);
    if (ptrPivot != NULL)
      ptrNode = ptrPivot;
  }
  return HeadNode (ptrNode);
}

extern Int4 TestTree (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return TREE_NULL;

  while (ptrNode != NULL)
  {
    if (ptrNode->ptrUp->ptrUp == ptrNode)
      return TREE_UNROOTED;
    ptrNode = ptrNode->ptrUp;
  }
  return TREE_ROOTED;
}

extern TreeNodePtr RootTree (TreeNodePtr ptrNode, Int4 left, Int4 right)
{
  TreeNodePtr ptrTemp;

  if (TestTree (ptrNode) == TREE_ROOTED)
    return NULL;

  ptrNode = ExploreUnrootedTreeNumber (ptrNode, left);
  if (ptrNode != NULL)
  {
    ptrTemp = TreeNodeNew ();

    if (ptrNode->ptrUp->ptrUp == ptrNode)
    {
      ptrTemp->ptrRight = ptrNode->ptrUp;
      ptrTemp->ptrLeft = ptrNode;
      ptrTemp->ptrRight->ptrUp = ptrTemp;
      ptrTemp->ptrLeft->ptrUp = ptrTemp;
      return ptrTemp;
    }

    while (ptrNode->ptrUp->ptrUp != ptrNode)
    {
/* if right then want to mirror and force left */
      if ((ptrNode->ptrRight != NULL) &&
          (ptrNode->ptrRight->nodenumber == right))
      {
        ptrTemp->ptrLeft = ptrNode->ptrRight;
        ptrNode->ptrRight = ptrNode->ptrLeft;
        ptrNode->ptrLeft = ptrTemp->ptrLeft;
        ptrTemp->ptrLeft = NULL;
      }

      if ((ptrNode->ptrLeft != NULL) &&
          (ptrNode->ptrLeft->nodenumber == right))
      {
        ptrTemp->ptrLeft = ptrNode->ptrLeft;
        ptrNode->ptrLeft->ptrUp = ptrTemp;
      }
      else /* will root the tree -- default is up */
      {
        if (ptrNode->ptrUp->ptrRight == ptrNode)
        {
/* if right then want to mirror and force left */
          ptrTemp->ptrRight = ptrNode->ptrUp->ptrRight;
          ptrNode->ptrUp->ptrRight = ptrNode->ptrUp->ptrLeft;
          ptrNode->ptrUp->ptrLeft = ptrTemp->ptrRight;
          ptrTemp->ptrRight = NULL;
        }
        if (ptrNode->ptrUp->ptrLeft == ptrNode)    /* must be */
        {
          ptrTemp->ptrLeft = ptrNode;
          ptrNode = ptrNode->ptrUp;
        }
      }
      ptrTemp->ptrRight = ptrNode;
      ptrNode->ptrLeft = ptrNode->ptrUp;
      ptrNode->ptrUp = ptrTemp;
      ptrNode = ptrTemp;
    }
    ptrNode->ptrLeft = ptrNode->ptrUp;
    while (ptrNode->ptrUp != NULL)
      ptrNode = ptrNode->ptrUp;
  }
  return ptrNode;
}

extern TreeNodePtr UnrootTree (TreeNodePtr ptrNode)
{
  TreeNodePtr ptrTemp;

  if (TestTree (ptrNode) == TREE_UNROOTED)
    return NULL;

  ptrNode = HeadNode (ptrNode);
  if (ptrNode != NULL)
  {
    ptrNode->ptrLeft->ptrUp = ptrNode->ptrRight;
    ptrNode->ptrRight->ptrUp = ptrNode->ptrLeft;
    ptrTemp = ptrNode->ptrLeft;
    ptrNode->ptrLeft = NULL;
    ptrNode->ptrRight = NULL;
    TreeNodeFree (ptrNode);
    ptrNode = ptrTemp;
  }
  return ptrNode;
}

extern TreeNodePtr TreeNodeNew (void)
{
  TreeNodePtr ptrNode;

  if ((ptrNode = (TreeNodePtr) MemNew (sizeof (TreeNode))) != NULL)
  {
    ptrNode->score = 0.0;
    ptrNode->flagPivot = FALSE;
    ptrNode->level = 0;
    ptrNode->nodenumber = 0;
    ptrNode->ptrUp = NULL;
    ptrNode->ptrLeft = NULL;
    ptrNode->ptrRight = NULL;
  }
  return ptrNode;
}

extern TreeNodePtr TreeNodeFree (TreeNodePtr ptrNode)
{
  MemFree (ptrNode->name);
  return (TreeNodePtr) MemFree (ptrNode);
}

extern TreeNodePtr TreeNodeFreeBranch (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return ptrNode;

  TreeNodeFreeBranch (ptrNode->ptrLeft);
  TreeNodeFreeBranch (ptrNode->ptrRight);

  return TreeNodeFree (ptrNode);
}

extern TreeNodePtr TreeNodeFreeTree (TreeNodePtr ptrNode)
{
  if (ptrNode == NULL)
    return ptrNode;

  return TreeNodeFreeBranch (HeadNode (ptrNode));
}
