// Fl_Overlay_Window.C

// A window using double-buffering and able to draw an overlay
// on top of that.  Uses the hardware to draw the overlay if
// possible, otherwise it just draws in the front buffer.

#include <config.h>
#include <FL/Fl.H>
#include <FL/Fl_Overlay_Window.H>
#include <FL/fl_draw.H>
#include <FL/x.H>

void Fl_Overlay_Window::show() {
  Fl_Double_Window::show();
  if (overlay_ && overlay_ != this) overlay_->show();
}

void Fl_Overlay_Window::hide() {
  Fl_Double_Window::hide();
}

void Fl_Overlay_Window::flush() {
  if (overlay_ != this) {
    Fl_Double_Window::flush();
    return;
  }

  // if overlay is damaged, make Fl_Double_Window update everywhere,
  // thus copying the entire window from back buffer and erasing overlay:
  if (damage()&8) {expose(2, 0, 0, w(), h()); clear_damage(damage()&~8);}

  // draw background, which may erase parts of overlay:
  Fl_Double_Window::flush();

  // and then draw the overlay:
  make_current();
  draw_overlay();
}

void Fl_Overlay_Window::resize(int X, int Y, int W, int H) {
  Fl_Double_Window::resize(X,Y,W,H);
  if (overlay_ && overlay_!=this) overlay_->resize(0,0,w(),h());
}

Fl_Overlay_Window::~Fl_Overlay_Window() {
  hide();
//  delete overlay; this is done by ~Fl_Group
}

#if !HAVE_OVERLAY

int Fl_Overlay_Window::can_do_overlay() {return 0;}

void Fl_Overlay_Window::redraw_overlay() {overlay_ = this; damage(8);}

#else

XVisualInfo *fl_overlay_visual(int);

class _Fl_Overlay : public Fl_Window {
  friend class Fl_Overlay_Window;
  void flush();
  void show();
public:
  _Fl_Overlay(int x, int y, int w, int h) :
    Fl_Window(x,y,w,h) {deactivate();}
};

static XVisualInfo *vis;
static Colormap colormap_;
static int tried;

int Fl_Overlay_Window::can_do_overlay() {
  if (!tried) {
    vis = fl_overlay_visual(1);
    if (!vis) vis = fl_overlay_visual(0);
    tried = 1;
  }
  return vis != 0;
}

void _Fl_Overlay::show() {
  if (shown()) {Fl_Window::show(); return;}
  if (!colormap_)
    colormap_ =
      XCreateColormap(fl_display, RootWindow(fl_display,fl_screen),
		      vis->visual, AllocNone);
  fl_background_pixel = 0; // clear
  Fl_X::make_xid(this, vis, colormap_);
  fl_background_pixel = -1;
}

extern uchar fl_overlay; // changes how fl_color(x) works

void _Fl_Overlay::flush() {
  fl_window = fl_xid(this);
  static GC gc;	// gc shared by all overlays
  if (!gc) gc = XCreateGC(fl_display, fl_xid(this), 0, 0);
  fl_gc = gc;
  // set the colors *after* the -bg switches have been parsed:
  if (tried == 1) {
    fl_alloc_color(colormap_,FL_RED);
    fl_alloc_color(colormap_,FL_BLACK);
    fl_alloc_color(colormap_,FL_WHITE);
    tried = 2;
  }
  fl_overlay = 1;
  Fl_Overlay_Window *w = (Fl_Overlay_Window *)parent();
  if (damage()&~6) {
    XClearWindow(fl_display, fl_xid(this));
    w->draw_overlay();
  } else {
    Fl_X *i = Fl_X::i(this);
    fl_clip_region(i->region); i->region = 0;
    w->draw_overlay();
    fl_pop_clip();
  }
  fl_overlay = 0;
}

void Fl_Overlay_Window::redraw_overlay() {
  if (!fl_display) return; // this prevents fluid -c from opening display
  if (!overlay_) {
    if (can_do_overlay()) {
      Fl_Group::current(this);
      overlay_ = new _Fl_Overlay(0,0,w(),h());
      Fl_Group::current(0);
    } else {
      overlay_ = this;	// fake the overlay
    }
  }
  if (shown()) {
    if (overlay_ == this)
      damage(8);
    else if (!overlay_->shown())
      overlay_->show();
    else
      overlay_->redraw();
  }
}

#endif

// End of Fl_Overlay_Window.C
