/*	Fl_Function_Type_Type.C

	Type describing a C function output by Fluid.

*/

#include <FL/Fl.H>
#include "Fl_Type.H"
#include <FL/fl_show_input.H>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

////////////////////////////////////////////////////////////////
// quick check of any C code for legality, returns an error message
// these currently require comments to parse correctly!

static char buffer[128]; // for error messages

// check a quoted string ending in either " or ' or >:
const char *_q_check(const char * & c, int type) {
  for (;;) switch (*c++) {
  case '\0':
    sprintf(buffer,"missing %c",type);
    return buffer;
  case '\\':
    if (*c) c++;
    break;
  default:
    if (*(c-1) == type) return 0;
  }
}

// check normal code, match braces and parenthesis:
const char *_c_check(const char * & c, int type) {
  const char *d;
  for (;;) switch (*c++) {
  case 0:
    if (!type) return 0;
    sprintf(buffer, "missing %c", type);
    return buffer;
  case '{':
    if (type) goto UNEXPECTED;
    d = _c_check(c,'}'); if (d) return d; break;
  case '(':
    d = _c_check(c,')'); if (d) return d; break;
  case '\"':
    d = _q_check(c,'\"'); if (d) return d; break;
  case '\'':
    d = _q_check(c,'\''); if (d) return d; break;
  case '}':
  case ')':
  case '#':
  UNEXPECTED:
    if (type == *(c-1)) return 0;
    sprintf(buffer, "unexpected %c", *(c-1));
    return buffer;
  }
}

const char *c_check(const char *c, int type) {
  return _c_check(c,type);
}

////////////////////////////////////////////////////////////////

class Fl_Function_Type : public Fl_Type {
  const char* return_type;
  char public_;
public:
  Fl_Type *make();
  void write_declare();
  void write_code1();
  void write_code2();
  void open();
  int ismain() {return name_ == 0;}
  virtual const char *type_name() {return "Function";}
  virtual const char *title() {
    return name() ? name() : "main()";
  }
  int is_parent() {return 1;}
  int is_code_block() {return 1;}
  void write_properties();
  void read_property(const char *);
};

Fl_Type *Fl_Function_Type::make() {
  Fl_Type *p = Fl_Type::current;
  while (p && !p->is_decl_block()) p = p->parent;
  Fl_Function_Type *o = new Fl_Function_Type();
  o->name("make_window()");
  o->return_type = 0;
  o->add(p);
  o->factory = this;
  o->public_ = 1;
  return o;
}

void Fl_Function_Type::write_properties() {
  Fl_Type::write_properties();
  if (!public_) write_string("private");
  if (return_type) {
    write_string("return_type");
    write_word(return_type);
  }
}

void Fl_Function_Type::read_property(const char *c) {
  if (!strcmp(c,"private")) {
    public_ = 0;
  } else if (!strcmp(c,"return_type")) {
    storestring(read_word(),return_type);
  } else {
    Fl_Type::read_property(c);
  }
}

#include "function_panel.H"
#include <FL/fl_ask.H>

void Fl_Function_Type::open() {
  if (!function_panel) make_function_panel();
  f_return_type_input->static_value(return_type);
  f_name_input->static_value(name());
  f_public_button->value(public_);
  function_panel->show();
  const char* message = 0;
  for (;;) { // repeat as long as there are errors
    if (message) fl_alert(message);
    for (;;) {
      Fl_Widget* w = Fl::readqueue();
      if (w == f_panel_cancel) goto BREAK2;
      else if (w == f_panel_ok) break;
      else if (!w) Fl::wait();
    }
    const char*c = f_name_input->value();
    while (isspace(*c)) c++;
    message = c_check(c); if (message) continue;
    const char *d = c;
    for (; *d != '('; d++) if (isspace(*d) || !*d) break;
    if (*c && *d != '(') {
      message = "must be name(arguments), try again:"; continue;
    }
    c = f_return_type_input->value();
    message = c_check(c); if (message) continue;
    name(f_name_input->value());
    storestring(c, return_type);
    public_ = f_public_button->value();
    break;
  }
 BREAK2:
  function_panel->hide();
}

Fl_Function_Type Fl_Function_type;

extern FILE *code_file;
extern FILE *header_file;

void Fl_Function_Type::write_declare() {
  ::write_declare("#include <FL/Fl.H>");
}

static int havewidgets = 0;

void Fl_Function_Type::write_code1() {
// we need to see if it creates any widgets, otherwise we will get
// an "unused variable" error:
  Fl_Type *child;
  for (child = next; child && child->level > level; child = child->next)
    if (child->is_widget()) {havewidgets = 1; break;}
  fprintf(code_file, "\n");
  if (ismain())
    fprintf(code_file, "int main(int argc, char **argv) {\n");
  else {
    const char* t = return_type;
    if (!t) t = havewidgets ? "Fl_Window*" : "void";
    if (public_) fprintf(header_file, "%s %s;\n", t, name());
    else fprintf(code_file, "static ");
    fprintf(code_file, "%s %s {\n", t, name());
  }
  if (havewidgets) fprintf(code_file, " Fl_Window *w;\n");
}

void Fl_Function_Type::write_code2() {
  if (ismain()) {
    if (havewidgets) fprintf(code_file, " w->show(argc, argv);\n");
    fprintf(code_file, " return Fl::run();\n}\n");
  } else if (havewidgets)
    fprintf(code_file," return w;\n}\n");
  else
    fprintf(code_file,"}\n");
}

////////////////////////////////////////////////////////////////

class Fl_Code_Type : public Fl_Type {
public:
  Fl_Type *make();
  void write_declare();
  void write_code1();
  void write_code2();
  void open();
  virtual const char *type_name() {return "code";}
  int is_code_block() {return 0;}
};

Fl_Type *Fl_Code_Type::make() {
  Fl_Type *p = Fl_Type::current;
  while (p && !p->is_code_block()) p = p->parent;
  if (!p) {
    fl_message("Please select a function");
    return 0;
  }
  Fl_Code_Type *o = new Fl_Code_Type();
  o->name("printf(\"Hello, World!\\n\");");
  o->add(p);
  o->factory = this;
  return o;
}

void Fl_Code_Type::open() {
  if (!code_panel) make_code_panel();
  code_input->static_value(name());
  code_panel->show();
  const char* message = 0;
  for (;;) { // repeat as long as there are errors
    if (message) fl_alert(message);
    for (;;) {
      Fl_Widget* w = Fl::readqueue();
      if (w == code_panel_cancel) goto BREAK2;
      else if (w == code_panel_ok) break;
      else if (!w) Fl::wait();
    }
    const char*c = code_input->value();
    message = c_check(c); if (message) continue;
    name(c);
    break;
  }
 BREAK2:
  code_panel->hide();
}

Fl_Code_Type Fl_Code_type;

extern FILE *code_file;
extern FILE *header_file;

void Fl_Code_Type::write_declare() {}

void Fl_Code_Type::write_code1() {
  const char* c = name();
  if (!c) return;
  fprintf(code_file, "%s", c);
//  if (c[strlen(c)-1] != ';') fprintf(code_file, ";");
  fprintf(code_file, "\n");
}

void Fl_Code_Type::write_code2() {}

////////////////////////////////////////////////////////////////

class Fl_CodeBlock_Type : public Fl_Type {
  const char* after;
public:
  Fl_Type *make();
  void write_declare();
  void write_code1();
  void write_code2();
  void open();
  virtual const char *type_name() {return "codeblock";}
  int is_code_block() {return 1;}
  int is_parent() {return 1;}
  void write_properties();
  void read_property(const char *);
};

Fl_Type *Fl_CodeBlock_Type::make() {
  Fl_Type *p = Fl_Type::current;
  while (p && !p->is_code_block()) p = p->parent;
  if (!p) {
    fl_message("Please select a function");
    return 0;
  }
  Fl_CodeBlock_Type *o = new Fl_CodeBlock_Type();
  o->name("if (test())");
  o->after = 0;
  o->add(p);
  o->factory = this;
  return o;
}

void Fl_CodeBlock_Type::write_properties() {
  Fl_Type::write_properties();
  if (after) {
    write_string("after");
    write_word(after);
  }
}

void Fl_CodeBlock_Type::read_property(const char *c) {
  if (!strcmp(c,"after")) {
    storestring(read_word(),after);
  } else {
    Fl_Type::read_property(c);
  }
}

void Fl_CodeBlock_Type::open() {
  if (!codeblock_panel) make_codeblock_panel();
  code_before_input->static_value(name());
  code_after_input->static_value(after);
  codeblock_panel->show();
  const char* message = 0;
  for (;;) { // repeat as long as there are errors
    if (message) fl_alert(message);
    for (;;) {
      Fl_Widget* w = Fl::readqueue();
      if (w == codeblock_panel_cancel) goto BREAK2;
      else if (w == codeblock_panel_ok) break;
      else if (!w) Fl::wait();
    }
    const char*c = code_before_input->value();
    message = c_check(c); if (message) continue;
    name(c);
    c = code_after_input->value();
    message = c_check(c); if (message) continue;
    storestring(c, after);
    break;
  }
 BREAK2:
  codeblock_panel->hide();
}

Fl_CodeBlock_Type Fl_CodeBlock_type;

extern FILE *code_file;
extern FILE *header_file;

void Fl_CodeBlock_Type::write_declare() {}

void Fl_CodeBlock_Type::write_code1() {
  const char* c = name();
  fprintf(code_file, "%s {\n", c ? c : "");
}

void Fl_CodeBlock_Type::write_code2() {
  if (after) fprintf(code_file, "} %s\n", after);
  else fprintf(code_file, "}\n");
}

////////////////////////////////////////////////////////////////

class Fl_Decl_Type : public Fl_Type {
  char public_;
public:
  Fl_Type *make();
  void write_declare();
  void write_code1();
  void write_code2();
  void open();
  virtual const char *type_name() {return "decl";}
  void write_properties();
  void read_property(const char *);
};

Fl_Type *Fl_Decl_Type::make() {
  Fl_Type *p = Fl_Type::current;
  while (p && !p->is_decl_block()) p = p->parent;
  Fl_Decl_Type *o = new Fl_Decl_Type();
  o->public_ = 0;
  o->name("int x;");
  o->add(p);
  o->factory = this;
  return o;
}

void Fl_Decl_Type::write_properties() {
  Fl_Type::write_properties();
  if (public_) write_string("public");
}

void Fl_Decl_Type::read_property(const char *c) {
  if (!strcmp(c,"public")) {
    public_ = 1;
  } else {
    Fl_Type::read_property(c);
  }
}

void Fl_Decl_Type::open() {
  if (!decl_panel) make_decl_panel();
  decl_input->static_value(name());
  decl_public_button->value(public_);
  decl_panel->show();
  const char* message = 0;
  for (;;) { // repeat as long as there are errors
    if (message) fl_alert(message);
    for (;;) {
      Fl_Widget* w = Fl::readqueue();
      if (w == decl_panel_cancel) goto BREAK2;
      else if (w == decl_panel_ok) break;
      else if (!w) Fl::wait();
    }
    const char*c = decl_input->value();
    while (isspace(*c)) c++;
    message = c_check(c&&c[0]=='#' ? c+1 : c);
    if (message) continue;
    name(c);
    public_ = decl_public_button->value();
    break;
  }
 BREAK2:
  decl_panel->hide();
}

Fl_Decl_Type Fl_Decl_type;

extern FILE *code_file;
extern FILE *header_file;

void Fl_Decl_Type::write_declare() {}

void Fl_Decl_Type::write_code1() {
  const char* c = name();
  if (!c) return;
  if (public_) {
    if (isalpha(*c) && strncmp(c,"extern", 6))
      fprintf(header_file, "extern %s\n", name());
    else
      fprintf(header_file, "%s\n", name());
  }
  if (!public_ && isalpha(*c) && strncmp(c,"extern", 6))
    fprintf(code_file, "static %s\n", name());
  else
    fprintf(code_file, "%s\n", c);
}

void Fl_Decl_Type::write_code2() {}

////////////////////////////////////////////////////////////////

class Fl_DeclBlock_Type : public Fl_Type {
  const char* after;
public:
  Fl_Type *make();
  void write_declare();
  void write_code1();
  void write_code2();
  void open();
  virtual const char *type_name() {return "declblock";}
  void write_properties();
  void read_property(const char *);
  int is_parent() {return 1;}
  int is_decl_block() {return 1;}
};

Fl_Type *Fl_DeclBlock_Type::make() {
  Fl_Type *p = Fl_Type::current;
  while (p && !p->is_decl_block()) p = p->parent;
  Fl_DeclBlock_Type *o = new Fl_DeclBlock_Type();
  o->name("#if 1");
  o->after = strdup("#endif");
  o->add(p);
  o->factory = this;
  return o;
}

void Fl_DeclBlock_Type::write_properties() {
  Fl_Type::write_properties();
  write_string("after");
  write_word(after);
}

void Fl_DeclBlock_Type::read_property(const char *c) {
  if (!strcmp(c,"after")) {
    storestring(read_word(),after);
  } else {
    Fl_Type::read_property(c);
  }
}

void Fl_DeclBlock_Type::open() {
  if (!declblock_panel) make_declblock_panel();
  decl_before_input->static_value(name());
  decl_after_input->static_value(after);
  declblock_panel->show();
  const char* message = 0;
  for (;;) { // repeat as long as there are errors
    if (message) fl_alert(message);
    for (;;) {
      Fl_Widget* w = Fl::readqueue();
      if (w == declblock_panel_cancel) goto BREAK2;
      else if (w == declblock_panel_ok) break;
      else if (!w) Fl::wait();
    }
    const char*c = decl_before_input->value();
    while (isspace(*c)) c++;
    message = c_check(c&&c[0]=='#' ? c+1 : c);
    if (message) continue;
    name(c);
    c = decl_after_input->value();
    while (isspace(*c)) c++;
    message = c_check(c&&c[0]=='#' ? c+1 : c);
    if (message) continue;
    storestring(c,after);
    break;
  }
 BREAK2:
  declblock_panel->hide();
}

Fl_DeclBlock_Type Fl_DeclBlock_type;

extern FILE *code_file;
extern FILE *header_file;

void Fl_DeclBlock_Type::write_declare() {}

void Fl_DeclBlock_Type::write_code1() {
  const char* c = name();
  if (c) fprintf(code_file, "%s\n", c);
}

void Fl_DeclBlock_Type::write_code2() {
  const char* c = after;
  if (c) fprintf(code_file, "%s\n", c);
}
