/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Verwendet: GTK, Imlib                                                     *
 * Modul:     gtkuti.c                                                       *
 *            utilities to handle some gtk widgets                           *
 * Autor:     Andreas Tille                                                  *
 * Datum:     23.09.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <sys/param.h>
#include <X11/Xlib.h>
#include "paul.h"
#include "callback.h"
 
#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

/*****************************************************************************************
 * Useful things for user interactivity                                                  *
 *****************************************************************************************/

void AreYouSure(void *data, GtkSignalFunc func, char *question)
/* If the user is sure in dead the function will be executed
 */
{
   GtkWidget *window = NULL;
   GtkWidget *text, *box, *button, *hbox;

   window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_position(GTK_WINDOW(window), GTK_WIN_POS_MOUSE);
   gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);
   gtk_window_set_title(GTK_WINDOW(window), "Are you sure?");

   box = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(box), 10);
   gtk_container_add(GTK_CONTAINER(window), box);

   text = gtk_label_new(question);
   gtk_box_pack_start(GTK_BOX(box), text, TRUE, TRUE, 0);

   hbox = gtk_hbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(hbox), 10);
   gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);
   
   button = gtk_button_new_with_label("Yes");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked", func, data);
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), 
                             GTK_OBJECT(window));
   gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
   GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
   gtk_widget_grab_default (button);

   button = gtk_button_new_with_label("No");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), 
                             GTK_OBJECT(window));
   gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
   GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);

   gtk_widget_show_all(window);
}

GtkWidget *CreateOkButton(GtkWidget *window, GtkWidget *box, void *userdata, 
                          GtkSignalFunc f1, void *d1, GtkSignalFunc f2, void *d2)
/* OK button with data to connect and two functions to call
 * --- Parameter: ---
 * GtkWidget     *window   : parent window
 * GtkWidget     *box      : box to draw button in
 * void          *userdata : data to connect with button or NULL
 * GtkSignalFunc  f1       : function to call first when clicked
 * void          *d1       : data for f1 call
 * GtkSignalFunc  f2       : second function to call when clicked (in most cases NULL)
 * void          *d2       : data for f2 or NULL
 * --- Return: ---
 * GtkWidget     *CreateOkButton() : button widget od NULL if fail
 */
{
   GtkWidget *button;

   if ( !GTK_IS_WINDOW(window) ) return NULL;
   if ( !GTK_IS_BOX(box) )       return NULL;
   
   button = gtk_button_new_with_label("OK");
   if ( userdata ) gtk_object_set_user_data(GTK_OBJECT(button), userdata);
   if ( f1 ) 
      gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(f1), d1);
   if ( f2 ) 
      gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(f2), d2);
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                             GTK_OBJECT(window));
   gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 5);
   return button;
}
			 
static void MyGtkWidgetDestroy(GtkWidget *widget)
/* gtk_widget_destroy + widget = NULL
 */
{
   g_return_if_fail(widget != NULL);
   g_return_if_fail(GTK_IS_WIDGET(widget));
    
   gtk_object_destroy((GtkObject*)widget);
   widget = NULL;
}

GtkWidget *CreateCancelButton(GtkWidget *window, GtkWidget *box)
/* OK button with data to connect and two functions to call
 * --- Parameter: ---
 * GtkWidget     *window               : parent window
 * GtkWidget     *box                  : box to draw button in
 * --- Return: ---
 * GtkWidget     *CreateCancelButton() : button widget od NULL if fail
 */
{
   GtkWidget *button;

   if ( !GTK_IS_WINDOW(window) ) return NULL;
   if ( !GTK_IS_BOX(box) )       return NULL;
   
   button = gtk_button_new_with_label("Cancel");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(MyGtkWidgetDestroy),
                             GTK_OBJECT(window));
   gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 5);
   return button;
}
			 
/*****************************************************************************************
 * Functions to build a table of parameter entries                                       *
 *****************************************************************************************/

static void ParameterChanged(GtkButton *button, gpointer data)
/* called when parameters were changed in menu
 * --- Parameter: ---
 * GtkButton *button: OK-Button of menu window, parameter structure connected
 * gpointer   data  : spin button which holds new value
 */
{
   register GtkWidget *zw;
   GtkTable           *table;
   PARA_IN            *para_in;
   register ITEM      *para;
   GList              *pl;
   int                 i;

   para_in = (PARA_IN *)gtk_object_get_user_data(GTK_OBJECT(button));
   para    = para_in->para;
   pl      = (table  = GTK_TABLE(data))->children;

   for ( i = para_in->np; i; ) {
      --i;
      if ( pl && GTK_IS_SPIN_BUTTON(zw = ((GtkTableChild *)(pl->data))->widget) 
              && GTK_IS_LABEL(((GtkTableChild *)((pl->next)->data))->widget) ) 
         switch ( (para+i)->typ ) {
	 case P_UCHAR:  *((unsigned char *)((para+i)->val))
                           = (unsigned char)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(zw));
                        break;
	 case P_INT:    *((int *)((para+i)->val))
                           = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(zw));
                        break;
	 case P_DOUBLE: *((double *)((para+i)->val)) 
                           = (double)gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(zw));
                        break;
         default:  g_warning("Unknown parameter type ignored.");
         }
      else 
         g_warning("Can't obtain %s", (para+i)->name);
      if ( !(pl = (pl->next)->next) ) break;
   }
   free(para);
   free(para_in);
}

static void ParameterTableEntry(GtkWidget *table, int i, ITEM *para)
/* build table of parameters
 * --- Parameter: ---
 * GtkWidget *table : table to draw in
 * char      *desc  : description for label
 * int        i     : row to draw in
 * gfloat     value : value according to the description
 * gfloat     lower : lower boundary
 * gfloat     upper : upper boundary
 * ffloat     step  : step when using spinbutton
 * int        prec  : precission of value
 * int        size  : size of value entry
 */
{
   GtkWidget     *label, *spinner;
   gfloat         value = 0, step = 0;
   int            digits = 0, size = 100;

   label   = gtk_label_new(para->name);
   gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
   gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, i, i+1);

   switch ( para->typ ) {
      case P_UCHAR:  value  = (gfloat)(*((unsigned char *)(para->val)));
                     step   = 1;
                     digits = 0;
                     size   = 50;
                     break;
      case P_INT:    value  = (gfloat)(*((int *)(para->val))),
                     step   = 1;
                     digits = 0;
                     size   = 80;
                     break;
      case P_DOUBLE: value  = (gfloat)(*((double *)(para->val))),
                     step   = 0.1;
                     digits = 2;
                     size   = 100;
                     break;
      default:  g_warning("Unknown parameter type (%i) ignored.", para->typ);
   }

   para->adj = (GtkAdjustment *)gtk_adjustment_new(value, para->lower, para->upper, step, 0.0, 0.0);
   spinner   = gtk_spin_button_new(para->adj, step, digits);
   gtk_widget_set_usize(spinner, size, 0);
   gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spinner), GTK_UPDATE_ALWAYS);
   if ( para->valchg ) {
      gtk_object_set_user_data(GTK_OBJECT(para->adj), para->val);
      gtk_signal_connect(GTK_OBJECT(para->adj), "value_changed", GTK_SIGNAL_FUNC(para->valchg), spinner);
   }
   gtk_table_attach_defaults(GTK_TABLE(table), spinner, 1, 2, i, i+1);
}

GtkWidget *BuildParameterTable(GtkWidget *box, ITEM *para, int np)
/* Build a table of parameters
 * GtkWidget *box                  : box to store the table in  
 * ITEM      *para                 : parameters
 * int        np                   : number of parameters
 * --- Return: ---
 * GtkWidget *BuildParameterTable(): created table widget to obtain values later on
 */
{
   GtkWidget *table;     /* Stack the items inside subsections */
   int        i;

   table = gtk_table_new(2, np, FALSE);
   gtk_container_border_width(GTK_CONTAINER(table), 5);
   gtk_box_pack_start(GTK_BOX(box), table, TRUE, FALSE, 0);

   for ( i = 0; i < np; i++ ) ParameterTableEntry(table, i, para + i);
   return table;
}

void BuildParameterInput(PARA_IN *para_in, PAUL *p, GtkSignalFunc add_func)
/* Build table of items to input
 * --- Parameter: ---
 * PARA_IN         *para_in : structure with parameters to change
 * GtkMenuCallback  add_func: if != NULL, do this func after parameter change
 */
{
   static GtkWidget *window = NULL;
   GtkWidget     *frame,     /* frame of subsections               */
                 *hbox,      /* row of items                       */
                 *table;
   
   if ( window ) {
      gtk_widget_destroy(window);
      window = NULL;
      return;
   }

   window = gtk_dialog_new();
   gtk_window_position(GTK_WINDOW(window), GTK_WIN_POS_MOUSE);
   gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                         &window);
   gtk_window_set_title(GTK_WINDOW(window), para_in->title);

   gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(window)->vbox), 5);

   frame = gtk_frame_new(para_in->frame);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(window)->vbox), frame);
   hbox = gtk_hbox_new(FALSE, 0);
   gtk_container_add(GTK_CONTAINER(frame), hbox);

   table = BuildParameterTable(hbox, para_in->para, para_in->np);
      
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(window)->action_area), hbox);
  
   CreateOkButton(window, hbox, para_in, GTK_SIGNAL_FUNC(ParameterChanged), table, add_func, p);
   CreateCancelButton(window, hbox);

   gtk_widget_show_all(window);
}

/*****************************************************************************************
 * Functions to build radiobuttons to select parameters                                  *
 *****************************************************************************************/

static void RadioParameterChanged(GtkButton *button, RADIO_PARA_IN *para_in)
/* called when parameters were changed in menu
 * --- Parameter: ---
 * GtkButton     *button : OK-Button of radio button widget (can be ignored)
 * RADIO_PARA_IN *para_in: para_in->def holds the whished value temporarily
 */
{
   *(para_in->val) = para_in->def;
   free(para_in->radio);
   free(para_in);
}

static void ButtonFunc(GtkWidget *button, unsigned char *val)
{
   *val = *((unsigned char *)gtk_object_get_user_data(GTK_OBJECT(button)));
}

void RadioButtonParameter(RADIO_PARA_IN *para_in)
{
   static    GtkWidget *window = NULL;
   GtkWidget           *frame,     /* frame of subsections               */
                       *hbox,      /* row of items                       */
                       *main_vbox, /* what should this mean ?? :)        */
                       *vbox,      /* inner box containing radio buttons */
                       *button;    /* Radio buttons                      */
   register  RADIO     *para;
   int                  i, def;

   if (!window) {
      window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
      gtk_signal_connect(GTK_OBJECT (window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			  &window);
      gtk_window_set_title(GTK_WINDOW (window), para_in->title);
      gtk_container_border_width(GTK_CONTAINER (window), 0);

      main_vbox = gtk_vbox_new(FALSE, 5);
      gtk_container_border_width(GTK_CONTAINER(main_vbox), 10);
      gtk_container_add(GTK_CONTAINER(window), main_vbox);
      
      frame = gtk_frame_new(para_in->frame);
      gtk_box_pack_start(GTK_BOX(main_vbox), frame, TRUE, TRUE, 0);
      
      vbox = gtk_vbox_new(FALSE, 5);
      gtk_container_border_width(GTK_CONTAINER(vbox), 10);
      gtk_container_add(GTK_CONTAINER(frame), vbox);

      para   = para_in->radio;
      button = NULL;
      def = para_in->def; /* store this value because it could be changed while  *
                           * creating the buttons!!!                             */
      for ( i = 0; i < para_in->np; i++ ) {
         button = gtk_radio_button_new_with_label( 
                     button ? gtk_radio_button_group(GTK_RADIO_BUTTON(button)) : NULL, 
                    (para+i)->name);
         gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
         gtk_object_set_user_data(GTK_OBJECT(button), &((para+i)->val));
         gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(ButtonFunc), 
                            &(para_in->def));
         if ( i == def ) gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), TRUE);
      }
      
      hbox = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, TRUE, 0);
  
      CreateOkButton(window, hbox, NULL, GTK_SIGNAL_FUNC(RadioParameterChanged), para_in, 
                     NULL, NULL);
      CreateCancelButton(window, hbox);
   }

   if (!GTK_WIDGET_VISIBLE(window) ) gtk_widget_show_all(window);
   else                              gtk_widget_destroy(window);
}


/*****************************************************************************************
 * Functions to manipulate image file list                                               *
 *****************************************************************************************/

void AddImageToFilelist(PAUL *p, PICTURE *bild)
/* Add one image name to pauls list of images
 */
{
   GtkWidget *list_item;
   GList     *pl = NULL;

   if ( !(p->piclist) ) {
      p->piclist = g_list_append(NULL, bild);
      CreateShowPixmaps(p->activ = p->piclist);
      CreatePicture(p);
   } else {
      if ( NBILDER(p->piclist) == 1 && !DoMatchPictures(p->opt->f) ) 
         SetAnimateButton(p, GTK_SIGNAL_FUNC(PictureAnimate), ANIMATE);
      p->piclist = g_list_append(p->piclist, bild);
      p->activ   = g_list_last(p->piclist);
   }
   ApplyPicture(p->show->window, BILD(p->activ));

   list_item = CreateListItem(p->activ);
   pl        = g_list_append(pl, list_item);
   if ( Info(p->opt->f) ) FileInfo(NULL, p);

   gtk_widget_show(list_item);
   gtk_object_set_data(GTK_OBJECT(list_item), PICTURE_DataKey, g_list_last(p->piclist));
   gtk_list_append_items(GTK_LIST(p->filelist), pl);

   if( !(bild->im) ) assert((bild->im = 
          at_imlib_set_image_from_data(bild->DATA, bild->W, bild->H)));

   FileListSelectionStatus = 1;  /* trick to get a single item selected */
   gtk_list_select_item(GTK_LIST(p->filelist), GetListPosition(p->piclist, p->activ));
}

void RemoveImageFromFilelist(PAUL *p, GtkWidget *list_item)
/* Remove one image name from pauls list of images
 * --- Parameter: ---
 * PAUL *p              : 
 * GtkWidget *list_item : list_item widget to remove from list
 */
{
   GList     *pl;
   
   if ( !(pl = (p->activ)->next) ) pl = (p->activ)->prev;
   p->piclist = g_list_remove_link(p->piclist, p->activ);
   p->activ = pl;  /* make following image activ */
   gtk_list_unselect_child(p->filelist, list_item);
   gtk_widget_destroy(GTK_WIDGET(list_item));

   if ( !(p->piclist) || NBILDER(p->piclist) == 1) {
      if ( !p->piclist ) gtk_widget_destroy(GTK_WIDGET(p->show));
      SetAnimateButton(p, NULL, NULL);
      if ( !p->piclist ) {
         if ( Info(p->opt->f) ) FileInfo(NULL, p);
         return;
      }
   }
   if ( (GTK_LIST(p->filelist)->selection) ) {
      p->activ = LI2PL(GTK_OBJECT((GTK_LIST(p->filelist)->selection)->data));
   } else {
      /* if there aren't any selected files, let the last clicked be selected */
      if ( (list_item = SearchListItem(GTK_LIST(p->filelist)->children, p->activ)) 
           && list_item->state == GTK_STATE_NORMAL ) {
         (p->filelist)->selection = g_list_prepend((p->filelist)->selection, list_item);
         gtk_widget_ref(list_item);
         gtk_list_item_select(GTK_LIST_ITEM(list_item));
      }
   } 
   if ( Info(p->opt->f) ) FileInfo(NULL, p);
   ApplyPicture(p->show->window, BILD(p->activ));
}

GtkWidget *CreateListItem(GList *pl)
{
   GtkWidget *list_item;
   PICTURE   *bild = BILD(pl);
   char       buf[256];
   
   strncpy(buf, bild->file, 250);
   if ( bild->ext ) strcat(strcat(buf, "."), bild->ext);
   list_item   = gtk_list_item_new();
   bild->label = gtk_label_new(buf);
   gtk_misc_set_alignment(GTK_MISC(bild->label), 0.0, 0.5);
   gtk_container_add(GTK_CONTAINER(list_item), bild->label);
   gtk_widget_show(bild->label);

   return list_item;
}

void CreateGtkListOfImages(PAUL *p)
/*
 * PAUL    *p               : list of images
 */
{
   GList     *dlist, *pl;
   GtkWidget *list_item;

   p->filelist = (GtkList *)gtk_list_new();
   gtk_widget_show((GtkWidget *)p->filelist);

/*
   gtk_widget_set_events(GTK_WIDGET(p->filelist), gtk_widget_get_events(GTK_WIDGET(p->filelist)) | 
                         GDK_KEY_PRESS_MASK | GDK_KEY_PRESS);
   gtk_signal_connect(GTK_OBJECT(p->filelist), "key_press_event", 
                      GTK_SIGNAL_FUNC(FileListKeyPressed), p); 
*/
   gtk_signal_connect(GTK_OBJECT(p->filelist), "button_press_event", 
                      GTK_SIGNAL_FUNC(FileListButtonPressed), p); 
   gtk_signal_connect(GTK_OBJECT(p->filelist), "select_child", 
                      GTK_SIGNAL_FUNC(FileListSelectChild), p);
   gtk_object_set_user_data(GTK_OBJECT(p->filelist), p);

   dlist=NULL;
   for ( pl = p->piclist; pl; pl = pl->next ) {
      list_item = CreateListItem(pl);
      gtk_widget_show(list_item);
      gtk_object_set_data(GTK_OBJECT(list_item), PICTURE_DataKey, pl);
      dlist     = g_list_append(dlist, list_item);
   }
   if ( Info(p->opt->f) ) FileInfo(NULL, p);
   gtk_list_set_selection_mode((GtkList *)p->filelist, GTK_SELECTION_MULTIPLE);
   gtk_list_append_items(GTK_LIST(p->filelist), dlist);
}

static GList *GetListFromSelection(GList *sel)
/* obtain image list of given selctions in file list
 * --- Parameter: ---
 * GList *sel                    : selected list items in file list
 * --- Return: ---
 * GList *GetListFromSelection() : List of selected images
 */
{
   GList   *gl, *new_pl = NULL;
   
   if ( (gl = sel) ) do {
      new_pl = g_list_append(new_pl, LI2BILD(GTK_OBJECT((gl)->data)));
   } while ( (gl = gl->next) );
   return new_pl;
}

GtkWidget *SearchListItem(GList *filelist, GList *search)
/* Search for the list item which contains a specified image
 * --- Parameter: ---
 * GList *filelist             : list of files
 * GList *search               : image to search
 * --- Return: ---
 * GtkWidget *SearchListItem() : list_item widget containing the searched picture
 */
{
   GList     *gl;
   GtkWidget *list_item;

   if ( (gl = filelist) ) do {
      list_item = gl->data;
      if ( search == LI2PL(list_item) ) return list_item;
   } while ( (gl = gl->next) );
   return NULL;
}

void MakeNewFileList(PAUL *p)
/* Creates a really new filelist (think twice about if this is necessary or if it is
 * enouth to go the way of ApplyPaulFunc()) and creates new gdk_imlib structures
 * (once more: think about if this is necessary)
 * This function makes sense when moving images, because all images were changed
 * completely
 */
{
   gtk_widget_destroy(GTK_WIDGET(p->filelist));
   CreateGtkListOfImages(p);
   gtk_container_add(GTK_CONTAINER(p->w), (GtkWidget *)p->filelist);

   CreateShowPixmaps(p->piclist);
   if ( !(p->activ) ) p->activ = p->piclist;
   UPDATE_IMAGE = 1;
   gtk_list_select_item(GTK_LIST(p->filelist), GetListPosition(p->piclist, p->activ));
   if ( Info(p->opt->f) ) FileInfo(NULL, p);
}


/*****************************************************************************************
 * Functions which are useful for almost all callbacks                                   *
 *****************************************************************************************/

void RefreshImage(PICTURE *bild)
/* make new entry in filelist and inform imlib about changes
 */
{
   char       buf[256];

   if ( !IS_PICTURE(bild) ) return;
   strncpy(buf, bild->file, 250);
   if ( bild->ext ) strcat(strcat(buf, "."), bild->ext);
   gtk_label_set(GTK_LABEL(bild->label), buf);
   if ( bild->im ) gdk_imlib_changed_image(bild->im);
   else 
      assert((bild->im = 
             at_imlib_set_image_from_data(bild->DATA, bild->W, bild->H)));
}

void ApplyPaulFunc(PAUL *p, void *func, int func_type, int which)
/* Apply function to marked images of PAUL
 */
{
   GList   *new_pl = NULL, *pl;

   if ( which == MARK )      new_pl = GetListFromSelection(GTK_LIST(p->filelist)->selection);
   else if ( which == THIS ) new_pl = g_list_append(NULL, BILD(p->activ));
   /* if ( which == ALL ) new_pl remains NULL !!! */
   
   if ( func_type == PAUL_FUNC_TYPE ) {
      GList *save_piclist = p->piclist;

      if ( new_pl ) p->piclist = new_pl;
      ((PAUL_FUNC)func)(p);
      p->piclist = save_piclist;
   } else {
      if ( new_pl ) ((PICLIST_FUNC)func)(new_pl);
      else          ((PICLIST_FUNC)func)(p->piclist);  /* this is done for ALL */
   }
   
   pl = NULL;
   if ( which == MARK ) pl = GTK_LIST(p->filelist)->selection;
   if ( which == ALL  ) pl = GTK_LIST(p->filelist)->children;
   if ( pl ) 
      do {
         RefreshImage(LI2BILD(pl->data));
      } while ( (pl = pl->next) );
   else
      RefreshImage(BILD(p->activ));

   ApplyPicture(p->show->window, BILD(p->activ));

   if ( new_pl ) g_list_free(new_pl);
}


/*****************************************************************************************
 * Functions to handle operation image                                                   *
 *****************************************************************************************/

void OperationToList(gpointer *realdata, gpointer *flagdata)
/* move operation image back to the image list
 * --- Parameter: ---
 * gpointer *realdata : list widget with single list element as operation image
 *                      PAUL *p set as user data    _OR_
 *                      PAUL *p if flagdata == NULL                        
 * gpointer *flagdata : NULL if realdate == PAUL *p, else if realdata is list widget
 * 
 */
{
   PAUL *p;
   
   if ( ANIMATION_SPEED != SINGLE ) {
      g_warning("Please avoid inserting operation image while animation is running");
      return;
   }
   if ( flagdata ) {
      g_return_if_fail(GTK_IS_LIST(realdata));
      p = (PAUL *)gtk_object_get_user_data(GTK_OBJECT(realdata));
   } else
      p = (PAUL *)realdata;
   g_return_if_fail(IS_PICTURE(p->op));
  
   if ( p->op->label ) gtk_widget_destroy(p->op->label);
   AddImageToFilelist(p, p->op);
   p->op = NULL;
   
   CreateOperationFrame(p);
}

void CreateOperationFrame(PAUL *p)
/* Set/Unset frame for name of operation image
 * --- Parameter: ---
 * PAUL          *p     :
 */
{
   static GtkWidget *frame     = NULL,
                    *list      = NULL,
                    *item      = NULL,
                    *label     = NULL;
   GtkWidget        *p_table    = GTK_WIDGET(GTK_TABLE((p->w)->parent));
   
   if ( !p->op ) {
      if ( frame ) gtk_widget_destroy(frame);
      frame = list = label = NULL;
      gtk_widget_show_all(p_table);
      return;
   }
   if ( !p->op->file ) p->op->file = strdup("Unknown");
   if ( frame ) {
      if ( label ) gtk_label_set(GTK_LABEL(label), p->op->file);
      else         g_warning("Frame has no label");
   } else {
      char buf[MAXPATHLEN];
      
      frame = gtk_frame_new("Operation image");
      gtk_table_attach_defaults(GTK_TABLE(p_table), frame, 0, 1, 1, 2);
      list  = gtk_list_new();
      gtk_container_add(GTK_CONTAINER(frame), list);
      gtk_signal_connect(GTK_OBJECT(list), "button_press_event", 
                         GTK_SIGNAL_FUNC(OperationToList), p);
      gtk_object_set_user_data(GTK_OBJECT(list), p);
      item  = gtk_list_item_new();
      gtk_container_add(GTK_CONTAINER(list), item);
      strcpy(buf, p->op->file);
      if ( p->op->ext ) strcat(strcat(buf, "."), p->op->ext);
      label = gtk_label_new(buf);
      gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
      gtk_container_add(GTK_CONTAINER(item), label);
   }
   gtk_widget_show_all(p_table);
}

void CreateOperationImageWindow(PAUL *p)
/*
 */
{
   PICTURE *bild = p->op;
     
   CreateOperationFrame(p);
   bild->label = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_widget_set_usize(bild->label, bild->W, bild->H);
   gtk_widget_show(bild->label);
   gtk_signal_connect(GTK_OBJECT(bild->label), "key_press_event", 
                      GTK_SIGNAL_FUNC(PictureKeyPressed), p);
   if ( !(bild->im) ) assert( (bild->im = 
                              at_imlib_set_image_from_data(bild->DATA, bild->W, bild->H)) );
   ApplyPicture(bild->label->window, bild);
   gtk_window_set_title(GTK_WINDOW(bild->label), "Operating image");
}

/*****************************************************************************************
 * Functions to handle animate button                                                    *
 *****************************************************************************************/

void SetAnimateButton(PAUL *p, GtkSignalFunc func, const char *label)
/* Set/Unset button at bottom line
 * --- Parameter: ---
 * PAUL          *p     :
 * GtkSignalFunc  func  : callback function to call when pressed or NULL
 * char          *label : label of button
 */
{
   static int        handler    = 0;
   static GtkWidget *button_box = NULL;
   GtkWidget        *p_table    = GTK_WIDGET(GTK_TABLE((p->w)->parent));
   
   if ( handler && p->ani_but ) {
      gtk_signal_disconnect(GTK_OBJECT(p->ani_but), handler);
      if ( gtk_signal_n_emissions_by_name(GTK_OBJECT(p->ani_but), "clicked") )
         gtk_signal_emit_stop_by_name(GTK_OBJECT(p->ani_but), "clicked");
   }

   if ( !label || !func ) {
      gtk_widget_show(p_table);
      handler = 0;
      if ( p->ani_but ) {
         gtk_widget_destroy(p->ani_but);
         p->ani_but = NULL;
      }
      if ( button_box ) {
         gtk_widget_destroy(button_box);
         button_box = NULL;
      }
      return;
   }
   
   if ( p->ani_but ) gtk_label_set(GTK_LABEL(GTK_BUTTON(p->ani_but)->child), label);
   else {
      p->ani_but = gtk_button_new_with_label(label);
      if ( !button_box ) {
         button_box = gtk_hbox_new(TRUE, 10);
         gtk_table_attach_defaults(GTK_TABLE(p_table), button_box, 0, 1, 2, 3);
      }
      gtk_container_add(GTK_CONTAINER(button_box), p->ani_but);
   }
   gtk_widget_show_all(p_table);
   
   handler = gtk_signal_connect(GTK_OBJECT(p->ani_but), "clicked", func, p);
   ButtonSensitive(p);
}

void ButtonSensitive(PAUL *p)
/* Make animate button sensitive or insensitive
 */
{
   GList *pl;
   
   if ( !p || !(p->ani_but) ) return;
   pl = (p->filelist)->selection;
   if ( (!pl || !pl->next) && !DoMatchPictures(p->opt->f) ) 
      gtk_widget_set_sensitive(p->ani_but, FALSE);
   else
      gtk_widget_set_sensitive(p->ani_but, TRUE);
}

