/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Uses:      fftw (Fastest Fourier Transform in the West from MIT)          *
 * Modul:     fft.c                                                          *
 *            fast fourier transform                                         *
 * Autor:     Andreas Tille                                                  *
 * Datum:     04.10.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <rfftw.h>

#include "paul.h" 

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

fftw_real *AllocFftwTarget(int w, int h)
/* allocate array of real
 * in case of even h add two rows, else add one row to enable in-place transformation
 */
{
   fftw_real *fp;
   
   assert ( (fp = calloc(w * (h + (h & 1 ? 1 : 2)), sizeof(fftw_real))) );
   return fp;
}

int MakeFloat(PICTURE *bild, fftw_real *fdata)
/* converts image data of a gray (i.e. green for paul) image into real
 * there will be no check, whether the picture is real gray, 
 * the green component is simply used
 * --- Parameter: ---
 * PICTURE   *bild        : gray images
 * fftw_real *fdata       : buffer to store float data
 * --- Return: ---
 * fftw_real *fdata       : image data as float
 * int        MakeFloat() : 0 if OK
 */
{
   register  fftw_real *fp;
   register  int        storepix = bild->storepix;
   register  unsigned   char *ap, *fip;

   if ( bild->id != PAUL_ID ) return -1;
   fp = fdata;

   for ( fip = (ap = bild->DATA + (storepix == 3 ? 1 : 0)) + bild->size * storepix; 
         ap < fip; ap += storepix, fp++ ) 
      *fp = (fftw_real)(*ap);
   return 0;
}

int MakeImageData(PICTURE *bild, fftw_real *fdata)
/* converts float data to unsigned char image data
 * use green values if all floats are >= 0 
 * use blue color for all values > 0 and red color for values < 0
 * --- Parameter: ---
 * PICTURE   *bild            : picture structure to store the data
 * fftw_real *fdata           : imaga data as float 
 * --- Return: ---
 * PICTURE   *bild            : picture structure with new image data
 * int        MakeImageData() : 0 if OK
 */
{
   fftw_real                *fip, fmax, fmin;
   register  fftw_real      *fp;
   register  unsigned  char *ap, *bp;

   if ( bild->id != PAUL_ID ) return -1;

   fmax = fmin = *(fp = fdata);
   for ( fip = fp++ + bild->size; fp < fip; fp++ ) {
      if ( *fp < fmin ) fmin = *fp;
      if ( *fp > fmax ) fmax = *fp;
   }
   
   assert ( (ap = malloc(3 * bild->size )) );
   bild->storepix = bild->spp = 3;
   NewImage(bild, bild->W, bild->H, ap);

   if ( fmax <= 0.0 && fmin <= 0.0 ) { /* invert values if all are negativ */
      for ( fip = (fp = fdata) + bild->size; fp < fip; fp++ ) *fp *= -1.0;
      fmax *= -1.0;
      fmin *= -1.0;
   }
   
   bp  = ap + 2;   /* for the negativ values take red color */
   if ( fmax >= 0.0 && fmin >= 0.0 ) ap++;  /* if all are positiv take green */
printf("%s(%i) fmax = %g, fmin = %g.\n", __FILE__, __LINE__, fmax, fmin);
    
   for ( fip = (fp = fdata) + bild->size; fp < fip; ap += 3, bp += 3, fp++ ) {
      if ( *fp < 0.0 ) *bp = (unsigned char)rint(255 * *fp / fmin);
      else             *ap = (unsigned char)rint(255 * *fp / fmax);
   }

   return 0;
}

int MakeNormalizedImageData(PICTURE *bild, fftw_real *fdata)
/* converts float data to unsigned char image data
 * use green values if all floats are >= 0 
 * use blue color for all values > 0 and red color for values < 0
 * --- Parameter: ---
 * PICTURE   *bild            : picture structure to store the data
 * fftw_real *fdata           : imaga data as float 
 * --- Return: ---
 * PICTURE   *bild            : picture structure with new image data
 * int        MakeImageData() : 0 if OK
 */
{
   fftw_real                *fip, scale = 1.0 / (bild->W * bild->H);
   register  fftw_real      *fp;
   register  int             i;
   register  unsigned  char *ap, *bp;

   if ( bild->id != PAUL_ID ) return -1;

   assert ( (ap = malloc(3 * bild->size )) );
   bild->storepix = bild->spp = 3;
   NewImage(bild, bild->W, bild->H, ap);

   bp  = ap + 2;   /* for the negativ values take red color */
    
   for ( fip = (fp = fdata) + bild->size; fp < fip; ap += 3, bp += 3, fp++ ) {
   if ( (i = rint(scale * *fp)) < 0 )
      *bp = (unsigned char)( -i >  255 ? 255 : i);
   else 
      *ap = (unsigned char)(  i >  255 ? 255 : i);
   }

   return 0;
}

int FastFourierTransform(PAUL *p)
/* calculate fast fourier transforms of images
 * --- Parameter: ---
 * PAUL *p                      : list of images
 * --- Return: ---
 * int   FastFourierTransform() : 0 if OK
 */
{
   static    rfftwnd_plan  plan = NULL, back = NULL;
   PICTURE                *bild;
   GList                  *pl;
   fftw_real              *fft, *real;
   register  fftw_real    *sp, *tp, *fip; 
   int                     i;
   
   if ( !NBILDER(p->piclist) ) return 0;

   for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) {
      fft = AllocFftwTarget(bild->W, bild->H);
      if ( MakeFloat(bild, fft) ) continue;
      if ( plan && (plan->n[0] != bild->W || plan->n[1] != ((bild->H >> 1) + 1)) ) {
	 rfftwnd_destroy_plan(plan);
         rfftwnd_destroy_plan(back);
         plan = NULL;
      }
      if ( !plan ) {
         plan  = rfftw2d_create_plan(bild->W, bild->H, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE | FFTW_IN_PLACE);
         back  = rfftw2d_create_plan(bild->W, bild->H, FFTW_COMPLEX_TO_REAL, FFTW_ESTIMATE | FFTW_IN_PLACE);
      }
      rfftwnd_one_real_to_complex(plan, fft, NULL);
      printf("Now start back transformation.\n");
      rfftwnd_one_complex_to_real(back, fft, NULL);
      /*
      real = AllocFftwTarget(bild->W, bild->H);

      for ( fip = (sp = fft + bild->W) + bild->size, tp = real + (bild->W >> 2); sp < fip; 
            sp += bild->W, tp += bild->W ) {
         for ( i = 0; i < bild->W; i++ )
            *(tp + i) = *(sp + (i<<1));
      }
      */
      MakeNormalizedImageData(bild, fft);
      free(fft);
      /* free(real); */
   }
   
   return 0;
}









