// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
   OString.h

   String Class

   Fri Aug 21 1992 /SCFR::Kepler.
   Reviewed: Wed Sep 16 1992

   Revision: June 1994
   Stephane Rehel, SCFR::Kepler, rehel@ensisun.imag.fr

   Revision: July 28th, 1995
   Stephane Rehel [SCFR::Kepler], Stephane.Rehel@imag.fr

   August 9th, 1995 /SR
     add insert() and remove()

   Revision: April 9th, 1996
     change filename strgcls.h to OString.h
     add strgcls.h binding
*/


#ifndef __OString_h
#define __OString_h

#include <string.h>
#include <assert.h>

#ifndef __boolean_h
#include "boolean.h"
#endif


/////////////////////////////////////////////////////////////////////////////
// OString Class
//
class OString
{
public:
  static OString nil; // unassigned string

  static const OString& empty();

private:

  char* theString;
  size_t theLength; // length of string, not of the char buffer
  size_t bufferLength;

public:

  OString()
    {
    theString= 0;
    }
  OString( size_t length )
    {
    theString= 0;
    _alloc( length );
    }
  OString( const char* initStrg )
    {
    theString= 0;
    operator= ( initStrg );
    }
  OString( const char* initStrg, int strgLength )
    {
    theString= 0;
    assert( initStrg != 0 );
    set( initStrg, strgLength );
    }
  OString( const OString& os )
    {
    theString= 0;
    operator= ( os );
    }
  ~OString()
    {
    destroy();
    }

  void destroy();
  OString& operator=( const char* initStrg )
    {
    assert( initStrg != 0 );
    set( initStrg, strlen(initStrg) );
    return *this;
    }
  OString& operator=( const OString& os )
    {
    assert( os.theString != 0 );
    set( os.theString, os.theLength );
    return *this;
    }

  void alloc( size_t bufferLength );
  void realloc( size_t newLength );

private:

  void set( const char* initStrg, size_t strgLength );
  void _alloc( size_t length );

public:

// access
  boolean assigned() const
    {
    return theString != 0;
    }
  int length() const
    {
    return int(theLength);
    }
  boolean isEmpty() const
    {
    return theLength == 0;
    }

  const char* get() const
    {
    assert( assigned() );
    return theString;
    }
  char* _get() const
    {
    assert( assigned() );
    return theString;
    }

  boolean operator==( const OString& os ) const
    {
    assert( assigned() );
    assert( os.assigned() );
    return theLength == os.theLength &&
           strcmp( theString, os.theString ) == 0;
    }
  boolean operator==( const char* strg ) const
    {
    assert( assigned() );
    assert( strg != 0 );
    return strcmp( theString, strg ) == 0;
    }
  boolean operator!=( const OString& os ) const
    {
    assert( assigned() );
    assert( os.assigned() );
    return theLength != os.theLength ||
           strcmp( theString, os.theString ) != 0;
    }

/*
  boolean operator || ( const OString& os ) const
    {
    return stricmp( theString, os.theString ) == 0;
    }
*/

  char operator[] ( size_t i ) const
    {
    assert( assigned() );
    return (i>=theLength) ? 0 : theString[i];
    }

  friend OString operator + ( const OString& os1, const OString& os2 );
  OString operator + ( char ch );
  OString& operator += ( const OString& os );
  OString& operator += ( const char* s );
  OString& operator += ( char ch );
  OString& operator += ( int i );

  // get string between indices i1 and i2
  // if i1 < 0 then i1 = 0
  // if i2 >= theLength then i2 = thenLength - 1
  // if i1 > i2 then return ""
  OString getSub( int i1, int i2 ) const;

  // get the first 'len' chars of string
  OString getFirst( int len ) const
    {
    return getSub( 0, len-1 );
    }

  // get the string between index i1 and the end
  OString getLast( int i1 ) const
    {
    // we set i2=length() so that i1 may equal length() too
    return getSub( i1, length() );
    }

  void insert( size_t i, char ch ); // 0 <= i <= length()
  void remove( size_t i ); // 0 <= i
  void fill( size_t length, int ch );

  // 0 <= i < theLength
  void put( size_t i, char ch );

  // return first index if char is found
  // if not found return -1
  int locateChar( char ch ) const;

  static boolean glob( const char* expression, const char* pattern );
  boolean glob( const char* pattern ) const;
  boolean glob( const OString& pattern ) const;
  boolean isPattern() const;

  OString getFirstWord() const;
  OString trim() const;

  static OString itoa( int i );

  void toUpper();
  void toLower();
};

/////////////////////////////////////////////////////////////////////////////

inline boolean operator< ( const OString& os1, const OString& os2 )
{
  assert( os1.assigned() );
  assert( os2.assigned() );
  return strcmp( os1.get(), os2.get()) < 0;
}

/////////////////////////////////////////////////////////////////////////////

inline boolean operator<=( const OString& os1, const OString& os2 )
{
  assert( os1.assigned() );
  assert( os2.assigned() );
  return strcmp( os1.get(), os2.get() ) <= 0;
}

/////////////////////////////////////////////////////////////////////////////

inline boolean operator> ( const OString& os1, const OString& os2 )
{
  assert( os1.assigned() );
  assert( os2.assigned() );
  return strcmp( os1.get(), os2.get() ) > 0;
}

/////////////////////////////////////////////////////////////////////////////

inline boolean operator>=( const OString& os1, const OString& os2 )
{
  assert( os1.assigned() );
  assert( os2.assigned() );
  return strcmp( os1.get(), os2.get() ) >= 0;
}

/////////////////////////////////////////////////////////////////////////////

//ostream& operator<<( ostream& outStream, const OString& os );

OString operator + ( const OString& os, int i );
OString operator + ( int i, const OString& os );

/////////////////////////////////////////////////////////////////////////////

#endif // ifndef __OString_h
