/* GGlyph - a graphical utility for managing one's collection of Type
   1 fonts (and eventually, other sorts of fonts) under X11.

   Copyright (c) 1998 David Huggins-Daines
   <bn711@freenet.carleton.ca>.

   You are permitted to copy, distribute, and modify this file under the
   terms of the GNU General Public License, version 2, or any later
   version.  See the file COPYING for more details. */

/*  viewer_window.c - the pop-up windows for viewing fonts */

#include <gtk/gtk.h>
#include <stdio.h>
#include "gglyph.h"
#include "viewer_window.h"
#include "fonts.h"

typedef struct _ViewerWindow ViewerWindow;

struct _ViewerWindow 
{
  GtkWidget *window;
  
  GtkObject *size;
  GtkWidget *frame, *preview, *antialias;
  GtkWidget *name, *family, *weight, *mono, *xlfd;

  FontFileRec *ffile;
};

static void cb_viewer_sample_sheet (GtkWidget *button, FontFileRec
				    *ffile);
static void cb_viewer_drop (GtkWidget *w, GdkEvent *event,
			    ViewerWindow *viewer);
static void set_value (GtkObject *adj, ViewerWindow *viewer);
static gint viewer_set_font (ViewerWindow *viewer,
			     GtkType1FontSpec spec, gfloat size,
			     gfloat slant, gfloat extend);

extern gint debug_level;
extern gchar *wdir_drop_types[], *viewer_title;

static void set_value (GtkObject *adj, ViewerWindow *viewer)
{
  if (adj == viewer->size)
    gtk_type1_preview_set_size (GTK_TYPE1_PREVIEW(viewer->preview),
				GTK_ADJUSTMENT(adj)->value);
  else if (adj == GTK_OBJECT(viewer->antialias))
    gtk_type1_preview_set_aa (GTK_TYPE1_PREVIEW(viewer->preview),
			      GTK_TOGGLE_BUTTON(adj)->active);
}

static gint viewer_set_font (ViewerWindow *viewer,
			     GtkType1FontSpec spec, gfloat size,
			     gfloat slant, gfloat extend)
{
  gint retval, font_id;
  
  g_return_val_if_fail (viewer != NULL, -1);

  retval = gtk_type1_preview_set_font (GTK_TYPE1_PREVIEW (viewer->preview),
				       spec, size,
				       slant, extend);
  if (retval == -1)
    return -1;

  font_id = GTK_TYPE1_PREVIEW (viewer->preview)->font_id;
  gtk_label_set (GTK_LABEL (viewer->name),
		 T1_GetFontName (font_id));
  gtk_label_set (GTK_LABEL (viewer->family),
		 T1_GetFamilyName (font_id));
  gtk_label_set (GTK_LABEL (viewer->weight),
		 T1_GetWeight (font_id));
  gtk_label_set (GTK_LABEL (viewer->mono),
		 T1_GetIsFixedPitch (font_id) ? "yes" : "no");
  
  gtk_adjustment_set_value (GTK_ADJUSTMENT(viewer->size), size);

  return 0;
}

void new_viewer_window (FontFileRec *ffile)
{
  GtkWidget *box, *table, *hbox, *label, *slider, *button;
  ViewerWindow *viewer;
  GtkType1FontSpec spec;

  viewer = g_new (ViewerWindow, 1);
  
  /* Create window for output */
  viewer->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_usize (viewer->window, 400, 400);
  gtk_window_set_title (GTK_WINDOW(viewer->window), ffile->pathname);
  gtk_container_border_width (GTK_CONTAINER(viewer->window), 0);
  /* allow_shrink, allow_grow, auto_shrink */
  gtk_window_set_policy (GTK_WINDOW (viewer->window), FALSE, TRUE,
			 TRUE);

  /* homogeneous, spacing */
  box = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER(box), 10);
  gtk_container_add (GTK_CONTAINER (viewer->window), box);
  gtk_widget_show (box);  

  viewer->preview = gtk_type1_preview_new();
  gtk_signal_connect (GTK_OBJECT (viewer->preview),
		      "drop_data_available_event",
		      GTK_SIGNAL_FUNC(cb_viewer_drop), viewer);
  
  viewer->frame = gtk_frame_new (NULL);
  
  gtk_widget_show (viewer->preview);
  gtk_container_add (GTK_CONTAINER(viewer->frame), viewer->preview);
  gtk_widget_show (viewer->frame);
  
  /* expand=TRUE - fill the box
     fill=TRUE - expand the viewer to fill the box.
  */
  gtk_box_pack_start (GTK_BOX(box), viewer->frame, TRUE, TRUE, 0);

  table = gtk_table_new (4, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);
  gtk_table_set_col_spacings (GTK_TABLE(table), 10);

  gtk_box_pack_start (GTK_BOX(box), table, FALSE, FALSE, 10);
  gtk_widget_show (table);
  
  hbox = gtk_hbox_new (FALSE, 10);
  label = gtk_label_new ("Fontname:");
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
  viewer->name = gtk_label_new ("");
  gtk_widget_show (viewer->name);
  gtk_box_pack_start (GTK_BOX(hbox), viewer->name, FALSE, FALSE, 0);
  gtk_widget_show (hbox);
  gtk_table_attach_defaults (GTK_TABLE (table), hbox, 0, 1, 0, 1);
/*   gtk_box_pack_start (GTK_BOX(box), hbox, FALSE, FALSE, 2); */

  hbox = gtk_hbox_new (FALSE, 10);
  label = gtk_label_new ("Family:");
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
  viewer->family = gtk_label_new ("");
  gtk_widget_show (viewer->family);
  gtk_box_pack_start (GTK_BOX(hbox), viewer->family, FALSE, FALSE, 0);
  gtk_widget_show (hbox);
  gtk_table_attach_defaults (GTK_TABLE (table), hbox, 0, 1, 1, 2);

  hbox = gtk_hbox_new (FALSE, 10);
  label = gtk_label_new ("Weight:");
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
  viewer->weight = gtk_label_new ("");
  gtk_widget_show (viewer->weight);
  gtk_box_pack_start (GTK_BOX(hbox), viewer->weight, FALSE, FALSE, 0);
  gtk_widget_show (hbox);
  gtk_table_attach_defaults (GTK_TABLE (table), hbox, 0, 1, 2, 3);

  hbox = gtk_hbox_new (FALSE, 10);
  label = gtk_label_new ("Monospaced:");
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
  viewer->mono = gtk_label_new ("");
  gtk_widget_show (viewer->mono);
  gtk_box_pack_start (GTK_BOX(hbox), viewer->mono, FALSE, FALSE, 0);
  gtk_widget_show (hbox);
  gtk_table_attach_defaults (GTK_TABLE (table), hbox, 0, 1, 3, 4);

  button = gtk_button_new_with_label ("View Sample Sheet");
  gtk_object_set_user_data (GTK_OBJECT (button), GINT_TO_POINTER
			    (VIEW_SAMPLE));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC
		      (cb_viewer_sample_sheet), ffile);
  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 0, 2, GTK_SHRINK,
		    GTK_SHRINK, 0, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  
  button = gtk_button_new_with_label ("Print Sample Sheet");
  gtk_object_set_user_data (GTK_OBJECT (button), GINT_TO_POINTER
			    (PRINT_SAMPLE));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC
		      (cb_viewer_sample_sheet), ffile);
  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 2, 4, GTK_SHRINK,
		    GTK_SHRINK, 0, 0);
  gtk_widget_show (button);

  if (ffile->xlfd){
    /* FIXME: this always gets clipped at the edges, and I have no
       idea why */
    viewer->xlfd = gtk_label_new (ffile->xlfd);
    gtk_box_pack_start (GTK_BOX (box), viewer->xlfd, FALSE, FALSE, 10);
    gtk_widget_show (viewer->xlfd);
  }

  hbox = gtk_hbox_new (FALSE, 0);
  label = gtk_label_new ("Size:");
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 10);
  viewer->size = gtk_adjustment_new (DEFAULT_SIZE, 6, 150, 1, 4, 0);
  gtk_signal_connect (GTK_OBJECT(viewer->size), "value_changed",
		      GTK_SIGNAL_FUNC(set_value), viewer);
  slider = gtk_hscale_new (GTK_ADJUSTMENT(viewer->size));
  gtk_scale_set_digits (GTK_SCALE(slider), 0);
  gtk_range_set_update_policy (GTK_RANGE(slider),
			       GTK_UPDATE_DISCONTINUOUS);
  gtk_widget_show (slider);
  gtk_box_pack_start (GTK_BOX(hbox), slider, TRUE, TRUE, 10);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX(box), hbox, FALSE, FALSE, 2);
  
  viewer->antialias = gtk_check_button_new_with_label ("Antialias");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(viewer->antialias),
			       GTK_TYPE1_PREVIEW(viewer->preview)->aa);
  
  gtk_signal_connect (GTK_OBJECT(viewer->antialias), "toggled",
		      GTK_SIGNAL_FUNC (set_value), viewer);
  gtk_widget_show (viewer->antialias);
  gtk_box_pack_start (GTK_BOX(box), viewer->antialias, FALSE, FALSE, 2);
  
  spec.type = GTK_TYPE1_FONTID;
  spec.font.t1_fontid = ffile->t1_fontid;
  viewer_set_font (viewer, spec, DEFAULT_SIZE, 0.0, 1.0);
  viewer->ffile = ffile;
  gtk_widget_show (viewer->window);
  
  gtk_widget_dnd_drop_set (viewer->preview, TRUE, wdir_drop_types, 1,
			   FALSE);
}

static void cb_viewer_drop (GtkWidget *w, GdkEvent *event,
			    ViewerWindow *viewer)
{
  FontFileRec *ffile;
  GtkType1FontSpec spec;
  gchar *pathname = (gchar *)event->dropdataavailable.data;

  ffile = find_font (pathname);
  
  if (ffile == NULL){
    fprintf (stderr, "Recieved bad drop data %s\n", pathname);
    return;
  }

  if (debug_level){
    g_print ("Dropped font is %s, record %p, id %d\n", pathname,
	     ffile, ffile->t1_fontid);
  }
  
  spec.type = GTK_TYPE1_FONTID;
  spec.font.t1_fontid = ffile->t1_fontid;
  viewer_set_font (viewer, spec, GTK_ADJUSTMENT
		   (viewer->size)->value, 0, 1);
  gtk_window_set_title (GTK_WINDOW(viewer->window), pathname);
  if (ffile->xlfd){
    gtk_label_set (GTK_LABEL(viewer->xlfd),
		   ffile->xlfd);
    gtk_widget_show (viewer->xlfd);
  }
  else {
    gtk_widget_hide (viewer->xlfd);
  }
    
  viewer->ffile = ffile;
}

gint cb_viewer_window (FontFileRec *ffile)
{
  new_viewer_window (ffile);

  return FALSE;
}

static void cb_viewer_sample_sheet (GtkWidget *button, FontFileRec *ffile)
{
  SampleAction act;

  act = GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT
						   (button)));

  do_sample_sheet (ffile, act);
}



