/* gtk_type1_preview.c: Type1 font display-box widget using t1lib.
   (c) 1998 David Huggins-Daines <bn711@freenet.carleton.ca>

   You may modify and distribute this file under the terms of the GNU
   General Public License, version 2, or any later version, at your
   convenience. See the file COPYING for details. */

#include "gtk_type1_preview.h"
#include "gdk_t1lib.h"
#include <string.h>

#define TEST_STRING "The quick brown fox jumps over the lazy dog."
#define BORDER_WIDTH 5

static void gtk_type1_preview_class_init (GtkType1PreviewClass
					  *klass);
static void gtk_type1_preview_init (GtkType1Preview *preview);
static gint gtk_type1_preview_expose (GtkWidget *w, GdkEventExpose
				      *evt);
static void gtk_type1_preview_size_allocate (GtkWidget *widget,
					     GtkAllocation
					     *allocation);

static void gtk_type1_preview_realize (GtkWidget *widget);
static void set_justified_line (GtkType1Preview *preview, GdkGC *gc,
				gint *origin, gint cwidth, gchar *lstart,
				gint len, gint spacecount);

guint gtk_type1_preview_get_type (void)
{
  static guint gtktype1preview_type = 0;
  if (!gtktype1preview_type) {
    GtkTypeInfo gtktype1preview_info = {
      "GtkType1Preview",
      sizeof(GtkType1Preview),
      sizeof(GtkType1PreviewClass),
      (GtkClassInitFunc) gtk_type1_preview_class_init,
      (GtkObjectInitFunc) gtk_type1_preview_init,
      (GtkArgSetFunc) NULL,
      (GtkArgGetFunc) NULL,
    };
    gtktype1preview_type = gtk_type_unique (gtk_widget_get_type(),
					    &gtktype1preview_info);
  }
  return gtktype1preview_type;
}

static void gtk_type1_preview_class_init (GtkType1PreviewClass *klass)
{
  GtkWidgetClass *widget_class;

  if (CheckForInit())
    gdk_t1lib_init(NO_LOGFILE);
  
  widget_class = (GtkWidgetClass*) klass;

  widget_class->realize = gtk_type1_preview_realize;
  widget_class->size_allocate = gtk_type1_preview_size_allocate;
  widget_class->expose_event = gtk_type1_preview_expose;
}

static void gtk_type1_preview_init (GtkType1Preview *preview)
{
  GTK_WIDGET_SET_FLAGS (preview, GTK_BASIC);
  
  preview->font_id = 0;
  preview->size = 0.0;
  preview->slant = 0.0;
  preview->extend = 0.0;
  preview->pixmap = NULL;
  preview->aa = TRUE;
  preview->clean = TRUE;
}

static void gtk_type1_preview_realize (GtkWidget *widget)
{
  GtkType1Preview *preview;
  GdkWindowAttr attributes;
  gint attributes_mask;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TYPE1_PREVIEW (widget));

  preview = GTK_TYPE1_PREVIEW (widget);
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, preview);

  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window,
			    GTK_STATE_NORMAL);
  
  GTK_TYPE1_PREVIEW(widget)->pixmap =
    gdk_pixmap_new (widget->window,
		    widget->allocation.width - BORDER_WIDTH * 2,
		    widget->allocation.height - BORDER_WIDTH * 2,
		    gdk_visual_get_best_depth());
}

static void gtk_type1_preview_size_allocate (GtkWidget *widget,
					     GtkAllocation *allocation)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TYPE1_PREVIEW (widget));
  g_return_if_fail (allocation != NULL);
  
  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);
      
      if (GTK_TYPE1_PREVIEW(widget)->pixmap)
	gdk_pixmap_unref (GTK_TYPE1_PREVIEW(widget)->pixmap);
      
      GTK_TYPE1_PREVIEW(widget)->pixmap =
	gdk_pixmap_new (widget->window,
			allocation->width - BORDER_WIDTH * 2,
			allocation->height - BORDER_WIDTH * 2,
			gdk_visual_get_best_depth());
      
      gtk_type1_preview_expose (widget, NULL);
    }
}

GtkWidget *gtk_type1_preview_new (void)
{
  GtkType1Preview *preview;

  preview = gtk_type_new (gtk_type1_preview_get_type());

  return GTK_WIDGET(preview);
}

gint gtk_type1_preview_set_font (GtkType1Preview *preview,
				 GtkType1FontSpec spec, gfloat size,
				 gfloat slant, gfloat extend)
{
  gint font_id;

  if (spec.type == GTK_TYPE1_FONTID){
    font_id = spec.font.t1_fontid;
    if (font_id < 0 || font_id > T1_Get_no_fonts())
      return -1;
  }
  else if (spec.type == GTK_TYPE1_FILE){
    font_id = T1_AddFont (spec.font.filename);
    if (font_id < 0)
      return -1;
  }
  else {
    /* WTF */
    return -1;
  }

  /* just in case */
  
  T1_DeleteAllSizes (preview->font_id);
  
  preview->font_id = font_id;
  preview->size = size;
  preview->slant = slant;
  preview->extend = extend;

  if (slant || extend) {
    T1_DeleteAllSizes (font_id);
    T1_LoadFont (font_id);
    if (slant)
      T1_SlantFont (font_id, slant);
    if (extend)
      T1_ExtendFont (font_id, extend);
  }
  
  preview->clean = FALSE;

  if (GTK_WIDGET_DRAWABLE(GTK_WIDGET(preview)))
    gtk_type1_preview_expose (GTK_WIDGET(preview), NULL);

  return 0;
}

gint gtk_type1_preview_set_size (GtkType1Preview *preview, gfloat
				 size)
{
  if (size != preview->size){
    preview->size = size;
    preview->clean = FALSE;
    
    if (GTK_WIDGET_DRAWABLE(GTK_WIDGET(preview)))
      gtk_type1_preview_expose (GTK_WIDGET(preview), NULL);
  }
  return 0;
}
  
gint gtk_type1_preview_set_extend (GtkType1Preview *preview, gfloat
				   extend)
{
  if (extend != preview->extend){
    preview->extend = extend;
    T1_DeleteAllSizes (preview->font_id);
    T1_LoadFont (preview->font_id);
    T1_ExtendFont (preview->font_id, extend);
    preview->clean = FALSE;
    
    if (GTK_WIDGET_DRAWABLE(GTK_WIDGET(preview)))
      gtk_type1_preview_expose (GTK_WIDGET(preview), NULL);
  }
  return 0;
}
  
gint gtk_type1_preview_set_slant (GtkType1Preview *preview, gfloat
				  slant)
{
  if (slant != preview->slant){
    preview->slant = slant;
    T1_DeleteAllSizes (preview->font_id);
    T1_LoadFont (preview->font_id);
    T1_SlantFont (preview->font_id, slant);
    preview->clean = FALSE;
    
    if (GTK_WIDGET_DRAWABLE(GTK_WIDGET(preview)))
      gtk_type1_preview_expose (GTK_WIDGET(preview), NULL);
  }
  return 0;
}
  
gint gtk_type1_preview_set_aa (GtkType1Preview *preview, gint aa)
{
  if (aa != preview->aa){
    preview->aa = aa;
    T1_DeleteAllSizes (preview->font_id);
    preview->clean = FALSE;
    
    if (GTK_WIDGET_DRAWABLE(GTK_WIDGET(preview)))
      gtk_type1_preview_expose (GTK_WIDGET(preview), NULL);
  }
  return 0;
}

static gint gtk_type1_preview_expose (GtkWidget *w, GdkEventExpose
				      *evt)
{
  GtkType1Preview *preview;
  gint width, height;
  
  g_assert (w != NULL);
  g_return_val_if_fail (GTK_IS_TYPE1_PREVIEW(w), FALSE);

  preview = GTK_TYPE1_PREVIEW(w);
  
  if (preview->size == 0.0)
    return TRUE;
  
  if (w->allocation.width == 0 || w->allocation.height == 0)
    return TRUE;
  
  width = w->allocation.width - (BORDER_WIDTH * 2);
  height = w->allocation.height - (BORDER_WIDTH * 2);

  if (preview->clean && evt){
    gdk_window_clear_area (w->window, evt->area.x, evt->area.y,
			   evt->area.width, evt->area.height);
    gdk_draw_pixmap (w->window, w->style->text_gc[GTK_WIDGET_STATE(w)],
		     preview->pixmap, 0, 0,
		     BORDER_WIDTH, BORDER_WIDTH, width, height);
    return TRUE;
  }
  else {
    /* love that GCC */
    gint lines, len, twidth, cwidth, cheight, origin, spacecount;
    gchar *lstart, *wstart;
    GdkGC *gc;

    /* how many lines will fit? */
    lines = height / preview->size;
  
    if (lines == 0)
      return TRUE;
    
    /* calculate the charspace width and height of the preview area.
       device resolution is assumed to be 72 dpi for now. */
    cwidth = 1000 * width / preview->size;
    cheight = 1000 * height / preview->size;

    gdk_draw_rectangle (preview->pixmap, w->style->bg_gc[GTK_WIDGET_STATE(w)],
			TRUE, 0, 0, width, height);
    
    gc = gdk_gc_new (w->window);
    gdk_gc_copy (w->style->text_gc[GTK_WIDGET_STATE(w)], gc);
    gdk_gc_set_background (gc, &w->style->bg[GTK_WIDGET_STATE(w)]);

    /* set the pointers to the beginning of the string */
    wstart = lstart = TEST_STRING;
    /* reset the space, length and total width counters */
    len = twidth = spacecount = 0;
    origin = 0;

    /* break lines and draw them */
    while (TRUE){
      /* get the next word */
      gchar *wend = strchr (wstart, ' ');
      gint wlen = wend ? (wend - wstart) : strlen (wstart);

      /* get its width */
      twidth += T1_GetStringWidth (preview->font_id, wstart,
				   wlen,
				   0, TRUE);

      
      /* brain-dead line-wrapping algorithm: if we exceed 80% of the
         total charspace width, break the line and draw it, unless
	 there are no spaces, in which case, don't even bother
<	 breaking the line. */
      if (twidth > (cwidth * 8 / 10)){
	/* the origin gets incremented by this function */
	if (spacecount){
	  set_justified_line (preview, gc, &origin, cwidth, lstart,
			      wstart - lstart - 1, spacecount - 1);
	
	  /* reset counters */
	  len = twidth = spacecount = 0;
	  /* move lstart to wstart */
	  lstart = wstart;
	}
	else {
	  set_justified_line (preview, gc, &origin, cwidth, lstart,
			      wstart - lstart + wlen, spacecount);
	  /* reset counters */
	  len = twidth = spacecount = 0;
	  /* DON'T move lstart to wstart - move it past the space */
	  wstart += wlen + 1;
	  lstart = wstart;
	}
      }
      /* if we're at the end, draw the line and get out */
      else if (wend == NULL) {
	/* the origin gets incremented by this function */
	set_justified_line (preview, gc, &origin, cwidth, lstart,
			    wstart - lstart + wlen, spacecount);
	break;
      }
      else {
	/* jump past the space and bump up the counter */
	wstart += wlen + 1;
	spacecount++;
      }
    }
    
    gdk_draw_pixmap (w->window, gc,
		     preview->pixmap, 0, 0,
		     BORDER_WIDTH, BORDER_WIDTH, width, height);
    gdk_gc_unref (gc);

    preview->clean = TRUE;
    
    return TRUE;
  }
}

static void set_justified_line (GtkType1Preview *preview, GdkGC *gc,
				gint *origin, gint cwidth, gchar
				*lstart, gint len, gint spacecount)
{
  gint spacewidth, baseline;
  gfloat spaceoff;
  BBox bbox;

  /* get the width of a space */
  spacewidth = T1_GetCharWidth (preview->font_id, ' ');
  
  /* get the width of the line */

  bbox = T1_GetStringBBox (preview->font_id, lstart, len, 0.0,
			   TRUE);

  /* to dodge floating-point exceptions */
  spaceoff = spacecount ? (float) (cwidth - bbox.urx)
    / spacecount : spacecount; 

  /* transform the bbox */
  bbox.urx = (bbox.urx + (spaceoff * spacecount)) * preview->size / 1000;
  bbox.ury = bbox.ury * preview->size / 1000;
  bbox.llx = bbox.llx * preview->size / 1000;
  bbox.lly = bbox.lly * preview->size / 1000;

  baseline = *origin + bbox.ury;
  
  if (preview->aa)
    gdk_t1lib_set_string_aa (preview->pixmap,
			     gc, FALSE,
			     0, baseline,
			     preview->font_id, lstart,
			     len, spaceoff,
			     T1_KERNING,
			     preview->size, 0.0);
  else
    gdk_t1lib_set_string (preview->pixmap,
			  gc, FALSE,
			  0, baseline,
			  preview->font_id, lstart,
			  len, spaceoff,
			  T1_KERNING,
			  preview->size, 0.0);

  /* now move down a line */
  *origin += preview->size;
}
