/* Risk! for Linux..
 * board.c
 *
 * Copyright (C) 1998 Joel Nordell
 *
 * This file is protected under the GPL license.
 */

#include <stdio.h>
#include "board.h"
#include "board_xpm.h"
#include "gamedefs.h"
#include "imap.h"
#include "game.h"

extern GdkPixmap *BoardPixmap;
extern GdkPixmap *BoardBackup;

static GtkWidget *TheFixed = NULL; /* For easy access to fixed widget */
static GtkWidget *TheBoard = NULL; /* For easy access to board widget */

gint CurrentCountry = -1;
gint CountryDown = -2;    // Which country is being "pressed"

GdkFont *ArmyFont = NULL; // Font for army display

void redraw_board_pixmap(GtkWidget *widget, gpointer data) {
  GdkRectangle update_rect;

  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = 640;
  update_rect.height= 381;

  gdk_draw_pixmap(BoardPixmap,
                  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                  BoardBackup,
                  0, 0, 0, 0, 
                  update_rect.width, update_rect.height);  
}

void colorize(GtkWidget *widget, gpointer data) {
  GtkWidget *board = data;
  GdkRectangle update_rect;
  int i;

  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = 640;
  update_rect.height= 381;

  redraw_board_pixmap(widget, data);

  for (i=0; i<Continents[NORTH_AMERICA].num_countries; i++)
    change_country_color(board, Continents[NORTH_AMERICA].countries[i],
                       65535, 65535, 0);
  for (i=0; i<Continents[SOUTH_AMERICA].num_countries; i++)
    change_country_color(board, Continents[SOUTH_AMERICA].countries[i],
                       65535, 10000, 10000);
  for (i=0; i<Continents[AFRICA].num_countries; i++)
    change_country_color(board, Continents[AFRICA].countries[i],
                      45767, 37767, 0);
  for (i=0; i<Continents[EUROPE].num_countries; i++)
    change_country_color(board, Continents[EUROPE].countries[i],
                       7000, 25535, 65535);
  for (i=0; i<Continents[ASIA].num_countries; i++)
    change_country_color(board, Continents[ASIA].countries[i],
                       0, 65535, 0);
  for (i=0; i<Continents[AUSTRALIA].num_countries; i++)
    change_country_color(board, Continents[AUSTRALIA].countries[i],
                       65535, 10000, 65535);

//  gtk_widget_draw(board, &update_rect);
}

static int sensitive = 1;
void set_sensitivity(GtkWidget *widget, gpointer data) {
  if (sensitive) {
    gtk_widget_set_sensitive((GtkWidget *)data, FALSE);
    sensitive = 0;
  } else {
    gtk_widget_set_sensitive((GtkWidget *)data, TRUE);
    sensitive = 1;
  }
}

void whiten(GtkWidget *widget, gpointer data) {
  GtkWidget *board = data;
  GdkRectangle update_rect;
  int i;

  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = 640;
  update_rect.height= 381;

  redraw_board_pixmap(widget, data);

  for (i=0; i<NUM_COUNTRIES; i++)
    change_country_color(board, i, 65535, 65535, 65535);

//  gtk_widget_draw(board, &update_rect);
}

void draw_army_label(GtkWidget *widget, int country, GdkColor *color, GdkRectangle *rect) {
  int n;
  char text[4];
  int x, y, w, h;
  int r, g, b, v;
  GdkRectangle update_rect;
  GdkGC *gc;

  // Get font
  if (ArmyFont == NULL)
    ArmyFont = gdk_font_load("-*-clean-bold-r-*-*-12-*-*-*-*-*-*-*");

  // Create two-char string from integer
  n = Countries[country].num_armies;
  if (n > 99) n = 99;
  sprintf(text, "%02d", n);

  // Decide whether to use white or black to write the text
  r = color->red; g = color->green; b = color->blue;
  if (r >= g && r >= b) v = r;
  else if (g >= r && g >= b) v = g;
  else if (b >= r && b >= g) v = b;

  if ( v < (65535 * 0.50))
    gc = widget->style->white_gc;
  else
    gc = widget->style->black_gc;

  // Special cases -- island countries with text in ocean
  if (country == ICELAND || country == GREAT_BRITAIN || country == JAPAN ||
      country == INDONESIA || country == NEW_GUINEA || country == MADAGASCAR) {

    // Black GC for black text
    gc = widget->style->black_gc;

    // Replace rectangle from backing pixmap
    gdk_draw_pixmap(BoardPixmap, gc, BoardBackup,
                    Countries[country].xtext,
                    Countries[country].ytext-12,
                    Countries[country].xtext,
                    Countries[country].ytext-12,
                    14, 12);
  }

  // Draw the string
  gdk_draw_string(BoardPixmap, ArmyFont, gc,
                  Countries[country].xtext,
                  Countries[country].ytext,
                  text);

  // Figure out if we need to send a gtk_widget_draw for the text rectangle.
  // Works like this:  Text is in a certain size rectangle.  If text coords
  // are outside the update rectangle for the pixmap, we need to also update
  // the rectangle for the text.

  x = Countries[country].xtext;
  y = Countries[country].ytext;
  w = 20;
  h = 12;

  update_rect.x = x;
  update_rect.y = y-h;
  update_rect.width = w;
  update_rect.height = h;

  if ( (update_rect.x < rect->x) ||
       (update_rect.y < rect->y) ||
       (update_rect.x+update_rect.width > (rect->x+rect->width)) ||
       (update_rect.y+update_rect.height > (rect->y+rect->height)) ) {
    gtk_widget_draw(widget, &update_rect);
  }
}

void draw_country_abs_color(GtkWidget *widget, int country, int r, int g, int b) {
  GdkRectangle update_rect;
  GdkBitmap *bitmap;
  GdkGC *gc;
  GdkColor color;
  int w, h;

  // Make sure we're not trying to draw an invalid country
  if (country == -1) return;
  if (country >= NUM_COUNTRIES) return;
 
  // Create gc and customize its color
  gc = gdk_gc_new(widget->window);
  gdk_gc_copy(gc, widget->style->white_gc);

  // Set color according to function's arguments
  color.pixel = 0;
  color.red = r;
  color.green = g;
  color.blue = b;

  gdk_color_alloc(gdk_window_get_colormap(widget->window), &color);

  // Set foreground color of the gc
  gdk_gc_set_foreground(gc, &color);

  // Get dimensions from country_xbm data
  w = Countries[country].width;
  h = Countries[country].height;

  // Create bitmap mask from country_xbm data
  // First check to see if bitmap already exists
  if (Countries[country].bitmap != NULL) {
    bitmap = Countries[country].bitmap;
  } else {
    bitmap = gdk_bitmap_create_from_data (widget->window,
                          country_xbm_data[country], 
                          w, h);
    Countries[country].bitmap = bitmap;
  }

  // Set rectangle position
  update_rect.x = Countries[country].xpos;
  update_rect.y = Countries[country].ypos;
  update_rect.width = w;
  update_rect.height = h;

  // Set clipping mask according to bitmap
  gdk_gc_set_clip_origin(gc, update_rect.x, update_rect.y);
  gdk_gc_set_clip_mask(gc, bitmap);

  // Draw the rectangle
  gdk_draw_rectangle(BoardPixmap,
                     gc,
                     TRUE,
                     update_rect.x, update_rect.y,
                     update_rect.width, update_rect.height);

  // Draw the text (number of armies) on the country
  draw_army_label(widget, country, &color, &update_rect);

  // Tell the parent widget to draw itself
  gtk_widget_draw(widget, &update_rect);

  // Delete the gc we just created
  g_free(gc);
}

void draw_country_abs_white(GtkWidget *widget, int country) {
  // Make sure we're not drawing an invalid country
  if (country == -1) return;

  draw_country_abs_color(widget, country, 65535, 65535, 65535);
}

void draw_country_color(GtkWidget *widget, int country, int r, int g, int b) {

  // Make sure we're not drawing an invalid country
  if (country == -1) return;

  draw_country_abs_color(widget, country,
                         r * Countries[country].lightness,
                         g * Countries[country].lightness,
                         b * Countries[country].lightness);
}

void draw_country_shaded(GtkWidget *widget, int country, int shade) {
  float red;
  float green;
  float blue;
  int r;
  int g;
  int b;

  // Make sure we're not drawing an invalid country
  if (country == -1) return;

  r = Countries[country].color.red;
  g = Countries[country].color.green;
  b = Countries[country].color.blue;

  if (r >= g) {
    if (r >= b) { // Red is highest
      green = ((float)g/(float)r);
      blue = ((float)b/(float)r);
      g = green*shade;
      b = blue*shade;
      r = shade;
    } else if (b >= g) { // Blue is highest
      green = ((float)g/(float)b);
      red = ((float)r/(float)b);
      g = green*shade;
      r = red*shade;
      b = shade;
    } 
  } else if (g > b) { // Green is highest
    blue = ((float)b/(float)g);
    red = ((float)r/(float)g);
    b = blue*shade;
    r = red*shade;
    g = shade;
  } else { // Blue is highest
    green = ((float)g/(float)b);
    red = ((float)r/(float)b);
    g = green*shade;
    r = red*shade;
    b = shade;
  }

  draw_country_abs_color(widget, country, r, g, b);
}

void draw_country_light(GtkWidget *widget, int country) {
  int r, g, b;
  r = Countries[country].color.red+12767;
  g = Countries[country].color.green+12767;
  b = Countries[country].color.blue+12767;

  if (r > (65535)) r = 65535;
  if (g > (65535)) g = 65535;
  if (b > (65535)) b = 65535;

  draw_country_abs_color(widget, country, r, g, b);
//  draw_country_shaded(widget, country, 65535);
}

void draw_country_dark(GtkWidget *widget, int country) {
/*
  int shade = (Countries[country].color.red +
               Countries[country].color.green +
               Countries[country].color.blue) / 3;
*/
  draw_country_color(widget, country,
                     Countries[country].color.red/2,
                     Countries[country].color.green/2,
                     Countries[country].color.blue/2);
}

void draw_country(GtkWidget *widget, int country) {
  // Make sure we're not drawing an invalid country
  if (country == -1) return;

  draw_country_color(widget, country,
                     Countries[country].color.red,
                     Countries[country].color.green,
                     Countries[country].color.blue);
}

void change_country_color(GtkWidget *widget, int country, int r, int g, int b) {
  Countries[country].color.red = r;
  Countries[country].color.green = g;
  Countries[country].color.blue = b;

  draw_country(widget, country);
}

/* Refill the screen from the backing pixmap */
static gint
expose_event(GtkWidget *widget, GdkEventExpose *event)
{
  gdk_draw_pixmap(widget->window,
                  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                  BoardPixmap,
                  event->area.x, event->area.y,
                  event->area.x, event->area.y,
                  event->area.width, event->area.height);
  return FALSE;
}

void highlight_neighbors(GtkWidget *widget, int country) {
  int i;

  if (country == -1) return;
  for (i=0; i<Countries[country].num_neighbors; i++) {
    draw_country_dark(widget, Countries[country].neighbors[i]);
  }
}

void dehighlight_neighbors(GtkWidget *widget, int country) {
  int i;

  if (country == -1) return;
  for (i=0; i<Countries[country].num_neighbors; i++) {
    draw_country(widget, Countries[country].neighbors[i]);
  }
}

void country_prelight(GtkWidget *widget, int country) {
  if (country == -1) {
    set_country_status("No Country");
    return;
  }

  set_country_status(Countries[country].name);
  draw_country_light(widget, country);
}

void country_activate(GtkWidget *widget, int country) {
  if (country == -1) return;
  draw_country_dark(widget, country);
}

void country_deactivate(GtkWidget *widget, int country) {
  if (country == -1) return;
  draw_country(widget, country);
}

void country_button_down(GtkWidget *widget, int country) {
  CountryDown = country;
  country_activate(widget, country);
}

void country_button_up(GtkWidget *widget, int country) {
  if (CountryDown == country) {
    country_click(widget, country);
  }
  // Now highlight whatever country the mouse is over!
  country_prelight(widget, country);
  CountryDown = -2; // -2 means mouse not down
}

void country_mouse_enter(GtkWidget *widget, int country) {
  if (CountryDown == country) {
    country_activate(widget, country);
  } else if (CountryDown == -2) {
    country_prelight(widget, country);
  }
}

void country_mouse_leave(GtkWidget *widget, int country) {
  if (CountryDown == country) {
    country_deactivate(widget, country);
  } else if (CountryDown == -2) {
    country_deactivate(widget, country);
  }
}

void button_on_board(GtkWidget *widget, GdkEventButton *event, gpointer data) {
    gint x, y;
    gint country;
   
    gdk_window_get_pointer(widget->window, &x, &y, NULL);
    country = which_country(x, y);
    country_button_down(widget, country);
}

void release_on_board(GtkWidget *widget, GdkEventButton *event, gpointer data) {
    gint x, y;
    gint country;
   
    gdk_window_get_pointer(widget->window, &x, &y, NULL);
    country = which_country(x, y);
    country_button_up(widget, country);
}

void mouse_motion(GtkWidget *widget, GdkEvent *event, gpointer data) {
  gint x, y;
  gint country;

  gdk_window_get_pointer(widget->window, &x, &y, NULL);

  country = which_country(x, y);

  set_coords(x, y);

  if (country != CurrentCountry) { // Moved to a new country
    country_mouse_leave(widget, CurrentCountry); // Left old country
    country_mouse_enter(widget, country);        // Enter new country
    CurrentCountry = country;
  }
}

void button_callback(GtkWidget *parent, gpointer data) {
  static int visible = 0;

  if (visible) {
    gtk_widget_hide(data);
    visible = 0;
  } else {
    gtk_widget_show(data);
    visible = 1;
  }
}

GtkWidget *board_new(GtkWidget *parent) {
  GtkWidget *board;

  GtkWidget *vbox;
  GtkWidget *eventbox;

  GtkWidget *button;
  GtkWidget *button2;
  GtkWidget *button3;

  int width, height;

  create_board_pixmap(parent); // Create the pixmap...
  gdk_window_get_size(BoardPixmap, &width, &height);

  TheBoard = gtk_drawing_area_new ();
  gtk_widget_set_usize(TheBoard, width, height);
  gtk_widget_show(TheBoard);

  TheFixed = gtk_fixed_new();
  gtk_widget_show(TheFixed);

  gtk_widget_set_events(TheBoard, GDK_EXPOSURE_MASK
                        | GDK_LEAVE_NOTIFY_MASK
                        | GDK_BUTTON_PRESS_MASK
                        | GDK_BUTTON_RELEASE_MASK
                        | GDK_POINTER_MOTION_MASK
                        | GDK_POINTER_MOTION_HINT_MASK);

  button = gtk_button_new_with_label("Colorize!");
  gtk_widget_set_usize(button, 64, 0);
  gtk_widget_show(button);

  button2 = gtk_button_new_with_label("Whiten!");
  gtk_widget_set_usize(button2, 64, 0);
  gtk_widget_show(button2);

  button3 = gtk_toggle_button_new_with_label("Insensitive");
  gtk_widget_set_usize(button3, 64, 0);
  gtk_widget_show(button3);

  gtk_fixed_put(GTK_FIXED(TheFixed), TheBoard, 0, 0);
  gtk_fixed_put(GTK_FIXED(TheFixed), button, 45, 329);
  gtk_fixed_put(GTK_FIXED(TheFixed), button2, 45, 355);
  gtk_fixed_put(GTK_FIXED(TheFixed), button3, 45, 303);

  gtk_signal_connect(GTK_OBJECT(TheBoard), "expose_event",
                     (GtkSignalFunc)expose_event, NULL);
  gtk_signal_connect(GTK_OBJECT(TheBoard), "button_press_event",
                     GTK_SIGNAL_FUNC(button_on_board), NULL);
  gtk_signal_connect(GTK_OBJECT(TheBoard), "button_release_event",
                     GTK_SIGNAL_FUNC(release_on_board), NULL);
  gtk_signal_connect(GTK_OBJECT(TheBoard), "motion_notify_event",
                     GTK_SIGNAL_FUNC(mouse_motion), NULL);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(colorize), TheBoard);
  gtk_signal_connect(GTK_OBJECT(button2), "clicked",
                     GTK_SIGNAL_FUNC(whiten), TheBoard);
  gtk_signal_connect(GTK_OBJECT(button3), "toggled",
                     GTK_SIGNAL_FUNC(set_sensitivity), TheBoard);
  return (TheFixed);
}
