/***********************************************************************/
/* SHOW.C - Functions involving displaying the data.                   */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1997 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3802 0800
 * Bellara                       http://www.gu.edu.au/gext/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: show.c 2.1 1995/06/24 16:31:13 MH Rel MH $
*/

#include <the.h>
#include <proto.h>

#if defined(USE_EXTCURSES)
# include <cur04.h>
#endif

#include <time.h>

/*------------------------ function definitions -----------------------*/
#ifdef HAVE_PROTO
static void build_lines(CHARTYPE,short,LINE *,short,short);
static void show_lines(CHARTYPE);
static void show_a_line(CHARTYPE,short);
static void set_prefix_contents(CHARTYPE,LINE *,short,LINETYPE);
static void show_hex_line(CHARTYPE,short);
#else
static void build_lines();
static void show_lines();
static void show_a_line();
static void set_prefix_contents();
static void show_hex_line();
#endif

static LINE *hexshow_curr=NULL;

/***********************************************************************/
#ifdef HAVE_PROTO
void show_heading(CHARTYPE scrno)
#else
void show_heading(scrno)
CHARTYPE scrno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
#if !defined(MULTIPLE_PSEUDO_FILES)
 extern CHARTYPE dir_path[MAX_FILE_NAME+1];
 extern CHARTYPE dir_files[MAX_FILE_NAME+1];
 extern CHARTYPE rexx_macro_name[MAX_FILE_NAME+1];
#endif
 extern bool curses_started;
 extern bool horizontal;
 extern CHARTYPE display_screens;
 extern short compatible_look;
 extern CHARTYPE *the_version;
/*--------------------------- local data ------------------------------*/
 short fpath_len=0,max_name=0;
 LENGTHTYPE x=0;
 LINETYPE line_number=0L;
 CHARTYPE buffer[60];
 CHARTYPE display_path[MAX_FILE_NAME+1];
#ifdef __PDCURSES__
 CHARTYPE title[MAX_FILE_NAME+1];
 static CHARTYPE old_title[MAX_FILE_NAME+1];
#endif
 CHARTYPE *fpath = display_path;
 short num_to_delete=0,num_to_start=0,col=0;
 short size_col=0,line_col=0;
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_heading");
#endif
/*---------------------------------------------------------------------*/
/* If IDLINE is not to be displayed, exit now.                         */
/*---------------------------------------------------------------------*/
 if (!SCREEN_VIEW(scrno)->id_line)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Reset idline to blank.                                              */
/*---------------------------------------------------------------------*/
 wattrset(SCREEN_WINDOW_IDLINE(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_IDLINE));
 wmove(SCREEN_WINDOW_IDLINE(scrno),0,0);
 my_wclrtoeol(SCREEN_WINDOW_IDLINE(scrno));
/*---------------------------------------------------------------------*/
/* Get line,col values only if POSITION is ON...                       */
/*---------------------------------------------------------------------*/
 if (SCREEN_VIEW(scrno)->position_status)
    (void)get_current_position(scrno,&line_number,&x);
/*---------------------------------------------------------------------*/
/* Set up buffer for line,col,size and alt values for vertical screens.*/
/*---------------------------------------------------------------------*/
 if (display_screens != 1 && !horizontal)
   {
    if (SCREEN_VIEW(scrno)->position_status)
      {
       if (compatible_look == COMPAT_XEDIT)
         {
          sprintf((DEFCHAR *)buffer,"S=%-.1ld L=%-.1ld C=%-.1d A=%d,%d",
                      SCREEN_FILE(scrno)->number_lines,
                      line_number,x,
                      SCREEN_FILE(scrno)->autosave_alt,
                      SCREEN_FILE(scrno)->save_alt);
         }
       else
         {
          sprintf((DEFCHAR *)buffer,"L=%-.1ld C=%-.1d S=%-.1ld A=%d,%d",
                      line_number,x,
                      SCREEN_FILE(scrno)->number_lines,
                      SCREEN_FILE(scrno)->autosave_alt,
                      SCREEN_FILE(scrno)->save_alt);
         }
      }
    else
      {
       sprintf((DEFCHAR *)buffer,"S=%-.1ld A=%d,%d",
                      SCREEN_FILE(scrno)->number_lines,
                      SCREEN_FILE(scrno)->autosave_alt,
                      SCREEN_FILE(scrno)->save_alt);
      }
    max_name = (screen[scrno].screen_cols-1) - strlen((DEFCHAR *)buffer);
   }
 else
   {
    if (SCREEN_VIEW(scrno)->position_status)
       max_name = (screen[scrno].screen_cols-48);
    else
       max_name = (screen[scrno].screen_cols-27);
    if (compatible_look == COMPAT_XEDIT)
      {
       size_col = screen[scrno].screen_cols-46;
       line_col = screen[scrno].screen_cols-46+12;
      }
    else
      {
       size_col = screen[scrno].screen_cols-46+21;
       line_col = screen[scrno].screen_cols-46;
      }
   }
/*---------------------------------------------------------------------*/
/* Determine which portion of filename can be displayed.               */
/*---------------------------------------------------------------------*/
#if defined(MULTIPLE_PSEUDO_FILES)
 strcpy((DEFCHAR *)display_path,(DEFCHAR *)SCREEN_FILE(scrno)->fpath);
 strcat((DEFCHAR *)display_path,(DEFCHAR *)SCREEN_FILE(scrno)->fname);
#else
 switch(SCREEN_FILE(scrno)->pseudo_file)
   {
    case PSEUDO_DIR:
         strcpy((DEFCHAR *)display_path,"DIR: ");
         strcat((DEFCHAR *)display_path,(DEFCHAR *)dir_path);
         strcat((DEFCHAR *)display_path,(DEFCHAR *)dir_files);
         break;
    case PSEUDO_REXX:
         strcpy((DEFCHAR *)display_path,"Output from: ");
         strcat((DEFCHAR *)display_path,(DEFCHAR *)rexx_macro_name);
         break;
    case PSEUDO_KEY:
         strcpy((DEFCHAR *)display_path,"Key definitions:");
         break;
    default:
         if (SCREEN_FILE(scrno)->display_actual_filename)
           {
            strcpy((DEFCHAR *)display_path,(DEFCHAR *)SCREEN_FILE(scrno)->fpath);
            strcat((DEFCHAR *)display_path,(DEFCHAR *)SCREEN_FILE(scrno)->fname);
           }
         else
           {
            strcpy((DEFCHAR *)display_path,(DEFCHAR *)SCREEN_FILE(scrno)->actualfname);
           }
         break;
   }
#endif
#ifdef __PDCURSES__
 sprintf((DEFCHAR *)title, "THE %s - %s", the_version, (DEFCHAR *)display_path );
 /* only display the title if different from previous one */
 if (strcmp((DEFCHAR *)title,(DEFCHAR *)old_title) != 0)
   {
    PDC_set_title( (DEFCHAR *)title );
    strcpy((DEFCHAR *)old_title,(DEFCHAR *)title);
   }
#endif

 fpath = strrmdup(strtrans(display_path,ISLASH,ESLASH),ESLASH);
 fpath_len = strlen((DEFCHAR *)fpath);
 if (fpath_len > max_name)
   {
    num_to_delete = fpath_len - max_name + 2;
    num_to_start = (strlen((DEFCHAR *)fpath)/2)-(num_to_delete/2);
    for (i=0;i<num_to_start;i++)
      {
       mvwaddch(SCREEN_WINDOW_IDLINE(scrno),0,i,display_path[i]);
      }
    col = i+2;
    waddstr(SCREEN_WINDOW_IDLINE(scrno),"<>");
    for (i=num_to_start+num_to_delete;i<strlen((DEFCHAR *)fpath);i++,col++)
      {
       mvwaddch(SCREEN_WINDOW_IDLINE(scrno),0,col,display_path[i]);
      }
   }
 else
  {
   wmove(SCREEN_WINDOW_IDLINE(scrno),0,0);
   wprintw(SCREEN_WINDOW_IDLINE(scrno),"%s",fpath);
  }
 if (display_screens != 1 && !horizontal)
   {
    wmove(SCREEN_WINDOW_IDLINE(scrno),0,max_name+1);
    wprintw(SCREEN_WINDOW_IDLINE(scrno),"%-s",buffer);
   }
 else
   {
    if (SCREEN_VIEW(scrno)->position_status)
      {
       wmove(SCREEN_WINDOW_IDLINE(scrno),0,line_col);
       wprintw(SCREEN_WINDOW_IDLINE(scrno),"Line=%-6.1ld Col=%-5.1d",line_number,x);
      }
    wmove(SCREEN_WINDOW_IDLINE(scrno),0,size_col);
    wprintw(SCREEN_WINDOW_IDLINE(scrno),"Size=%-6.1ld",SCREEN_FILE(scrno)->number_lines);
    wmove(SCREEN_WINDOW_IDLINE(scrno),0,screen[scrno].screen_cols-46+32);
    wprintw(SCREEN_WINDOW_IDLINE(scrno),"Alt=          ");
    wmove(SCREEN_WINDOW_IDLINE(scrno),0,screen[scrno].screen_cols-46+36);
    wprintw(SCREEN_WINDOW_IDLINE(scrno),"%d,%d",SCREEN_FILE(scrno)->autosave_alt,
                             SCREEN_FILE(scrno)->save_alt);
   }
#ifdef MSWIN
 Win31Scroll(x,line_number,SCREEN_FILE(scrno)->number_lines,SCREEN_FILE(scrno)->max_line_length);
#endif
 wnoutrefresh(SCREEN_WINDOW_IDLINE(scrno));
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void show_statarea(void)
#else
void show_statarea()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern CHARTYPE *rec;
 extern CHARTYPE *cmd_rec;
 extern CHARTYPE *pre_rec;
 extern CHARTYPE *the_version;
 extern CHARTYPE *the_copyright;
 extern bool CLOCKx;
 extern bool HEXDISPLAYx;
 extern bool INSERTMODEx;
 extern CHARTYPE number_of_files;
 extern WINDOW *statarea;
 extern bool colour_support;
 extern bool initial;
 extern bool rexx_support;
 extern ROWTYPE STATUSLINEx;
/*--------------------------- local data ------------------------------*/
 register int i=0;
 short y=0,x=0;
 short key=0;
 WINDOW *w=NULL;
 time_t timer;
 struct tm *tblock=NULL;
 char rel[10];                                 /* not initialised - OK */
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_statarea");
#endif
/*---------------------------------------------------------------------*/
/* If the status line is off, just exit...                             */
/*---------------------------------------------------------------------*/
 if (STATUSLINEx == 'O')
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* If GUI option set for status line...                                */
/*---------------------------------------------------------------------*/
 if (STATUSLINEx == 'G')
   {
#ifdef MSWIN
   Show_GUI_footing();
# ifdef TRACE
    trace_return();
# endif
    return;
#endif

#ifdef TRACE
    trace_return();
#endif
    return;
   }
 w = CURRENT_WINDOW;
/*---------------------------------------------------------------------*/
/* Display THE version.                                                */
/*---------------------------------------------------------------------*/
 strcpy(rel,"THE ");
 strcat(rel,(DEFCHAR *)the_version);
/* strcat(rel,the_release+10);*/
 mvwaddstr(statarea,0,0,rel);
/*---------------------------------------------------------------------*/
/* Clear from files to clock position...                               */
/*---------------------------------------------------------------------*/
 for (i=12;i<53;i++)
   {
    wmove(statarea,0,i);
    waddch(statarea,' ');
   }
/*---------------------------------------------------------------------*/
/* Display number of files or copyright on startup.                    */
/*---------------------------------------------------------------------*/
 wmove(statarea,0,12);
 if (initial)
    waddstr(statarea,(DEFCHAR *)the_copyright);
 else
    wprintw(statarea,"Files=%d   Width=%d",number_of_files,max_line_length);
/*---------------------------------------------------------------------*/
/* Display any pending prefix command warning                          */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->first_ppc != NULL
 &&  CURRENT_FILE->first_ppc->ppc_cmd_idx != (-1)
 &&  CURRENT_FILE->first_ppc->ppc_cmd_idx != (-2))
   {
    wmove(statarea,0,20);
    wprintw(statarea,"'%s' pending...",get_prefix_command(CURRENT_FILE->first_ppc->ppc_cmd_idx));
   }
/*---------------------------------------------------------------------*/
/* Display CLOCK.                                                      */
/*---------------------------------------------------------------------*/
 if (CLOCKx)
   {
    timer = time(NULL);
    tblock = localtime(&timer);
    wmove(statarea,0,53);

    wprintw(statarea,"%2d:%02.2d%s",
           (tblock->tm_hour > 12) ? (tblock->tm_hour-12) : (tblock->tm_hour),
            tblock->tm_min,
           (tblock->tm_hour >= 12) ? ("pm") : ("am"));
   }
/*---------------------------------------------------------------------*/
/* Display HEXDISPLAY.                                                 */
/*---------------------------------------------------------------------*/
 if (HEXDISPLAYx)
   {
    getyx(CURRENT_WINDOW,y,x);
    switch(CURRENT_VIEW->current_window)
      {
       case WINDOW_FILEAREA:
            key = (short)(*(rec+CURRENT_VIEW->verify_col-1+x) & A_CHARTEXT);
            break;
       case WINDOW_COMMAND:
            key = (short)(*(cmd_rec+x) & A_CHARTEXT);
            break;
       case WINDOW_PREFIX:
            key = (short)(*(pre_rec+x) & A_CHARTEXT);
            break;
      }
    wmove(statarea,0,61);
    wprintw(statarea,"' '=%2.2X/%3.3d  ",key,key);
    wmove(statarea,0,62);
    put_char(statarea,key,ADDCHAR);
   }
/*---------------------------------------------------------------------*/
/* Display colour setting.                                             */
/*---------------------------------------------------------------------*/
 wmove(statarea,0,73);
#ifdef A_COLOR
 if (colour_support)
    waddch(statarea,'C');
 else
    waddch(statarea,'c');
#else
 waddch(statarea,'M');
#endif
/*---------------------------------------------------------------------*/
/* Display REXX support character.                                     */
/*---------------------------------------------------------------------*/
 wmove(statarea,0,74);
 if (rexx_support)
    waddch(statarea,'R');
 else
    waddch(statarea,' ');
/*---------------------------------------------------------------------*/
/* Display INSERTMODE toggle.                                          */
/*---------------------------------------------------------------------*/
 wmove(statarea,0,76);
 if (INSERTMODEx)
    waddstr(statarea,"Ins");
 else
    waddstr(statarea,"   ");
/*---------------------------------------------------------------------*/
/* Refresh the STATUS LINE.                                            */
/*---------------------------------------------------------------------*/
 wnoutrefresh(statarea);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void clear_statarea(void)
#else
void clear_statarea()
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern WINDOW *statarea;
 extern ROWTYPE STATUSLINEx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    clear_statarea");
#endif
/*---------------------------------------------------------------------*/
/* If the status line is not displayed, don't do anything.             */
/*---------------------------------------------------------------------*/
 switch(STATUSLINEx)
   {
    case 'T':
    case 'B':
         wmove(statarea,0,0);
         my_wclrtoeol(statarea);
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void redraw_window(WINDOW *win)
#else
void redraw_window(win)
WINDOW *win;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0;
 chtype ch=0;
 short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    redraw_window");
#endif
 getyx(win,y,x);
 for (i=0;i<getmaxx(win);i++)
     for (j=0;j<getmaxy(win);j++)
        {
         wmove(win,j,i);
         ch = (chtype)(winch(win) & A_CHARTEXT);
         put_char(win,ch,ADDCHAR);
        }
 wmove(win,y,x);
#ifdef TRACE
 trace_return();
#endif
 return;
}
#if NOT_USED
/***********************************************************************/
#ifdef HAVE_PROTO
void repaint_screen(void)
#else
void repaint_screen()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    repaint_screen");
#endif

 getyx(CURRENT_WINDOW,y,x);
 y = get_row_for_focus_line(current_screen,CURRENT_VIEW->focus_line,
                            CURRENT_VIEW->current_row);
 if (x > CURRENT_SCREEN.cols[WINDOW_FILEAREA])
    x = 0;
 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 build_screen(current_screen);
 display_screen(current_screen);
 cleanup_command_line();
/* show_heading();*/
 wmove(CURRENT_WINDOW,y,x);

#ifdef TRACE
 trace_return();
#endif
 return;
}
#endif

/***********************************************************************/
#ifdef HAVE_PROTO
void build_screen(CHARTYPE scrno)
#else
void build_screen(scrno)
CHARTYPE scrno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
extern bool SBx;
/*--------------------------- local data ------------------------------*/
 LINE *curr=NULL;
 LINE *save_curr=NULL;
 short crow = SCREEN_VIEW(scrno)->current_row;
 LINETYPE cline = SCREEN_VIEW(scrno)->current_line;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    build_screen");
#endif
 hexshow_curr = save_curr = curr = lll_find(SCREEN_FILE(scrno)->first_line,SCREEN_FILE(scrno)->last_line,
                                            cline,SCREEN_FILE(scrno)->number_lines);
/*---------------------------------------------------------------------*/
/* Build   the file contents from the current line to the bottom of the*/
/* window.                                                             */
/*---------------------------------------------------------------------*/
 build_lines(scrno,DIRECTION_FORWARD,curr,screen[scrno].rows[WINDOW_FILEAREA]-crow,crow);
/*---------------------------------------------------------------------*/
/* Build   the file contents from the current line to the top of the   */
/* window.                                                             */
/*---------------------------------------------------------------------*/
 curr = save_curr->prev;
 build_lines(scrno,DIRECTION_BACKWARD,curr,crow,crow-1);
#if defined(HAVE_SB_INIT)
 if (SBx 
 && scrno == current_screen)
   {
    sb_set_vert(2+CURRENT_FILE->number_lines +CURRENT_SCREEN.rows[WINDOW_FILEAREA],
                CURRENT_SCREEN.rows[WINDOW_FILEAREA],
                CURRENT_VIEW->current_line);
    sb_set_horz(max_line_length+CURRENT_SCREEN.cols[WINDOW_FILEAREA],
                CURRENT_SCREEN.cols[WINDOW_FILEAREA],
                CURRENT_VIEW->verify_col);
    sb_refresh();
   }
#endif
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void display_screen(CHARTYPE scrno)
#else
void display_screen(scrno)
CHARTYPE scrno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool in_macro;
 extern bool in_repeat;
 extern bool batch_only;
 extern bool curses_started;
 extern bool first_screen_display;
/*--------------------------- local data ------------------------------*/
 unsigned short x=0,y=0;
 unsigned short savex=0,savey=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    display_screen");
#endif
/*---------------------------------------------------------------------*/
/* We don't display the screen if we are in a macro, running in batch, */
/* runninf REPEAT command, or curses hasn't started yet...             */
/*---------------------------------------------------------------------*/
 if (batch_only
 ||  in_macro
 ||  in_repeat
 || !curses_started)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 first_screen_display = TRUE;
/*---------------------------------------------------------------------*/
/* Turn off the cursor.                                                */
/*---------------------------------------------------------------------*/
 draw_cursor(FALSE);
/*---------------------------------------------------------------------*/
/* Display the IDLINE window...                                        */
/*---------------------------------------------------------------------*/
 show_heading(scrno);
/*---------------------------------------------------------------------*/
/* Display the ARROW and CMDLINE if on...                              */
/*---------------------------------------------------------------------*/
 if (SCREEN_WINDOW_ARROW(scrno) != NULL)
   {
    wattrset(SCREEN_WINDOW_ARROW(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_ARROW));
    redraw_window(SCREEN_WINDOW_ARROW(scrno));
    touchwin(SCREEN_WINDOW_ARROW(scrno));
    wnoutrefresh(SCREEN_WINDOW_ARROW(scrno));
   }
 if (SCREEN_WINDOW_COMMAND(scrno) != NULL)
   {
    wattrset(SCREEN_WINDOW_COMMAND(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_CMDLINE));
    redraw_window(SCREEN_WINDOW_COMMAND(scrno));
    touchwin(SCREEN_WINDOW_COMMAND(scrno));
    wnoutrefresh(SCREEN_WINDOW_COMMAND(scrno));
   }
/*---------------------------------------------------------------------*/
/* Save the position of previous window if on command line.            */
/*---------------------------------------------------------------------*/
 if (SCREEN_VIEW(scrno)->current_window == WINDOW_COMMAND)
    getyx(SCREEN_PREV_WINDOW(scrno),savey,savex);
 getyx(SCREEN_WINDOW(scrno),y,x);
/*---------------------------------------------------------------------*/
/* Display the built lines...                                          */
/*---------------------------------------------------------------------*/
 show_lines(scrno);
/*---------------------------------------------------------------------*/
/* Refresh the windows.                                                */
/*---------------------------------------------------------------------*/
 if (SCREEN_WINDOW_PREFIX(scrno) != NULL)
     wnoutrefresh(SCREEN_WINDOW_PREFIX(scrno));
 if (SCREEN_WINDOW_GAP(scrno) != NULL)
     wnoutrefresh(SCREEN_WINDOW_GAP(scrno));
 wrefresh(SCREEN_WINDOW_FILEAREA(scrno));
/*---------------------------------------------------------------------*/
/* Lastly, turn the cursor back on again.                              */
/*---------------------------------------------------------------------*/
 draw_cursor(TRUE);
/*---------------------------------------------------------------------*/
/* Restore the position of previous window if on command line.         */
/*---------------------------------------------------------------------*/
 if (SCREEN_VIEW(scrno)->current_window == WINDOW_COMMAND)
    wmove(SCREEN_PREV_WINDOW(scrno),savey,savex);
 wmove(SCREEN_WINDOW(scrno),y,x);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
static void build_lines(CHARTYPE scrno,short direction,LINE *curr,
                         short rows,short start_row)
#else
static void build_lines(scrno,direction,curr,rows,start_row)
CHARTYPE scrno;
short direction;
LINE *curr;
short rows,start_row;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
 extern VIEW_DETAILS *vd_mark;
 extern short compatible_feel;
 extern CHARTYPE display_screens;
/*--------------------------- local data ------------------------------*/
 LINETYPE cline = SCREEN_VIEW(scrno)->current_line;
 RESERVED *curr_rsrvd=NULL;
 short num_shadow_lines=0;
 bool marked=FALSE,current=FALSE;
 short tab_actual_row=0;
 short scale_actual_row=0;
 short hexshow_actual_start_row=0;
 bool display_rec=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    build_lines");
#endif
/*---------------------------------------------------------------------*/
/* Determine if the contents of "rec" should be used to display the    */
/* focus line.                                                         */
/*---------------------------------------------------------------------*/
 if (display_screens > 1)
   {
    if (scrno == current_screen
    ||  SCREEN_FILE(current_screen) == SCREEN_FILE(other_screen))
       display_rec = TRUE;
    else
       display_rec = FALSE;
   }
 else
    display_rec = TRUE;
/*---------------------------------------------------------------------*/
/* Determine the row that is the focus line.                           */
/*---------------------------------------------------------------------*/
 if (direction == DIRECTION_BACKWARD)
    cline--;
 num_shadow_lines = 0;
/*---------------------------------------------------------------------*/
/* These only need to be calculated once.                              */
/*---------------------------------------------------------------------*/
 tab_actual_row=calculate_actual_row(SCREEN_VIEW(scrno)->tab_base,SCREEN_VIEW(scrno)->tab_off,screen[scrno].rows[WINDOW_FILEAREA],TRUE);
 scale_actual_row=calculate_actual_row(SCREEN_VIEW(scrno)->scale_base,SCREEN_VIEW(scrno)->scale_off,screen[scrno].rows[WINDOW_FILEAREA],TRUE);
/*---------------------------------------------------------------------*/
/* Now, for each row to be displayed...                                */
/*---------------------------------------------------------------------*/
 while(rows)
   {
    screen[scrno].sl[start_row].full_length = FALSE;
    screen[scrno].sl[start_row].number_lines_excluded = 0;
    screen[scrno].sl[start_row].other_start_col = 0;
    screen[scrno].sl[start_row].other_end_col = 0;
    screen[scrno].sl[start_row].line_type = LINE_LINE;
    screen[scrno].sl[start_row].main_enterable = TRUE;
    screen[scrno].sl[start_row].prefix_enterable = TRUE;
    screen[scrno].sl[start_row].highlight = FALSE;
/*---------------------------------------------------------------------*/
/* If HEXSHOW is ON...                                                 */
/*---------------------------------------------------------------------*/
    if (SCREEN_VIEW(scrno)->hexshow_on)
      {
       hexshow_actual_start_row=calculate_actual_row(SCREEN_VIEW(scrno)->hexshow_base,SCREEN_VIEW(scrno)->hexshow_off,screen[scrno].rows[WINDOW_FILEAREA],TRUE);
       if (hexshow_actual_start_row == start_row
       ||  hexshow_actual_start_row+1 == start_row)
         {
          screen[scrno].sl[start_row].line_type = LINE_HEXSHOW;
          screen[scrno].sl[start_row].full_length = TRUE;
          screen[scrno].sl[start_row].line_number = (-1L);
          screen[scrno].sl[start_row].main_enterable = FALSE;
          screen[scrno].sl[start_row].prefix_enterable = FALSE;
          screen[scrno].sl[start_row].normal_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_SHADOW);
          if (scrno == current_screen)
            {
             if (SCREEN_VIEW(scrno)->current_line == SCREEN_VIEW(scrno)->focus_line)
               {
                screen[scrno].sl[start_row].contents = rec;
                screen[scrno].sl[start_row].length = rec_len;
               }
             else
               {
                screen[scrno].sl[start_row].contents = hexshow_curr->line;
                screen[scrno].sl[start_row].length = hexshow_curr->length;
               }
            }
          else
            {
             if (SCREEN_VIEW(scrno)->current_line == SCREEN_VIEW(current_screen)->focus_line
             &&  display_rec)
               {
                screen[scrno].sl[start_row].contents = rec;
                screen[scrno].sl[start_row].length = rec_len;
               }
             else
               {
                screen[scrno].sl[start_row].contents = hexshow_curr->line;
                screen[scrno].sl[start_row].length = hexshow_curr->length;
               }
            }
          if (hexshow_actual_start_row == start_row)
             screen[scrno].sl[start_row].other_start_col = 0;
          else
             screen[scrno].sl[start_row].other_start_col = 1;
          if (SCREEN_VIEW(scrno)->prefix) /* display prefix if on */
            {
             memset(screen[scrno].sl[start_row].prefix,' ',SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap);
             screen[scrno].sl[start_row].prefix[SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap] = '\0';
             screen[scrno].sl[start_row].prefix_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_PREFIX);
             if (SCREEN_VIEW(scrno)->prefix_gap)
               {
                screen[scrno].sl[start_row].gap_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_GAP);
                strcpy((DEFCHAR*)screen[scrno].sl[start_row].gap,"");
               }
            }
          start_row += direction;
          rows--;
          continue;
         }
      }
/*---------------------------------------------------------------------*/
/* If the current line is a reserved line...                           */
/*---------------------------------------------------------------------*/
    if (SCREEN_FILE(scrno)->first_reserved)    /* at least one reserved line */
      {
       if ((curr_rsrvd = find_reserved_line(scrno,TRUE,start_row,0,0)) != NULL)
         {
          screen[scrno].sl[start_row].line_type = LINE_RESERVED;
          screen[scrno].sl[start_row].full_length = TRUE;
          screen[scrno].sl[start_row].line_number = (-1L);
          screen[scrno].sl[start_row].current = (LINE *)NULL;
          screen[scrno].sl[start_row].main_enterable = FALSE;
          screen[scrno].sl[start_row].prefix_enterable = FALSE;
          if (SCREEN_VIEW(scrno)->prefix)               /* display prefix if on */
            {
             screen[scrno].sl[start_row].prefix_colour = set_colour(curr_rsrvd->attr);
             screen[scrno].sl[start_row].gap_colour = set_colour(curr_rsrvd->attr);
             if ((SCREEN_VIEW(scrno)->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT)
               {
                memset(screen[scrno].sl[start_row].prefix,' ',SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap);
                memcpy(screen[scrno].sl[start_row].prefix,curr_rsrvd->line,min(curr_rsrvd->length,SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap));
                screen[scrno].sl[start_row].prefix[SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap] = '\0';
                strcpy((DEFCHAR*)screen[scrno].sl[start_row].gap,"");
                if (curr_rsrvd->length <= SCREEN_VIEW(scrno)->prefix_width)
                  {
                   screen[scrno].sl[start_row].contents = NULL;
                   screen[scrno].sl[start_row].length = 0;
                  }
                else
                  {
                   screen[scrno].sl[start_row].contents = curr_rsrvd->line+SCREEN_VIEW(scrno)->prefix_width;
                   screen[scrno].sl[start_row].length = curr_rsrvd->length-SCREEN_VIEW(scrno)->prefix_width;
                  }
                if (curr_rsrvd->length > SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap)
                  {
                   memcpy(screen[scrno].sl[start_row].gap,
                          curr_rsrvd->line+(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap),
                          min(SCREEN_VIEW(scrno)->prefix_gap,
                              curr_rsrvd->length-(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap)));
                   screen[scrno].sl[start_row].gap[min(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap,curr_rsrvd->length-(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap))] = '\0';
                  }
               }
             else
               {
                screen[scrno].sl[start_row].contents = curr_rsrvd->line;
                screen[scrno].sl[start_row].length = min(curr_rsrvd->length,screen[scrno].cols[WINDOW_FILEAREA]);
                memset(screen[scrno].sl[start_row].prefix,' ',SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap);
                screen[scrno].sl[start_row].prefix[SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap] = '\0';
                strcpy((DEFCHAR*)screen[scrno].sl[start_row].gap,"");
                if (curr_rsrvd->length > screen[scrno].cols[WINDOW_FILEAREA])
                  {
                   /*
                    * Reserved line contents may be in prefix area or gap or both
                    * Add to gap if there is one
                    */
                   if (SCREEN_VIEW(scrno)->prefix_gap)
                     {
                      memcpy(screen[scrno].sl[start_row].gap,
                             curr_rsrvd->line+screen[scrno].cols[WINDOW_FILEAREA],
                             min(curr_rsrvd->length-screen[scrno].cols[WINDOW_FILEAREA],
                                 SCREEN_VIEW(scrno)->prefix_gap));
                      screen[scrno].sl[start_row].gap[min(curr_rsrvd->length-screen[scrno].cols[WINDOW_FILEAREA],SCREEN_VIEW(scrno)->prefix_gap)] = '\0';
                     }
                   if (curr_rsrvd->length > screen[scrno].cols[WINDOW_FILEAREA]+(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap))
                     {
                     /*
                      * Add to prefix area
                      */
                      memcpy(screen[scrno].sl[start_row].prefix,
                             curr_rsrvd->line+screen[scrno].cols[WINDOW_FILEAREA]+(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap),
                             min(curr_rsrvd->length-screen[scrno].cols[WINDOW_FILEAREA]+(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap),
                                 SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap));
                      screen[scrno].sl[start_row].prefix[min(curr_rsrvd->length-screen[scrno].cols[WINDOW_FILEAREA]+(SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap),SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap)] = '\0';
                     }
                  }
               }
            }
          else
            {
             screen[scrno].sl[start_row].contents = curr_rsrvd->line;
             screen[scrno].sl[start_row].length = curr_rsrvd->length;
            }
          screen[scrno].sl[start_row].normal_colour = set_colour(curr_rsrvd->attr);
          screen[scrno].sl[start_row].other_colour = set_colour(curr_rsrvd->attr);
          start_row += direction;
          rows--;
          continue;
         }
      }
/*---------------------------------------------------------------------*/
/* If the current line is the scale or tab line...                     */
/*---------------------------------------------------------------------*/
    if ((SCREEN_VIEW(scrno)->scale_on && scale_actual_row == start_row)
    || (SCREEN_VIEW(scrno)->tab_on && tab_actual_row == start_row))
      {
       screen[scrno].sl[start_row].contents = NULL;
       screen[scrno].sl[start_row].length = 0;
       screen[scrno].sl[start_row].line_number = (-1L);
       screen[scrno].sl[start_row].current = (LINE *)NULL;
       screen[scrno].sl[start_row].main_enterable = FALSE;
       screen[scrno].sl[start_row].prefix_enterable = FALSE;
       if (SCREEN_VIEW(scrno)->prefix) /* display prefix if on */
         {
          memset(screen[scrno].sl[start_row].prefix,' ',SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap);
          screen[scrno].sl[start_row].prefix[SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap] = '\0';
          screen[scrno].sl[start_row].prefix_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_PREFIX);
          screen[scrno].sl[start_row].gap_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_GAP);
          strcpy((DEFCHAR*)screen[scrno].sl[start_row].gap,"");
         }
       if (SCREEN_VIEW(scrno)->tab_on && tab_actual_row == start_row)
         {
          screen[scrno].sl[start_row].line_type |= LINE_TABLINE;
          screen[scrno].sl[start_row].normal_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_TABLINE);
         }
       if (SCREEN_VIEW(scrno)->scale_on && scale_actual_row == start_row)
         {
          screen[scrno].sl[start_row].line_type |= LINE_SCALE;
          screen[scrno].sl[start_row].normal_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_SCALE);
         }
       start_row += direction;
       rows--;
       continue;
      }
/*---------------------------------------------------------------------*/
/* If the current line is above or below TOF or EOF, set all to blank. */
/*---------------------------------------------------------------------*/
    if (curr == NULL)
      {
       screen[scrno].sl[start_row].contents = NULL;
       screen[scrno].sl[start_row].length = 0;
       screen[scrno].sl[start_row].normal_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_FILEAREA);
       screen[scrno].sl[start_row].line_type = (direction == DIRECTION_BACKWARD) ? LINE_OUT_OF_BOUNDS_ABOVE : LINE_OUT_OF_BOUNDS_BELOW;
       screen[scrno].sl[start_row].line_number = (-1L);
       screen[scrno].sl[start_row].current = (LINE *)NULL;
       screen[scrno].sl[start_row].main_enterable = FALSE;
       screen[scrno].sl[start_row].prefix_enterable = FALSE;
       if (SCREEN_VIEW(scrno)->prefix) /* display prefix if on */
         {
          memset(screen[scrno].sl[start_row].prefix,' ',SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap);
          screen[scrno].sl[start_row].prefix[SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap] = '\0';
          screen[scrno].sl[start_row].prefix_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_PREFIX);
          screen[scrno].sl[start_row].gap_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_GAP);
          strcpy((DEFCHAR*)screen[scrno].sl[start_row].gap,"");
         }
       start_row += direction;
       rows--;
       continue;
      }
/*---------------------------------------------------------------------*/
/* If the current line is excluded, increment a running total.         */
/* Ignore the line if on TOF or BOF.                                   */
/*---------------------------------------------------------------------*/
    if (curr->next != NULL                           /* Bottom of file */
    &&  curr->prev != NULL)                             /* Top of file */
      {
       if (in_scope(SCREEN_VIEW(scrno),curr)
       || cline == SCREEN_VIEW(scrno)->current_line
       || curr->pre != NULL)
          ;
       else
         {
          if (num_shadow_lines == 0
          && direction == DIRECTION_FORWARD)
            {
             set_prefix_contents(scrno,curr,start_row,cline);
             screen[scrno].sl[start_row].line_number = cline;
             screen[scrno].sl[start_row].current = curr;
            }
          num_shadow_lines++;
          cline += (LINETYPE)direction;
          if (direction == DIRECTION_FORWARD)
             curr = curr->next;
          else
             curr = curr->prev;
          continue;
         }
      }
/*---------------------------------------------------------------------*/
/* If we get here, we have to determine if a shadow line is to be      */
/* displayed or not.                                                   */
/*---------------------------------------------------------------------*/
    if (SCREEN_VIEW(scrno)->shadow
    && num_shadow_lines > 0)
      {
       screen[scrno].sl[start_row].length = 0;
       if (direction != DIRECTION_FORWARD)
         {
          set_prefix_contents(scrno,curr->next,start_row,cline+1);
          screen[scrno].sl[start_row].line_number = cline+1;
          screen[scrno].sl[start_row].current = curr;
         }
       screen[scrno].sl[start_row].normal_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_SHADOW);
       screen[scrno].sl[start_row].full_length = TRUE;
       screen[scrno].sl[start_row].number_lines_excluded = num_shadow_lines;
       screen[scrno].sl[start_row].line_type = LINE_SHADOW;
       if (compatible_feel == COMPAT_XEDIT)
          screen[scrno].sl[start_row].main_enterable = FALSE;
       num_shadow_lines = 0;
       start_row += direction;
       rows--;
       continue;
      }
/*---------------------------------------------------------------------*/
/* Determine if line being processed is the current line.              */
/*---------------------------------------------------------------------*/
    if (cline == SCREEN_VIEW(scrno)->current_line)
       current = TRUE;
    else
       current = FALSE;
/*---------------------------------------------------------------------*/
/* Determine if line being processed is in a marked block.             */
/*---------------------------------------------------------------------*/
    if (MARK_VIEW != (VIEW_DETAILS *)NULL
    &&  MARK_VIEW == SCREEN_VIEW(scrno))
      {
       if (cline >= MARK_VIEW->mark_start_line
       &&  cline <= MARK_VIEW->mark_end_line)
          marked = TRUE;
       else
          marked = FALSE;
      }
/*---------------------------------------------------------------------*/
/* The remainder is for lines that are to be displayed.                */
/*---------------------------------------------------------------------*/
    screen[scrno].sl[start_row].line_number = cline;
    screen[scrno].sl[start_row].current = curr;
/*---------------------------------------------------------------------*/
/* If the current row to be displayed is the focus line, display       */
/* the working area, rec and rec_len instead of the entry in the LL.   */
/*---------------------------------------------------------------------*/
#if 1
    if (scrno == current_screen)
      {
       if (cline == SCREEN_VIEW(scrno)->focus_line
       &&  display_rec)
         {
          screen[scrno].sl[start_row].contents = rec;
          screen[scrno].sl[start_row].length = rec_len;
         }
       else
         {
          screen[scrno].sl[start_row].contents = curr->line;
          screen[scrno].sl[start_row].length = curr->length;
         }
      }
    else
      {
       if (cline == SCREEN_VIEW(current_screen)->focus_line
       &&  display_rec)
         {
          screen[scrno].sl[start_row].contents = rec;
          screen[scrno].sl[start_row].length = rec_len;
         }
       else
         {
          screen[scrno].sl[start_row].contents = curr->line;
          screen[scrno].sl[start_row].length = curr->length;
         }
      }
#else
    if (cline == SCREEN_VIEW(scrno)->focus_line
    &&  display_rec)
      {
       screen[scrno].sl[start_row].contents = rec;
       screen[scrno].sl[start_row].length = rec_len;
      }
    else
      {
       screen[scrno].sl[start_row].contents = curr->line;
       screen[scrno].sl[start_row].length = curr->length;
      }
#endif
    set_prefix_contents(scrno,curr,start_row,cline);
/*---------------------------------------------------------------------*/
/* Set up TOF and EOF lines...                                         */
/*---------------------------------------------------------------------*/
    if (curr->next == NULL                           /* Bottom of file */
    ||  curr->prev == NULL)                             /* Top of file */
      {
       screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CTOFEOF) :
                                                   set_colour(SCREEN_FILE(scrno)->attr+ATTR_TOFEOF);
#ifndef REMOVED_FOR_CONSISTANCY
       if (compatible_feel == COMPAT_XEDIT)
          screen[scrno].sl[start_row].main_enterable = FALSE;
#endif
       screen[scrno].sl[start_row].line_type = (curr->next==NULL)?LINE_EOF:LINE_TOF; /* MH12 */
      }
    else
      {
       if (SCREEN_VIEW(scrno)->highlight)
         {
          switch(SCREEN_VIEW(scrno)->highlight)
            {
             case HIGHLIGHT_TAG:
                  if (curr->tag_flag)
                     screen[scrno].sl[start_row].highlight = TRUE;
                  break;
             case HIGHLIGHT_ALT:
                  if (curr->new_flag
                  ||  curr->changed_flag)
                     screen[scrno].sl[start_row].highlight = TRUE;
                  break;
             case HIGHLIGHT_SELECT:
                  if (curr->select >= SCREEN_VIEW(scrno)->highlight_low
                  &&  curr->select <= SCREEN_VIEW(scrno)->highlight_high)
                     screen[scrno].sl[start_row].highlight = TRUE;
                  break;
             default:
                  break;
            }
         }
       if (marked)
         {
          switch(MARK_VIEW->mark_type)
            {
             case M_LINE:
                  screen[scrno].sl[start_row].other_start_col = (-1);
                  screen[scrno].sl[start_row].other_end_col = (-1);
                  screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CBLOCK) :
                                                               set_colour(SCREEN_FILE(scrno)->attr+ATTR_BLOCK);
                  screen[scrno].sl[start_row].full_length = TRUE;
                  break;
             case M_BOX:
             case M_COLUMN:
             case M_WORD:
                  screen[scrno].sl[start_row].other_start_col = MARK_VIEW->mark_start_col - 1;
                  screen[scrno].sl[start_row].other_end_col = MARK_VIEW->mark_end_col - 1;
                  if (screen[scrno].sl[start_row].highlight)
                     screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CHIGHLIGHT) :
                                                      set_colour(SCREEN_FILE(scrno)->attr+ATTR_HIGHLIGHT);
                  else
                     screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CURLINE) :
                                                               set_colour(SCREEN_FILE(scrno)->attr+ATTR_FILEAREA);
                  screen[scrno].sl[start_row].other_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CBLOCK) :
                                                              set_colour(SCREEN_FILE(scrno)->attr+ATTR_BLOCK);
                  break;
             case M_STREAM:
                  if (screen[scrno].sl[start_row].highlight)
                     screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CHIGHLIGHT) :
                                                      set_colour(SCREEN_FILE(scrno)->attr+ATTR_HIGHLIGHT);
                  else
                     screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CURLINE) :
                                                               set_colour(SCREEN_FILE(scrno)->attr+ATTR_FILEAREA);
                  screen[scrno].sl[start_row].other_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CBLOCK) :
                                                              set_colour(SCREEN_FILE(scrno)->attr+ATTR_BLOCK);
                  screen[scrno].sl[start_row].other_end_col = MAX_INT;
                  screen[scrno].sl[start_row].other_start_col = 0;
                  if (cline == MARK_VIEW->mark_start_line)
                     screen[scrno].sl[start_row].other_start_col = MARK_VIEW->mark_start_col - 1;
                  if (cline == MARK_VIEW->mark_end_line)
                     screen[scrno].sl[start_row].other_end_col = MARK_VIEW->mark_end_col - 1;
                  if (cline > MARK_VIEW->mark_start_line
                  &&  cline < MARK_VIEW->mark_end_line)
                    {
                     screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CBLOCK) :
                                                               set_colour(SCREEN_FILE(scrno)->attr+ATTR_BLOCK);
                     screen[scrno].sl[start_row].full_length = TRUE;
                    }
                  break;
            }
         }
       else
         {
          if (screen[scrno].sl[start_row].highlight)
            {
             screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CHIGHLIGHT) :
                                                      set_colour(SCREEN_FILE(scrno)->attr+ATTR_HIGHLIGHT);
             screen[scrno].sl[start_row].other_colour = screen[scrno].sl[start_row].normal_colour;
            }
          else
            {
             screen[scrno].sl[start_row].normal_colour = (current) ? set_colour(SCREEN_FILE(scrno)->attr+ATTR_CURLINE) :
                                                      set_colour(SCREEN_FILE(scrno)->attr+ATTR_FILEAREA);
             screen[scrno].sl[start_row].other_colour = screen[scrno].sl[start_row].normal_colour;
            }
         }
      }
    start_row += direction;
    rows--;

    cline += (LINETYPE)direction;
    if (direction == DIRECTION_FORWARD)
       curr = curr->next;
    else
       curr = curr->prev;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
static void show_lines(CHARTYPE scrno)
#else
static void show_lines(scrno)
CHARTYPE scrno;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0;
 short true_col=0;
 short max_cols = min(screen[scrno].cols[WINDOW_FILEAREA],SCREEN_VIEW(scrno)->verify_end-SCREEN_VIEW(scrno)->verify_start+1);
 short off=0,num_tens=0;
 unsigned short y=0,x=0;
 CHARTYPE tens[5];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_lines");
#endif
 for (i=0;i<screen[scrno].rows[WINDOW_FILEAREA];i++)
   {
/*---------------------------------------------------------------------*/
/* Display the contents of the prefix area (if on).                    */
/*---------------------------------------------------------------------*/
    if (SCREEN_VIEW(scrno)->prefix)
      {
       wattrset(SCREEN_WINDOW_PREFIX(scrno),screen[scrno].sl[i].prefix_colour);
       wmove(SCREEN_WINDOW_PREFIX(scrno),i,0);
       my_wclrtoeol(SCREEN_WINDOW_PREFIX(scrno));
       put_string(SCREEN_WINDOW_PREFIX(scrno),i,0,screen[scrno].sl[i].prefix,strlen((DEFCHAR *)screen[scrno].sl[i].prefix));
       if (SCREEN_VIEW(scrno)->prefix_gap)
         {
          wattrset(SCREEN_WINDOW_GAP(scrno),screen[scrno].sl[i].gap_colour);
          wmove(SCREEN_WINDOW_GAP(scrno),i,0);
          my_wclrtoeol(SCREEN_WINDOW_GAP(scrno));
          put_string(SCREEN_WINDOW_GAP(scrno),i,0,screen[scrno].sl[i].gap,strlen((DEFCHAR *)screen[scrno].sl[i].gap));
         }
      }
    wmove(SCREEN_WINDOW_FILEAREA(scrno),i,0);
    wattrset(SCREEN_WINDOW_FILEAREA(scrno),screen[scrno].sl[i].normal_colour);
/*---------------------------------------------------------------------*/
/* Display any shadow line. No need to test to see if SHADOW is ON as  */
/* number_excluded_lines would not be > 0 if SHADOW OFF.               */
/*---------------------------------------------------------------------*/
    if (screen[scrno].sl[i].number_lines_excluded > 0)
      {
       for (j=0;j<screen[scrno].cols[WINDOW_FILEAREA];j++)
          mvwaddch(SCREEN_WINDOW_FILEAREA(scrno),i,j,'-');
       wmove(SCREEN_WINDOW_FILEAREA(scrno),i,max(0,(screen[scrno].cols[WINDOW_FILEAREA]/2)-14));
       wprintw(SCREEN_WINDOW_FILEAREA(scrno),"%4d line(s) not displayed ",screen[scrno].sl[i].number_lines_excluded);
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display SCALE and/or TABLINE...                                     */
/*---------------------------------------------------------------------*/
    if (screen[scrno].sl[i].line_type & LINE_SCALE
    ||  screen[scrno].sl[i].line_type & LINE_TABLINE)
      {
       my_wclrtoeol(SCREEN_WINDOW_FILEAREA(scrno));
       for (j=0;j<max_cols;j++)
         {
          wmove(SCREEN_WINDOW_FILEAREA(scrno),i,j);
          true_col = j + SCREEN_VIEW(scrno)->verify_col-1;
/*---------------------------------------------------------------------*/
/* Display '|' in current column position if this is the scale line.   */
/*---------------------------------------------------------------------*/
          if (screen[scrno].sl[i].line_type & LINE_SCALE
          &&  SCREEN_VIEW(scrno)->current_column == true_col+1)
            {
             waddch(SCREEN_WINDOW_FILEAREA(scrno),'|');
             continue;
            }
/*---------------------------------------------------------------------*/
/* Display 'T' in each tab column. This overrides all other characters */
/* except column position.                                             */
/*---------------------------------------------------------------------*/
          if (screen[scrno].sl[i].line_type & LINE_TABLINE)
            {
             if (is_tab_col(true_col+1))
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),'T');
                continue;
               }
            }
/*---------------------------------------------------------------------*/
/* Only display the following if it is a scale line...                 */
/*---------------------------------------------------------------------*/
          if (screen[scrno].sl[i].line_type & LINE_SCALE)
            {
             if (SCREEN_VIEW(scrno)->margin_left-1 == true_col)
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),'[');
                continue;
               }
             if (SCREEN_VIEW(scrno)->margin_right-1 == true_col)
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),']');
                continue;
               }
             if (SCREEN_VIEW(scrno)->margin_indent_offset_status
             &&  SCREEN_VIEW(scrno)->margin_left+SCREEN_VIEW(scrno)->margin_indent-1 == true_col)
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),'p');
                continue;
               }
             if (!SCREEN_VIEW(scrno)->margin_indent_offset_status
             &&  SCREEN_VIEW(scrno)->margin_indent-1 == true_col)
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),'p');
                continue;
               }
             if (SCREEN_VIEW(scrno)->zone_start-1 == true_col)
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),'<');
                continue;
               }
             if (SCREEN_VIEW(scrno)->zone_end-1 == true_col)
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),'>');
/*              break;*/
                continue;
               }
             if (true_col % 10 == 9)
               {
                sprintf((DEFCHAR *)tens,"%d",(true_col / 10) + 1);
                num_tens = strlen((DEFCHAR *)tens);
                getyx(SCREEN_WINDOW_FILEAREA(scrno),y,x);
                if ((short)x-(num_tens-1) < 0)
                  {
                   off = -((short)x-(num_tens-1));
                   x = 0;
                  }
                else
                  {
                   off = 0;
                   x -= (num_tens-1);
                  }
                wmove(SCREEN_WINDOW_FILEAREA(scrno),y,x);
                waddstr(SCREEN_WINDOW_FILEAREA(scrno),(DEFCHAR *)tens+off);
                continue;
               }
             if (true_col % 5 == 4)
               {
                waddch(SCREEN_WINDOW_FILEAREA(scrno),'+');
                continue;
               }
             waddch(SCREEN_WINDOW_FILEAREA(scrno),'.');
            }
         }
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display HEXSHOW line...                                             */
/*---------------------------------------------------------------------*/
    if (screen[scrno].sl[i].line_type & LINE_HEXSHOW)
      {
       my_wclrtoeol(SCREEN_WINDOW_FILEAREA(scrno));
       show_hex_line(scrno,i);
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display TOF or EOF line.                                            */
/*---------------------------------------------------------------------*/
    if (screen[scrno].sl[i].line_type == LINE_TOF)
      {
       my_wclrtoeol(SCREEN_WINDOW_FILEAREA(scrno));
       waddstr(SCREEN_WINDOW_FILEAREA(scrno),(DEFCHAR *)TOP_OF_FILE);
       continue;
      }
    if (screen[scrno].sl[i].line_type == LINE_EOF)
      {
       my_wclrtoeol(SCREEN_WINDOW_FILEAREA(scrno));
       waddstr(SCREEN_WINDOW_FILEAREA(scrno),(DEFCHAR *)BOTTOM_OF_FILE);
       continue;
      }
/*---------------------------------------------------------------------*/
/* Display marked LINE block line(s).                                  */
/*---------------------------------------------------------------------*/
    show_a_line(scrno,i);
   }
 if (SCREEN_WINDOW_PREFIX(scrno) != NULL)
    wattrset(SCREEN_WINDOW_PREFIX(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_PENDING));
 if (SCREEN_WINDOW_GAP(scrno) != NULL)
    wattrset(SCREEN_WINDOW_GAP(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_GAP));
 wattrset(SCREEN_WINDOW_FILEAREA(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_FILEAREA));
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
static void show_a_line(CHARTYPE scrno,short row)
#else
static void show_a_line(scrno,row)
CHARTYPE scrno;
short row;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 chtype ch=0;
 LENGTHTYPE vcol=0,vend=0,vlen=0;
 LENGTHTYPE length=0;
 CHARTYPE *line=NULL;
#if defined(USE_EXTCURSES) || defined(HAVE_BSD_CURSES)
 chtype attr=0,old_attr=0;
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_a_line");
#endif
/*---------------------------------------------------------------------*/
/* If the line to be displayed is a reserved line, set the columns to  */
/* be displayed so that the full line is displayed.                    */
/*---------------------------------------------------------------------*/
 if (screen[scrno].sl[row].line_type == LINE_RESERVED)
   {
    vcol = 0;
    vend = screen[scrno].cols[WINDOW_FILEAREA];
    vlen = screen[scrno].cols[WINDOW_FILEAREA];
   }
 else
   {
    vcol = SCREEN_VIEW(scrno)->verify_col - 1;
    vend = SCREEN_VIEW(scrno)->verify_end - 1;
    vlen = SCREEN_VIEW(scrno)->verify_end - SCREEN_VIEW(scrno)->verify_start + 1;
   }
 length = screen[scrno].sl[row].length;
 line = screen[scrno].sl[row].contents;
/*---------------------------------------------------------------------*/
/* If the contents are NULL then clear to eol in normal colour.        */
/*---------------------------------------------------------------------*/
 if (line == NULL)
   {
    my_wclrtoeol(SCREEN_WINDOW_FILEAREA(scrno));
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 wmove(SCREEN_WINDOW_FILEAREA(scrno),row,0);
 for (i=0;i<screen[scrno].cols[WINDOW_FILEAREA];i++)
   {
/*---------------------------------------------------------------------*/
/* If the last character in the line has been displayed, display blank.*/
/*---------------------------------------------------------------------*/
    if (i+vcol < length)
       ch = *(line+i+vcol);
    else
       ch = ' ';
/*---------------------------------------------------------------------*/
/* Determine colour of character to be displayed. BOX blocks are sorted*/
/* out here.                                                           */
/*---------------------------------------------------------------------*/
    if (vcol+i >= screen[scrno].sl[row].other_start_col
    &&  vcol+i <= screen[scrno].sl[row].other_end_col)
#if defined(USE_EXTCURSES) || defined(HAVE_BSD_CURSES)
       attr = screen[scrno].sl[row].other_colour;
#else
       ch = ch | screen[scrno].sl[row].other_colour;
#endif
    else
#if defined(USE_EXTCURSES) || defined(HAVE_BSD_CURSES)
       attr = screen[scrno].sl[row].normal_colour;
#else
       ch = ch | screen[scrno].sl[row].normal_colour;
#endif
/*---------------------------------------------------------------------*/
/* If we have gone past the VERIFY END column, display a blank in the  */
/* normal colour.                                                      */
/*---------------------------------------------------------------------*/
    if (i >= vlen)
      {
#if defined(USE_EXTCURSES) || defined(HAVE_BSD_CURSES)
       ch = ' ';
       attr = screen[scrno].sl[row].normal_colour;
#else
       ch = ' ' | screen[scrno].sl[row].normal_colour;
#endif
      }
/*---------------------------------------------------------------------*/
/* Go and display the character...                                     */
/*---------------------------------------------------------------------*/
#if defined(USE_EXTCURSES) || defined(HAVE_BSD_CURSES)
    if (attr != old_attr)
      {
       wattrset(SCREEN_WINDOW_FILEAREA(scrno),attr);
       old_attr = attr;
      }
#endif
    put_char(SCREEN_WINDOW_FILEAREA(scrno),ch,ADDCHAR);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
static void set_prefix_contents(CHARTYPE scrno,LINE *curr,short start_row,LINETYPE cline)
#else
static void set_prefix_contents(scrno,curr,start_row,cline)
CHARTYPE scrno;
LINE *curr;
short start_row;
LINETYPE cline;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 CHARTYPE *ptr=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    set_prefix_contents");
#endif
 ptr = screen[scrno].sl[start_row].prefix;

 if (SCREEN_VIEW(scrno)->prefix)
   {
    if (curr->pre != NULL)                /* prefix command pending... */
/*  && !blank_field(curr->pre->ppc_command))*/    /* ... and not blank */
      {
       strcpy((DEFCHAR *)ptr,(DEFCHAR *)curr->pre->ppc_command);
       screen[scrno].sl[start_row].prefix_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_PENDING);
      }
    else                             /* no prefix command on this line */
      {
       memset(ptr,' ',SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap);
       screen[scrno].sl[start_row].prefix[SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap] = '\0';
       screen[scrno].sl[start_row].prefix_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_PREFIX);
       if (SCREEN_VIEW(scrno)->number)
         {
          if ((SCREEN_VIEW(scrno)->prefix&PREFIX_STATUS_MASK) == PREFIX_ON)
             sprintf((DEFCHAR *)ptr,"%*.*ld",
                    SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap,
                    SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap,
                    cline);
          else
             sprintf((DEFCHAR *)ptr,"%*ld",
                    SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap,
                    cline);
         }
       else
         {
          if ((SCREEN_VIEW(scrno)->prefix&PREFIX_STATUS_MASK) == PREFIX_ON)
             memset(ptr,'=',
                    SCREEN_VIEW(scrno)->prefix_width-SCREEN_VIEW(scrno)->prefix_gap);
         }
      }
   /*
    * clear the gap
    */
    strcpy((DEFCHAR*)screen[scrno].sl[start_row].gap,"");
    screen[scrno].sl[start_row].gap_colour = set_colour(SCREEN_FILE(scrno)->attr+ATTR_GAP);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
static void show_hex_line(CHARTYPE scrno,short row)
#else
static void show_hex_line(scrno,row)
CHARTYPE scrno;
short row;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 chtype ch=0;
 LENGTHTYPE vcol=0,vend=0,vlen=0;
 LENGTHTYPE length=0;
 CHARTYPE *line=NULL;
 int num=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    show_hex_line");
#endif
/*---------------------------------------------------------------------*/
/* Set up columns to display...                                        */
/*---------------------------------------------------------------------*/
 vcol = SCREEN_VIEW(scrno)->verify_col - 1;
 vend = SCREEN_VIEW(scrno)->verify_end - 1;
 vlen = SCREEN_VIEW(scrno)->verify_end - SCREEN_VIEW(scrno)->verify_start + 1;
 length = screen[scrno].sl[row].length;
 line = screen[scrno].sl[row].contents;
#if defined(USE_EXTCURSES) || defined(HAVE_BSD_CURSES)
 wattrset(SCREEN_WINDOW_FILEAREA(scrno),screen[scrno].sl[row].normal_colour);
#endif
 wmove(SCREEN_WINDOW_FILEAREA(scrno),row,0);
 for (i=0;i<screen[scrno].cols[WINDOW_FILEAREA];i++)
   {
/*---------------------------------------------------------------------*/
/* If the last character in the line has been displayed, display blank.*/
/*---------------------------------------------------------------------*/
    if (i+vcol < length)
       ch = *(line+i+vcol);
    else
       ch = ' ';
/*---------------------------------------------------------------------*/
/* If we have gone past the VERIFY END column, display a blank.        */
/*---------------------------------------------------------------------*/
    if (i >= vlen)
       ch = ' ';
/*---------------------------------------------------------------------*/
/* Calculate the HEX character to display based on which HEXSHOW line  */
/* is being displayed.                                                 */
/*---------------------------------------------------------------------*/
    if (screen[scrno].sl[row].other_start_col == 0)
       num = (int)((ch / 16) + (int)'0');
    else
       num = (int)((ch % 16) + (int)'0');
    num = (num > (int)'9') ? num + 7 : num;
#if defined(USE_EXTCURSES) || defined(HAVE_BSD_CURSES)
    ch = (chtype)num;
#else
    ch = (chtype)num | screen[scrno].sl[row].normal_colour;
#endif
/*---------------------------------------------------------------------*/
/* Go and display the character...                                     */
/*---------------------------------------------------------------------*/
    put_char(SCREEN_WINDOW_FILEAREA(scrno),ch,ADDCHAR);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}

/***********************************************************************/
#ifdef HAVE_PROTO
void touch_screen(CHARTYPE scrno)
#else
void touch_screen(scrno)
CHARTYPE scrno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:touch_screen");
#endif
 for (i=0;i<VIEW_WINDOWS;i++)
   {
    if (screen[scrno].win[i] != (WINDOW *)NULL)
       touchwin(screen[scrno].win[i]);
   }
#if 0
 touchwin(CURRENT_WINDOW_FILEAREA);
 if (CURRENT_WINDOW_PREFIX != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_PREFIX);
 if (CURRENT_WINDOW_GAP != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_GAP);
 if (CURRENT_WINDOW_IDLINE != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_IDLINE);
 if (CURRENT_WINDOW_ARROW != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_ARROW);
 if (CURRENT_WINDOW_COMMAND != (WINDOW *)NULL)
    touchwin(CURRENT_WINDOW_COMMAND);
#endif

#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void refresh_screen(CHARTYPE scrno)
#else
void refresh_screen(scrno)
CHARTYPE scrno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:refresh_screen");
#endif
/*---------------------------------------------------------------------*/
/* Turn off the cursor.                                                */
/*---------------------------------------------------------------------*/
/* MH  draw_cursor(FALSE); */
 show_heading(scrno);
 wnoutrefresh(SCREEN_WINDOW_FILEAREA(scrno));
 if (SCREEN_WINDOW_PREFIX(scrno) != (WINDOW *)NULL)
    wnoutrefresh(SCREEN_WINDOW_PREFIX(scrno));
 if (SCREEN_WINDOW_GAP(scrno) != (WINDOW *)NULL)
    wnoutrefresh(SCREEN_WINDOW_GAP(scrno));
 if (SCREEN_WINDOW_ARROW(scrno) != (WINDOW *)NULL)
   {
    touchwin(SCREEN_WINDOW_ARROW(scrno));
    wnoutrefresh(SCREEN_WINDOW_ARROW(scrno));
   }
 if (SCREEN_WINDOW_COMMAND(scrno) != (WINDOW *)NULL)
    wnoutrefresh(SCREEN_WINDOW_COMMAND(scrno));
 wnoutrefresh(SCREEN_WINDOW(scrno));
/*---------------------------------------------------------------------*/
/* Turn on the cursor.                                                 */
/*---------------------------------------------------------------------*/
/*    draw_cursor(TRUE);*/
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void redraw_screen(CHARTYPE scrno)
#else
void redraw_screen(scrno)
CHARTYPE scrno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commutil.c:redraw_screen");
#endif
 if (curses_started)
   {
/*---------------------------------------------------------------------*/
/* Turn off the cursor.                                                */
/*---------------------------------------------------------------------*/
/* MH    draw_cursor(FALSE); */
    if (SCREEN_WINDOW_COMMAND(scrno) != NULL)
      {
       wattrset(SCREEN_WINDOW_COMMAND(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_CMDLINE));
       touchwin(SCREEN_WINDOW_COMMAND(scrno));
       wnoutrefresh(SCREEN_WINDOW_COMMAND(scrno));
      }
    if (SCREEN_WINDOW_ARROW(scrno) != NULL)
      {
       wattrset(SCREEN_WINDOW_ARROW(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_ARROW));
       redraw_window(SCREEN_WINDOW_ARROW(scrno));
       wnoutrefresh(SCREEN_WINDOW_ARROW(scrno));
      }
    if (SCREEN_WINDOW_IDLINE(scrno) != NULL)
      {
       wattrset(SCREEN_WINDOW_IDLINE(scrno),set_colour(SCREEN_FILE(scrno)->attr+ATTR_IDLINE));
       redraw_window(SCREEN_WINDOW_IDLINE(scrno));
      }
    if (SCREEN_WINDOW_PREFIX(scrno) != NULL)
       touchwin(SCREEN_WINDOW_PREFIX(scrno));
    if (SCREEN_WINDOW_GAP(scrno) != NULL)
       touchwin(SCREEN_WINDOW_GAP(scrno));
    touchwin(SCREEN_WINDOW_FILEAREA(scrno));
/*---------------------------------------------------------------------*/
/* Turn on the cursor.                                                 */
/*---------------------------------------------------------------------*/
/* MH    draw_cursor(TRUE); */
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}

/***********************************************************************/
#ifdef HAVE_PROTO
bool line_in_view(CHARTYPE scrno,LINETYPE line_number)
#else
bool line_in_view(scrno,line_number)
CHARTYPE scrno;
LINETYPE line_number;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 bool result=FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    line_in_view");
#endif
 for (i=0;i<screen[scrno].rows[WINDOW_FILEAREA];i++)
   {
    if (screen[scrno].sl[i].line_number == line_number)
      {
       result = TRUE;
       break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(result);
}
/***********************************************************************/
#ifdef HAVE_PROTO
bool column_in_view(CHARTYPE scrno,LENGTHTYPE column_number)
#else
bool column_in_view(scrno,column_number)
CHARTYPE scrno;
LENGTHTYPE column_number;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 bool result=FALSE;
 LENGTHTYPE min_file_col=0,max_file_col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    column_in_view");
#endif
 min_file_col = screen[scrno].screen_view->verify_col - 1;
 max_file_col = screen[scrno].screen_view->verify_col + screen[scrno].cols[WINDOW_FILEAREA] - 2;

 if (column_number >= min_file_col
 &&  column_number <= max_file_col)            /* new column in display */
    result = TRUE;
#ifdef TRACE
 trace_return();
#endif
 return(result);
}
/***********************************************************************/
#ifdef HAVE_PROTO
LINETYPE find_next_current_line(LINETYPE num_pages,short direction)
#else
LINETYPE find_next_current_line(num_pages,direction)
LINETYPE num_pages;
short direction;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
 LINETYPE cline = CURRENT_VIEW->current_line;
 short rows=0,num_display_lines=0,num_shadow_lines=0;
 LINE *curr=NULL;
 RESERVED *curr_reserved=CURRENT_FILE->first_reserved;
 short tab_actual_row=calculate_actual_row(CURRENT_VIEW->tab_base,CURRENT_VIEW->tab_off,CURRENT_SCREEN.rows[WINDOW_FILEAREA],TRUE);
 short scale_actual_row=calculate_actual_row(CURRENT_VIEW->scale_base,CURRENT_VIEW->scale_off,CURRENT_SCREEN.rows[WINDOW_FILEAREA],TRUE);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    find_next_current_line");
#endif
/*---------------------------------------------------------------------*/
/* Determine the number of file lines displayed...                     */
/*---------------------------------------------------------------------*/
 num_display_lines = (CURRENT_SCREEN.rows[WINDOW_FILEAREA]) - 1;
 for (i=0;curr_reserved!=NULL;i++)
     curr_reserved = curr_reserved->next;
 num_display_lines -= i;
 if (CURRENT_VIEW->scale_on)
    num_display_lines--;
 if (CURRENT_VIEW->tab_on)
    num_display_lines--;
 if (CURRENT_VIEW->hexshow_on)
    num_display_lines = num_display_lines - 2;
 if (CURRENT_VIEW->scale_on
 &&  CURRENT_VIEW->tab_on
 &&  tab_actual_row == scale_actual_row)
    num_display_lines++;

 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,cline,CURRENT_FILE->number_lines);
 while(num_pages)
   {
    rows = num_display_lines;
    while(rows)
      {
/*---------------------------------------------------------------------*/
/* If the current line is above or below TOF or EOF, set all to blank. */
/*---------------------------------------------------------------------*/
       if (curr == NULL)
         {
          cline = (direction == DIRECTION_FORWARD) ? CURRENT_FILE->number_lines + 1L : 0L;
          num_pages = 1L;
          break;
         }
/*---------------------------------------------------------------------*/
/* If the current line is excluded, increment a running total.         */
/* Ignore the line if on TOF or BOF.                                   */
/*---------------------------------------------------------------------*/
       if (curr->next != NULL                           /* Bottom of file */
       &&  curr->prev != NULL)                             /* Top of file */
         {
          if (!in_scope(CURRENT_VIEW,curr))
            {
             num_shadow_lines++;
             cline += (LINETYPE)direction;
             if (direction == DIRECTION_FORWARD)
                curr = curr->next;
             else
                curr = curr->prev;
             continue;
            }
         }
/*---------------------------------------------------------------------*/
/* If we get here, we have to determine if a shadow line is to be      */
/* displayed or not.                                                   */
/*---------------------------------------------------------------------*/
       if (CURRENT_VIEW->shadow
       && num_shadow_lines > 0)
         {
          num_shadow_lines = 0;
          rows--;
          continue;
         }
       rows--;
       cline += (LINETYPE)direction;
       if (direction == DIRECTION_FORWARD)
          curr = curr->next;
       else
          curr = curr->prev;
      }
    num_pages--;
   }
 if (direction == DIRECTION_FORWARD
 &&  cline > CURRENT_FILE->number_lines+1L)
    cline = CURRENT_FILE->number_lines+1L;
 if (direction == DIRECTION_BACKWARD
 &&  cline < 0L)
    cline = 0L;
 cline = find_next_in_scope(CURRENT_VIEW,(LINE *)NULL,cline,direction);
#ifdef TRACE
 trace_return();
#endif
 return(cline);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short get_row_for_focus_line(CHARTYPE scrno,LINETYPE fl,short cr)
#else
short get_row_for_focus_line(scrno,fl,cr)
CHARTYPE scrno;
LINETYPE fl;
short cr;
#endif
/***********************************************************************/
/*---------------------------------------------------------------------*/
/* Returns the row within the main window where the focus line is      */
/* placed. If the focus line is off the screen, or out of bounds of the*/
/* current size of the file; <0 or >number_lines, this returns the     */
/* current row.                                                        */
/*---------------------------------------------------------------------*/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    get_row_for_focus_line");
#endif
 for (i=0;i<screen[scrno].rows[WINDOW_FILEAREA];i++)
   {
    if (screen[scrno].sl[i].line_number == fl)
      {
#ifdef TRACE
       trace_return();
#endif
       return(i);
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(cr);
}
/***********************************************************************/
#ifdef HAVE_PROTO
LINETYPE get_focus_line_in_view(CHARTYPE scrno,LINETYPE fl,unsigned short row)
#else
LINETYPE get_focus_line_in_view(scrno,fl,row)
CHARTYPE scrno;
LINETYPE fl;
unsigned short row;
#endif
/***********************************************************************/
/*---------------------------------------------------------------------*/
/* Returns a new focus line if the specified focus line is no longer   */
/* in view, or the same line number if that line is still in view.     */
/*---------------------------------------------------------------------*/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register unsigned short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    get_focus_line_in_view");
#endif
 for (i=row;i<screen[scrno].rows[WINDOW_FILEAREA];i++)
   {
    if (screen[scrno].sl[i].line_number != (-1L))
      {
#ifdef TRACE
       trace_return();
#endif
       return(screen[scrno].sl[i].line_number);
      }
   }
 for (i=row;i>0;i--)
   {
    if (screen[scrno].sl[i].line_number != (-1L))
      {
#ifdef TRACE
       trace_return();
#endif
       return(screen[scrno].sl[i].line_number);
      }
   }
/*---------------------------------------------------------------------*/
/* We should never get here as there would be no editable lines in view*/
/*---------------------------------------------------------------------*/
#ifdef TRACE
 trace_return();
#endif
 return(fl);
}
/***********************************************************************/
#ifdef HAVE_PROTO
LINETYPE calculate_focus_line(LINETYPE fl,LINETYPE cl)
#else
LINETYPE calculate_focus_line(fl,cl)
LINETYPE fl,cl;
#endif
/***********************************************************************/
/*---------------------------------------------------------------------*/
/* Returns the new focus line. If the focus line is still in the       */
/* window, it stays as is. If not,the focus   line becomes the current */
/* line.                                                               */
/*---------------------------------------------------------------------*/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINETYPE new_fl=(-1L);
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    calculate_focus_line");
#endif
 for (i=0;i<CURRENT_SCREEN.rows[WINDOW_FILEAREA];i++)
  {
   if (CURRENT_SCREEN.sl[i].line_number == fl
   &&  (CURRENT_SCREEN.sl[i].line_type == LINE_LINE
     || CURRENT_SCREEN.sl[i].line_type == LINE_TOF   /* MH12 */
     || CURRENT_SCREEN.sl[i].line_type == LINE_EOF)) /* MH12 */
     {
      new_fl = fl;
      break;
     }
  }
 if (new_fl == (-1L))
    new_fl = cl;
#ifdef TRACE
 trace_return();
#endif
 return(new_fl);
}
/***********************************************************************/
#ifdef HAVE_PROTO
void get_current_position(CHARTYPE scrno,LINETYPE *line,LENGTHTYPE *col)
#else
void get_current_position(scrno,line,col)
CHARTYPE scrno;
LINETYPE *line;
LENGTHTYPE *col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    get_current_position");
#endif
 getyx(SCREEN_WINDOW(scrno),y,x);
 switch(SCREEN_VIEW(scrno)->current_window)
   {
    case WINDOW_COMMAND:
         *line = SCREEN_VIEW(scrno)->current_line;
         *col = (LENGTHTYPE)(x+1);
         break;
    case WINDOW_FILEAREA:
         *line = SCREEN_VIEW(scrno)->focus_line;
         *col = (LENGTHTYPE)x + SCREEN_VIEW(scrno)->verify_col;
         break;
    case WINDOW_PREFIX:
         *line = SCREEN_VIEW(scrno)->focus_line;
         *col = (LENGTHTYPE)(x+1);
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void calculate_new_column(COLTYPE current_screen_col,LENGTHTYPE current_verify_col,
                          LENGTHTYPE new_file_col,COLTYPE *new_screen_col, LENGTHTYPE *new_verify_col)
#else
void calculate_new_column(current_screen_col,current_verify_col,new_file_col,new_screen_col,new_verify_col)
COLTYPE current_screen_col;
LENGTHTYPE current_verify_col,new_file_col;
COLTYPE *new_screen_col;
LENGTHTYPE *new_verify_col;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINETYPE x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    calculate_new_column");
#endif
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    *new_screen_col = (LENGTHTYPE)(new_file_col);
#ifdef TRACE
    trace_return();
#endif
    return;
   }
#if 0
 min_file_col = CURRENT_VIEW->verify_col - 1;
 max_file_col = CURRENT_VIEW->verify_col + CURRENT_SCREEN.cols[WINDOW_FILEAREA] - 2;

 if (new_file_col >= min_file_col
 &&  new_file_col <= max_file_col)            /* new column in display */
#else
 if (column_in_view(current_screen,new_file_col))
#endif
   {
    *new_screen_col = (LENGTHTYPE)(new_file_col - (current_verify_col - 1));
    *new_verify_col = current_verify_col;
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* To get here, we have new verify column...                           */
/*---------------------------------------------------------------------*/
 x = CURRENT_SCREEN.cols[WINDOW_FILEAREA] / 2;
 *new_verify_col = (LENGTHTYPE)max(1L,(LINETYPE)new_file_col - x + 2L);
 *new_screen_col = (LENGTHTYPE)((*new_verify_col == 1) ? new_file_col : x - 1);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
short prepare_view(CHARTYPE scrn)
#else
short prepare_view(scrn)
CHARTYPE scrn;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 int y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    prepare_view");
#endif
 SCREEN_VIEW(scrn)->current_row = calculate_actual_row(SCREEN_VIEW(scrn)->current_base,
                                  SCREEN_VIEW(scrn)->current_off,
                                  screen[scrn].rows[WINDOW_FILEAREA],TRUE);
 build_screen(scrn);
 if (!line_in_view(scrn,SCREEN_VIEW(scrn)->focus_line))
   {
    SCREEN_VIEW(scrn)->focus_line = SCREEN_VIEW(scrn)->current_line;
    pre_process_line(SCREEN_VIEW(scrn),SCREEN_VIEW(scrn)->focus_line,(LINE *)NULL);
    build_screen(scrn);
   }
 if (curses_started)
   {
    getyx(SCREEN_WINDOW_FILEAREA(scrn),y,x);
    y = get_row_for_focus_line(scrn,SCREEN_VIEW(scrn)->focus_line,
                               SCREEN_VIEW(scrn)->current_row);
 /* ensure column from WINDOW is in view */
    wmove(SCREEN_WINDOW_FILEAREA(scrn),y,x);
   }

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short advance_view(VIEW_DETAILS *next_view,short direction)
#else
short advance_view(next_view,direction)
VIEW_DETAILS *next_view;
short direction;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE number_of_files;
 extern CHARTYPE *cmd_rec;
 extern unsigned short cmd_rec_len;
 extern VIEW_DETAILS *vd_first;
 extern VIEW_DETAILS *vd_last;
 extern bool curses_started;
 extern CHARTYPE display_screens;
 extern WINDOW *statarea;
 extern WINDOW *divider;
 extern bool horizontal;
/*--------------------------- local data ------------------------------*/
 VIEW_DETAILS *save_current_view=next_view; /* point to passed view */
 CHARTYPE save_prefix=0;
 ROWTYPE save_cmd_line=0;
 short save_gap=0;
 bool save_id_line=0;
 int y=0,x=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    advance_view");
#endif
/*---------------------------------------------------------------------*/
/* If this is the only file, ignore the command...                     */
/*---------------------------------------------------------------------*/
 if (number_of_files < 2)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If we already have a current view, save some details of it...       */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW)
   {
    save_prefix=CURRENT_VIEW->prefix;
    save_gap=CURRENT_VIEW->prefix_gap;
    save_cmd_line=CURRENT_VIEW->cmd_line;
    save_id_line=CURRENT_VIEW->id_line;
   }
 memset(cmd_rec,' ',max_line_length);
 cmd_rec_len = 0;
/*---------------------------------------------------------------------*/
/* If we have not passed a "next" view determine what the next view    */
/* will be...                                                          */
/*---------------------------------------------------------------------*/
 if (!save_current_view)
   {
    post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL,TRUE);
/*---------------------------------------------------------------------*/
/* Get a temporary pointer to the "next" view in the linked list.      */
/*---------------------------------------------------------------------*/
    if (direction == DIRECTION_FORWARD)
      {
       if (CURRENT_VIEW->next == (VIEW_DETAILS *)NULL)
          save_current_view = vd_first;
       else
          save_current_view = CURRENT_VIEW->next;
      }
    else
      {
       if (CURRENT_VIEW->prev == (VIEW_DETAILS *)NULL)
          save_current_view = vd_last;
       else
          save_current_view = CURRENT_VIEW->prev;
      }
   }
/*---------------------------------------------------------------------*/
/* Save the position of the cursor for the current view before getting */
/* the contents of the new file...                                     */
/*---------------------------------------------------------------------*/
 if (curses_started)
   {
    if (CURRENT_WINDOW_COMMAND != NULL)
      {
       wmove(CURRENT_WINDOW_COMMAND,0,0);
       my_wclrtoeol(CURRENT_WINDOW_COMMAND);
      }
    getyx(CURRENT_WINDOW_FILEAREA,CURRENT_VIEW->y[WINDOW_FILEAREA],CURRENT_VIEW->x[WINDOW_FILEAREA]);
    if (CURRENT_WINDOW_PREFIX != NULL)
       getyx(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
   }
/*---------------------------------------------------------------------*/
/* If more than one screen is displayed and the file displayed in each */
/* screen is the same, remove the 'current' view from the linked list, */
/* making the next view the current one. Only do this is the "next"    */
/* view is not the view in the other screen.                           */
/*---------------------------------------------------------------------*/
 if (display_screens > 1)
   {
    if (CURRENT_SCREEN.screen_view->file_for_view == OTHER_SCREEN.screen_view->file_for_view)
      {
       if (CURRENT_VIEW->file_for_view == save_current_view->file_for_view)
         {
          if (direction == DIRECTION_FORWARD)
            {
             if (save_current_view->next == (VIEW_DETAILS *)NULL)
                save_current_view = vd_first;
             else
                save_current_view = save_current_view->next;
            }
          else
            {
             if (save_current_view->prev == (VIEW_DETAILS *)NULL)
                save_current_view = vd_last;
             else
                save_current_view = save_current_view->prev;
            }
         }
       free_a_view();
       CURRENT_VIEW = CURRENT_SCREEN.screen_view = save_current_view;
       OTHER_FILE->file_views--;
      }
    else
/*---------------------------------------------------------------------*/
/* First check if the file in the next view is the same as the file    */
/* being displayed in the other screen...                              */
/*---------------------------------------------------------------------*/
      {
       if (save_current_view->file_for_view == OTHER_FILE)
         {
          CURRENT_VIEW = CURRENT_SCREEN.screen_view = save_current_view;
          if ((rc = defaults_for_other_files(OTHER_VIEW)) != RC_OK)
            {
#ifdef TRACE
             trace_return();
#endif
             return(rc);
            }
          CURRENT_SCREEN.screen_view = CURRENT_VIEW;
          CURRENT_FILE = CURRENT_SCREEN.screen_view->file_for_view = OTHER_FILE;
          CURRENT_FILE->file_views++;
         }
       else
          CURRENT_VIEW = CURRENT_SCREEN.screen_view = save_current_view;
      }
   }
 else  /* only one screen being displayed...less hassle */
   {
/*---------------------------------------------------------------------*/
/* Make the current view the "next" one determined above.              */
/*---------------------------------------------------------------------*/
    CURRENT_VIEW = CURRENT_SCREEN.screen_view = save_current_view;
   }
/*---------------------------------------------------------------------*/
/* If the position of the prefix or command line for the new view is   */
/* different from the previous view, rebuild the windows...            */
/*---------------------------------------------------------------------*/
 if ((save_prefix&PREFIX_LOCATION_MASK) != (CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK)
 ||  save_gap != CURRENT_VIEW->prefix_gap
 ||  save_cmd_line != CURRENT_VIEW->cmd_line
 ||  save_id_line != CURRENT_VIEW->id_line)
   {
    set_screen_defaults();
    if (curses_started)
      {
       if (set_up_windows(current_screen) != RC_OK)
         {
#ifdef TRACE
         trace_return();
#endif
         return(RC_OK);
        }
      }
   }
/*---------------------------------------------------------------------*/
/* Re-calculate CURLINE for the new view in case the CURLINE is no     */
/* longer in the display area.                                         */
/*---------------------------------------------------------------------*/
 prepare_view(current_screen);

 pre_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 build_screen(current_screen); 
 display_screen(current_screen);
 if (curses_started)
   {
    if (statarea != NULL)
      {
       wattrset(statarea,set_colour(CURRENT_FILE->attr+ATTR_STATAREA));
       redraw_window(statarea);
       touchwin(statarea);
      }
    if (divider != NULL)
      {
       if (display_screens > 1
       && !horizontal)
          wattrset(divider,set_colour(CURRENT_FILE->attr+ATTR_DIVIDER));
       touchwin(divider);
       wnoutrefresh(divider);
      }
    wmove(CURRENT_WINDOW_FILEAREA,CURRENT_VIEW->y[WINDOW_FILEAREA],CURRENT_VIEW->x[WINDOW_FILEAREA]);
    if (CURRENT_WINDOW_PREFIX != NULL)
       wmove(CURRENT_WINDOW_PREFIX,CURRENT_VIEW->y[WINDOW_PREFIX],CURRENT_VIEW->x[WINDOW_PREFIX]);
    getyx(CURRENT_WINDOW,y,x);
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}

#if defined(CAN_RESIZE) || defined(OS2) || defined(WIN32)
/***********************************************************************/
#ifdef HAVE_PROTO
short THE_Resize(int rows, int cols)
#else
short THE_Resize(rows,cols)
int rows,cols;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE display_screens;
 extern short terminal_lines;
 extern short terminal_cols;
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 register int i=0;
 int rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    THE_Resize");
#endif
/*---------------------------------------------------------------------*/
/* This function is called as the result of a screen resize.           */
/*---------------------------------------------------------------------*/
 resize_term(rows,cols);     /* LINES and COLS are NOT used in XCURSES */
 terminal_lines = LINES;
 terminal_cols = COLS;
#ifdef HAVE_BSD_CURSES
 terminal_lines--;
#endif
 set_screen_defaults();
 if (curses_started)
   {
    for (i=0;i<display_screens;i++)
      {
       if ((rc = set_up_windows(i)) != RC_OK)
         {
#ifdef TRACE
          trace_return();
#endif
          return(rc);
         }
      }
   }
 create_statusline_window();
 return (RC_OK);
}
#endif

#if defined(HAVE_BROKEN_SYSVR4_CURSES)
/***********************************************************************/
#ifdef HAVE_PROTO
short force_curses_background(void)
#else
short force_curses_background()
int rows,cols;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool colour_support;
/*--------------------------- local data ------------------------------*/
 int rc=RC_OK;
 short fg=0,bg=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("show.c:    force_curses_background");
#endif
/*---------------------------------------------------------------------*/
/* This function is called to ensure that the background colour of the */
/* first line on the screen is set to that which THE requests.  Some   */
/* curses implementations, notably Solaris 2.5, fail to set the        */
/* background to black.                                                */
/*---------------------------------------------------------------------*/
 if (colour_support)
   {
    pair_content(1,&fg,&bg);
    init_pair(1,COLOR_BLACK,COLOR_WHITE);
    move(0,0);
    attrset(COLOR_PAIR(1));
    addch(' ');
    init_pair(1,fg,bg);
   }
 return (RC_OK);
}
#endif
