/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  ORBit: A CORBA v2.2 ORB
 *
 *  Copyright (C) 1998 Richard H. Porter, and Red Hat Software
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Dick Porter <dick@cymru.net>
 *          Elliot Lee <sopwith@redhat.com>
 *
 */

#include <string.h>
#include <stdio.h>
#include <assert.h>

#include "orbit.h"

/* Stupid inheritance... :) we have to do the same thing five times or so,
   might as well make a macro for it. */

#define DO_POLICY_GET_VALUE(x) PortableServer_##x##Value PortableServer_##x##__get_value(PortableServer_##x obj, CORBA_Environment *ev) { \
if(!obj) { \
ev->_major = ex_CORBA_BAD_PARAM; \
goto error_exit; \
} \
return obj->value; \
error_exit: \
CORBA_exception_set_system(ev, 0, CORBA_COMPLETED_NO); \
return 0; \
}

DO_POLICY_GET_VALUE(ThreadPolicy)
DO_POLICY_GET_VALUE(LifespanPolicy)
DO_POLICY_GET_VALUE(IdUniquenessPolicy)
DO_POLICY_GET_VALUE(IdAssignmentPolicy)
DO_POLICY_GET_VALUE(ImplicitActivationPolicy)
DO_POLICY_GET_VALUE(ServantRetentionPolicy)
DO_POLICY_GET_VALUE(RequestProcessingPolicy)

/* make emacs happy; */

PortableServer_POAManager_State
PortableServer_POAManager_get_state(PortableServer_POAManager obj,
				    CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return -1;
	}

	return obj->state;
}

/**** PortableServer_POAManager_activate
      Inputs: 'obj' - a POAManager to activate
      Outputs: '*ev' - result of the activate operation

      Side effect: Clears the 'held_requests' lists for all POA's
                   associated with the 'obj' POAManager.

      Description: Sets the POAManager state to 'ACTIVE', then
                   goes through all the POA's associated with this
		   POAManager, and makes them re-process their
		   'held_requests'
 */
void
PortableServer_POAManager_activate(PortableServer_POAManager obj,
				   CORBA_Environment *ev)
{
	GSList *todo;
	GSList *curitem;
	PortableServer_POA curpoa;

	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	if(obj->state == PortableServer_POAManager_INACTIVE) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POAManager_AdapterInactive,
				    NULL);
		return;
	}

	obj->state = PortableServer_POAManager_ACTIVE;

	for(curitem = obj->poa_collection; curitem;
	    curitem = g_slist_next(curitem)) {
		curpoa = (PortableServer_POA)curitem->data;

		todo = curpoa->held_requests;
		curpoa->held_requests = NULL;

		g_slist_foreach(todo, (GFunc)ORBit_POA_handle_request,
				curpoa);

		g_slist_free(todo);
	}
}

void
PortableServer_POAManager_hold_requests(PortableServer_POAManager obj,
					CORBA_boolean wait_for_completion,
					CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	if(obj->state == PortableServer_POAManager_INACTIVE) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POAManager_AdapterInactive,
				    NULL);
		return;
	}

	obj->state = PortableServer_POAManager_HOLDING;
	if(!wait_for_completion)
		g_warning("hold_requests not finished - don't know how to kill outstanding request fulfillments");
}

void
PortableServer_POAManager_discard_requests(PortableServer_POAManager obj,
					   CORBA_boolean wait_for_completion,
					   CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	if(obj->state == PortableServer_POAManager_INACTIVE) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POAManager_AdapterInactive,
				    NULL);
		return;
	}

	obj->state = PortableServer_POAManager_DISCARDING;
	if(!wait_for_completion)
		g_warning("discard_requests not finished - don't know how to kill outstanding request fulfillments");
}

void
PortableServer_POAManager_deactivate(PortableServer_POAManager obj,
				     CORBA_boolean etherealize_objects,
				     CORBA_boolean wait_for_completion,
				     CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	if(obj->state == PortableServer_POAManager_INACTIVE) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POAManager_AdapterInactive,
				    NULL);
		return;
	}

	obj->state = PortableServer_POAManager_INACTIVE;

	if(etherealize_objects)
		g_slist_foreach(obj->poa_collection, (GFunc)ORBit_POA_etherealize_objects, ev);
}


CORBA_boolean
PortableServer_AdapterActivator_unknown_adapter(PortableServer_AdapterActivator obj,
						PortableServer_POA parent,
						CORBA_char *name,
						CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(CORBA_FALSE);
}


/**** PortableServer_ServantActivator_incarnate
 */
PortableServer_Servant

PortableServer_ServantActivator_incarnate
(PortableServer_ServantActivator obj,
 PortableServer_ObjectId *oid,
 PortableServer_POA adapter,
 CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

void
PortableServer_ServantActivator_etherealize
(PortableServer_ServantActivator obj,
 PortableServer_ObjectId *oid, PortableServer_POA adapter,
 PortableServer_Servant serv,
 CORBA_boolean cleanup_in_progress,
 CORBA_boolean remaining_activations,
 CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return;
}

PortableServer_POA
PortableServer_POA_create_POA
               (PortableServer_POA poa,
		CORBA_char *adapter_name,
		PortableServer_POAManager a_POAManager,
		CORBA_PolicyList* policies,
		CORBA_Environment *ev)
{
	PortableServer_POA new_poa = NULL;

	if(policies)
		g_warning("non-default POA policies aren't implemented yet");

	new_poa = ORBit_POA_new(poa->orb,
				adapter_name, a_POAManager, policies, ev);
	new_poa->parent_poa = poa;

	if(ev->_major == CORBA_NO_EXCEPTION)
		ORBit_POA_add_child(poa, new_poa, ev);

	return new_poa;
}

/**** PortableServer_POA_find_POA
      Inputs: 'obj' - a POA
              'activate_it' - whether to activate unknown POA's

      Outputs: 'child_poa'

      Description: Finds (and optionally activates) a child POA of 'obj'
                   with the specified names.

      TODO: Activate non-existent adapters if asked.

 */
PortableServer_POA
PortableServer_POA_find_POA(PortableServer_POA obj,
			    CORBA_char *adapter_name,
			    CORBA_boolean activate_it,
			    CORBA_Environment *ev)
{
	GSList *curitem;
	PortableServer_POA child_poa;

	for(curitem = obj->child_POAs; curitem;
	    curitem = g_slist_next(curitem)) {
		child_poa = (PortableServer_POA)curitem->data;
		if(!strcmp(child_poa->the_name, adapter_name))
			return child_poa;
	}

	if(activate_it)
		g_warning("Don't yet know how to activate POA named \"%s\"",
			  adapter_name);

	CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
			    ex_PortableServer_POA_AdapterNonExistent,
			    NULL);	

	return NULL;
}

/**** PortableServer_POA_destroy
      Inputs: 'obj' - the POA to be destroyed
              'etherealize_objects' - flag indicating whether any servant
	                              manager should be asked to etherealize
				      objects in the active object map
	      'wait_for_completion' - flag indicating whether to wait for
	                              requests currently being handled
 */
void
PortableServer_POA_destroy(PortableServer_POA obj,
			   CORBA_boolean etherealize_objects,
			   CORBA_boolean wait_for_completion,
			   CORBA_Environment *ev)
{
	if(etherealize_objects || !wait_for_completion)
		g_warning("PortableServer_POA_destroy not yet fully implemented; ignoring flags");

	if(ORBIT_ROOT_OBJECT(obj)->refs > 1)
		g_warning("POA has multiple refs [%d]",
			  ORBIT_ROOT_OBJECT(obj)->refs);

	CORBA_Object_release((CORBA_Object)obj, ev);
}

PortableServer_ThreadPolicy PortableServer_POA_create_thread_policy(PortableServer_POA obj, PortableServer_ThreadPolicyValue value, CORBA_Environment *ev)
{
	PortableServer_ThreadPolicy retval;

	retval = g_new(struct PortableServer_ThreadPolicy_type, 1);
	ORBit_policy_object_init((CORBA_Policy)retval,
				 PortableServer_THREAD_POLICY_ID, ev);

	retval->value = value;

	return (PortableServer_ThreadPolicy)CORBA_Object_duplicate((CORBA_Object)retval, ev);
}

PortableServer_LifespanPolicy PortableServer_POA_create_lifespan_policy(PortableServer_POA obj, PortableServer_LifespanPolicyValue value, CORBA_Environment *ev)
{
	PortableServer_LifespanPolicy retval;

	retval = g_new(struct PortableServer_LifespanPolicy_type, 1);
	ORBit_policy_object_init((CORBA_Policy)retval,
				 PortableServer_LIFESPAN_POLICY_ID, ev);

	retval->value = value;

	return (PortableServer_LifespanPolicy)CORBA_Object_duplicate((CORBA_Object)retval, ev);
}

PortableServer_IdUniquenessPolicy PortableServer_POA_create_id_uniqueness_policy(PortableServer_POA obj, PortableServer_IdUniquenessPolicyValue value, CORBA_Environment *ev)
{
	PortableServer_IdUniquenessPolicy retval;

	retval = g_new(struct PortableServer_IdUniquenessPolicy_type, 1);
	ORBit_policy_object_init((CORBA_Policy)retval,
				 PortableServer_ID_UNIQUENESS_POLICY_ID,
				 ev);

	retval->value = value;

	return (PortableServer_IdUniquenessPolicy)CORBA_Object_duplicate((CORBA_Object)retval, ev);
}

PortableServer_IdAssignmentPolicy PortableServer_POA_create_id_assignment_policy(PortableServer_POA obj, PortableServer_IdAssignmentPolicyValue value, CORBA_Environment *ev)
{
	PortableServer_IdAssignmentPolicy retval;

	retval = g_new(struct PortableServer_IdAssignmentPolicy_type, 1);
	ORBit_policy_object_init((CORBA_Policy)retval,
				 PortableServer_ID_ASSIGNMENT_POLICY_ID, ev);

	retval->value = value;

	return (PortableServer_IdAssignmentPolicy)CORBA_Object_duplicate((CORBA_Object)retval, ev);
}

PortableServer_ImplicitActivationPolicy PortableServer_POA_create_implicit_activation_policy(PortableServer_POA obj, PortableServer_ImplicitActivationPolicyValue value, CORBA_Environment *ev)
{
	PortableServer_ImplicitActivationPolicy retval;

	retval = g_new(struct PortableServer_ImplicitActivationPolicy_type, 1);
	ORBit_policy_object_init((CORBA_Policy)retval,
				 PortableServer_IMPLICIT_ACTIVATION_POLICY_ID, ev);

	retval->value = value;

	return (PortableServer_ImplicitActivationPolicy)CORBA_Object_duplicate((CORBA_Object)retval, ev);
}

PortableServer_ServantRetentionPolicy PortableServer_POA_create_servant_retention_policy(PortableServer_POA obj, PortableServer_ServantRetentionPolicyValue value, CORBA_Environment *ev)
{
	PortableServer_ServantRetentionPolicy retval;

	retval = g_new(struct PortableServer_ServantRetentionPolicy_type, 1);
	ORBit_policy_object_init((CORBA_Policy)retval,
				 PortableServer_SERVANT_RETENTION_POLICY_ID, ev);

	retval->value = value;

	return (PortableServer_ServantRetentionPolicy)CORBA_Object_duplicate((CORBA_Object)retval, ev);
}

PortableServer_RequestProcessingPolicy PortableServer_POA_create_request_processing_policy(PortableServer_POA obj, PortableServer_RequestProcessingPolicyValue value, CORBA_Environment *ev)
{
	PortableServer_RequestProcessingPolicy retval;

	retval = g_new(struct PortableServer_RequestProcessingPolicy_type, 1);
	ORBit_policy_object_init((CORBA_Policy)retval,
				 PortableServer_REQUEST_PROCESSING_POLICY_ID, ev);

	retval->value = value;

	return (PortableServer_RequestProcessingPolicy)CORBA_Object_duplicate((CORBA_Object)retval, ev);
}

CORBA_char *PortableServer_POA__get_the_name(PortableServer_POA obj, CORBA_Environment *ev)
{
	g_assert(obj);
	g_assert(obj->the_name);
	return obj->the_name;
}

PortableServer_POA
PortableServer_POA__get_the_parent(PortableServer_POA obj,
				   CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return NULL;
	}

	return obj->parent_poa;
}

PortableServer_POAManager
PortableServer_POA__get_the_POAManager(PortableServer_POA obj,
				       CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return NULL;
	}

	return obj->the_POAManager;
}

PortableServer_AdapterActivator PortableServer_POA__get_the_activator(PortableServer_POA obj, CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return NULL;
	}

	return obj->the_activator;
}

void PortableServer_POA__set_the_activator(PortableServer_POA obj, PortableServer_AdapterActivator the_activator, CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	obj->the_activator = the_activator;
}

PortableServer_ServantManager PortableServer_POA_get_servant_manager(PortableServer_POA obj, CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev, ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return NULL;
	}

	if(obj->request_processing != PortableServer_USE_SERVANT_MANAGER) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
		return NULL;
	}

	return obj->servant_manager;
}

void PortableServer_POA_set_servant_manager(PortableServer_POA obj, PortableServer_ServantManager imgr, CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev, ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	if(obj->request_processing != PortableServer_USE_SERVANT_MANAGER) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
		return;
	}

	obj->servant_manager = imgr;
}

PortableServer_Servant PortableServer_POA_get_servant(PortableServer_POA obj, CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev, ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return NULL;
	}

	if(obj->request_processing != PortableServer_USE_DEFAULT_SERVANT) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
		return NULL;
	}

	return obj->default_servant;
}

void PortableServer_POA_set_servant(PortableServer_POA obj, PortableServer_Servant p_servant, CORBA_Environment *ev)
{
	if(!obj) {
		CORBA_exception_set_system(ev, ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	if(obj->request_processing != PortableServer_USE_DEFAULT_SERVANT) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
		return;
	}

	obj->default_servant = p_servant;
}

CORBA_ORB
get_orb_for_poa(PortableServer_POA poa)
{
	if(poa->orb)
		return poa->orb;
	if(poa->parent_poa)
		return get_orb_for_poa(poa->parent_poa);

	return CORBA_OBJECT_NIL;
}

PortableServer_ObjectId *
PortableServer_POA_activate_object(PortableServer_POA obj,
				   PortableServer_Servant p_servant,
				   CORBA_Environment *ev)
{
	PortableServer_ServantBase *servant;
	PortableServer_ObjectId *new_objid;
	ORBit_POAObject *new_obj;

	servant = p_servant;

	if(obj->servant_retention != PortableServer_RETAIN
	   || obj->id_assignment != PortableServer_SYSTEM_ID) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
		return NULL;
	}

	new_obj = g_new(ORBit_POAObject, 1);
	new_obj->object_id = (PortableServer_ObjectId*)CORBA_sequence_octet__alloc();

	new_objid =
		ORBit_POA_allocate_oid(obj,
				      ORBIT_OBJECT_KEY(servant->_private)->class_info->class_name);

	new_obj->object_id->_buffer = CORBA_octet_allocbuf(new_objid->_length);
	new_obj->object_id->_length = new_objid->_length;
	memcpy(new_obj->object_id->_buffer, new_objid->_buffer,
	       new_objid->_length);
	CORBA_sequence_set_release(new_obj->object_id, CORBA_TRUE);

	new_obj->servant = p_servant;
	ORBIT_OBJECT_KEY(servant->_private)->object = new_obj;
	new_obj->orb = get_orb_for_poa(obj);

	g_hash_table_insert(obj->active_object_map,
			    new_obj->object_id,
			    new_obj);

	return new_objid;
}

void
PortableServer_POA_activate_object_with_id(PortableServer_POA obj,
					   PortableServer_ObjectId *id,
					   PortableServer_Servant p_servant,
					   CORBA_Environment *ev)
{
	PortableServer_ServantBase *servant = p_servant;
	ORBit_POAObject *newobj;

	if(!obj || !id || !p_servant
	   || id->_buffer[id->_length - 1]
	   || strchr(id->_buffer, '/')) {
		CORBA_exception_set_system(ev, ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	newobj = g_hash_table_lookup(obj->active_object_map,
				     id);

	if(!newobj) {
		newobj = g_new(ORBit_POAObject, 1);
		newobj->object_id = (PortableServer_ObjectId *)CORBA_sequence_octet__alloc();
		newobj->object_id->_length = id->_length;
		newobj->object_id->_buffer = CORBA_octet_allocbuf(id->_length);
		newobj->object_id->_release = CORBA_TRUE;
		memcpy(newobj->object_id->_buffer, id->_buffer, id->_length);
		newobj->poa = obj;
		newobj->orb = get_orb_for_poa(obj);
	}

	newobj->servant = p_servant;

	g_hash_table_insert(obj->active_object_map,
			    newobj->object_id,
			    newobj);

	ORBIT_OBJECT_KEY(servant->_private)->object = newobj;
}

void
PortableServer_POA_deactivate_object(PortableServer_POA obj,
				     PortableServer_ObjectId *oid,
				     CORBA_Environment *ev)
{
	ORBit_POAObject *oldobj;

	if(!obj || !oid) {
		CORBA_exception_set_system(ev, ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return;
	}

	oldobj = g_hash_table_lookup(obj->active_object_map,
				     oid);

	if(!oldobj) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_ObjectNotActive,
				    NULL);
		return;
	}

	g_hash_table_remove(obj->active_object_map, oid);

	if(obj->request_processing == PortableServer_USE_SERVANT_MANAGER) {
		POA_PortableServer_ServantActivator__epv *epv;
		POA_PortableServer_ServantActivator *sm;

		sm = (POA_PortableServer_ServantActivator *)obj->servant_manager;
		epv = sm->vepv->PortableServer_ServantActivator_epv;
		epv->etherealize(sm, oldobj->object_id, obj,
				 oldobj->servant,
				 CORBA_FALSE,
				 CORBA_FALSE,
				 ev);
	}

	CORBA_free(oldobj->object_id);

	g_free(oldobj);
}

CORBA_Object
PortableServer_POA_create_reference(PortableServer_POA obj,
				    CORBA_RepositoryId intf,
				    CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

CORBA_Object PortableServer_POA_create_reference_with_id(PortableServer_POA obj, PortableServer_ObjectId *oid, CORBA_RepositoryId intf, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

PortableServer_ObjectId *PortableServer_POA_servant_to_id(PortableServer_POA obj, PortableServer_Servant p_servant, CORBA_Environment *ev)
{
	PortableServer_ServantBase *serv = p_servant;
	g_return_val_if_fail(p_servant != NULL, NULL);
	return ORBIT_OBJECT_KEY(serv->_private)->object->object_id;
}

CORBA_Object PortableServer_POA_servant_to_reference(PortableServer_POA obj, PortableServer_Servant p_servant, CORBA_Environment *ev)
{
	CORBA_Object retval;

	retval = PortableServer_POA_id_to_reference(obj,
						    ORBIT_OBJECT_KEY(((PortableServer_ServantBase*)p_servant)->_private)->object->object_id, ev);

	return retval;
}

PortableServer_Servant PortableServer_POA_reference_to_servant(PortableServer_POA obj, CORBA_Object reference, CORBA_Environment *ev)
{
	ORBit_POAObject *objinfo;

	g_assert(!"NYI");

	if(obj->request_processing != PortableServer_USE_DEFAULT_SERVANT
	   && obj->servant_retention != PortableServer_RETAIN) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
	}

	objinfo = g_hash_table_lookup(obj->active_object_map,
				      strrchr(reference->objinfo.iopinfo->object_key._buffer, '/') + 1);

	if(!objinfo) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_ObjectNotActive,
				    NULL);
	}

	return objinfo->servant;
}

PortableServer_ObjectId *PortableServer_POA_reference_to_id(PortableServer_POA obj, CORBA_Object reference, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

PortableServer_Servant PortableServer_POA_id_to_servant(PortableServer_POA obj, PortableServer_ObjectId *oid, CORBA_Environment *ev)
{
	ORBit_POAObject *objinfo;

	if(obj->servant_retention != PortableServer_RETAIN) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
		return NULL;
	}

	objinfo = g_hash_table_lookup(obj->active_object_map,
				      oid);

	if(!objinfo) {
		CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
				    ex_PortableServer_POA_WrongPolicy,
				    NULL);
		return NULL;
	}

	return objinfo->servant;
}

CORBA_Object PortableServer_POA_id_to_reference(PortableServer_POA obj,
						PortableServer_ObjectId *oid,
						CORBA_Environment *ev)
{
	GHashTable *objinfo;
	CORBA_Object retval;
	CORBA_ORB orb;
	ORBit_POAObject *pobj;
	TAG_INTERNET_IOP_info iopinfo = {1,0,NULL,0,{0,0,NULL,0}};
	TAG_ORBIT_SPECIFIC_info orbitinfo = { "", 0 };
	ORBit_ObjectKey *objkey = NULL;

	objinfo = g_hash_table_new(g_str_hash, g_str_equal);

	orb = obj->the_POAManager->orb;

	g_assert(!oid->_buffer[oid->_length - 1]);

	pobj = g_hash_table_lookup(obj->active_object_map, oid);

	if(pobj && pobj->servant) {
		objkey = ORBIT_OBJECT_KEY(((PortableServer_ServantBase *)pobj->servant)->_private);
		g_hash_table_insert(objinfo, "type_id",
				    (gpointer)objkey->class_info->class_name);

	} else {
		g_hash_table_insert(objinfo, "type_id", "Null");
	}
	ORBit_POA_find_object_key_for_oid(obj, oid, &iopinfo.object_key);
	if(orb->cnx.ipv4) {
		iopinfo.host = IIOP_CONNECTION(orb->cnx.ipv4)->u.ipv4.hostname;
		iopinfo.port = ntohs(IIOP_CONNECTION(orb->cnx.ipv4)->u.ipv4.location.sin_port);
	}
	g_hash_table_insert(objinfo, "IOP_TAG_INTERNET_IOP", &iopinfo);

	if(orb->cnx.ipv6 || orb->cnx.usock) {
#ifdef HAVE_IPV6
		if(orb->cnx.ipv6) {
			orbitinfo.ipv6_port =
				ntohs(IIOP_CONNECTION(orb->cnx.ipv6)->u.ipv6.location.sin_port);
		}
#endif
		if(orb->cnx.usock) {
			orbitinfo.unix_sock_path =
				IIOP_CONNECTION(orb->cnx.usock)->u.usock.sun_path;
		}
		g_hash_table_insert(objinfo, "IOP_TAG_ORBIT_SPECIFIC", &orbitinfo);
	}

	retval = ORBit_create_object_with_info(objinfo, orb, ev);

	g_hash_table_destroy(objinfo);
	CORBA_free(iopinfo.object_key._buffer);

	if(retval != CORBA_OBJECT_NIL
	   && ev->_major == CORBA_NO_EXCEPTION
	   && objkey) {
		retval->vepv = g_new0(gpointer, ORBit_class_assignment_counter + 1);
		objkey->class_info->init_local_objref(retval, pobj->servant);
		retval->servant = pobj->servant;
	} else
		retval->vepv = retval->servant = NULL;

	return retval;
}


PortableServer_POA PortableServer_Current_get_POA(PortableServer_Current obj, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

PortableServer_ObjectId *PortableServer_Current_get_object_id(PortableServer_Current obj, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}


CORBA_char *PortableServer_ObjectId_to_string(PortableServer_ObjectId *id, CORBA_Environment *env)
{
	return CORBA_string_dup(id->_buffer);
}

CORBA_wchar *PortableServer_ObjectId_to_wstring(PortableServer_ObjectId *id, CORBA_Environment *env)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

PortableServer_ObjectId *PortableServer_string_to_ObjectId(CORBA_char *str, CORBA_Environment *env)
{
	PortableServer_ObjectId *retval;

	retval = (PortableServer_ObjectId *)CORBA_sequence_octet__alloc();

	retval->_length = strlen(str) + 1;
	retval->_buffer = CORBA_octet_allocbuf(retval->_length);

	memcpy(retval->_buffer, str, retval->_length);

	return retval;
}

PortableServer_ObjectId *PortableServer_wstring_to_ObjectId(CORBA_wchar *str, CORBA_Environment *env)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}


PortableServer_POA PortableServer_ServantBase__default_POA(PortableServer_Servant servant, CORBA_Environment *ev)
{
	g_return_val_if_fail(servant, NULL);

	return ORBIT_OBJECT_KEY(((PortableServer_ServantBase *)servant)->_private)->object->poa;
}

void PortableServer_ServantLocator_preinvoke(PortableServer_ObjectId *oid, PortableServer_POA adapter, CORBA_Identifier op_name, PortableServer_ServantLocator_Cookie *cookie)
{
	g_assert(!"Not yet implemented");
	return;
}

void PortableServer_ServantLocator_postinvoke(PortableServer_ObjectId *oid, PortableServer_POA adapter, CORBA_Identifier op_name, PortableServer_ServantLocator_Cookie cookie, PortableServer_Servant servant)
{
	g_assert(!"Not yet implemented");
	return;
}

void PortableServer_ServantBase__init(PortableServer_Servant servant,
				      CORBA_Environment *ev)
{
	PortableServer_ServantBase *serv = servant;

	if(!serv->_private) /* If not already initialized, create the place to
			       stick our info */
		serv->_private = g_new0(ORBit_ObjectKey, 1);
}

void PortableServer_ServantBase__fini(PortableServer_Servant servant,
				      CORBA_Environment *ev)
{
	PortableServer_ServantBase *serv = servant;

	g_free(serv->_private);
	serv->_private = NULL;
}


/************************ ServerRequest stuff ********************/

CORBA_Identifier CORBA_ServerRequest_operation(CORBA_ServerRequest req, CORBA_Environment *env)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

CORBA_Context *CORBA_ServerRequest_ctx(CORBA_ServerRequest req, CORBA_Environment *env)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

void CORBA_ServerRequest_arguments(CORBA_ServerRequest req, CORBA_NVList *parameters, CORBA_Environment *env)
{
	g_assert(!"Not yet implemented");
	return;
}

void CORBA_ServerRequest_set_result(CORBA_ServerRequest req, CORBA_any *value, CORBA_Environment *env)
{
	g_assert(!"Not yet implemented");
	return;
}

void CORBA_ServerRequest_set_exception(CORBA_ServerRequest req, CORBA_exception_type major, CORBA_any *value, CORBA_Environment *env)
{
	g_assert(!"Not yet implemented");
	return;
}

void
POA_PortableServer_ServantActivator__init(PortableServer_Servant servant,
					  CORBA_Environment * ev)
{
   static const PortableServer_ClassInfo class_info =
   {NULL,
    "IDL:omg.org/PortableServer/ServantActivator:1.0",
    NULL };

   PortableServer_ServantBase__init(((PortableServer_ServantBase *) servant), ev);

   ORBIT_OBJECT_KEY(((PortableServer_ServantBase *)servant)->_private)->class_info = (gpointer)&class_info;
}

void
POA_PortableServer_ServantActivator__fini(PortableServer_Servant servant,
					  CORBA_Environment * ev)
{
   PortableServer_ServantBase__fini(servant, ev);
}

void 
POA_PortableServer_ServantLocator__init(PortableServer_Servant servant,
					CORBA_Environment * ev)
{
   static const PortableServer_ClassInfo class_info =
   {NULL,
    "IDL:omg.org/PortableServer/ServantLocator:1.0",
    NULL };

   PortableServer_ServantBase__init(((PortableServer_ServantBase *) servant), ev);

   ORBIT_OBJECT_KEY(((PortableServer_ServantBase *)servant)->_private)->class_info = (gpointer)&class_info;
}

void 
POA_PortableServer_ServantLocator__fini(PortableServer_Servant servant,
					CORBA_Environment * ev)
{
   PortableServer_ServantBase__fini(servant, ev);
}
