/*
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	RBD/GDB
 *
 *	$Log:	recon.c,v $
 * Revision 6.1  96/11/23  19:37:34  nevin
 * Ohio Release
 * 
 * Revision 6.0  96/02/29  14:21:45  gdburns
 * Ohio Release
 * 
 * Revision 5.2.1.2  96/02/28  18:42:24  gdburns
 * major overhaul
 * 
 * Revision 5.2.1.1  94/10/18  02:17:20  raja
 * Upgrade to new network description.
 * 
 * Revision 5.2  94/08/22  14:18:54  gdburns
 * Ohio Release
 * 
 * Revision 5.1.1.1  94/08/12  14:50:37  raja
 * Fixed do_recon() return error.
 * 
 * Revision 5.1  94/05/18  13:01:43  gdburns
 * Ohio Release
 * 
 * Revision 2.3  94/04/22  12:50:38  gdburns
 * Ohio Release
 * 
 *	Function:	- LAM reconnaissance tool
 *			- tests if Trollius is bootable on the LAN
 *			- uses "tkill -N"
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <lamnet.h>
#include <net.h>
#include <terror.h>

/*
 * static variables
 */
static int		fl_debug;		/* debug mode */
static int		fl_verbose;		/* verbose mode */

static char		*usage = "recon [-adhv] [<bhost>]";

/* 
 * external functions
 */
extern char		*findpath();
extern int		bhostparse();
extern int		inetexec();
extern int		_lam_few();

/* 
 * static functions
 */
static void		help();
static char		*findbhost();

/*
 *	main
 */
int
main(argc, argv)  

int			argc;
char			*argv[];

{
	FILE		*fp;		/* boot schema file pointer */
	char		*bhost;		/* boot schema filename */
	char		**cmdv;		/* test command argv */
	int		cmdn;		/* test command argc */
	int		i;
	int		idup;		/* duplicate node index */
	int		iorigin;	/* index of origin node */
	int		r;		/* child return error code */
	int		global_ret;	/* global return code */
	int		nlamnet;	/* lamnet size */
	struct lamnode	*lamnet;	/* node description array */
/*
 * Parse the command line.
 */
	validopts("adhv");

	if (do_args(&argc, argv)) {
		fprintf(stderr, "usage: %s\n", usage);
		exit(EUSAGE);
	}

	if ((errno = (argc <= 2) ? 0 : EUSAGE)) {
		fprintf(stderr, "usage: %s\n", usage);
		exit(errno);
	}
 
	if (opt_taken('h')) {
		help();
		exit(0);
	}

	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;
/* 
 * Locate the system file.
 */
	bhost = findbhost(argc, argv);
/*
 * Open the system file.
 */
	fp = fopen(bhost, "r");

	if (fp == 0) {
		perror("recon (fopen)");
		exit(errno);
	}
/*
 * Parse the system file.
 */
	if (bhostparse(fp, &lamnet, &nlamnet)) {
		fprintf(stderr, "recon: cannot parse host file: ");
		terror("");
		exit(errno);
	}
/*
 * Close the system file.
 */
	if (fclose(fp)) {
		perror("recon (fclose)");
		exit(errno);
	}
/*
 * Locate the host nodes.
 */
	if (lamnet_findhosts(lamnet, nlamnet)) {

		if (errno == EBADHOST) {
			terror("recon");
		} else {
			terror("recon (lamnet_findhosts)");
		}

		exit(errno);
	}
/*
 * Make rudimentary check for duplicate hosts.
 */
	if (lamnet_dups(lamnet, nlamnet, &idup)) {
		fprintf(stderr, "recon: %s: duplicated host",
				lamnet[idup].lnd_hname);
		exit(EINVAL);
	}
/*
 * Find the origin node.
 */
	iorigin = lamnet_findorig(lamnet, nlamnet);

	if (iorigin < 0) {
		fprintf(stderr, "recon: local host not present");
		exit(EINVAL);
	}

	lamnet[iorigin].lnd_type |= NT_ORIGIN | NT_ME;

	DBUG("recon: found %d host node(s)\n", nlamnet);
	DBUG("recon: origin node is %d\n", iorigin);
/*
 * Build the reconnaissance command.
 * We use "tkill -N" as a test.
 */
	cmdn = 0;
	cmdv = 0;
	argvadd(&cmdn, &cmdv, DEFTTKILL);
	argvadd(&cmdn, &cmdv, "-N");
/*
 * Loop over all host nodes.
 */
	global_ret = 0;

	for (i = 0; i < nlamnet; ++i) {
		VERBOSE("recon: testing n%d (%s)\n",
				lamnet[i].lnd_nodeid, lamnet[i].lnd_hname);

		if (lamnet[i].lnd_type & NT_ORIGIN) {
			r = _lam_few(cmdv);

			if (r) {
				errno = r;
			}
		} else {
			r = inetexec(lamnet[i].lnd_hname,
					lamnet[i].lnd_uname, cmdv);
		}

		if (r) {
			global_ret = errno;
			fprintf(stderr, "recon: \"%s\" cannot be booted.\n",
					lamnet[i].lnd_hname);

			if (errno != EUNKNOWN) {
				terror("recon");
			}

			if (! opt_taken('a')) {
				return(global_ret);
			}
		}
	}
	
	return(global_ret);
}

/*
 *	findbhost
 *
 *	Function:	- locates boot schema file
 *	Accepts:	- argc of remaining command line
 *			- argv of remaining command line
 *	Returns:	- full pathname of boot schema file
 */
static char *
findbhost(cmdc, cmdv)

int			cmdc;
char			**cmdv;

{
	char		*bhost;
	char		*full;		/* temporary for full pathname */
	char		**pathv;
	int		pathc;
/*
 * Set the directories for the boot schema file search.
 */
	pathc = 0;
	pathv = 0;
	argvadd(&pathc, &pathv, "");
	argvadd(&pathc, &pathv, "$TROLLIUSHOME/boot");
	argvadd(&pathc, &pathv, "$LAMHOME/boot");
	argvadd(&pathc, &pathv, DEFPBHOST);
/*
 * Set the boot schema file names.
 */
	if (cmdc == 2) {
		bhost = cmdv[1];
	} else if ((bhost = getenv("LAMBHOST"))) {
	} else if ((bhost = getenv("TROLLIUSBHOST"))) {
	} else {
		bhost = DEFFBHOST;
	}
/*
 * Search the directories.
 */
	full = findpath(bhost, pathv, R_OK);

	if (full == 0) {
		fprintf(stderr, "recon: cannot locate \"%s\"\n", bhost);
		exit(errno);
	}

	DBUG("recon: boot schema file: %s\n", full);
	return(full);
}

/*
 *	help
 *
 *	Function:	- prints helpful information on this command
 */
static void 
help()

{
	printf("\nSynopsis:\trecon [options] [<bhost>]\n");
	printf("\nDescription:\tCheck if LAM can be booted.\n");
	printf("\nOptions:\t-h\t\tPrint this message.\n");
	printf("\t\t-a\t\tReport all host errors.\n");
	printf("\t\t-d\t\tTurn on debugging.\n");
	printf("\t\t-v\t\tBe verbose.\n");
	printf("\t\t<bhost>\t\tUse <bhost> as boot schema.\n");
}
