/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: lamboot.c,v 6.1 96/11/23 19:37:24 nevin Rel $
 *
 *	Function:	- fully-connected LAM booting tool
 *			- boots Trollius on a network of UNIX hosts
 *			- hosts specified using host file syntax
 *			- uses hboot
 */

#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <lamnet.h>
#include <net.h>
#include <sfh.h>
#include <terror.h>
#include <patchlevel.h>

/*
 * static variables
 */
static char		*bhost;		/* boot schema file */
static int		fl_debug;	/* debug mode */
static int		fl_verbose;	/* verbose mode */
static int		nboot;		/* # nodes booted */
static int		nrun;		/* # nodes running */

static char		*usage = "lamboot [-dhvxHV] [<bhost>]";

/*
 * external functions
 */
extern int		_lam_few();
extern int		bhostparse();
extern int		lambootagent();

/* 
 * static functions
 */
static void		bail();
static void		help();
static void		wipe();
static char		*findbhost();


int
main(argc, argv)  

int			argc;
char			*argv[];

{
	FILE		*fp;		/* boot schema file pointer */
	int		idup;		/* index of duplicate node */
	int		iorigin;	/* index of origin node */
	int		nlamnet;	/* lamnet size */
	struct lamnode	*lamnet;	/* network description array */
/*
 * Parse the command line.
 */
	validopts("dhvxHV");

	if (do_args(&argc, argv)) {
		fprintf(stderr, "usage: %s\n", usage);
		exit(EUSAGE);
	}

	if ((errno = (argc <= 2) ? 0 : EUSAGE)) {
		fprintf(stderr, "usage: %s\n", usage);
		exit(errno);
	}
 
	if (opt_taken('h')) {
		help();
		exit(0);
	}

	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;

	if (opt_taken('V')) {
		printf("\nLAM 6.1 - Ohio Supercomputer Center\n\n");
		printf("\tPatchlevel:\t%d\n", LAMPATCHLEVEL);
		printf("\tOS:\t\t%s\n", OS);
		printf("\tCPU:\t\t%s\n", CPU);
		printf("\tRPI:\t\t%s\n", RPI);
		printf("\tSHM:\t\t%s\n\n", SHM);
		exit(0);
	} else {
		if (! opt_taken('H')) {
			printf("\nLAM 6.1 - Ohio Supercomputer Center\n\n");
		}
	}
/* 
 * Locate the system file.
 */
	bhost = findbhost(argc, argv);
/*
 * Open the system file.
 */
	fp = fopen(bhost, "r");
	if (fp == 0) lampanic("lamboot (fopen)");
/*
 * Parse the system file.
 */
	if (bhostparse(fp, &lamnet, &nlamnet)) {
		fprintf(stderr, "lamboot: cannot parse host file: ");
		lampanic("");
	}
/*
 * Close the system file.
 */
	if (fclose(fp)) lampanic("lamboot (fclose)");
/*
 * Locate the host nodes.
 */
	if (lamnet_findhosts(lamnet, nlamnet)) {

		if (errno == EBADHOST) {
			lampanic("lamboot");
		} else {
			lampanic("lamboot (lamnet_findhosts)");
		}
	}
/*
 * Make rudimentary check for duplicate hosts.
 */
	if (lamnet_dups(lamnet, nlamnet, &idup)) {
		fprintf(stderr, "lamboot: %s: duplicated host\n",
				lamnet[idup].lnd_hname);
		exit(EINVAL);
	}
/*
 * Find the origin node.
 */
	iorigin = lamnet_findorig(lamnet, nlamnet);

	if (iorigin < 0) {
		fprintf(stderr, "lamboot: local host not present\n");
		exit(EINVAL);
	}

	lamnet[iorigin].lnd_type |= NT_ORIGIN | NT_ME;

	DBUG("lamboot: found %d host node(s)\n", nlamnet);
	DBUG("lamboot: origin node is %d\n", iorigin);
/*
 * Clean up on interrupt.
 */
	if (signal(SIGINT, bail) == SIG_ERR) lampanic("lamboot (signal)");
/*
 * Boot the system.
 */
	if (lambootagent(lamnet, nlamnet, &nboot, &nrun)) {

		if (nrun < nlamnet) {
			fprintf(stderr, "lamboot: %s failed on n%d (%s)\n",
					DEFTHBOOT, lamnet[nrun].lnd_nodeid,
					lamnet[nrun].lnd_hname);

			if (errno != EUNKNOWN) {
				terror("lamboot");
			}
		} else {
			terror("lamboot");
		}

		bail();
	}

	return(0);
}

/*
 *	findbhost
 *
 *	Function:	- locates boot schema file
 *	Accepts:	- argc of remaining command line
 *			- argv of remaining command line
 *	Returns:	- full pathname of boot schema file
 */
static char *
findbhost(cmdc, cmdv)

int			cmdc;
char			**cmdv;

{
	char		*fname;
	char		*full;		/* temporary for full pathname */
	char		**pathv;
	int		pathc;
/*
 * Set the directories for the boot schema file search.
 */
	pathc = 0;
	pathv = 0;
	argvadd(&pathc, &pathv, "");
	argvadd(&pathc, &pathv, "$TROLLIUSHOME/boot");
	argvadd(&pathc, &pathv, "$LAMHOME/boot");
	argvadd(&pathc, &pathv, DEFPBHOST);
/*
 * Set the boot schema file names.
 */
	if (cmdc == 2) {
		fname = cmdv[1];
	} else if ((fname = getenv("LAMBHOST"))) {
	} else if ((fname = getenv("TROLLIUSBHOST"))) {
	} else {
		fname = DEFFBHOST;
	}
/*
 * Search the directories.
 */
	full = sfh_path_find(fname, pathv, R_OK);

	if (full == 0) {
		fprintf(stderr, "lamboot: cannot locate \"%s\"\n", fname);
		exit(errno);
	}

	DBUG("lamboot: boot schema file: %s\n", full);
	return(full);
}

/*
 *	bail
 *
 *	Function:	- cleans up and bails out
 *	Returns:	- does not return, exits with error code
 */
static void
bail()

{
	int		err_save;	/* saved error code */

	err_save = errno;
	VERBOSE("\n");
	wipe();
	exit(err_save);
}

/*
 *	wipe
 *
 *	Function:	- executes the wipe tool
 *			- kills all host Trollius sessions
 */
static void
wipe()

{
	int		cmdn;
	int		r;
	char		**cmdv;
	char		buf[16];

	if (nboot <= 0) {
		return;
	}

	cmdn = 0;
	cmdv = 0;
	argvadd(&cmdn, &cmdv, DEFTWIPE);

	if (fl_verbose) {
		argvadd(&cmdn, &cmdv, "-v");
	}

	argvadd(&cmdn, &cmdv, "-n");
	sprintf(buf, "%d", nboot);
	argvadd(&cmdn, &cmdv, buf);

	argvadd(&cmdn, &cmdv, bhost);

	VERBOSE("%s ...\n", DEFTWIPE);

	r = _lam_few(cmdv);

	if (r) {
		errno = r;
		terror("lamboot (wipe)");
		exit(errno);
	}
}

/*
 *	help
 *
 *	Function:	- prints helpful information on this command
 */
static void 
help()

{
	printf("\nSynopsis:\tlamboot [options] [<bhost>]\n");
	printf("\nDescription:\tSet up a LAM multicomputer.\n");
	printf("\nOptions:\t-h\t\tPrint this message.\n");
	printf("\t\t-d\t\tPrint debugging messages.\n");
	printf("\t\t-v\t\tBe verbose.\n");
	printf("\t\t-x\t\tRun nodes in fault tolerant mode.\n");
	printf("\t\t-H\t\tDon't print the header.\n");
	printf("\t\t-V\t\tPrint version and exit without booting.\n");
	printf("\t\t<bhost>\t\tUse <bhost> as boot schema.\n");
}
